# Develop OCaml Software

On this page, you will learn how to use
BSD Owl Scripts (BPS) to:

- compile and install simple OCaml programs;
- compile and install simple OCaml libraries;
- prepare and install documentation generated with `ocamldoc`;
- prepare and install a custom toplevel;
- generate lexers and parsers with `ocamllex` and `ocamlyacc`;
- use 3rd party libraries in your programs;
- generate objects with debugging symbols;
- generate objects with profiling information;
- use `autoconf` in your project;
- organise a complex project in several directories;
- produce GPG-signed tarballs;
- simultaneously build a project with various configurations.


# Compile and install simple OCaml programs

We assume you have a really simple program `wordcount`, your own
implementation of the UNIX `wc(1)` utility.  The source code is a
single file `wordcount.ml`.


## The first time

Create a directory to hold your files and put your source there.
Along the source, create a `Makefile` with the following content:

	PROGRAM=    wordcount
	.include "ocaml.prog.mk"


## Building

You can now `make` your program and produce a `wordcount` binary.  The
complete output of the make process looks like this:

	$ make
	make depend
	ocamldep  wordcount.ml > .depend
	make build
	ocamlc -c -o wordcount.cmo wordcount.ml
	ocamlc -o wordcount.cb wordcount.cmo
	cp wordcount.cb wordcount
	make doc

When you call `make` without argument it is the same thing as
`make all` which decomposes as `make depend` and `make build` as you
see.  You can test your program, edit it and re-`make` it.


## Installing

Once you are satisfied with the results, you can install it with `make
install`.  It will call `su` to gain root privileges and install your
program under `/usr/local`, the value of *PREFIX*

	$ make install
	===> Switching to root credentials for target (install)
	Password:
	/usr/bin/install -c -d /usr/local/bin
	install -o root -g wheel -m 555 wordcount /usr/local/bin

You can check the value of the *PREFIX* variable, or any other
variable, with `make -V` as in

	$ make -V PREFIX
	/usr/local

If you want to install your program to another location like
`${HOME}/bin` you only need to change the *PREFIX*.  You can make the
change permanent by adding a `PREFIX=${HOME}` line to your `Makefile`:

	PROGRAM=    wordcount
	PREFIX=     ${HOME}
	.include "ocaml.prog.mk"

The order of variable declarations is not important but they have to
come before the `.include` line.  It is also possible to use
`PREFIX=${HOME}` just once by adding it on the command line without
editing the `Makefile`:

	make PREFIX=${HOME} install
	/usr/bin/install -c -d /home/michael/bin
	install -o michael -g michael -m 550 wordcount /home/michael/bin

Note that since you have write access to the *PREFIX* directory, it is
not necessary to gain root privileges for this installation.


## Cleaning

Last you can remove object code from the directory with

	$ make clean
	rm -f  wordcount.cmo wordcount.cmi wordcount.cb wordcount

If you look closely, you will notice that the `.depend` file is not
removed:

	$ ls -A
	.depend      Makefile     wordcount.ml

This is on purpose, and if you also want to get rid of the `.depend`
file you can use the more powerful mantra

	$ make realclean
	rm -f  wordcount.cmo wordcount.cmi wordcount.cb wordcount
	rm -f  .depend


## Several source files

If your program consists of several files `ancillary.ml` and
`application.ml`, you need to list them in the *SRCS* variable, as in:

	PROGRAM= 	wordcount
	SRCS+=		ancillary.ml
	SRCS+=		application.ml
	.include "ocaml.prog.mk"

When you list explicitly source files in the *SRCS* variable, you are
not obliged to use the same name for your source file and your
program, so if you use the name `main.ml` instead of `wordcount.ml`
your `Makefile` would be

	PROGRAM=    wordcount
	SRCS=       main.ml
	.include "ocaml.prog.mk"

While dependencies between modules are computed with `ocamldep` so
that modules are compiled as needed, the order in which the files are
listed in *SRCS* is used by the linker.  It is thus important to list
files in an order suited to the linking phase.


## Interfaces without implementation

If your implementation files are accompanied by interface files, these
are automatically detected and handled appropriately.  It is also
possible to compile a program against a module without implementation,
by listing the implementation in the *SRCS* variable.  With the `Makefile`

	PROGRAM=    wordcount
	SRCS+=      ancillary.mli
	SRCS+=      application.ml
	.include "ocaml.prog.mk"

you can compile `moduleB.ml` using the interface defined in
`moduleA.mli` without having implemented the interface.  (Of course,
you will not be able to link `wordcount`.)


# Compile and install simple OCaml libraries

We assume that you have a really simple OCaml library `newton`
consisting of an implementation file `newton.ml` and an interface file
`newton.mli`.


## The first time

The corresponding `Makefile` is then

	LIBRARY=    newton
	SRCS+=      newton.ml
	.include "ocaml.lib.mk"

Note that it is mandatory to list all implementation files in the
*SRCS* variable: in contrast to the module `ocaml.prog.mk` used to
compile OCaml programs, the module `ocaml.lib.ml` does not try to
automotically add the obvious `newton.ml` to your library.  This is
because libraries are essentially archive files and you are expected
to list files that are to be put in the archive.


## Building

As for programs, you can `make` the library:

	$ make
	make depend
	ocamldep  newton.ml newton.mli > .depend
	make build
	ocamlc -c -o newton.cmo newton.ml
	ocamlc -a -o newton.cma newton.cmo
	make doc


## Installing

You can now test your library and if you are satisfied, decide to
install it with `make install`.  The library will be installed in
`${LIBDIR}` that is

	$ make -V LIBDIR
	/usr/local/lib/ocaml

unless you selected another *PREFIX* as described above in _Compile
and install simple OCaml programs._ Alternatively, you can define a
*PACKAGE* name and library files will go in
`${PREFIX}/lib/ocaml/site-lib/${PACKAGE}` as in the following example:

	$ make PACKAGE=mypackage install
	===> Switching to root credentials for target (install)
	Password:
	/usr/bin/install -c -d /usr/local/lib/ocaml/site-lib/mypackage
	install -o root -g wheel -m 444 newton.cma /usr/local/lib/ocaml/site-lib/mypackage
	install -o root -g wheel -m 444 newton.cmi /usr/local/lib/ocaml/site-lib/mypackage


## Cleaning

Finally, you can `make clean` or `make distclean` your working
directory.


## Ad-hoc tests

Assume that you have written a simple program `test_newton.ml` that
you use to test your library.  You can then use a command line like

	$ make && ocaml newton.cma test_newton.ml

to rebuild your library and run your test with a single command line.
However each developer action should be doable in a single
command, which allows to focus on the task _run the test_ instead of
of the steps involved in the task. Here is how you can define an
ad-hoc `test` target in the `Makefile`:

	LIBRARY= newton
	SRCS+= newton.ml

	test: newton.cma .PHONY
		ocaml newton.cma test_newton.ml

	.include "ocaml.lib.mk"

The two new lines read as “in order to perform the _test_ please
refresh `newton.cma` and call `ocaml newton.cma test_newton.ml`.  The
`.PHONY` keyword tells `make(1)` that the target _test_ will not
produce a file called `test`, but rather defines a task to perform.
You can then `make test`:

	$ make test
	ocamlc -c -o newton.cmi newton.mli
	ocamlc -c -o newton.cmo newton.ml
	ocamlc -a -o newton.cma newton.cmo
	ocaml newton.cma test_newton.ml
	06   1.00000000
	05   2.00000000
	04   1.66666667
	03   1.61904762
	02   1.61803445
	01   1.61803399
	00   1.61803399

The lines starting with numbers are the output of the test, it
computes the golden ratio using Newton's method.


## META file

If you have prepared a META file so that your library can be found by
`ocamlfind(1)` you only need to include `ocaml.meta.mk` right before
the `ocaml.lib.mk`.  If for some reason, you prefer having the `META`
file in its own directory, a sample `Makefile` would be

	LIBDIR?= ${PREFIX}/lib/ocaml/site-lib${PACKAGEDIR}
	.include "ocaml.meta.mk"


## Packed modules

You might prefer producing a packed module instead of a library. The
setup is very similar to the one you use for a library:

	PACK=       libNewton
	SRCS+=      newton.ml
	.include "ocaml.pack.mk"


# Prepare and install documentation

The generation of documentation with `ocamldoc` is supported.  We
build over the first `newton` library example and assume the file
`newton.mli` contains documentation annotations that can be used by
`ocamldoc`.  You can edit your `Makefile` as follows:

	LIBRARY=    newton
	SRCS+=      newton.ml
	USE_ODOC=	yes
	ODOC_NAME=	newtontk
	ODOC_TITLE=	Newton's method
	.include "ocaml.lib.mk"

Setting *USE_ODOC* to `yes` tells `ocaml.lib.mk` it should use
`ocamldoc` to generate on-line documentation, the *ODOC_NAME* is an
identifiant used to construct file names for the output, and the value
of *ODOC_TITLE* is used as title in the generated documentation.  This
latter variable is not mandatory and if you feel lazy, you can left it
uninitialised. You are now ready to `make` everything, which will
rebuild the library and generate the documentation:

	$ make
	make depend
	make build
	ocamlc -c -o newton.cmi newton.mli
	ocamlc -c -o newton.cmo newton.ml
	ocamlc -a -o newton.cma newton.cmo
	make doc
	ocamldoc -t "Newton's method" -dump newtontk.odoc newton.ml newton.mli
	rm -R -f newtontk_html.temp newtontk_html
	mkdir newtontk_html.temp
	ocamldoc -t "Newton's method" -html -d newtontk_html.temp newton.ml newton.mli
	mv newtontk_html.temp newtontk_html

The documentation generation step has two products, an `ocamldoc` dump
that you can use in other `ocamldoc` runs with the `-load` option and
a directory `${ODOC_NAME}_html` holding the HTML generated
documentation.  You can tune the products you want to generate by
setting appropriately the *ODOC_FORMAT* variable as described in the
`ocaml.odoc.lib` file.  There you will also see how to tune the
generated output by adding a custom CSS file or a charset other than
ISO-8859-1 in your input files.

If you want to rebuild the library without rebuilding the
documentation, you can use the `build` target instead of the `all`
target implied by an empty list of arguments to `make`, as in

	$ make build
	ocamlc -c -o newton.cmi newton.mli
	ocamlc -c -o newton.cmo newton.ml
	ocamlc -a -o newton.cma newton.cmo

Conversely `make doc` will only regenerate the documentation.


# Prepare and install a custom toplevel

Here is an example of `Makefile` that will allow you to prepare a
toplevel linked against the `unix` and `str` libraries and the
`initialise_toplevel` module:

	TOPLEVEL = toplevel
	SRCS = initialise_toplevel.ml
	LIBS = unix
	LIBS+= str
	.include "ocaml.toplevel.mk"

There is an interface to options several of `opcamlmktop` please refer
to the `ocaml.toplevel.mk` file.

**Note** We would like to provide a more capable and useful toplevel
production support. See this ticket:

  https://bitbucket.org/michipili/bsdowl/issue/7/powerful-ocamltoplevelmk


# Generate lexers and parsers

The standard tools `ocamllex(1)` and `ocamlyacc(1)` are supported by
our suite.  In order to interpret and compile lexers and parsers, you
only need to list them as sources of your program as in the following
example:

	PROGRAM=	minibasic

	SRCS =		main.ml
	SRCS+=		basic_types.ml
	SRCS+=		basic_parser.mly
	SRCS+=		basic_lexer.mll

	.include "ocaml.prog.mk"

If you wrote interface files for the generated implementtation files,
these will automatically be generated.


# Use 3rd party libraries

Linking against 3rd party libraries distributed with a `META` file
compatible with `ocamlfind(1)` is supported through the *PKGS*
variable:

	PROGRAM=	wordcount

	SRCS+=		extras.ml
	SRCS+=		wordcount.ml
	PKGS=		str

	.include "ocaml.prog.mk"

Predicates can be added with the *PREDICATES* variable, they are
passed to `ocamlfind(1)` with the `-p` flag.

Libraries which are not supporting `ocamlfind(1)` must be listed in
the *LIBS* variable _and_ their installation directory in *DIRS*. This
convention is illustrated in the next `Makefile`:

	PROGRAM=	golden_ratio

	SRCS=		main.ml

	LIBS+=		nums
	LIBS+=		newton
	LIBS+=		fibonacci

	DIRS=		${.CURDIR}/../newton
	DIRS+=		${.CURDIR}/../fibonacci

	.include "ocaml.prog.mk"

Note that the `nums` library is part of the OCaml distribution and its
installation directory does not need to be listed in the `DIRS`
variable.


# Generate objects with debugging symbols

Generation of objects containing debugging symbols is controlled by
the *WITH_DEBUG* knob.  When its value is set to `yes` all objects,
executables and libraries are compiled or linked with the `-g` flag.


# Generate objects with profiling information

Generation of objects containing debugging symbols is controlled by
the *WITH_PROFILE* knob.  When its value is set to `yes` all objects,
executables and libraries are compiled or linked with profiling
front-ends of the OCaml compiler.


# Use `autoconf` in your project

We show how to use `autoconf(1)` to let the user configure
installation paths for your program.  The `./configure` script
generated by the following `configure.ac` will produce a `Makefile.inc`
and a `standardDirectory.ml` file by replacing variables in
`Makefile.inc.in` and `standardDirectory.ml.in`.

	AC_INIT(program.ml)
	AC_OUTPUT(Makefile.inc)
	AC_OUTPUT(standardDirectory.ml)

It is better to produce an auxiliary `Makefile.inc` file with
`autoconf` instead of the primary `Makefile` so that the project tree
remains in a usable state, even if it is not configured.

There is `autoconf` macros available for OCaml-based projects, that
will identify OCaml tools, and test for the availability of libraries:

	http://forge.ocamlcore.org/projects/ocaml-autoconf/


# Organise a complex project

We show how to use BSD Owl Scripts to organise the build of a
small project consisting of two libraries and a program: a first
library `newton` provides a service to compute the golden ratio using
Newton's method, a second library `fibonacci` offers a computation
method based on Fibonacci numbers.  The `golden_ratio` program uses
these two service to compare the approximations.


## Simple aggregate and delegate pattern

First, create a master directory and three subdirectories `newton`,
`fibonacci` and `golden_ratio` holding your software components.
Along with the sources, these folder contain `Makefile`s as described
in _Compile and install simple OCaml libraries_ and _Use 3rd party
libraries_ above.  Once you are done, create the master `Makefile` in
the master directory:

	SUBDIR+=	fibonacci
	SUBDIR+=	newton
	SUBDIR+=	golden_ratio

	.include "bps.subdir.mk"

The special `bps.subdir.mk` will delegate most targets to the
subdirectories listed in *SUBDIR*.  Thus you can `make` to produce
libraries and program, `make install` and `make clean` also work
similarly.


## Projects

Besides delegating the common targets `all`, `install` and `clean`
(for the most importants), there is much more project related tasks
that can be handled my `Makefiles`.  We provide a directives file
`bps.project.mk` which is much more powerful than `bps.subdir.mk` and
can be used to support your project development by automating other
tasks.  You can turn the previous `bps.subdir.mk`-based `Makefile`
into a project just by adding declaration of a `NAME`, a `VERSION` and
an `AUTHOR`—and of course editing the `.include` line:

	PROJECT=	golden_ratio
	VERSION=	1.0
	AUTHOR=		Michael Grünewald

	SUBDIR+=	fibonacci
	SUBDIR+=	newton
	SUBDIR+=	golden_ratio

	.include "bps.project.mk"


## Preparing project tarballs

Issuing `make dist` will `distclean` the master directory and prepare
source tarballs in *PROJECTDISTDIR* for several archive formats.  The
name of the archive is deduced from *PROJECT* and *VERSION*.

Some files present in the source directory can be filtered out from
the tarball distribution by adding them to the *PROJECTDISTEXCLUDE*
variable.

Additional files that should be copied in the distribution
directory—as a `ChangeLog` or `README`—can be specified in
*PROJECTDIST*.


## Signing projects tarballs

The `prepublish` target will try to sign tarballs with GPG, using the
key identified by *AUTHOR*. The signature file are stored in
*PROJECTDISTDIR* along the distfiles produced in the last step.
Additional files that should be signed can be specified in
*PROJECTDISTSIGN*.


## Entering developer subshell

The developer subshell is a normal interactive shell where some
environement variables are set up to ease development.  They are most
useful if your project defines a project library, which is located in
the `Library` directory inside the project master directory, or in any
other location specified by *PROJECTLIBRARY*.

If the project library contains a `Make` or `Mk` directory, or if the
project master directory contains a `Mk` subdirectory, these are added
to the lookup path of `make`.  Thus, it is very easy to
[define a library](ProjectLibrary) of `Makefile` for the various products of
your project.

If the project library contains an `Ancillary` directory, it added to
the path, so that the `${PROJECTLIBRARY}/Ancillary` is a convenient
place to [store scripts](ProjectLibrary) used in your project.


## Manage compilation configurations

It is common to use several configuration sets when working on a
project: sometimes the classical triad _debug_, _profile_ and
_release_ are enough, sometimes more complicated setups are required.
While there is no special provision (yet) for managing the compilation
configurations in a project it is easy to implement.  Here is how to:

- Define several compilation setups.
- Simultaneously prepare products with different distinct setups.

Assume we have three compilation setups _debug_, _profile_ and
_release._ For each of these setup we create a corresponding
`Makefile`, so `Makefile.debug`, `Makefile.profile` and
`Makefile.release` stored in *PROJECTLIBRARYMAKE*.

Once you have created these files, you only need to export

	MAKEINITRC=Makefile.debug

to work in the _debug_ compilation setup, for instance. Before
changing this environment variable, it is advisable to `clean` your
project tree—or to take advantage of the more advanced feature
`MAKEOBJDIR`.

The simultaneous build of your products with distinct compilation
setups is achieved by adding these lines to your
master `Makefile` (before the `.include` line):

	PROJECTSETUP=	debug profile release

	universe:
	.for setup in ${PROJECTSETUP}
		${ENVTOOL} MAKEINITRC=Makefile.${setup} MAKEOBJDIR=${.CURDIR}/obj/${setup} ${MAKE} all
	.endfor

Issuing `make universe` will then create an `obj` directory containing
subdirectories `debug`, `profile` and `release` holding compilation
products.

Depending on your workflow, there might be more useful ways to combine
these features together.


## Prepare and install documentation for complex projects

We can use `ocamldoc` to generate the documentation of a large
project, which sources span across several directories.

First, we require `ocamldoc` to generate an *ocamldoc dump* for each
subpackage, which is achieved by parametrising the *ODOC_FORMAT*
variable.  Here is how the `Makefile` of the `newton` library in our
`golden_ratio` project now looks like:

	LIBRARY=    newton
	SRCS+=      newton.ml
	USE_ODOC=	yes
	ODOC_NAME=	newtontk
	ODOC_TITLE=	Newton's method
	ODOC_FORMAT=odoc
	.include "ocaml.lib.mk"

The implicit value for *ODOC_FORMAT* is `odoc html` so that the above
setting merely inhibits the production of HTML documentation for
`newton`. This documentation is not useful anymore, since it is a
subset of the global documentation whose production we are organising.

In the project master directory, create a `manual` directory and
create a `Makefile` there along these lines:

	ODOC_TITLE= Golden Ratio

	DIRS+= ../newton
	MANUAL+= newton.odoc

	DIRS+= ../fibonacci
	MANUAL+= fibonacci.odoc

	.include "ocaml.manual.mk"

As a final step, integrate the `manual` directory to your project by
appending it to the *SUBDIR* list in your master `Makefile`.  Making
`all` or will from now on produce HTML documentation!  The `all`
target is split as `depend build doc` so, if you want to skip
documentation generation, you can use `make build` instead of `make
all` or `make`.
