!>=========================================================================
!!
!!  Module:
!!
!!  epswrite_hdf5_m     Originally by JRD     Last Modified 12/2014 (FHJ)
!!
!!    Routines to write header info for epsmat files in HDF5 format.
!!
!!=========================================================================

#include "f_defs.h"

module epswrite_hdf5_m
#ifdef HDF5
  use hdf5
  use global_m
  use hdf5_io_m
  use wfn_io_hdf5_m
  implicit none

  private

  public :: &
    eps_hdf5_setup, &
    set_subspace_neigen_q, &   
    set_qpt_done, &
    is_qpt_done

contains


subroutine eps_hdf5_setup(kp, gvec, syms, crys, pol, qgrid, nq, qpts, nmtx, &
  nmtx_max, nband, name, restart)
  type(kpoints), intent(in) :: kp
  type(gspace), intent(in) :: gvec
  type(symmetry), intent(in) :: syms
  type(crystal), intent(in) :: crys
  type(polarizability), intent(in) :: pol
  integer, intent(in) :: qgrid(3)
  integer, intent(in) :: nq
  real(DP), intent(in) :: qpts(:,:) !< (3,nq)
  integer, intent(in) :: nmtx(:) !< (nq)
  integer, intent(in) :: nmtx_max
  integer, intent(in) :: nband
  character(len=*), intent(in) :: name
  logical, intent(inout), optional :: restart

  integer(HID_T) :: file_id
  integer(HID_T) :: dspace
  integer(HID_T) :: dset_id
  integer(HSIZE_T) :: dims(6)

  integer :: error, ii, intdata(1)
  integer :: neigen_of_q(nq)
  logical :: qpt_done(nq)
  real(DP), allocatable :: tmp(:,:)
  real(DP) :: freqs_tmp(2,pol%nfreq)
  real(DP) :: realdata(1)
  logical :: restart_, file_exists, file_ok
  character(len=3) :: sheader='WFN'

  PUSH_SUB(eps_hdf5_setup)

  restart_=.false.
  if (present(restart)) restart_ = restart

  ! FHJ: try to restart the calculation, if possible and desired.
  ! We ignore the restart flags if the file doesn`t exist. However, the code
  ! dies if the file exists and is incompatible or looks corrupted.
  if (restart_) then
    call try_restart()
    if (file_ok) return
  endif

  ! FHJ: Set up file: write MF header and create groups
  write(6,'(1x,2a)') "Initializing ", trim(name)
  call setup_hdf5_mf_file(trim(name))
  call write_hdf5_header_type(trim(name), sheader, SCALARSIZE, kp, gvec, syms, crys)
  call write_hdf5_gvectors(trim(name), gvec%ng, gvec%components)
  call h5fopen_f(trim(name), H5F_ACC_RDWR_F, file_id, error)
  call hdf5_create_group(file_id, 'eps_header', error)
  call hdf5_create_group(file_id, 'eps_header/params', error)
  call hdf5_create_group(file_id, 'eps_header/qpoints', error)
  call hdf5_create_group(file_id, 'eps_header/freqs', error)
  call hdf5_create_group(file_id, 'eps_header/gspace', error)
  call hdf5_create_group(file_id, 'mats', error)

  if( pol%subspace ) then
    call hdf5_create_group(file_id, 'eps_header/subspace', error)
  endif

  call hdf5_write_int(file_id, 'eps_header/versionnumber', VER_EPS_HDF5, error)
  call hdf5_write_int(file_id, 'eps_header/flavor', SCALARSIZE, error)

  ! FHJ: General datasets
  call hdf5_write_int(file_id, 'eps_header/params/matrix_type', pol%matrix_type, error)
  call hdf5_write_logical(file_id, 'eps_header/params/has_advanced', pol%has_advanced, error)
  call hdf5_write_int(file_id, 'eps_header/params/nmatrix', pol%nmatrix, error)
  call hdf5_write_int(file_id, 'eps_header/params/matrix_flavor', pol%matrix_flavor, error)
  call hdf5_write_int(file_id, 'eps_header/params/icutv', pol%icutv, error)
  call hdf5_write_double(file_id, 'eps_header/params/ecuts', pol%ecuts, error)
  call hdf5_write_int(file_id, 'eps_header/params/nband', nband, error)
  call hdf5_write_double(file_id, 'eps_header/params/efermi', pol%efermi/ryd, error)
  call hdf5_write_int(file_id, 'eps_header/params/intraband_flag', pol%intraband_flag, error)
  call hdf5_write_double(file_id, 'eps_header/params/intraband_overlap_min', pol%intraband_overlap_min, error)
  call hdf5_write_logical(file_id, 'eps_header/params/subsample', pol%subsample, error)
  call hdf5_write_logical(file_id, 'eps_header/params/subspace', pol%subspace, error)

  ! FHJ: Q-points-related datasets
  qpt_done(:) = .false.
  call hdf5_write_int(file_id, 'eps_header/qpoints/nq', nq, error)
  call hdf5_write_double_array(file_id, 'eps_header/qpoints/qpts', (/3,nq/), qpts, error)
  call hdf5_write_int_array(file_id, 'eps_header/qpoints/qgrid', (/3/), qgrid, error)
  call hdf5_write_logical_array(file_id, 'eps_header/qpoints/qpt_done', (/nq/), qpt_done, error)

  ! FHJ: Frequency-related datasets
  call hdf5_write_int(file_id, 'eps_header/freqs/freq_dep', pol%freq_dep, error)
  call hdf5_write_int(file_id, 'eps_header/freqs/nfreq', pol%nfreq, error)
  call hdf5_write_int(file_id, 'eps_header/freqs/nfreq_imag', pol%nfreq_imag, error)
  do ii=1,pol%nfreq !FHJ: TODO - have an unique complex freq grid in the future!
    freqs_tmp(1,ii) = pol%dFreqGrid(ii) + dble(pol%dFreqBrd(ii))
    freqs_tmp(2,ii) = IMAG(pol%dFreqBrd(ii))
  enddo
  call hdf5_write_double_array(file_id, 'eps_header/freqs/freqs', (/2, pol%nfreq/), freqs_tmp, error)

  ! FHJ: G-vectors-related datasets
  call hdf5_write_int_array(file_id, 'eps_header/gspace/nmtx', (/nq/), nmtx, error)
  call hdf5_write_int(file_id, 'eps_header/gspace/nmtx_max',  nmtx_max, error)
  call hdf5_create_dset(file_id, 'eps_header/gspace/ekin', H5T_NATIVE_DOUBLE, (/gvec%ng,nq/), error)
  ! FHJ: Epsmat version 3+ includes the bare Coulomb interaction as well.
  ! We use this to get the advanced epsilon from the retarded matrix.
  call hdf5_create_dset(file_id, 'eps_header/gspace/vcoul', H5T_NATIVE_DOUBLE, (/nmtx_max,nq/), error)
  call hdf5_create_dset(file_id, 'eps_header/gspace/gind_eps2rho', H5T_NATIVE_INTEGER, (/gvec%ng,nq/), error)
  call hdf5_create_dset(file_id, 'eps_header/gspace/gind_rho2eps', H5T_NATIVE_INTEGER, (/gvec%ng,nq/), error)

  !DVF: subspace approximation-related datasets
  if( pol%subspace ) then
    call hdf5_write_logical(file_id, 'eps_header/subspace/keep_full_eps_static', pol%keep_full_eps_static, error)
    call hdf5_write_logical(file_id, 'eps_header/subspace/matrix_in_subspace_basis', pol%matrix_in_subspace_basis, error)
    call hdf5_write_double(file_id, 'eps_header/subspace/eps_eigenvalue_cutoff', pol%chi_eigenvalue_cutoff, error)
    call hdf5_write_int(file_id, 'eps_header/subspace/neig_max',  pol%neig_sub_input, error)
    ! use same strategy as qpt_done, since the number of eigenvector will never exceed neig_sub_input, we 
    ! initially set all of them to neig_sub_input and we update the actual value when calculated
    neigen_of_q(:) = pol%neig_sub_input
    call hdf5_write_int_array(file_id, 'eps_header/subspace/neig', (/nq/), neigen_of_q, error)
  endif

  ! FHJ: Matrix-elements-related datasets
  call hdf5_create_dset(file_id, 'mats/matrix', H5T_NATIVE_DOUBLE, &
    (/pol%matrix_flavor,nmtx_max,nmtx_max,pol%nfreq,pol%nmatrix,nq/), error)
  call hdf5_create_dset(file_id, 'mats/matrix-diagonal', H5T_NATIVE_DOUBLE, &
    (/pol%matrix_flavor,nmtx_max,nq/), error)

  if( pol%subspace .and. pol%matrix_in_subspace_basis ) then
    ! 
    ! here pol%nmatrix should always be ONE, this may change if we want to write
    ! the subspace chi, which can be spin polarized, which not sure if make any sense.
    ! 
    call hdf5_create_dset(file_id, 'mats/matrix_fulleps0', H5T_NATIVE_DOUBLE, &
       (/pol%matrix_flavor,nmtx_max,nmtx_max,1,pol%nmatrix,nq/), error)
    !XXX
    !XXX actual (max) size
    call hdf5_create_dset(file_id, 'mats/matrix_eigenvec', H5T_NATIVE_DOUBLE, &
        (/pol%matrix_flavor,nmtx_max,pol%neig_sub_input,1,pol%nmatrix,nq/), error)
    !XXX
    !XXX full matrix
    !XXX call hdf5_create_dset(file_id, 'mats/matrix_eigenvec', H5T_NATIVE_DOUBLE, &
    !XXX     (/pol%matrix_flavor,nmtx_max,nmtx_max,1,pol%nmatrix,nq/), error)
    !XXX
    call hdf5_create_dset(file_id, 'mats/matrix_subspace', H5T_NATIVE_DOUBLE, &
       (/pol%matrix_flavor,pol%neig_sub_input,pol%neig_sub_input,pol%nfreq,pol%nmatrix,nq/), error)
  end if

  call h5fclose_f(file_id, error)

  POP_SUB(eps_hdf5_setup)

contains

  subroutine try_restart()
    integer :: nspin_old, nq_old
    real(DP) :: qpts_old(3,nq)
    integer :: freq_dep_old, nfreq_old
    integer :: ng_old, gvecs_old(3,gvec%ng), nmtx_old(nq), matrix_flavor_old, nmatrix_old
    integer :: neig_max_old
    logical :: subspace_exists, subspace_old, matrix_in_subspace_basis_old, keep_full_eps_static_old
    logical :: subspace_error

    subspace_error = .false.
    write(6,'(1x,2a)') "Trying to restart file ", trim(name)
    inquire(file=trim(name), exist=file_exists)
    if (file_exists) then
      call h5fopen_f(trim(name), H5F_ACC_RDONLY_F, file_id, error)
      if (error==0) then
        ! FHJ: Consistency check.
        call h5lexists_f(file_id, 'eps_header/qpoints/qpt_done', file_ok, error)
        if (file_ok) then
          call hdf5_read_int(file_id, 'mf_header/kpoints/nspin', nspin_old, error)
          if (error==0) call hdf5_read_int(file_id, &
            'eps_header/qpoints/nq', nq_old, error)
          if (error==0) call hdf5_read_double_array(file_id, &
            'eps_header/qpoints/qpts', (/3,nq/), qpts_old, error)
          if (error==0) call hdf5_read_int(file_id, &
            'eps_header/freqs/freq_dep', freq_dep_old, error)
          if (error==0) call hdf5_read_int(file_id, &
            'eps_header/freqs/nfreq', nfreq_old, error)
          if (error==0) call hdf5_read_int(file_id, &
            'mf_header/gspace/ng', ng_old, error)
          if (error==0) call hdf5_read_int_array(file_id, &
            'mf_header/gspace/components', (/3,gvec%ng/), gvecs_old, error)
          if (error==0) call hdf5_read_int_array(file_id, &
            'eps_header/gspace/nmtx', (/nq/), nmtx_old, error)
          if (error==0) call hdf5_read_int(file_id, &
            'eps_header/params/matrix_flavor', matrix_flavor_old, error)
          if (error==0) call hdf5_read_int(file_id, &
            'eps_header/params/nmatrix', nmatrix_old, error)
          file_ok = (error==0) .and. (nspin_old==kp%nspin .and. nq_old==nq .and. &
            all(dabs(qpts_old-qpts)<TOL_ZERO) .and. freq_dep_old==pol%freq_dep .and. &
            nfreq_old==pol%nfreq .and. ng_old==gvec%ng .and. all(gvecs_old==gvec%components) .and. &
            all(nmtx_old==nmtx) .and. matrix_flavor_old==pol%matrix_flavor .and. &
            nmatrix_old==pol%nmatrix)
          ! additional check in case of subspace
          if ( file_ok ) then
            ! check the existence of the parameter (this ensure that the routine will 
            ! work also for old .h5 files)
            call h5lexists_f(file_id, 'eps_header/params/subspace', subspace_exists, error)
            if (subspace_exists .and. error == 0) then
              call hdf5_read_logical(file_id, 'eps_header/params/subspace', subspace_old, error)
              ! eps_header/params/subspace exists, check new and old flags
              file_ok = (error==0) .and. (pol%subspace .eqv. subspace_old)
              if( file_ok ) then
                if( subspace_old .and. error == 0) then
                  ! if the flags are equivalent and we are doing a subspace calculation
                  ! check the remaining parameters
                  ! this complication is due to the fact that the subspace section is not 
                  ! created in non subspace calculation...
                  call hdf5_read_int(file_id, 'eps_header/subspace/neig_max',  neig_max_old, error)
                  if (error==0) call hdf5_read_logical(file_id, &
                     'eps_header/subspace/matrix_in_subspace_basis', matrix_in_subspace_basis_old, error)
                  if (error==0) call hdf5_read_logical(file_id, &
                     'eps_header/subspace/keep_full_eps_static', keep_full_eps_static_old, error)
                  file_ok = (error==0) &
                            .and. (neig_max_old == pol%neig_sub_input) &
                            .and. (matrix_in_subspace_basis_old .eqv. pol%matrix_in_subspace_basis) &
                            .and. (keep_full_eps_static_old     .eqv. pol%keep_full_eps_static)
                  if( .not. file_ok ) subspace_error = .true.
                end if
              else
                if(pol%subspace) then
                  write(6,'(1x,A)') "ERROR: Trying to restart a subspace calc from a non-subspace calc"
                else
                  write(6,'(1x,A)') "ERROR: Trying to restart a non-subspace calc from a subspace calc"
                end if
                subspace_error = .true.
                file_ok = .false.
              end if
            else
              if (pol%subspace) then
                ! eps_header/params/subspace don`t exist but we are trying to restart a subspace calc.
                write(6,'(1x,A)') "ERROR: Not a subspace initialized .h5 file"
                subspace_error = .true.
                file_ok = .false.
              end if
            end if
          end if
        endif
        call h5fclose_f(file_id, error)
        if (file_ok) then
          ! FHJ: File *seems* alright, we don`t have to initialize it
          write(6,'(1x,2a)') "Everything looks ok: restarting file ", trim(name)
          return
        endif
        write(0,*)
        write(0,'(3a)') "ERROR: file ", trim(name), " is incompatible with the current calculation."
        if( subspace_error ) then 
          write(0,*) "ERROR is most likely due to incompatible .h5 files within subspace approximation"
        end if
        write(0,*) 'Values from file vs. calculation:'
        write(0,*) 'nspin:', nspin_old, kp%nspin
        write(0,*) 'nq:', nq_old, nq
        write(0,*) 'qpts (same?):', all(dabs(qpts_old-qpts)<TOL_ZERO)
        write(0,*) 'freq_dep:', freq_dep_old, pol%freq_dep
        write(0,*) 'nfreq:', nfreq_old, pol%nfreq

        write(0,*) 'ng:', ng_old, gvec%ng
        write(0,*) 'gvecs (same?):', all(gvecs_old==gvec%components)
        write(0,*) 'nmtx (same?):', all(nmtx_old==nmtx)
        write(0,*) 'matrix_flavor:', matrix_flavor_old, pol%matrix_flavor
        write(0,*) 'nmatrix:', nmatrix_old, pol%nmatrix
        write(0,*)
        write(0,*) 'NOTE: you should only trust the first pair of values that disagree.'
        write(0,*)
        call die("file "//trim(name)//" is incompatible with the current calculation.", only_root_writes=.true.)
      else
        write(0,'(3a,i0,a)') "ERROR: ", trim(name), " is not a valid HDF5 file (error code: ", error," )."
        write(0,'(a)') 'Make sure the file is not corrupted'
        call die("file "//trim(name)//" looks corrupted", only_root_writes=.true.)
      endif
    endif
    file_ok = .false.
    write(0,'(3a)') "WARNING: file ", trim(name), " doesn`t exist. We`ll start the calculation from scratch."   
    restart_ = .false.
    if (present(restart)) restart = .false.

  end subroutine try_restart

end subroutine eps_hdf5_setup


subroutine setup_subspace_mats_hdf5(fname, pol,iq)
  character(len=*), intent(in) :: fname
  integer, intent(in) :: iq
  type(polarizability), intent(in) :: pol

  integer(HID_T) :: file_id
  integer :: error
  character(len=20) :: subspace_name
! DVF: we create different subgroups in the mats group for the
! subspace matrices and basis vectors at each q-point. This is 
! necessary because we don`t know the size of the subspace basis
! at each q-point before we actually do the calculation like we do 
! for the reciprocal space basis. So, we need to allocate the matrices  
! right after we have run scalapack/ELPA, and then create the needed 
! subgroups and dsets. That is the purpose of this routine. 

! The above description assumes we specify the eigenvector tolerance and 
! not the number of eigenvectors. Specifying the tolerance is best practice 
! since the eigenvector tolerance is directly related to the convergence of 
! one`s calculation, and specifying the tolerance requires no prior knowledge 
! of the size of the g-space, and hence is more automatic for users.

  PUSH_SUB(setup_subspace_mats_hdf5)

  call h5fopen_f(trim(fname), H5F_ACC_RDWR_F, file_id, error)
  ! DVF: subspace matrix-elements-related datasets
  write(subspace_name,'(a14,i4)') "mats/subspace_",iq
  call hdf5_create_group(file_id, trim(subspace_name), error)
  call hdf5_create_dset(file_id, trim(subspace_name) // '/matrix_sub', H5T_NATIVE_DOUBLE, &
    (/pol%matrix_flavor,pol%neig_sub,pol%neig_sub,pol%nfreq,pol%nmatrix,1/), error)
  call hdf5_create_dset(file_id, trim(subspace_name) // '/basis_sub', H5T_NATIVE_DOUBLE, &
    (/pol%matrix_flavor,pol%nmtx,pol%neig_sub,1,pol%nmatrix,1/), error)
  call h5fclose_f(file_id, error)

  POP_SUB(setup_subspace_mats_hdf5)

end subroutine setup_subspace_mats_hdf5

subroutine set_subspace_neigen_q(fname, iq, neigen)
  character(len=*), intent(in) :: fname
  integer, intent(in) :: iq, neigen

  integer(HID_T) :: file_id
  integer :: nq, error
  integer, allocatable :: neigen_of_q(:)

  PUSH_SUB(set_subspace_neigen_q)

  call open_file(99, trim(fname), status='old')
  call close_file(99)

  call h5fopen_f(trim(fname), H5F_ACC_RDWR_F, file_id, error)
  call hdf5_read_int(file_id, 'eps_header/qpoints/nq', nq, error)
  SAFE_ALLOCATE(neigen_of_q, (nq))

  call hdf5_read_int_array(file_id, 'eps_header/subspace/neig', (/nq/), neigen_of_q, error)
  neigen_of_q(iq) = neigen
  call hdf5_write_int_array(file_id, 'eps_header/subspace/neig', (/nq/), neigen_of_q, error)

  call h5fclose_f(file_id, error)
  SAFE_DEALLOCATE(neigen_of_q)

  POP_SUB(set_subspace_neigen_q)

end subroutine set_subspace_neigen_q

subroutine set_qpt_done(fname, iq)
  character(len=*), intent(in) :: fname
  integer, intent(in) :: iq

  integer(HID_T) :: file_id
  integer :: nq, error
  logical, allocatable :: qpt_done(:)

  PUSH_SUB(set_qpt_done)

  call open_file(99, trim(fname), status='old')
  call close_file(99)
  
  call h5fopen_f(trim(fname), H5F_ACC_RDWR_F, file_id, error)
  call hdf5_read_int(file_id, 'eps_header/qpoints/nq', nq, error)
  SAFE_ALLOCATE(qpt_done, (nq))
  call hdf5_read_logical_array(file_id, 'eps_header/qpoints/qpt_done', (/nq/), qpt_done, error)
  qpt_done(iq) = .true.
  call hdf5_write_logical_array(file_id, 'eps_header/qpoints/qpt_done', (/nq/), qpt_done, error)
  call h5fclose_f(file_id, error)
  SAFE_DEALLOCATE(qpt_done)

  POP_SUB(set_qpt_done)

end subroutine set_qpt_done


logical function is_qpt_done(fname, iq)
  character(len=*), intent(in) :: fname
  integer, intent(in) :: iq

  integer(HID_T) :: file_id
  integer :: nq, error
  logical, allocatable :: qpt_done(:)

  PUSH_SUB(is_qpt_done)

  call open_file(99, trim(fname), status='old')
  call close_file(99)

  call h5fopen_f(trim(fname), H5F_ACC_RDONLY_F, file_id, error)
  call hdf5_read_int(file_id, 'eps_header/qpoints/nq', nq, error)
  SAFE_ALLOCATE(qpt_done, (nq))
  call hdf5_read_logical_array(file_id, 'eps_header/qpoints/qpt_done', (/nq/), qpt_done, error)
  is_qpt_done = qpt_done(iq)
  call h5fclose_f(file_id, error)
  SAFE_DEALLOCATE(qpt_done)

  POP_SUB(is_qpt_done)

end function is_qpt_done

#endif
end module epswrite_hdf5_m
