
/*------------------------------------------------------------------------------

   surface.cpp
   generates an isosurface of the scalar field
   written by G. Samsonidze (October 2008)
   Objects used by ICM also refactored out into Common/wfn_utils.cpp by DAS

   based on marching cubes and marching tetrahedra codes by P. Bourke
   http://local.wasp.uwa.edu.au/~pbourke/geometry/polygonise/
   See surface.inp for details on input file.

------------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <limits.h>
#include "wfn_utils.h"

const int MAXBOND = 16;
const double ATOMSIZE = 0.20;
const double BONDRADIUS = 0.15;
const double BONDTOLERANCE = 0.45;
const double MINIMUMBONDINGDISTANCE = 0.40;
const double STRICTVALENCE = false;

typedef struct {
   CARTESIAN point[8];
   CARTESIAN vector[8];
   double scalar[8];
} CUBE;

typedef struct {
   CARTESIAN point[4];
   CARTESIAN vector[4];
   double scalar[4];
} TETRAHEDRON;

typedef struct {
   CARTESIAN point[3];
} TRIANGLE;

struct TRIANGLELIST {
   TRIANGLE triangle;
   TRIANGLELIST *node;
};

typedef struct {
   CARTESIAN vector[3];
} NORMAL;

struct NORMALLIST {
   NORMAL normal;
   NORMALLIST *node;
};

const int edge[12][2] = {
   {0, 1}, {1, 2}, {2, 3}, {3, 0},
   {4, 5}, {5, 6}, {6, 7}, {7, 4},
   {0, 4}, {1, 5}, {2, 6}, {3, 7}};

const int face[12][3] = {
   {0, 1, 2}, {0, 3, 2}, {4, 5, 6}, {4, 7, 6},
   {0, 1, 5}, {0, 4, 5}, {1, 2, 6}, {1, 5, 6},
   {2, 3, 7}, {2, 6, 7}, {3, 0, 4}, {3, 7, 4}};

const int vertex_tetrahedron[6][4] = {
   {0, 5, 1, 6}, {0, 1, 2, 6}, {0, 2, 3, 6},
   {0, 3, 7, 6}, {0, 7, 4, 6}, {0, 4, 5, 6}};

const int edge_tetrahedron[6][2] = {
   {0, 1}, {1, 2}, {2, 0}, {0, 3}, {1, 3}, {2, 3}};

const int edge_table_tetrahedron[16] = {
   0x0000, 0x000d, 0x0013, 0x001e, 0x0026, 0x002b, 0x0035, 0x0038,
   0x0038, 0x0035, 0x002b, 0x0026, 0x001e, 0x0013, 0x000d, 0x0000};

const int triangle_table_tetrahedron[16][7] = {
   {-1, -1, -1, -1, -1, -1, -1},
   { 0,  3,  2, -1, -1, -1, -1},
   { 0,  1,  4, -1, -1, -1, -1},
   { 1,  4,  2,  2,  4,  3, -1},
   { 1,  2,  5, -1, -1, -1, -1},
   { 0,  3,  5,  0,  5,  1, -1},
   { 0,  2,  5,  0,  5,  4, -1},
   { 5,  4,  3, -1, -1, -1, -1},
   { 3,  4,  5, -1, -1, -1, -1},
   { 4,  5,  0,  5,  2,  0, -1},
   { 1,  5,  0,  5,  3,  0, -1},
   { 5,  2,  1, -1, -1, -1, -1},
   { 3,  4,  2,  2,  4,  1, -1},
   { 4,  1,  0, -1, -1, -1, -1},
   { 2,  3,  0, -1, -1, -1, -1},
   {-1, -1, -1, -1, -1, -1, -1}};

const int edge_table_cube[256] = {
   0x0000, 0x0109, 0x0203, 0x030a, 0x0406, 0x050f, 0x0605, 0x070c,
   0x080c, 0x0905, 0x0a0f, 0x0b06, 0x0c0a, 0x0d03, 0x0e09, 0x0f00,
   0x0190, 0x0099, 0x0393, 0x029a, 0x0596, 0x049f, 0x0795, 0x069c,
   0x099c, 0x0895, 0x0b9f, 0x0a96, 0x0d9a, 0x0c93, 0x0f99, 0x0e90,
   0x0230, 0x0339, 0x0033, 0x013a, 0x0636, 0x073f, 0x0435, 0x053c,
   0x0a3c, 0x0b35, 0x083f, 0x0936, 0x0e3a, 0x0f33, 0x0c39, 0x0d30,
   0x03a0, 0x02a9, 0x01a3, 0x00aa, 0x07a6, 0x06af, 0x05a5, 0x04ac,
   0x0bac, 0x0aa5, 0x09af, 0x08a6, 0x0faa, 0x0ea3, 0x0da9, 0x0ca0,
   0x0460, 0x0569, 0x0663, 0x076a, 0x0066, 0x016f, 0x0265, 0x036c,
   0x0c6c, 0x0d65, 0x0e6f, 0x0f66, 0x086a, 0x0963, 0x0a69, 0x0b60,
   0x05f0, 0x04f9, 0x07f3, 0x06fa, 0x01f6, 0x00ff, 0x03f5, 0x02fc,
   0x0dfc, 0x0cf5, 0x0fff, 0x0ef6, 0x09fa, 0x08f3, 0x0bf9, 0x0af0,
   0x0650, 0x0759, 0x0453, 0x055a, 0x0256, 0x035f, 0x0055, 0x015c,
   0x0e5c, 0x0f55, 0x0c5f, 0x0d56, 0x0a5a, 0x0b53, 0x0859, 0x0950,
   0x07c0, 0x06c9, 0x05c3, 0x04ca, 0x03c6, 0x02cf, 0x01c5, 0x00cc,
   0x0fcc, 0x0ec5, 0x0dcf, 0x0cc6, 0x0bca, 0x0ac3, 0x09c9, 0x08c0,
   0x08c0, 0x09c9, 0x0ac3, 0x0bca, 0x0cc6, 0x0dcf, 0x0ec5, 0x0fcc,
   0x00cc, 0x01c5, 0x02cf, 0x03c6, 0x04ca, 0x05c3, 0x06c9, 0x07c0,
   0x0950, 0x0859, 0x0b53, 0x0a5a, 0x0d56, 0x0c5f, 0x0f55, 0x0e5c,
   0x015c, 0x0055, 0x035f, 0x0256, 0x055a, 0x0453, 0x0759, 0x0650,
   0x0af0, 0x0bf9, 0x08f3, 0x09fa, 0x0ef6, 0x0fff, 0x0cf5, 0x0dfc,
   0x02fc, 0x03f5, 0x00ff, 0x01f6, 0x06fa, 0x07f3, 0x04f9, 0x05f0,
   0x0b60, 0x0a69, 0x0963, 0x086a, 0x0f66, 0x0e6f, 0x0d65, 0x0c6c,
   0x036c, 0x0265, 0x016f, 0x0066, 0x076a, 0x0663, 0x0569, 0x0460,
   0x0ca0, 0x0da9, 0x0ea3, 0x0faa, 0x08a6, 0x09af, 0x0aa5, 0x0bac,
   0x04ac, 0x05a5, 0x06af, 0x07a6, 0x00aa, 0x01a3, 0x02a9, 0x03a0,
   0x0d30, 0x0c39, 0x0f33, 0x0e3a, 0x0936, 0x083f, 0x0b35, 0x0a3c,
   0x053c, 0x0435, 0x073f, 0x0636, 0x013a, 0x0033, 0x0339, 0x0230,
   0x0e90, 0x0f99, 0x0c93, 0x0d9a, 0x0a96, 0x0b9f, 0x0895, 0x099c,
   0x069c, 0x0795, 0x049f, 0x0596, 0x029a, 0x0393, 0x0099, 0x0190,
   0x0f00, 0x0e09, 0x0d03, 0x0c0a, 0x0b06, 0x0a0f, 0x0905, 0x080c,
   0x070c, 0x0605, 0x050f, 0x0406, 0x030a, 0x0203, 0x0109, 0x0000};

const int triangle_table_cube[256][16] = {
   {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 0,  8,  3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 0,  1,  9, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 1,  8,  3,  9,  8,  1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 1,  2, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 0,  8,  3,  1,  2, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 9,  2, 10,  0,  2,  9, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 2,  8,  3,  2, 10,  8, 10,  9,  8, -1, -1, -1, -1, -1, -1, -1},
   { 3, 11,  2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 0, 11,  2,  8, 11,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 1,  9,  0,  2,  3, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 1, 11,  2,  1,  9, 11,  9,  8, 11, -1, -1, -1, -1, -1, -1, -1},
   { 3, 10,  1, 11, 10,  3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 0, 10,  1,  0,  8, 10,  8, 11, 10, -1, -1, -1, -1, -1, -1, -1},
   { 3,  9,  0,  3, 11,  9, 11, 10,  9, -1, -1, -1, -1, -1, -1, -1},
   { 9,  8, 10, 10,  8, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 4,  7,  8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 4,  3,  0,  7,  3,  4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 0,  1,  9,  8,  4,  7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 4,  1,  9,  4,  7,  1,  7,  3,  1, -1, -1, -1, -1, -1, -1, -1},
   { 1,  2, 10,  8,  4,  7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 3,  4,  7,  3,  0,  4,  1,  2, 10, -1, -1, -1, -1, -1, -1, -1},
   { 9,  2, 10,  9,  0,  2,  8,  4,  7, -1, -1, -1, -1, -1, -1, -1},
   { 2, 10,  9,  2,  9,  7,  2,  7,  3,  7,  9,  4, -1, -1, -1, -1},
   { 8,  4,  7,  3, 11,  2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   {11,  4,  7, 11,  2,  4,  2,  0,  4, -1, -1, -1, -1, -1, -1, -1},
   { 9,  0,  1,  8,  4,  7,  2,  3, 11, -1, -1, -1, -1, -1, -1, -1},
   { 4,  7, 11,  9,  4, 11,  9, 11,  2,  9,  2,  1, -1, -1, -1, -1},
   { 3, 10,  1,  3, 11, 10,  7,  8,  4, -1, -1, -1, -1, -1, -1, -1},
   { 1, 11, 10,  1,  4, 11,  1,  0,  4,  7, 11,  4, -1, -1, -1, -1},
   { 4,  7,  8,  9,  0, 11,  9, 11, 10, 11,  0,  3, -1, -1, -1, -1},
   { 4,  7, 11,  4, 11,  9,  9, 11, 10, -1, -1, -1, -1, -1, -1, -1},
   { 9,  5,  4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 9,  5,  4,  0,  8,  3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 0,  5,  4,  1,  5,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 8,  5,  4,  8,  3,  5,  3,  1,  5, -1, -1, -1, -1, -1, -1, -1},
   { 1,  2, 10,  9,  5,  4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 3,  0,  8,  1,  2, 10,  4,  9,  5, -1, -1, -1, -1, -1, -1, -1},
   { 5,  2, 10,  5,  4,  2,  4,  0,  2, -1, -1, -1, -1, -1, -1, -1},
   { 2, 10,  5,  3,  2,  5,  3,  5,  4,  3,  4,  8, -1, -1, -1, -1},
   { 9,  5,  4,  2,  3, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 0, 11,  2,  0,  8, 11,  4,  9,  5, -1, -1, -1, -1, -1, -1, -1},
   { 0,  5,  4,  0,  1,  5,  2,  3, 11, -1, -1, -1, -1, -1, -1, -1},
   { 2,  1,  5,  2,  5,  8,  2,  8, 11,  4,  8,  5, -1, -1, -1, -1},
   {10,  3, 11, 10,  1,  3,  9,  5,  4, -1, -1, -1, -1, -1, -1, -1},
   { 4,  9,  5,  0,  8,  1,  8, 10,  1,  8, 11, 10, -1, -1, -1, -1},
   { 5,  4,  0,  5,  0, 11,  5, 11, 10, 11,  0,  3, -1, -1, -1, -1},
   { 5,  4,  8,  5,  8, 10, 10,  8, 11, -1, -1, -1, -1, -1, -1, -1},
   { 9,  7,  8,  5,  7,  9, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 9,  3,  0,  9,  5,  3,  5,  7,  3, -1, -1, -1, -1, -1, -1, -1},
   { 0,  7,  8,  0,  1,  7,  1,  5,  7, -1, -1, -1, -1, -1, -1, -1},
   { 1,  5,  3,  3,  5,  7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 9,  7,  8,  9,  5,  7, 10,  1,  2, -1, -1, -1, -1, -1, -1, -1},
   {10,  1,  2,  9,  5,  0,  5,  3,  0,  5,  7,  3, -1, -1, -1, -1},
   { 8,  0,  2,  8,  2,  5,  8,  5,  7, 10,  5,  2, -1, -1, -1, -1},
   { 2, 10,  5,  2,  5,  3,  3,  5,  7, -1, -1, -1, -1, -1, -1, -1},
   { 7,  9,  5,  7,  8,  9,  3, 11,  2, -1, -1, -1, -1, -1, -1, -1},
   { 9,  5,  7,  9,  7,  2,  9,  2,  0,  2,  7, 11, -1, -1, -1, -1},
   { 2,  3, 11,  0,  1,  8,  1,  7,  8,  1,  5,  7, -1, -1, -1, -1},
   {11,  2,  1, 11,  1,  7,  7,  1,  5, -1, -1, -1, -1, -1, -1, -1},
   { 9,  5,  8,  8,  5,  7, 10,  1,  3, 10,  3, 11, -1, -1, -1, -1},
   { 5,  7,  0,  5,  0,  9,  7, 11,  0,  1,  0, 10, 11, 10,  0, -1},
   {11, 10,  0, 11,  0,  3, 10,  5,  0,  8,  0,  7,  5,  7,  0, -1},
   {11, 10,  5,  7, 11,  5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   {10,  6,  5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 0,  8,  3,  5, 10,  6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 9,  0,  1,  5, 10,  6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 1,  8,  3,  1,  9,  8,  5, 10,  6, -1, -1, -1, -1, -1, -1, -1},
   { 1,  6,  5,  2,  6,  1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 1,  6,  5,  1,  2,  6,  3,  0,  8, -1, -1, -1, -1, -1, -1, -1},
   { 9,  6,  5,  9,  0,  6,  0,  2,  6, -1, -1, -1, -1, -1, -1, -1},
   { 5,  9,  8,  5,  8,  2,  5,  2,  6,  3,  2,  8, -1, -1, -1, -1},
   { 2,  3, 11, 10,  6,  5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   {11,  0,  8, 11,  2,  0, 10,  6,  5, -1, -1, -1, -1, -1, -1, -1},
   { 0,  1,  9,  2,  3, 11,  5, 10,  6, -1, -1, -1, -1, -1, -1, -1},
   { 5, 10,  6,  1,  9,  2,  9, 11,  2,  9,  8, 11, -1, -1, -1, -1},
   { 6,  3, 11,  6,  5,  3,  5,  1,  3, -1, -1, -1, -1, -1, -1, -1},
   { 0,  8, 11,  0, 11,  5,  0,  5,  1,  5, 11,  6, -1, -1, -1, -1},
   { 3, 11,  6,  0,  3,  6,  0,  6,  5,  0,  5,  9, -1, -1, -1, -1},
   { 6,  5,  9,  6,  9, 11, 11,  9,  8, -1, -1, -1, -1, -1, -1, -1},
   { 5, 10,  6,  4,  7,  8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 4,  3,  0,  4,  7,  3,  6,  5, 10, -1, -1, -1, -1, -1, -1, -1},
   { 1,  9,  0,  5, 10,  6,  8,  4,  7, -1, -1, -1, -1, -1, -1, -1},
   {10,  6,  5,  1,  9,  7,  1,  7,  3,  7,  9,  4, -1, -1, -1, -1},
   { 6,  1,  2,  6,  5,  1,  4,  7,  8, -1, -1, -1, -1, -1, -1, -1},
   { 1,  2,  5,  5,  2,  6,  3,  0,  4,  3,  4,  7, -1, -1, -1, -1},
   { 8,  4,  7,  9,  0,  5,  0,  6,  5,  0,  2,  6, -1, -1, -1, -1},
   { 7,  3,  9,  7,  9,  4,  3,  2,  9,  5,  9,  6,  2,  6,  9, -1},
   { 3, 11,  2,  7,  8,  4, 10,  6,  5, -1, -1, -1, -1, -1, -1, -1},
   { 5, 10,  6,  4,  7,  2,  4,  2,  0,  2,  7, 11, -1, -1, -1, -1},
   { 0,  1,  9,  4,  7,  8,  2,  3, 11,  5, 10,  6, -1, -1, -1, -1},
   { 9,  2,  1,  9, 11,  2,  9,  4, 11,  7, 11,  4,  5, 10,  6, -1},
   { 8,  4,  7,  3, 11,  5,  3,  5,  1,  5, 11,  6, -1, -1, -1, -1},
   { 5,  1, 11,  5, 11,  6,  1,  0, 11,  7, 11,  4,  0,  4, 11, -1},
   { 0,  5,  9,  0,  6,  5,  0,  3,  6, 11,  6,  3,  8,  4,  7, -1},
   { 6,  5,  9,  6,  9, 11,  4,  7,  9,  7, 11,  9, -1, -1, -1, -1},
   {10,  4,  9,  6,  4, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 4, 10,  6,  4,  9, 10,  0,  8,  3, -1, -1, -1, -1, -1, -1, -1},
   {10,  0,  1, 10,  6,  0,  6,  4,  0, -1, -1, -1, -1, -1, -1, -1},
   { 8,  3,  1,  8,  1,  6,  8,  6,  4,  6,  1, 10, -1, -1, -1, -1},
   { 1,  4,  9,  1,  2,  4,  2,  6,  4, -1, -1, -1, -1, -1, -1, -1},
   { 3,  0,  8,  1,  2,  9,  2,  4,  9,  2,  6,  4, -1, -1, -1, -1},
   { 0,  2,  4,  4,  2,  6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 8,  3,  2,  8,  2,  4,  4,  2,  6, -1, -1, -1, -1, -1, -1, -1},
   {10,  4,  9, 10,  6,  4, 11,  2,  3, -1, -1, -1, -1, -1, -1, -1},
   { 0,  8,  2,  2,  8, 11,  4,  9, 10,  4, 10,  6, -1, -1, -1, -1},
   { 3, 11,  2,  0,  1,  6,  0,  6,  4,  6,  1, 10, -1, -1, -1, -1},
   { 6,  4,  1,  6,  1, 10,  4,  8,  1,  2,  1, 11,  8, 11,  1, -1},
   { 9,  6,  4,  9,  3,  6,  9,  1,  3, 11,  6,  3, -1, -1, -1, -1},
   { 8, 11,  1,  8,  1,  0, 11,  6,  1,  9,  1,  4,  6,  4,  1, -1},
   { 3, 11,  6,  3,  6,  0,  0,  6,  4, -1, -1, -1, -1, -1, -1, -1},
   { 6,  4,  8, 11,  6,  8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 7, 10,  6,  7,  8, 10,  8,  9, 10, -1, -1, -1, -1, -1, -1, -1},
   { 0,  7,  3,  0, 10,  7,  0,  9, 10,  6,  7, 10, -1, -1, -1, -1},
   {10,  6,  7,  1, 10,  7,  1,  7,  8,  1,  8,  0, -1, -1, -1, -1},
   {10,  6,  7, 10,  7,  1,  1,  7,  3, -1, -1, -1, -1, -1, -1, -1},
   { 1,  2,  6,  1,  6,  8,  1,  8,  9,  8,  6,  7, -1, -1, -1, -1},
   { 2,  6,  9,  2,  9,  1,  6,  7,  9,  0,  9,  3,  7,  3,  9, -1},
   { 7,  8,  0,  7,  0,  6,  6,  0,  2, -1, -1, -1, -1, -1, -1, -1},
   { 7,  3,  2,  6,  7,  2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 2,  3, 11, 10,  6,  8, 10,  8,  9,  8,  6,  7, -1, -1, -1, -1},
   { 2,  0,  7,  2,  7, 11,  0,  9,  7,  6,  7, 10,  9, 10,  7, -1},
   { 1,  8,  0,  1,  7,  8,  1, 10,  7,  6,  7, 10,  2,  3, 11, -1},
   {11,  2,  1, 11,  1,  7, 10,  6,  1,  6,  7,  1, -1, -1, -1, -1},
   { 8,  9,  6,  8,  6,  7,  9,  1,  6, 11,  6,  3,  1,  3,  6, -1},
   { 0,  9,  1, 11,  6,  7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 7,  8,  0,  7,  0,  6,  3, 11,  0, 11,  6,  0, -1, -1, -1, -1},
   { 7, 11,  6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 7,  6, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 3,  0,  8, 11,  7,  6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 0,  1,  9, 11,  7,  6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 8,  1,  9,  8,  3,  1, 11,  7,  6, -1, -1, -1, -1, -1, -1, -1},
   {10,  1,  2,  6, 11,  7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 1,  2, 10,  3,  0,  8,  6, 11,  7, -1, -1, -1, -1, -1, -1, -1},
   { 2,  9,  0,  2, 10,  9,  6, 11,  7, -1, -1, -1, -1, -1, -1, -1},
   { 6, 11,  7,  2, 10,  3, 10,  8,  3, 10,  9,  8, -1, -1, -1, -1},
   { 7,  2,  3,  6,  2,  7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 7,  0,  8,  7,  6,  0,  6,  2,  0, -1, -1, -1, -1, -1, -1, -1},
   { 2,  7,  6,  2,  3,  7,  0,  1,  9, -1, -1, -1, -1, -1, -1, -1},
   { 1,  6,  2,  1,  8,  6,  1,  9,  8,  8,  7,  6, -1, -1, -1, -1},
   {10,  7,  6, 10,  1,  7,  1,  3,  7, -1, -1, -1, -1, -1, -1, -1},
   {10,  7,  6,  1,  7, 10,  1,  8,  7,  1,  0,  8, -1, -1, -1, -1},
   { 0,  3,  7,  0,  7, 10,  0, 10,  9,  6, 10,  7, -1, -1, -1, -1},
   { 7,  6, 10,  7, 10,  8,  8, 10,  9, -1, -1, -1, -1, -1, -1, -1},
   { 6,  8,  4, 11,  8,  6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 3,  6, 11,  3,  0,  6,  0,  4,  6, -1, -1, -1, -1, -1, -1, -1},
   { 8,  6, 11,  8,  4,  6,  9,  0,  1, -1, -1, -1, -1, -1, -1, -1},
   { 9,  4,  6,  9,  6,  3,  9,  3,  1, 11,  3,  6, -1, -1, -1, -1},
   { 6,  8,  4,  6, 11,  8,  2, 10,  1, -1, -1, -1, -1, -1, -1, -1},
   { 1,  2, 10,  3,  0, 11,  0,  6, 11,  0,  4,  6, -1, -1, -1, -1},
   { 4, 11,  8,  4,  6, 11,  0,  2,  9,  2, 10,  9, -1, -1, -1, -1},
   {10,  9,  3, 10,  3,  2,  9,  4,  3, 11,  3,  6,  4,  6,  3, -1},
   { 8,  2,  3,  8,  4,  2,  4,  6,  2, -1, -1, -1, -1, -1, -1, -1},
   { 0,  4,  2,  4,  6,  2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 1,  9,  0,  2,  3,  4,  2,  4,  6,  4,  3,  8, -1, -1, -1, -1},
   { 1,  9,  4,  1,  4,  2,  2,  4,  6, -1, -1, -1, -1, -1, -1, -1},
   { 8,  1,  3,  8,  6,  1,  8,  4,  6,  6, 10,  1, -1, -1, -1, -1},
   {10,  1,  0, 10,  0,  6,  6,  0,  4, -1, -1, -1, -1, -1, -1, -1},
   { 4,  6,  3,  4,  3,  8,  6, 10,  3,  0,  3,  9, 10,  9,  3, -1},
   {10,  9,  4,  6, 10,  4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 4,  9,  5,  7,  6, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 0,  8,  3,  4,  9,  5, 11,  7,  6, -1, -1, -1, -1, -1, -1, -1},
   { 5,  0,  1,  5,  4,  0,  7,  6, 11, -1, -1, -1, -1, -1, -1, -1},
   {11,  7,  6,  8,  3,  4,  3,  5,  4,  3,  1,  5, -1, -1, -1, -1},
   { 9,  5,  4, 10,  1,  2,  7,  6, 11, -1, -1, -1, -1, -1, -1, -1},
   { 6, 11,  7,  1,  2, 10,  0,  8,  3,  4,  9,  5, -1, -1, -1, -1},
   { 7,  6, 11,  5,  4, 10,  4,  2, 10,  4,  0,  2, -1, -1, -1, -1},
   { 3,  4,  8,  3,  5,  4,  3,  2,  5, 10,  5,  2, 11,  7,  6, -1},
   { 7,  2,  3,  7,  6,  2,  5,  4,  9, -1, -1, -1, -1, -1, -1, -1},
   { 9,  5,  4,  0,  8,  6,  0,  6,  2,  6,  8,  7, -1, -1, -1, -1},
   { 3,  6,  2,  3,  7,  6,  1,  5,  0,  5,  4,  0, -1, -1, -1, -1},
   { 6,  2,  8,  6,  8,  7,  2,  1,  8,  4,  8,  5,  1,  5,  8, -1},
   { 9,  5,  4, 10,  1,  6,  1,  7,  6,  1,  3,  7, -1, -1, -1, -1},
   { 1,  6, 10,  1,  7,  6,  1,  0,  7,  8,  7,  0,  9,  5,  4, -1},
   { 4,  0, 10,  4, 10,  5,  0,  3, 10,  6, 10,  7,  3,  7, 10, -1},
   { 7,  6, 10,  7, 10,  8,  5,  4, 10,  4,  8, 10, -1, -1, -1, -1},
   { 6,  9,  5,  6, 11,  9, 11,  8,  9, -1, -1, -1, -1, -1, -1, -1},
   { 3,  6, 11,  0,  6,  3,  0,  5,  6,  0,  9,  5, -1, -1, -1, -1},
   { 0, 11,  8,  0,  5, 11,  0,  1,  5,  5,  6, 11, -1, -1, -1, -1},
   { 6, 11,  3,  6,  3,  5,  5,  3,  1, -1, -1, -1, -1, -1, -1, -1},
   { 1,  2, 10,  9,  5, 11,  9, 11,  8, 11,  5,  6, -1, -1, -1, -1},
   { 0, 11,  3,  0,  6, 11,  0,  9,  6,  5,  6,  9,  1,  2, 10, -1},
   {11,  8,  5, 11,  5,  6,  8,  0,  5, 10,  5,  2,  0,  2,  5, -1},
   { 6, 11,  3,  6,  3,  5,  2, 10,  3, 10,  5,  3, -1, -1, -1, -1},
   { 5,  8,  9,  5,  2,  8,  5,  6,  2,  3,  8,  2, -1, -1, -1, -1},
   { 9,  5,  6,  9,  6,  0,  0,  6,  2, -1, -1, -1, -1, -1, -1, -1},
   { 1,  5,  8,  1,  8,  0,  5,  6,  8,  3,  8,  2,  6,  2,  8, -1},
   { 1,  5,  6,  2,  1,  6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 1,  3,  6,  1,  6, 10,  3,  8,  6,  5,  6,  9,  8,  9,  6, -1},
   {10,  1,  0, 10,  0,  6,  9,  5,  0,  5,  6,  0, -1, -1, -1, -1},
   { 0,  3,  8,  5,  6, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   {10,  5,  6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   {11,  5, 10,  7,  5, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   {11,  5, 10, 11,  7,  5,  8,  3,  0, -1, -1, -1, -1, -1, -1, -1},
   { 5, 11,  7,  5, 10, 11,  1,  9,  0, -1, -1, -1, -1, -1, -1, -1},
   {10,  7,  5, 10, 11,  7,  9,  8,  1,  8,  3,  1, -1, -1, -1, -1},
   {11,  1,  2, 11,  7,  1,  7,  5,  1, -1, -1, -1, -1, -1, -1, -1},
   { 0,  8,  3,  1,  2,  7,  1,  7,  5,  7,  2, 11, -1, -1, -1, -1},
   { 9,  7,  5,  9,  2,  7,  9,  0,  2,  2, 11,  7, -1, -1, -1, -1},
   { 7,  5,  2,  7,  2, 11,  5,  9,  2,  3,  2,  8,  9,  8,  2, -1},
   { 2,  5, 10,  2,  3,  5,  3,  7,  5, -1, -1, -1, -1, -1, -1, -1},
   { 8,  2,  0,  8,  5,  2,  8,  7,  5, 10,  2,  5, -1, -1, -1, -1},
   { 9,  0,  1,  5, 10,  3,  5,  3,  7,  3, 10,  2, -1, -1, -1, -1},
   { 9,  8,  2,  9,  2,  1,  8,  7,  2, 10,  2,  5,  7,  5,  2, -1},
   { 1,  3,  5,  3,  7,  5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 0,  8,  7,  0,  7,  1,  1,  7,  5, -1, -1, -1, -1, -1, -1, -1},
   { 9,  0,  3,  9,  3,  5,  5,  3,  7, -1, -1, -1, -1, -1, -1, -1},
   { 9,  8,  7,  5,  9,  7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 5,  8,  4,  5, 10,  8, 10, 11,  8, -1, -1, -1, -1, -1, -1, -1},
   { 5,  0,  4,  5, 11,  0,  5, 10, 11, 11,  3,  0, -1, -1, -1, -1},
   { 0,  1,  9,  8,  4, 10,  8, 10, 11, 10,  4,  5, -1, -1, -1, -1},
   {10, 11,  4, 10,  4,  5, 11,  3,  4,  9,  4,  1,  3,  1,  4, -1},
   { 2,  5,  1,  2,  8,  5,  2, 11,  8,  4,  5,  8, -1, -1, -1, -1},
   { 0,  4, 11,  0, 11,  3,  4,  5, 11,  2, 11,  1,  5,  1, 11, -1},
   { 0,  2,  5,  0,  5,  9,  2, 11,  5,  4,  5,  8, 11,  8,  5, -1},
   { 9,  4,  5,  2, 11,  3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 2,  5, 10,  3,  5,  2,  3,  4,  5,  3,  8,  4, -1, -1, -1, -1},
   { 5, 10,  2,  5,  2,  4,  4,  2,  0, -1, -1, -1, -1, -1, -1, -1},
   { 3, 10,  2,  3,  5, 10,  3,  8,  5,  4,  5,  8,  0,  1,  9, -1},
   { 5, 10,  2,  5,  2,  4,  1,  9,  2,  9,  4,  2, -1, -1, -1, -1},
   { 8,  4,  5,  8,  5,  3,  3,  5,  1, -1, -1, -1, -1, -1, -1, -1},
   { 0,  4,  5,  1,  0,  5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 8,  4,  5,  8,  5,  3,  9,  0,  5,  0,  3,  5, -1, -1, -1, -1},
   { 9,  4,  5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 4, 11,  7,  4,  9, 11,  9, 10, 11, -1, -1, -1, -1, -1, -1, -1},
   { 0,  8,  3,  4,  9,  7,  9, 11,  7,  9, 10, 11, -1, -1, -1, -1},
   { 1, 10, 11,  1, 11,  4,  1,  4,  0,  7,  4, 11, -1, -1, -1, -1},
   { 3,  1,  4,  3,  4,  8,  1, 10,  4,  7,  4, 11, 10, 11,  4, -1},
   { 4, 11,  7,  9, 11,  4,  9,  2, 11,  9,  1,  2, -1, -1, -1, -1},
   { 9,  7,  4,  9, 11,  7,  9,  1, 11,  2, 11,  1,  0,  8,  3, -1},
   {11,  7,  4, 11,  4,  2,  2,  4,  0, -1, -1, -1, -1, -1, -1, -1},
   {11,  7,  4, 11,  4,  2,  8,  3,  4,  3,  2,  4, -1, -1, -1, -1},
   { 2,  9, 10,  2,  7,  9,  2,  3,  7,  7,  4,  9, -1, -1, -1, -1},
   { 9, 10,  7,  9,  7,  4, 10,  2,  7,  8,  7,  0,  2,  0,  7, -1},
   { 3,  7, 10,  3, 10,  2,  7,  4, 10,  1, 10,  0,  4,  0, 10, -1},
   { 1, 10,  2,  8,  7,  4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 4,  9,  1,  4,  1,  7,  7,  1,  3, -1, -1, -1, -1, -1, -1, -1},
   { 4,  9,  1,  4,  1,  7,  0,  8,  1,  8,  7,  1, -1, -1, -1, -1},
   { 4,  0,  3,  7,  4,  3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 4,  8,  7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 9, 10,  8, 10, 11,  8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 3,  0,  9,  3,  9, 11, 11,  9, 10, -1, -1, -1, -1, -1, -1, -1},
   { 0,  1, 10,  0, 10,  8,  8, 10, 11, -1, -1, -1, -1, -1, -1, -1},
   { 3,  1, 10, 11,  3, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 1,  2, 11,  1, 11,  9,  9, 11,  8, -1, -1, -1, -1, -1, -1, -1},
   { 3,  0,  9,  3,  9, 11,  1,  2,  9,  2, 11,  9, -1, -1, -1, -1},
   { 0,  2, 11,  8,  0, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 3,  2, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 2,  3,  8,  2,  8, 10, 10,  8,  9, -1, -1, -1, -1, -1, -1, -1},
   { 9, 10,  2,  0,  9,  2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 2,  3,  8,  2,  8, 10,  0,  1,  8,  1, 10,  8, -1, -1, -1, -1},
   { 1, 10,  2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 1,  3,  8,  9,  1,  8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 0,  9,  1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   { 0,  3,  8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
   {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1}};

int *uas = NULL;
CARTESIAN *uap = NULL;
int *cs = NULL;
int *nb = NULL;
CARTESIAN **bonds = NULL;
int **sci = NULL;
CARTESIAN ***vector = NULL;
TRIANGLELIST *triangle[2] = {NULL, NULL};
NORMALLIST *normal[2] = {NULL, NULL};

int par_read(char *pfn, char *header, char *ifn, char *iff, char *ofn, char *off, double *isovalue, int *sign, int *power, int *algorithm, bool *smooth, bool *box, bool *basis, bool *uc, CARTESIAN *uco, CARTESIAN *ucv, int *ucf, bool *sc, CARTESIAN *sco, CARTESIAN *scv, int *scf, bool *sct)
{
   int i, ierr, icount = 0, icheck = 0;
   char s1[MAXCHAR], s2[MAXCHAR], s3[MAXCHAR];
   char* trash;
   FILE *h;

   h = fopen(pfn, "r");
   if (h == NULL)
      return -1;

   while (!feof(h))
   {
      strncpy(s1, "\0", 1);
      trash = fgets(s1, MAXCHAR, h);
      parse(s1, s2, s3);
      lowercase(s2);
      if (strcmp(s2, "header") == 0)
      {
         icount++;
         strncpy(header, s3, MAXCHAR);
         header[MAXCHAR - 1] = '\0';
      }
      else if (strcmp(s2, "inputfilename") == 0)
      {
         icount++;
         erasechar(s3, ' ');
         erasechar(s3, '\t');
         strncpy(ifn, s3, MAXCHAR);
         ifn[MAXCHAR - 1] = '\0';
      }
      else if (strcmp(s2, "inputfileformat") == 0)
      {
         icount++;
         erasechar(s3, ' ');
         erasechar(s3, '\t');
         strncpy(iff, s3, MAXCHAR);
         iff[MAXCHAR - 1] = '\0';
      }
      else if (strcmp(s2, "outputfilename") == 0)
      {
         icount++;
         erasechar(s3, ' ');
         erasechar(s3, '\t');
         strncpy(ofn, s3, MAXCHAR);
         ofn[MAXCHAR - 1] = '\0';
      }
      else if (strcmp(s2, "outputfileformat") == 0)
      {
         icount++;
         erasechar(s3, ' ');
         erasechar(s3, '\t');
         strncpy(off, s3, MAXCHAR);
         off[MAXCHAR - 1] = '\0';
      }
      else if (strcmp(s2, "isovalue") == 0)
      {
         icount++;
         *isovalue = atof(s3);
         if (*isovalue < EPS9 || *isovalue > 1.0 - EPS9)
            icheck--;
      }
      else if (strcmp(s2, "sign") == 0)
      {
         icount++;
         erasechar(s3, ' ');
         erasechar(s3, '\t');
         lowercase(s3);
         if (strcmp(s3, "positive") == 0)
            *sign = 0;
         else if (strcmp(s3, "negative") == 0)
            *sign = 1;
         else if (strcmp(s3, "both") == 0)
            *sign = 2;
         else
            icheck--;
      }
      else if (strcmp(s2, "power") == 0)
      {
         icount++;
         *power = atoi(s3);
         if (*power < 0 || *power > 2)
            icheck--;
      }
      else if (strcmp(s2, "algorithm") == 0)
      {
         icount++;
         erasechar(s3, ' ');
         erasechar(s3, '\t');
         lowercase(s3);
         if (strcmp(s3, "cube") == 0)
            *algorithm = 0;
         else if (strcmp(s3, "tetrahedron") == 0)
            *algorithm = 1;
         else
            icheck--;
      }
      else if (strcmp(s2, "smooth") == 0)
      {
         icount++;
         erasechar(s3, ' ');
         erasechar(s3, '\t');
         lowercase(s3);
         if (strcmp(s3, "f") == 0 || strcmp(s3, "false") == 0 || strcmp(s3, "n") == 0 || strcmp(s3, "no") == 0)
            *smooth = false;
         else if (strcmp(s3, "t") == 0 || strcmp(s3, "true") == 0 || strcmp(s3, "y") == 0 || strcmp(s3, "yes") == 0)
            *smooth = true;
         else
            icheck--;
      }
      else if (strcmp(s2, "box") == 0)
      {
         icount++;
         erasechar(s3, ' ');
         erasechar(s3, '\t');
         lowercase(s3);
         if (strcmp(s3, "f") == 0 || strcmp(s3, "false") == 0 || strcmp(s3, "n") == 0 || strcmp(s3, "no") == 0)
            *box = false;
         else if (strcmp(s3, "t") == 0 || strcmp(s3, "true") == 0 || strcmp(s3, "y") == 0 || strcmp(s3, "yes") == 0)
            *box = true;
         else
            icheck--;
      }
      else if (strcmp(s2, "basis") == 0)
      {
         icount++;
         erasechar(s3, ' ');
         erasechar(s3, '\t');
         lowercase(s3);
         if (strcmp(s3, "f") == 0 || strcmp(s3, "false") == 0 || strcmp(s3, "n") == 0 || strcmp(s3, "no") == 0)
            *basis = false;
         else if (strcmp(s3, "t") == 0 || strcmp(s3, "true") == 0 || strcmp(s3, "y") == 0 || strcmp(s3, "yes") == 0)
            *basis = true;
         else
            icheck--;
      }
      else if (strcmp(s2, "uc") == 0)
      {
         icount++;
         erasechar(s3, ' ');
         erasechar(s3, '\t');
         lowercase(s3);
         if (strcmp(s3, "f") == 0 || strcmp(s3, "false") == 0 || strcmp(s3, "n") == 0 || strcmp(s3, "no") == 0)
            *uc = false;
         else if (strcmp(s3, "t") == 0 || strcmp(s3, "true") == 0 || strcmp(s3, "y") == 0 || strcmp(s3, "yes") == 0)
            *uc = true;
         else
            icheck--;
      }
      else if (strcmp(s2, "uco") == 0)
      {
         icount++;
         ierr = fscanf(h, "%le%le%le\n", &uco->x, &uco->y, &uco->z);
      }
      else if (strcmp(s2, "ucv") == 0)
      {
         icount++;
         for (i = 0; i < 3; i++)
            ierr = fscanf(h, "%le%le%le\n", &ucv[i].x, &ucv[i].y, &ucv[i].z);
      }
      else if (strcmp(s2, "ucu") == 0)
      {
         icount++;
         erasechar(s3, ' ');
         erasechar(s3, '\t');
         lowercase(s3);
         if (strcmp(s3, "bohr") == 0)
            *ucf = 0;
         else if (strcmp(s3, "angstrom") == 0)
            *ucf = 1;
         else if (strcmp(s3, "latvec") == 0)
            *ucf = 2;
         else
            icheck--;
      }
      else if (strcmp(s2, "sc") == 0)
      {
         icount++;
         erasechar(s3, ' ');
         erasechar(s3, '\t');
         lowercase(s3);
         if (strcmp(s3, "f") == 0 || strcmp(s3, "false") == 0 || strcmp(s3, "n") == 0 || strcmp(s3, "no") == 0)
            *sc = false;
         else if (strcmp(s3, "t") == 0 || strcmp(s3, "true") == 0 || strcmp(s3, "y") == 0 || strcmp(s3, "yes") == 0)
            *sc = true;
         else
            icheck--;
      }
      else if (strcmp(s2, "sco") == 0)
      {
         icount++;
         ierr = fscanf(h, "%le%le%le\n", &sco->x, &sco->y, &sco->z);
      }
      else if (strcmp(s2, "scv") == 0)
      {
         icount++;
         for (i = 0; i < 3; i++)
            ierr = fscanf(h, "%le%le%le\n", &scv[i].x, &scv[i].y, &scv[i].z);
      }
      else if (strcmp(s2, "scu") == 0)
      {
         icount++;
         erasechar(s3, ' ');
         erasechar(s3, '\t');
         lowercase(s3);
         if (strcmp(s3, "bohr") == 0)
            *scf = 0;
         else if (strcmp(s3, "angstrom") == 0)
            *scf = 1;
         else if (strcmp(s3, "latvec") == 0)
            *scf = 2;
         else
            icheck--;
      }
      else if (strcmp(s2, "sct") == 0)
      {
         icount++;
         erasechar(s3, ' ');
         erasechar(s3, '\t');
         lowercase(s3);
         if (strcmp(s3, "f") == 0 || strcmp(s3, "false") == 0 || strcmp(s3, "n") == 0 || strcmp(s3, "no") == 0)
            *sct = false;
         else if (strcmp(s3, "t") == 0 || strcmp(s3, "true") == 0 || strcmp(s3, "y") == 0 || strcmp(s3, "yes") == 0)
            *sct = true;
         else
            icheck--;
      }
   }

   ierr = fclose(h);
   if (ierr != 0)
      return -1;

   if (icount != 21 || icheck < 0)
      return -1;

   return 0;
}

int atom_clone(int na, CARTESIAN *ucv, CARTESIAN sco, CARTESIAN *scv, int *scmin, int *scmax)
{
   bool flag;
   int i, j, k, l, m, n, s, t, apmin[3], apmax[3];
   double scvw[3], ucvw[3], p, psccenter[3], pscmin[3], pscmax[3], pa[3];
   CARTESIAN scvn[3], ucvn[3], sccenter, sccorner[8], sa;

   normal_make(scv, scvn, scvw);
   normal_make(ucv, ucvn, ucvw);

   sccenter.x = sco.x + scv[0].x * 0.5 + scv[1].x * 0.5 + scv[2].x * 0.5;
   sccenter.y = sco.y + scv[0].y * 0.5 + scv[1].y * 0.5 + scv[2].y * 0.5;
   sccenter.z = sco.z + scv[0].z * 0.5 + scv[1].z * 0.5 + scv[2].z * 0.5;

   for (i = 0; i < 8; i++)
   {
      sccorner[i].x = sccenter.x;
      sccorner[i].y = sccenter.y;
      sccorner[i].z = sccenter.z;
      for (j = 0; j < 3; j++)
      {
         sccorner[i].x += scv[j].x * (double(vertex[i][j]) - 0.5);
         sccorner[i].y += scv[j].y * (double(vertex[i][j]) - 0.5);
         sccorner[i].z += scv[j].z * (double(vertex[i][j]) - 0.5);
      }
   }

   for (i = 0; i < 3; i++)
   {
      scmin[i] = INT_MAX;
      scmax[i] = INT_MIN;
      for (j = 0; j < 8; j++)
      {
         p = (sccorner[j].x * ucvn[i].x + sccorner[j].y * ucvn[i].y + sccorner[j].z * ucvn[i].z) / ucvw[i];
         // this is similar to double_to_int but we want to convert
         // the interval (-2,-1] to -2, (-1,0] to -1, (0,1] to 1,
         // (1,2] to 2, that is, skipping the integer value of 0.
         if (p < 0.0)
            k = int(p - 1.0);
         else
            k = int(p + 1.0);
         if (k < scmin[i])
            scmin[i] = k;
         if (k > scmax[i])
            scmax[i] = k;
      }
   }

   for (i = 0; i < 3; i++)
   {
      apmin[i] = INT_MAX;
      apmax[i] = INT_MIN;
      for (j = 0; j < na; j++)
      {
         p = (ap[j].x * ucvn[i].x + ap[j].y * ucvn[i].y + ap[j].z * ucvn[i].z) / ucvw[i];
         // this is similar to double_to_int but we want to convert
         // the interval (-2,-1] to -2, (-1,0] to -1, (0,1] to 1,
         // (1,2] to 2, that is, skipping the integer value of 0.
         if (p < 0.0)
            k = int(p - 1.0);
         else
            k = int(p + 1.0);
         if (k < apmin[i])
            apmin[i] = k;
         if (k > apmax[i])
            apmax[i] = k;
      }
   }

   j = 0;
   for (i = 0; i < 3; i++)
   {
      if (abs(apmin[i]) > j)
         j = abs(apmin[i]);
      if (abs(apmax[i]) > j)
         j = abs(apmax[i]);
   }
   for (i = 0; i < 3; i++)
   {
      scmin[i] -= j;
      scmax[i] += j;
   }

   for (i = 0; i < 3; i++)
      psccenter[i] = sccenter.x * scvn[i].x + sccenter.y * scvn[i].y + sccenter.z * scvn[i].z;
   for (i = 0; i < 3; i++)
      pscmin[i] = psccenter[i] - 0.5 * scvw[i] - EPS9;
   for (i = 0; i < 3; i++)
      pscmax[i] = psccenter[i] + 0.5 * scvw[i] - EPS9;

   // loop over t is used to construct a supercell twice
   // t = 0 is a dry run intended to identify number of atoms in the supercell n
   // t = 1 is for allocating and filling arrays of atomic species and positions
   uas = as;
   uap = ap;
   n = 0;
   s = 0;
   for (t = 0; t < 2; t++)
   {
      if (t != 0)
      {
         as = new int[MAX(1,n)];
         ap = new CARTESIAN[MAX(1,n)];
         sci = new int*[MAX(1,n)];
         for (i = 0; i < n; i++)
            sci[i] = new int[4];
      }
      for (i = scmin[0]; i <= scmax[0]; i++)
      for (j = scmin[1]; j <= scmax[1]; j++)
      for (k = scmin[2]; k <= scmax[2]; k++)
      for (l = 0; l < na; l++)
      {
         sa.x = ucv[0].x * double(i) + ucv[1].x * double(j) + ucv[2].x * double(k) + uap[l].x;
         sa.y = ucv[0].y * double(i) + ucv[1].y * double(j) + ucv[2].y * double(k) + uap[l].y;
         sa.z = ucv[0].z * double(i) + ucv[1].z * double(j) + ucv[2].z * double(k) + uap[l].z;
         for (m = 0; m < 3; m++)
            pa[m] = sa.x * scvn[m].x + sa.y * scvn[m].y + sa.z * scvn[m].z;
         flag = true;
         for (m = 0; m < 3; m++)
            if (pa[m] < pscmin[m] || pa[m] > pscmax[m])
               flag = false;
         if (flag)
         {
            if (t == 0)
               n++;
            else
            {
               as[s] = uas[l];
               ap[s].x = sa.x;
               ap[s].y = sa.y;
               ap[s].z = sa.z;
               sci[s][0] = l;
               sci[s][1] = i;
               sci[s][2] = j;
               sci[s][3] = k;
               s++;
            }
         }
      }
   }

   return n;
}

void scalar_normal(int ni, int nj, int nk, CARTESIAN *sfs)
{
   int i, j, k, i1, j1, k1, i2, j2, k2;
   double weight, gradient[3];
   CARTESIAN normal;

   vector = new CARTESIAN**[ni];
   for (i = 0; i < ni; i++)
      vector[i] = new CARTESIAN*[nj];
   for (i = 0; i < ni; i++)
   for (j = 0; j < nj; j++)
      vector[i][j] = new CARTESIAN[nk];

   for (i = 0; i < ni; i++)
   {
      if (i == 0)
         i1 = i;
      else
         i1 = i - 1;
      if (i == ni - 1)
         i2 = i;
      else
         i2 = i + 1;
      for (j = 0; j < nj; j++)
      {
         if (j == 0)
            j1 = j;
         else
            j1 = j - 1;
         if (j == nj - 1)
            j2 = j;
         else
            j2 = j + 1;
         for (k = 0; k < nk; k++)
         {
            if (k == 0)
               k1 = k;
            else
               k1 = k - 1;
            if (k == nk - 1)
               k2 = k;
            else
               k2 = k + 1;
            gradient[0] = scalar[i2][j][k] - scalar[i1][j][k];
            gradient[1] = scalar[i][j2][k] - scalar[i][j1][k];
            gradient[2] = scalar[i][j][k2] - scalar[i][j][k1];
            normal.x = gradient[0] * sfs[0].x + gradient[1] * sfs[1].x + gradient[2] * sfs[2].x;
            normal.y = gradient[0] * sfs[0].y + gradient[1] * sfs[1].y + gradient[2] * sfs[2].y;
            normal.z = gradient[0] * sfs[0].z + gradient[1] * sfs[1].z + gradient[2] * sfs[2].z;
            weight = sqrt(normal.x * normal.x + normal.y * normal.y + normal.z * normal.z);
            if (weight < EPS9)
               weight = 1.0;
            normal.x = normal.x / weight;
            normal.y = normal.y / weight;
            normal.z = normal.z / weight;
            vector[i][j][k] = normal;
         }
      }
   }
}

int chemical_species(int na)
{
   bool flag;
   int i, j, ns = 0;
   int *species;

   species = new int[MAX(1,na)];
   for (i = 0; i < na; i++)
      species[i] = 0;
   for (i = 0; i < na; i++)
   {
      flag = true;
      for (j = 0; j < ns; j++)
         if (species[j] == as[i])
            flag = false;
      if (flag)
      {
         species[ns] = as[i];
         ns++;
      }
   }

   cs = new int[MAX(1,ns)];
   for (i = 0; i < ns; i++)
      cs[i] = species[i];
   delete [] species;

   return ns;
}

int bonds_st(int na, CARTESIAN *scv)
{
   int i, i1, i2, j1, j2, j3, v1;
   double r, r1, r2;
   CARTESIAN p;

   nb = new int[MAX(1,na)];
   bonds = new CARTESIAN*[MAX(1,na)];
   for (i = 0; i < na; i++)
      bonds[i] = new CARTESIAN[MAXBOND];

   for (i1 = 0; i1 < na; i1++)
   {
      v1 = periodic_table[as[i1]].valence;
      r1 = periodic_table[as[i1]].rcov;
      i = 0;
      for (i2 = 0; i2 < na; i2++)
      {
         r2 = periodic_table[as[i2]].rcov;
         for (j1 = -1; j1 < 2; j1++)
         for (j2 = -1; j2 < 2; j2++)
         for (j3 = -1; j3 < 2; j3++)
         if ((j3 != 0 || j2 != 0 || j1 != 0 || i2 != i1) && as[i2] != 0 && as[i1] != 0)
         {
            p.x = scv[2].x * double(j3) + scv[1].x * double(j2) + scv[0].x * double(j1) + ap[i2].x - ap[i1].x;
            p.y = scv[2].y * double(j3) + scv[1].y * double(j2) + scv[0].y * double(j1) + ap[i2].y - ap[i1].y;
            p.z = scv[2].z * double(j3) + scv[1].z * double(j2) + scv[0].z * double(j1) + ap[i2].z - ap[i1].z;
            r = sqrt(p.x * p.x + p.y * p.y + p.z * p.z);
            if (r < MINIMUMBONDINGDISTANCE)
               return -1;
            if (r < r1 + r2 + BONDTOLERANCE)
            {
               if (i >= MAXBOND)
                  return -1;
               if (STRICTVALENCE && i >= v1)
                  return -1;
               bonds[i1][i].x = ap[i1].x + p.x * r1 / (r1 + r2);
               bonds[i1][i].y = ap[i1].y + p.y * r1 / (r1 + r2);
               bonds[i1][i].z = ap[i1].z + p.z * r1 / (r1 + r2);
               i++;
            }
         }
      }
      nb[i1] = i;
   }

   return 0;
}

int bonds_ut(int na, int una, CARTESIAN *ucv, int *scmin, int *scmax)
{
   int i, i1, i2, j1, j2, j3, v1;
   double r, r1, r2;
   CARTESIAN p;

   nb = new int[MAX(1,na)];
   bonds = new CARTESIAN*[MAX(1,na)];
   for (i = 0; i < na; i++)
      bonds[i] = new CARTESIAN[MAXBOND];

   for (i1 = 0; i1 < na; i1++)
   {
      v1 = periodic_table[as[i1]].valence;
      r1 = periodic_table[as[i1]].rcov;
      i = 0;
      for (i2 = 0; i2 < una; i2++)
      {
         r2 = periodic_table[uas[i2]].rcov;
         for (j1 = scmin[0] - 1; j1 < scmax[0] + 2; j1++)
         for (j2 = scmin[1] - 1; j2 < scmax[1] + 2; j2++)
         for (j3 = scmin[2] - 1; j3 < scmax[2] + 2; j3++)
         if ((j3 != sci[i1][3] || j2 != sci[i1][2] || j1 != sci[i1][1] || i2 != sci[i1][0]) && uas[i2] != 0 && as[i1] != 0)
         {
            p.x = ucv[2].x * double(j3) + ucv[1].x * double(j2) + ucv[0].x * double(j1) + uap[i2].x - ap[i1].x;
            p.y = ucv[2].y * double(j3) + ucv[1].y * double(j2) + ucv[0].y * double(j1) + uap[i2].y - ap[i1].y;
            p.z = ucv[2].z * double(j3) + ucv[1].z * double(j2) + ucv[0].z * double(j1) + uap[i2].z - ap[i1].z;
            r = sqrt(p.x * p.x + p.y * p.y + p.z * p.z);
            if (r < MINIMUMBONDINGDISTANCE)
               return -1;
            if (r < r1 + r2 + BONDTOLERANCE)
            {
               if (i >= MAXBOND)
                  return -1;
               if (STRICTVALENCE && i >= v1)
                  return -1;
               bonds[i1][i].x = ap[i1].x + p.x * r1 / (r1 + r2);
               bonds[i1][i].y = ap[i1].y + p.y * r1 / (r1 + r2);
               bonds[i1][i].z = ap[i1].z + p.z * r1 / (r1 + r2);
               i++;
            }
         }
      }
      nb[i1] = i;
   }

   return 0;
}

CARTESIAN interpolate_vertex(double scalar, CARTESIAN cartesian1, CARTESIAN cartesian2, double scalar1, double scalar2)
{
   double weight1, weight2;
   CARTESIAN cartesian;

   if (fabs(scalar1 - scalar) < fabs(scalar2 - scalar1) * EPS9)
      return cartesian1;
   if (fabs(scalar2 - scalar) < fabs(scalar2 - scalar1) * EPS9)
      return cartesian2;

   if (fabs(scalar2 - scalar1) < EPS9)
   {
      weight1 = 0.5;
      weight2 = 0.5;
   }
   else
   {
      weight1 = (scalar2 - scalar) / (scalar2 - scalar1);
      weight2 = (scalar - scalar1) / (scalar2 - scalar1);
   }

   cartesian.x = weight1 * cartesian1.x + weight2 * cartesian2.x;
   cartesian.y = weight1 * cartesian1.y + weight2 * cartesian2.y;
   cartesian.z = weight1 * cartesian1.z + weight2 * cartesian2.z;

   return cartesian;
}

int polygonise_cube(CUBE cube, double isovalue, TRIANGLE *triangle, NORMAL *normal, int smooth)
{
   int i, j, k = 0, l = 1, m = 1, t = 0;
   CARTESIAN point[12], vector[12];

   for (i = 0; i < 8; i++)
   {
      if (cube.scalar[i] < isovalue)
         k |= l;
      l <<= 1;
   }

   if (edge_table_cube[k] == 0)
      return 0;

   for (i = 0; i < 12; i++)
   {
      if (edge_table_cube[k] & m)
      {
         point[i] = interpolate_vertex(isovalue, cube.point[edge[i][0]], cube.point[edge[i][1]], cube.scalar[edge[i][0]], cube.scalar[edge[i][1]]);
         if (smooth)
            vector[i] = interpolate_vertex(isovalue, cube.vector[edge[i][0]], cube.vector[edge[i][1]], cube.scalar[edge[i][0]], cube.scalar[edge[i][1]]);
      }
      m <<= 1;
   }

   for (i = 0; triangle_table_cube[k][i] != -1; i += 3)
   {
      for (j = 0; j < 3; j++)
         triangle[t].point[j] = point[triangle_table_cube[k][i + j]];
      if (smooth)
         for (j = 0; j < 3; j++)
            normal[t].vector[j] = vector[triangle_table_cube[k][i + j]];
      t++;
   }

   return t;
}

int polygonise_tetrahedron(TETRAHEDRON tetrahedron, double isovalue, TRIANGLE *triangle, NORMAL *normal, int smooth)
{
   bool flag[6];
   int i, j, k = 0, l = 1, m = 1, t = 0;
   CARTESIAN point[6], vector[6];

   for (i = 0; i < 4; i++)
   {
      if (tetrahedron.scalar[i] < isovalue)
         k |= l;
      l <<= 1;
   }

   if (edge_table_tetrahedron[k] == 0)
      return 0;

   for (i = 0; i < 6; i++)
   {
      if (edge_table_tetrahedron[k] & m)
         flag[i] = true;
      else
         flag[i] = false;
      m <<= 1;
   }

   for (i = 0; i < 6; i++)
   {
   if (flag[i])
      point[i] = interpolate_vertex(isovalue, tetrahedron.point[edge_tetrahedron[i][0]], tetrahedron.point[edge_tetrahedron[i][1]], tetrahedron.scalar[edge_tetrahedron[i][0]], tetrahedron.scalar[edge_tetrahedron[i][1]]);
   if (smooth && flag[i])
      vector[i] = interpolate_vertex(isovalue, tetrahedron.vector[edge_tetrahedron[i][0]], tetrahedron.vector[edge_tetrahedron[i][1]], tetrahedron.scalar[edge_tetrahedron[i][0]], tetrahedron.scalar[edge_tetrahedron[i][1]]);
   }

   for (i = 0; triangle_table_tetrahedron[k][i] != -1; i += 3)
   {
      for (j = 0; j < 3; j++)
         triangle[t].point[j] = point[triangle_table_tetrahedron[k][i + j]];
      if (smooth)
      for (j = 0; j < 3; j++)
         normal[t].vector[j] = vector[triangle_table_tetrahedron[k][i + j]];
      t++;
   }

   return t;
}

void polygonise(int ni, int nj, int nk, CARTESIAN sfo, CARTESIAN *sfs, double isovalue, int algorithm, int smooth, int sign)
{
   int i, j, k, c, a, t, na, nt, lo, hi;
   CUBE cube;
   TETRAHEDRON tetrahedron;
   TRIANGLE polygon_triangle[5];
   NORMAL polygon_normal[5];
   TRIANGLELIST *previous_triangle, *current_triangle, *first_triangle = NULL;
   NORMALLIST *previous_normal, *current_normal, *first_normal = NULL;

   if (algorithm == 0)
      na = 1;
   else
      na = 6;

   for (i = 0; i < ni - 1; i++)
   for (j = 0; j < nj - 1; j++)
   for (k = 0; k < nk - 1; k++)
   {
      lo = 0;
      hi = 0;
      for (c = 0; c < 8; c++)
      {
         if (scalar[i + vertex[c][0]][j + vertex[c][1]][k + vertex[c][2]] <= isovalue)
            lo++;
         if (scalar[i + vertex[c][0]][j + vertex[c][1]][k + vertex[c][2]] >= isovalue)
            hi++;
      }
      if (lo != 0 && hi != 0)
      {
         for (c = 0; c < 8; c++)
         {
            cube.point[c].x = sfo.x + double(i + vertex[c][0]) * sfs[0].x + double(j + vertex[c][1]) * sfs[1].x + double(k + vertex[c][2]) * sfs[2].x;
            cube.point[c].y = sfo.y + double(i + vertex[c][0]) * sfs[0].y + double(j + vertex[c][1]) * sfs[1].y + double(k + vertex[c][2]) * sfs[2].y;
            cube.point[c].z = sfo.z + double(i + vertex[c][0]) * sfs[0].z + double(j + vertex[c][1]) * sfs[1].z + double(k + vertex[c][2]) * sfs[2].z;
            if (smooth)
               cube.vector[c] = vector[i + vertex[c][0]][j + vertex[c][1]][k + vertex[c][2]];
            cube.scalar[c] = scalar[i + vertex[c][0]][j + vertex[c][1]][k + vertex[c][2]];
         }

         for (a = 0; a < na; a++)
         {
            if (algorithm == 0)
               nt = polygonise_cube(cube, isovalue, polygon_triangle, polygon_normal, smooth);
            else
            {
               for (t = 0; t < 4; t++)
               {
                  tetrahedron.point[t] = cube.point[vertex_tetrahedron[a][t]];
                  if (smooth)
                     tetrahedron.vector[t] = cube.vector[vertex_tetrahedron[a][t]];
                  tetrahedron.scalar[t] = cube.scalar[vertex_tetrahedron[a][t]];
               }
               nt = polygonise_tetrahedron(tetrahedron, isovalue, polygon_triangle, polygon_normal, smooth);
            }
            for (t = 0; t < nt; t++)
            {
               current_triangle = new TRIANGLELIST;
               current_triangle->triangle = polygon_triangle[t];
               current_triangle->node = NULL;
               if (first_triangle == NULL)
                  first_triangle = current_triangle;
               else
                  previous_triangle->node = current_triangle;
               previous_triangle = current_triangle;
               if (smooth)
               {
                  current_normal = new NORMALLIST;
                  current_normal->normal = polygon_normal[t];
                  current_normal->node = NULL;
                  if (first_normal == NULL)
                     first_normal = current_normal;
                  else
                     previous_normal->node = current_normal;
                  previous_normal = current_normal;
               }
            }
         }
      }
   }

   triangle[sign] = first_triangle;
   if (smooth)
      normal[sign] = first_normal;
}

int pov_write(char *ofn, char *header, int sign, bool smooth, bool box, bool basis, int na, int ns, CARTESIAN sco, CARTESIAN *scv)
{
   const CARTESIAN cameralocation = {-0.1, -0.5, 0.9};
   const CARTESIAN cameralookat = {0.0, 0.0, 0.0};
   const double camerafactor = 2.0;

   bool flag;
   int i, j, s, ierr;
   double scscale, cameradistance, camerascale, axislength, axisradius, ratom, rbond, color[3];
   double xmin = INF9, ymin = INF9, zmin = INF9, xmax = -INF9, ymax = -INF9, zmax = -INF9, dx, dy, dz;
   char symbol[4];
   CARTESIAN sccenter, sccorner[8], cameralocationsc, cameralookatsc;
   TRIANGLELIST *current_triangle;
   NORMALLIST *current_normal;
   FILE *h;

   h = fopen(ofn, "w");
   if (h == NULL)
      return -1;

   sccenter.x = sco.x + scv[0].x * 0.5 + scv[1].x * 0.5 + scv[2].x * 0.5;
   sccenter.y = sco.y + scv[0].y * 0.5 + scv[1].y * 0.5 + scv[2].y * 0.5;
   sccenter.z = sco.z + scv[0].z * 0.5 + scv[1].z * 0.5 + scv[2].z * 0.5;
   for (i = 0; i < 8; i++)
   {
      sccorner[i].x = sccenter.x;
      sccorner[i].y = sccenter.y;
      sccorner[i].z = sccenter.z;
      for (j = 0; j < 3; j++)
      {
         sccorner[i].x += scv[j].x * (double(vertex[i][j]) - 0.5);
         sccorner[i].y += scv[j].y * (double(vertex[i][j]) - 0.5);
         sccorner[i].z += scv[j].z * (double(vertex[i][j]) - 0.5);
      }
   }

   scscale = 0.0;
   for (i = 0; i < 3; i++)
      scscale += sqrt(scv[i].x * scv[i].x + scv[i].y * scv[i].y + scv[i].z * scv[i].z);
   scscale = scscale / 3.0;
   cameradistance = sqrt((cameralocation.x - cameralookat.x) * (cameralocation.x - cameralookat.x) + (cameralocation.y - cameralookat.y) * (cameralocation.y - cameralookat.y) + (cameralocation.z - cameralookat.z) * (cameralocation.z - cameralookat.z));
   camerascale = camerafactor * scscale / cameradistance;
   cameralocationsc.x = sccenter.x / camerascale + cameralocation.x;
   cameralocationsc.y = sccenter.y / camerascale + cameralocation.y;
   cameralocationsc.z = sccenter.z / camerascale + cameralocation.z;
   cameralookatsc.x = sccenter.x / camerascale + cameralookat.x;
   cameralookatsc.y = sccenter.y / camerascale + cameralookat.y;
   cameralookatsc.z = sccenter.z / camerascale + cameralookat.z;

   fprintf(h, "\n");
   fprintf(h, "// %s\n",header);
   fprintf(h, "\n");
   fprintf(h, "#declare camera_location = <%.2f, %.2f, %.2f>;\n", cameralocationsc.x, cameralocationsc.y, cameralocationsc.z);
   fprintf(h, "#declare camera_look_at = <%.2f, %.2f, %.2f>;\n", cameralookatsc.x, cameralookatsc.y, cameralookatsc.z);
   fprintf(h, "#declare camera_scale = %.2f;\n", camerascale);
   fprintf(h, "#declare light_location = camera_location - camera_look_at;\n");
   fprintf(h, "#declare light_scale = 1e6;\n");
   fprintf(h, "#declare color_light = rgb <2.00, 2.00, 2.00>;\n");
   fprintf(h, "#declare color_background = rgb <0.00, 0.00, 0.00>;\n");
   if (box)
   {
      fprintf(h, "#declare radius_frame = 0.01;\n");
      fprintf(h, "#declare color_frame = rgb <0.75, 0.75, 0.75>;\n");
      fprintf(h, "#declare color_box = rgbf <1.00, 1.00, 1.00, 0.75>;\n");
   }
   if (basis)
   {
      axislength = scscale / 3.0;
      axisradius = axislength / 50.0;
      fprintf(h, "#declare length_axis = %.2f;\n", axislength);
      fprintf(h, "#declare radius_axis = %.2f;\n", axisradius);
      fprintf(h, "#declare color_axis_x = rgb <1.00, 0.00, 0.00>;\n");
      fprintf(h, "#declare color_axis_y = rgb <0.00, 1.00, 0.00>;\n");
      fprintf(h, "#declare color_axis_z = rgb <0.00, 0.00, 1.00>;\n");
      fprintf(h, "#declare length_arrow = 0.2 * length_axis;\n");
      fprintf(h, "#declare radius_arrow = 2.0 * radius_axis;\n");
      fprintf(h, "#declare color_arrow_x = color_axis_x;\n");
      fprintf(h, "#declare color_arrow_y = color_axis_y;\n");
      fprintf(h, "#declare color_arrow_z = color_axis_z;\n");
   }
   for (i = 0; i < ns; i++)
   {
      strncpy(symbol, periodic_table[cs[i]].symbol, 4);
      symbol[3] = '\0';
      lowercase(symbol);
      ratom = periodic_table[cs[i]].rvdw * ATOMSIZE;
      rbond = BONDRADIUS;
      for (j = 0; j < 3; j++)
         color[j] = periodic_table[cs[i]].color[j];
      fprintf(h, "#declare radius_atom_%s = %.2f;\n", symbol, ratom);
      if (cs[i] != 0)
         fprintf(h, "#declare radius_bond_%s = %.2f;\n", symbol, rbond);
      fprintf(h, "#declare color_atom_%s = rgb <%.2f, %.2f, %.2f>;\n", symbol, color[0], color[1], color[2]);
      if (cs[i] != 0)
         fprintf(h, "#declare color_bond_%s = rgb <%.2f, %.2f, %.2f>;\n", symbol, color[0], color[1], color[2]);
   }
   if (sign == 2)
   {
      fprintf(h, "#declare color_isosurface_positive = rgbf <0.40, 0.40, 0.80, 0.75>;\n");
      fprintf(h, "#declare color_isosurface_negative = rgbf <0.80, 0.40, 0.40, 0.75>;\n");
   }
   else
      fprintf(h, "#declare color_isosurface = rgbf <0.40, 0.80, 0.40, 0.75>;\n");
   fprintf(h, "\n");
   fprintf(h, "camera { location camera_location sky <0.00, 0.00, 1.00> up <0.00, 0.00, 1.00> right <-1.33, 0.00, 0.00> direction <0.00, -1.00, 0.00> look_at camera_look_at scale camera_scale }\n");
   fprintf(h, "light_source { light_location color color_light shadowless scale light_scale }\n");
   fprintf(h, "background { color color_background }\n");
   fprintf(h, "\n");
   if (box)
   {
      fprintf(h, "union {\n");
      for (i = 0; i < 8; i++)
         fprintf(h, "sphere { <%.9f, %.9f, %.9f>, radius_frame }\n", sccorner[i].x, sccorner[i].y, sccorner[i].z);
      for (i = 0; i < 12; i++)
         fprintf(h, "cylinder { <%.9f, %.9f, %.9f>, <%.9f, %.9f, %.9f>, radius_frame }\n", sccorner[edge[i][0]].x, sccorner[edge[i][0]].y, sccorner[edge[i][0]].z, sccorner[edge[i][1]].x, sccorner[edge[i][1]].y, sccorner[edge[i][1]].z);
      fprintf(h, "texture { pigment { color color_frame } }\n");
      fprintf(h, "}\n");
      fprintf(h, "\n");
      fprintf(h, "union {\n");
      for (i = 0; i < 12; i++)
         fprintf(h, "triangle { <%.9f, %.9f, %.9f>, <%.9f, %.9f, %.9f>, <%.9f, %.9f, %.9f> }\n", sccorner[face[i][0]].x, sccorner[face[i][0]].y, sccorner[face[i][0]].z, sccorner[face[i][1]].x, sccorner[face[i][1]].y, sccorner[face[i][1]].z, sccorner[face[i][2]].x, sccorner[face[i][2]].y, sccorner[face[i][2]].z);
      fprintf(h, "texture { pigment { color color_box } }\n");
      fprintf(h, "}\n");
      fprintf(h, "\n");
   }
   if (basis)
   {
      fprintf(h, "union {\n");
      fprintf(h, "cylinder { <0.00, 0.00, 0.00>, <length_axis, 0.00, 0.00>, radius_axis texture { pigment { color color_axis_x } } }\n");
      fprintf(h, "cylinder { <0.00, 0.00, 0.00>, <0.00, length_axis, 0.00>, radius_axis texture { pigment { color color_axis_y } } }\n");
      fprintf(h, "cylinder { <0.00, 0.00, 0.00>, <0.00, 0.00, length_axis>, radius_axis texture { pigment { color color_axis_z } } }\n");
      fprintf(h, "cone { <length_axis, 0.00, 0.00>, radius_arrow <length_axis + length_arrow, 0.00, 0.00>, 0.00 texture { pigment { color color_arrow_x } } }\n");
      fprintf(h, "cone { <0.00, length_axis, 0.00>, radius_arrow <0.00, length_axis + length_arrow, 0.00>, 0.00 texture { pigment { color color_arrow_y } } }\n");
      fprintf(h, "cone { <0.00, 0.00, length_axis>, radius_arrow <0.00, 0.00, length_axis + length_arrow>, 0.00 texture { pigment { color color_arrow_z } } }\n");
      fprintf(h, "}\n");
      fprintf(h, "\n");
   }
   fprintf(h, "union {\n");
   for (i = 0; i < na; i++)
   {
      strncpy(symbol, periodic_table[as[i]].symbol, 4);
      symbol[3] = '\0';
      lowercase(symbol);
      fprintf(h, "sphere { <%.9f, %.9f, %.9f>, radius_atom_%s texture { pigment { color color_atom_%s } } }\n", ap[i].x, ap[i].y, ap[i].z, symbol, symbol);
   }
   fprintf(h, "}\n");
   fprintf(h, "\n");
   fprintf(h, "union {\n");
   for (i = 0; i < na; i++)
   {
      strncpy(symbol, periodic_table[as[i]].symbol, 4);
      symbol[3] = '\0';
      lowercase(symbol);
      for (j = 0; j < nb[i]; j++)
         fprintf(h, "cylinder { <%.9f, %.9f, %.9f>, <%.9f, %.9f, %.9f>, radius_bond_%s texture { pigment { color color_bond_%s } } }\n", ap[i].x, ap[i].y, ap[i].z, bonds[i][j].x, bonds[i][j].y, bonds[i][j].z, symbol, symbol);
   }
   fprintf(h, "}\n");
   fprintf(h, "\n");
   for (s = 0; s < 2; s++)
   {
      current_triangle = triangle[s];
      if (smooth)
         current_normal = normal[s];
      if (current_triangle != NULL)
      {
         fprintf(h, "mesh {\n");
         do
         {
            for (i = 0; i < 3; i++)
            {
               if (current_triangle->triangle.point[i].x < xmin) xmin = current_triangle->triangle.point[i].x;
               if (current_triangle->triangle.point[i].x > xmax) xmax = current_triangle->triangle.point[i].x;
               if (current_triangle->triangle.point[i].y < ymin) ymin = current_triangle->triangle.point[i].y;
               if (current_triangle->triangle.point[i].y > ymax) ymax = current_triangle->triangle.point[i].y;
               if (current_triangle->triangle.point[i].z < zmin) zmin = current_triangle->triangle.point[i].z;
               if (current_triangle->triangle.point[i].z > zmax) zmax = current_triangle->triangle.point[i].z;
            }
            if (smooth)
               flag = fabs(current_normal->normal.vector[0].x) + fabs(current_normal->normal.vector[0].y) + fabs(current_normal->normal.vector[0].z) < EPS9 || fabs(current_normal->normal.vector[1].x) + fabs(current_normal->normal.vector[1].y) + fabs(current_normal->normal.vector[1].z) < EPS9 || fabs(current_normal->normal.vector[2].x) + fabs(current_normal->normal.vector[2].y) + fabs(current_normal->normal.vector[2].z) < EPS9;
            else
               flag = true;
            if (flag)
               fprintf(h, "triangle { <%.9f, %.9f, %.9f>, <%.9f, %.9f, %.9f>, <%.9f, %.9f, %.9f> }\n", current_triangle->triangle.point[0].x, current_triangle->triangle.point[0].y, current_triangle->triangle.point[0].z, current_triangle->triangle.point[1].x, current_triangle->triangle.point[1].y, current_triangle->triangle.point[1].z, current_triangle->triangle.point[2].x, current_triangle->triangle.point[2].y, current_triangle->triangle.point[2].z);
            else
               fprintf(h, "smooth_triangle { <%.9f, %.9f, %.9f>, <%.9f, %.9f, %.9f>, <%.9f, %.9f, %.9f>, <%.9f, %.9f, %.9f>, <%.9f, %.9f, %.9f>, <%.9f, %.9f, %.9f> }\n", current_triangle->triangle.point[0].x, current_triangle->triangle.point[0].y, current_triangle->triangle.point[0].z, current_normal->normal.vector[0].x, current_normal->normal.vector[0].y, current_normal->normal.vector[0].z, current_triangle->triangle.point[1].x, current_triangle->triangle.point[1].y, current_triangle->triangle.point[1].z, current_normal->normal.vector[1].x, current_normal->normal.vector[1].y, current_normal->normal.vector[1].z, current_triangle->triangle.point[2].x, current_triangle->triangle.point[2].y, current_triangle->triangle.point[2].z, current_normal->normal.vector[2].x, current_normal->normal.vector[2].y, current_normal->normal.vector[2].z);
            current_triangle = current_triangle->node;
            if (smooth)
               current_normal = current_normal->node;
         }
         while (current_triangle != NULL);
         if (sign == 2)
         {
            if (s == 0)
               fprintf(h, "texture { pigment { color color_isosurface_positive } }\n");
            else
               fprintf(h, "texture { pigment { color color_isosurface_negative } }\n");
         }
         else
            fprintf(h, "texture { pigment { color color_isosurface } }\n");
         fprintf(h, "}\n");
         fprintf(h, "\n");
      }
   }

   ierr = fclose(h);
   if (ierr != 0)
      return -1;

   xmin = xmin / BOHR;
   xmax = xmax / BOHR;
   ymin = ymin / BOHR;
   ymax = ymax / BOHR;
   zmin = zmin / BOHR;
   zmax = zmax / BOHR;
   dx = xmax - xmin;
   dy = ymax - ymin;
   dz = zmax - zmin;
   printf("\n the minimal bounding box for the isosurface\n\n xmin = %.9f xmax = %.9f dx = %.9f bohr\n ymin = %.9f ymax = %.9f dy = %.9f bohr\n zmin = %.9f zmax = %.9f dz = %.9f bohr\n\n", xmin, xmax, dx, ymin, ymax, dy, zmin, zmax, dz);

   return 0;
}

int terminate(int ierr, int na, int ni, int nj)
{
   int i, j, s;
   TRIANGLELIST *current_triangle, *next_triangle;
   NORMALLIST *current_normal, *next_normal;

   if (as != NULL) delete [] as;
   if (ap != NULL) delete [] ap;
   if (uas != NULL) delete [] uas;
   if (uap != NULL) delete [] uap;
   if (cs != NULL) delete [] cs;
   if (nb != NULL) delete [] nb;

   if (bonds != NULL)
   {
      for (i = 0; i < na; i++)
         delete [] bonds[i];
      delete [] bonds;
   }

   if (sci != NULL)
   {
      for (i = 0; i < na; i++)
         delete [] sci[i];
      delete [] sci;
   }

   if (scalar != NULL)
   {
      for (i = 0; i < ni; i++)
      for (j = 0; j < nj; j++)
         delete [] scalar[i][j];
      for (i = 0; i < ni; i++)
         delete [] scalar[i];
      delete [] scalar;
   }

   if (vector != NULL)
   {
      for (i = 0; i < ni; i++)
      for (j = 0; j < nj; j++)
         delete [] vector[i][j];
      for (i = 0; i < ni; i++)
         delete [] vector[i];
      delete [] vector;
   }

   for (s = 0; s < 2; s++)
   {
      current_triangle = triangle[s];
      while (current_triangle != NULL)
      {
         next_triangle = current_triangle->node;
         delete current_triangle;
         current_triangle = next_triangle;
      }
   }

   for (s = 0; s < 2; s++)
   {
      current_normal = normal[s];
      while (current_normal != NULL)
      {
         next_normal = current_normal->node;
         delete current_normal;
         current_normal = next_normal;
      }
   }

   return ierr;
}

int main(int argc, char *argv[])
{
   bool smooth, box, basis, uc, sc, sct;
   int sign, power, algorithm, ucf, scf, ns, una;
   int scmin[3], scmax[3];
   int ierr = 0, na = 0, ni = 0, nj = 0, nk = 0;
   double isovalue;
   char pfn[MAXCHAR], ifn[MAXCHAR], iff[MAXCHAR], ofn[MAXCHAR], off[MAXCHAR], header[MAXCHAR];
   CARTESIAN sfo, sfv[3], sfs[3], uco, ucv[3], sco, scv[3];

   if (argc != 2)
   {
      fprintf(stderr, "\nUsage: surface.x surface.inp\n\n");
      return terminate(-1, na, ni, nj);
   }
   strncpy(pfn, argv[1], MAXCHAR);
   pfn[MAXCHAR - 1] = '\0';

   ierr = par_read(pfn, header, ifn, iff, ofn, off, &isovalue, &sign, &power, &algorithm, &smooth, &box, &basis, &uc, &uco, ucv, &ucf, &sc, &sco, scv, &scf, &sct);
   if (ierr != 0)
   {
      fprintf(stderr, "\nError: failed to read %s\n\n", pfn);
      return terminate(-1, na, ni, nj);
   }

   if (strcmp(iff, "cube") == 0)
      ierr = cub_read(ifn, &na, &ni, &nj, &nk, &sfo, sfv, sfs);
   else if (strcmp(iff, "xsf") == 0)
      ierr = xsf_read(ifn, &na, &ni, &nj, &nk, &sfo, sfv, sfs);
   else
   {
      fprintf(stderr, "\nError: unrecognized input format %s\n\n", iff);
      return terminate(-1, na, ni, nj);
   }
   if (ierr != 0)
   {
      fprintf(stderr, "\nError: failed to read %s\n\n", ifn);
      return terminate(-1, na, ni, nj);
   }

   cell_set(sfo, sfv, uc, &uco, ucv, ucf, sc, &sco, scv, scf);

   if (sc)
   {
      una = na;
      na = atom_clone(na, ucv, sco, scv, scmin, scmax);
      ierr = scalar_clone(&ni, &nj, &nk, &sfo, sfs, uco, ucv, sco, scv);
   }
   if (na < 0 || ierr != 0)
   {
      fprintf(stderr, "\nError: failed to build supercell\n\n");
      return terminate(-1, na, ni, nj);
   }

   if (smooth)
      scalar_normal(ni, nj, nk, sfs);

   ns = chemical_species(na);

   if (sct || !sc)
      ierr = bonds_st(na, scv);
   else
      ierr = bonds_ut(na, una, ucv, scmin, scmax);
   if (ierr != 0)
   {
      fprintf(stderr, "\nError: failed to build interatomic bonds\n\n");
      return terminate(-1, na, ni, nj);
   }

   isovalue = isovalue_scale(ni, nj, nk, power, isovalue);
   if (sign == 0 || sign == 2)
      polygonise(ni, nj, nk, sfo, sfs, isovalue, algorithm, smooth, 0);
   isovalue = -isovalue;
   if (sign == 1 || sign == 2)
      polygonise(ni, nj, nk, sfo, sfs, isovalue, algorithm, smooth, 1);

   if (strcmp(off, "povray") == 0)
      ierr = pov_write(ofn, header, sign, smooth, box, basis, na, ns, sco, scv);
   else
   {
      fprintf(stderr, "\nError: unrecognized output format %s\n\n", off);
      return terminate(-1, na, ni, nj);
   }
   if (ierr != 0)
   {
      fprintf(stderr, "\nError: failed to write %s\n\n", ofn);
      return terminate(-1, na, ni, nj);
   }

   return terminate(0, na, ni, nj);
}

