!==========================================================================
!
! Routines:
!
! (1) inread()  Originally By (SIB)             Last Modified 5/9/2008 (JRD)
!
!     Reads the input file and sets various values
!
!       ecuts is the epsilon_cutoff
!       cwfn%nband is number_bands
!       cwfn%band_index() gets setup from band_occupation
!       pol%qpt(1:3,1:pol%nq) is the q vector information
!
!       pol%icutv to 0 if use no truncation
!
!       pol%icutv to 2 if use spherical truncation
!          transval(1) means r cut
!
!       pol%icutv to 4 if use cell wire truncation
!       pol%icutv to 5 if use cell box truncation
!       pol%icutv to 6 if use cell slab truncation
!
!==============================================================================

#include "f_defs.h"

subroutine inread(pol,vwfn,cwfn)

  use global_m
  use scissors_m
  use inread_common_m
  implicit none

  type (polarizability), intent(out) :: pol
  type (valence_wfns), intent(out) :: vwfn
  type (conduction_wfns), intent(out) :: cwfn
  
  character*256 :: blockword,keyword,line,errmsg
  integer :: ii,itestq,jj,nqpt_read,iostat
  
  real(DP) :: div,qpt_read(3,MAX_KPTS)
  integer :: band_occ(MAX_BANDS), ifreqCounter
  real(DP) :: tmpFreq, freqStep
  logical :: funny_occs, found, on_steroids

! for default settings for full-frequency calculation(Adler-Wiser)
  integer :: flag_DeltaFreq, flag_FreqStepIncrease, flag_FreqCutoff2  
! for default settings for full-frequency calculation(Shishkin-Kresse)
  integer :: flag_InitSFreq, flag_DeltaSFreq, flag_SFreqStepIncrease
  integer :: flag_SFreqCutoff1, flag_SFreqCutoff2
  character(len=9) :: str_intra
!-----------------------------
! Set default values

  PUSH_SUB(inread)
  
#ifdef MPI
  ! Non-root nodes should wait for root to read the whole file.
  ! That way, we can be sure root gets a chance to write errors before
  ! any die call is issued by another node. Root calls MPI_Barrier below.
  if(peinf%inode /= 0) call MPI_Barrier(MPI_COMM_WORLD, mpierr)
#endif

  nqpt_read=0
  pol%nq=0
  pol%nq0=0
  pol%nq1=0
  pol%subsample=.false.
  band_occ=0
  call scissors_zero(pol%scis)
  vwfn%nband=0
  cwfn%nband=0
  pol%freq_dep=0
  pol%freq_dep_method=0
  pol%nFreq=1
  pol%dInitFreq=0.0d0
  pol%dDeltaFreq=0.0d0
  pol%dFreqStepIncrease=0.0d0 
  pol%dFreqCutoff1=0.0d0
  pol%dFreqCutoff2=0.0d0
  pol%dBrdning=0.0d0
  pol%stop_after_qpt=-1

  flag_DeltaFreq=0
  flag_FreqStepIncrease=0
  flag_FreqCutoff2=0
  
  pol%nSFreq=1
  pol%dInitSFreq=0.0d0
  pol%dDeltaSFreq=0.0d0
  pol%dSFreqStepIncrease=0.0d0
  pol%dSFreqCutoff1=0.0d0
  pol%dSFreqCutoff2=0.0d0

  flag_InitSFreq=0
  flag_DeltaSFreq=0
  flag_SFreqStepIncrease=0
  flag_SFreqCutoff1=0
  flag_SFreqCutoff2=0  

  pol%fullConvLog=0
  pol%icutv=0
  pol%iwritecoul=0
  pol%truncval(1)=0.0d0
  pol%truncval(2)=0.0d0
  pol%truncval(3)=0.0d0
  pol%ecuts=0.0d0
  pol%valueq0=0
  pol%iqexactlyzero=0
  pol%ncrit=0
#ifdef HDF5
  pol%use_hdf5 = .true.
#else
  pol%use_hdf5 = .false.
#endif
  pol%efermi_input=0.0d0
  pol%rfermi=.true.
  pol%gcomm=-1
  pol%os_opt_ffts=0
  pol%restart=.false.
  pol%min_fftgrid=.true.
  pol%lin_denominator=0d0
  pol%os_para_freqs=1
  pol%os_hdf5=.false.
  pol%iwriteint=1
  pol%skip_epsilon=.false.
  pol%skip_chi=.false.
  pol%freplacebz=.false.
  pol%fwritebz=.false.
  pol%degeneracy_check_override=.false.
  peinf%npools=0
  pol%eqp_corrections=.false.
  pol%intraband_flag=0
  pol%intraband_overlap_min=0.9d0
  pol%patched_sampling=.false.
      
!----------------- Never ending loop ---------------------------------------

! Actually the loop ends when the end of the file is reached


  do while(0.eq.0)

    read(55,'(a256)',iostat=iostat) line
    if(iostat < 0) exit

! Skip comment lines

    if(len_trim(line).eq.0) cycle
    if(line(1:1).eq.'#') cycle

! Determine keyword:

    keyword=line(1:scan(line," ")-1)
    line=adjustl(line(scan(line," ")+1:256))

! SIB: If we have a 'begin', then in scan the information up to 'end'
! For now, only 'qpoints' is recognized

    if(trim(keyword).eq.'begin') then
      blockword=line(1:scan(line," ")-1)
      ii=0
      do while(trim(line).ne.'end')
        read(55,'(a256)',end=105) line
        if(trim(line).ne.'end') then
          ii=ii+1
          call check_bounds_nkq(ii, 'q', 'begin qpoints')
          if(trim(blockword).eq.'qpoints') then
            read(line,*,iostat=iostat) (qpt_read(jj,ii),jj=1,3), div,itestq
            if(iostat /= 0) then
              write(errmsg,'(3a)') 'Unexpected characters were found while reading elements of the ', &
                trim(blockword),' block.'
              call die(errmsg, only_root_writes = .true.)
            endif
            if (itestq.ne.0) then
              if(itestq /= 1 .and. itestq /= 2) &
                call die("Illegal value for last column in qpoints. May only be 0, 1, 2.", only_root_writes = .true.)
              pol%nq0 = pol%nq0 + 1
              if (pol%valueq0/=0 .and. itestq/=pol%valueq0) &
                call die("All q->0 points must have the same value for last column.", only_root_writes=.true.)
              pol%valueq0=itestq
              if (all(abs(qpt_read(:,ii))<TOL_Zero)) then
                pol%iqexactlyzero = 1
                if (peinf%inode .eq. 0) write(6,921)
              endif
            endif
            qpt_read(1:3,ii)=qpt_read(1:3,ii)/div
          else
            write(errmsg,'(3a)') 'Unexpected blockword ', trim(blockword), ' was found in epsilon.inp.'
            call die(errmsg, only_root_writes = .true.)
          end if
        end if
      end do
      if(trim(blockword).eq.'qpoints') then
        nqpt_read=ii
      endif

! SIB: Other keywords than 'begin'

    elseif(trim(keyword).eq.'frequency_dependence') then
      read(line,*,err=110) pol%freq_dep
    elseif(trim(keyword).eq.'frequency_dependence_method') then
      read(line,*,err=110) pol%freq_dep_method
    elseif(trim(keyword).eq.'init_frequency') then
      read(line,*,err=110) pol%dInitFreq
    elseif(trim(keyword).eq.'delta_frequency') then
      read(line,*,err=110) pol%dDeltaFreq
      flag_DeltaFreq=1
    elseif(trim(keyword).eq.'delta_frequency_step') then
      read(line,*,err=110) pol%dFreqStepIncrease
      flag_FreqStepIncrease=1
    elseif(trim(keyword).eq.'frequency_low_cutoff') then
      read(line,*,err=110) pol%dFreqCutoff1
    elseif(trim(keyword).eq.'frequency_high_cutoff') then
      read(line,*,err=110) pol%dFreqCutoff2
      flag_FreqCutoff2=1
    elseif(trim(keyword).eq.'broadening') then
      read(line,*,err=110) pol%dBrdning
    elseif(trim(keyword).eq.'init_sfrequency') then
      read(line,*,err=110) pol%dInitSFreq
      flag_InitSFreq=1
    elseif(trim(keyword).eq.'delta_sfrequency') then
      read(line,*,err=110) pol%dDeltaSFreq
      flag_DeltaSFreq=1
    elseif(trim(keyword).eq.'delta_sfrequency_step') then
      read(line,*,err=110) pol%dSFreqStepIncrease
      flag_SFreqStepIncrease=1
    elseif(trim(keyword).eq.'sfrequency_low_cutoff') then
      read(line,*,err=110) pol%dSFreqCutoff1
      flag_SFreqCutoff1=1
    elseif(trim(keyword).eq.'sfrequency_high_cutoff') then
      read(line,*,err=110) pol%dSFreqCutoff2
      flag_SFreqCutoff2=1
    elseif(trim(keyword).eq.'full_chi_conv_log') then
      read(line,*,err=110) pol%fullConvLog
    elseif(trim(keyword).eq.'number_qpoints') then
      read(line,*,err=110) pol%nq ! FHJ: deprecated
    elseif(trim(keyword).eq.'number_valence_pools') then
      read(line,*,err=110) peinf%npools
    elseif(trim(keyword).eq.'skip_epsilon') then
      pol%skip_epsilon=.true.
    elseif(trim(keyword).eq.'skip_chi') then
      pol%skip_chi=.true.
    elseif(trim(keyword).eq.'spherical_truncation') then
      pol%icutv=2
    elseif(trim(keyword).eq.'cell_wire_truncation') then
      pol%icutv=4
    elseif(trim(keyword).eq.'cell_box_truncation') then
      pol%icutv=5
    elseif(trim(keyword).eq.'cell_slab_truncation') then
      pol%icutv=6
    elseif(trim(keyword).eq.'comm_mpi') then
      pol%iwriteint=1
    elseif(trim(keyword).eq.'comm_disk') then
      pol%iwriteint=0
    elseif(trim(keyword).eq.'gcomm_elements') then
      pol%gcomm=0
    elseif(trim(keyword).eq.'gcomm_matrix') then
      pol%gcomm=-1
    elseif(trim(keyword).eq.'dont_use_hdf5') then
      pol%use_hdf5 = .false.
    elseif(trim(keyword).eq.'no_min_fftgrid') then
      pol%min_fftgrid = .false.
    elseif(trim(keyword).eq.'restart') then
      pol%restart = .true.
    elseif(trim(keyword).eq.'stop_after_qpt') then
      read(line,*,err=110) pol%stop_after_qpt
    elseif(trim(keyword).eq.'write_vcoul') then
      pol%iwritecoul=1
    elseif(trim(keyword).eq.'coulomb_truncation_x') then
      read(line,*,err=110) pol%truncval(1)
    elseif(trim(keyword).eq.'coulomb_truncation_radius') then
      read(line,*,err=110) pol%truncval(1)
    elseif(trim(keyword).eq.'epsilon_cutoff') then
      read(line,*,err=110) pol%ecuts
    elseif(trim(keyword).eq.'number_bands') then
      read(line,*,err=110) cwfn%nband
      call check_bounds_nbands(cwfn%nband, 'number_bands')
    elseif(trim(keyword).eq.'wavefunction_cutoff') then
      if(peinf%inode.eq.0) then 
        write(0,123) 
      endif
123   format(1x,"WARNING: Found keyword wavefunction_cutoff in the input file.",/,&
        10x,"This option is obsolete, use program wfnreduce instead.",/)
    elseif(trim(keyword).eq.'band_occupation') then
      read(line,*,err=110) (band_occ(ii),ii=1,cwfn%nband)
    elseif(trim(keyword).eq.'number_partial_occup') then
      read(line,*,err=110) pol%ncrit
    elseif(trim(keyword).eq.'ncrit') then
      if(peinf%inode.eq.0) then 
        write(0,124) 
      endif
124   format(1x,"WARNING: Found keyword ncrit in the input file.",/,&
        10x,"This option is deprecated, use keyword number_partial_occup instead.",/)
      read(line,*,err=110) pol%ncrit
    elseif(trim(keyword).eq.'fermi_level') then
      read(line,*,err=110) pol%efermi_input
    elseif(trim(keyword).eq.'fermi_level_absolute') then
      pol%rfermi=.false.
    elseif(trim(keyword).eq.'fermi_level_relative') then
      pol%rfermi=.true.
    elseif(trim(keyword).eq.'fullbz_replace') then
      pol%freplacebz=.true.
    elseif(trim(keyword).eq.'fullbz_write') then
      pol%fwritebz=.true.
    elseif(trim(keyword).eq.'degeneracy_check_override') then
      pol%degeneracy_check_override=.true.
    elseif(trim(keyword).eq.'eqp_corrections') then
      pol%eqp_corrections=.true.
    else
      call scissors_inread(keyword, line, pol%scis, found)
      if(.not. found) then
        write(errmsg,'(3a)') 'Unexpected keyword ', trim(keyword), ' was found in epsilon.inp.'
        call die(errmsg, only_root_writes = .true.)
      endif
    end if
  enddo

! End of Big Input Loop

  if ((pol%freq_dep==2).and.(pol%freq_dep_method==1)) then
    if (peinf%inode==0) then
      write(0,*)
      write(0,*) 'WARNING: Shishkin and Kresse method for polarizability is under development.'
      write(0,*) 'Please, double check your results against the Adler-Wiser formulation.'
      write(0,*)
    endif
    if (pol%gcomm==-1) then
      pol%gcomm=0
      if (peinf%inode==0) then
        write(0,*)
        write(0,*) 'WARNING: Shishkin and Kresse method for polarizability does not work with gcomm_matrix.'
        write(0,*) 'Changing communication method to gcomm_elements.'
        write(0,*)
      endif
    endif
  endif

! Default settings for full-frequency calculation
  if (pol%freq_dep .eq. 2) then
    if (flag_DeltaFreq .eq. 0) then
      pol%dDeltaFreq=pol%dBrdning
      if (peinf%inode==0) then
        write(6,'(1x,a,1x,f11.6)') 'Default: delta_frequency =', pol%dDeltaFreq 
      endif 
    endif
      
    if (flag_FreqStepIncrease .eq. 0) then
      pol%dFreqStepIncrease=1.0
      if (peinf%inode==0) then
        write(6,'(1x,a,1x,f11.6)') 'Default: delta_frequency_step =', pol%dFreqStepIncrease
      endif
    endif

    if (flag_FreqCutoff2 .eq. 0) then
      pol%dFreqCutoff2=4*pol%dFreqCutoff1
      if (peinf%inode==0) then        
        write(6,'(1x,a,1x,f11.6)') 'Default: frequency_high_cutoff =', pol%dFreqCutoff2
      endif
    endif

    if (pol%freq_dep_method .eq. 1) then
      if (flag_InitSFreq .eq. 0) then
        pol%dInitSFreq=0D0
        if (peinf%inode==0) then
          write(6,'(1x,a,1x,f11.6)') 'Default: init_sfrequency =', pol%dInitSFreq
        endif
      endif
   
      if (flag_DeltaSFreq .eq. 0) then
        pol%dDeltaSFreq=pol%dDeltaFreq
        if (peinf%inode==0) then
          write(6,'(1x,a,1x,f11.6)') 'Default: delta_sfrequency =', pol%dDeltaSFreq
        endif
      endif
   
      if (flag_SFreqStepIncrease .eq. 0) then
        pol%dSFreqStepIncrease=0D0
        if (peinf%inode==0) then
          write(6,'(1x,a,1x,f11.6)') 'Default: delta_sfrequency_step =', pol%dSFreqStepIncrease
        endif
      endif

      if (flag_SFreqCutoff1 .eq. 0) then
        pol%dSFreqCutoff1=1D0
        if (peinf%inode==0) then
          write(6,'(1x,a,1x,f11.6)') 'Default: sfrequency_low_cutoff =', pol%dSFreqCutoff1
        endif
      endif

      if (flag_SFreqCutoff2 .eq. 0) then
        pol%dSFreqCutoff2=pol%dFreqCutoff1
        if (peinf%inode==0) then
          write(6,'(1x,a,1x,f11.6)') 'Default: sfrequency_high_cutoff =', pol%dSFreqCutoff2
        endif
      endif
    endif !FF and pol%freq_dep_method .eq. 1 
  endif
!-----------------------------------------------------------------------

!------------ This is where we end up if the file was read successfully -----------

  if (.not.pol%use_hdf5) pol%restart = .false.
  if (peinf%inode==0) then
    if (pol%restart) then
      write(6,'(/,1x,a,/)') 'We will restart the calculation from the last finished q-point.'
    else
      write(6,'(/,1x,a,/)') 'We will start a new calculation from scratch.'
    endif
  endif

  if (pol%iwriteint==0) then
    ! FHJ: min_fftgrid doesn`t support COMM_DISK
    pol%min_fftgrid=.false.
  endif


  if ((pol%freq_dep .eq. 2) .and. abs(pol%dDeltaFreq) .gt. TOL_Zero) then
    !SAFE_ALLOCATE(pol%dFreqGrid,(pol%nFreq))
    !SAFE_ALLOCATE(pol%dFreqBrd,(pol%nFreq))
    !do jj=1,pol%nFreq
    !  pol%dFreqGrid(jj)=dble(jj-1)*pol%dDeltaFreq + pol%dInitFreq
    !  pol%dFreqBrd(jj)=pol%dBrdning*(0.0,1.0)
    !enddo 

    tmpFreq = pol%dInitFreq
    ifreqCounter = 0
    freqStep = pol%dDeltaFreq
    do while (tmpFreq .le. pol%dFreqCutoff2)
      ifreqCounter = ifreqCounter+1
      if (tmpFreq .lt. pol%dFreqCutoff1) then
        tmpFreq=tmpFreq+pol%dDeltaFreq
      else 
        freqstep = freqstep + pol%dFreqStepIncrease
        tmpFreq=tmpFreq+freqStep
      endif
    enddo
    
    pol%nFreq = iFreqCounter
    SAFE_ALLOCATE(pol%dFreqGrid,(pol%nFreq))
    SAFE_ALLOCATE(pol%dFreqBrd,(pol%nFreq))
    
    tmpFreq = pol%dInitFreq
    ifreqCounter = 0
    freqStep = pol%dDeltaFreq
    do while (tmpFreq .le. pol%dFreqCutoff2)
      ifreqCounter = ifreqCounter+1
      pol%dFreqGrid(iFreqCounter)=tmpFreq
      pol%dFreqBrd(iFreqCounter)=pol%dBrdning*(0.0,1.0)
      
      if (tmpFreq .lt. pol%dFreqCutoff1) then
        tmpFreq=tmpFreq+pol%dDeltaFreq
      else 
        freqstep = freqstep + pol%dFreqStepIncrease
        tmpFreq=tmpFreq+freqStep
      endif
    enddo
  
    if(pol%freq_dep_method .eq. 1) then

      tmpFreq = pol%dInitSFreq
      ifreqCounter = 0
      freqStep = pol%dDeltaSFreq
      do while (tmpFreq .le. pol%dSFreqCutoff2)
        ifreqCounter = ifreqCounter+1
        if (tmpFreq .lt. pol%dSFreqCutoff1) then
          tmpFreq=tmpFreq+pol%dDeltaSFreq
        else
          freqstep = freqstep + pol%dSFreqStepIncrease
          tmpFreq=tmpFreq+freqStep
        endif
      enddo

      pol%nSFreq = ifreqCounter
      if (peinf%inode.eq.0) then 
        print*, 'pol%nSFreq',pol%nSFreq
      endif

      SAFE_ALLOCATE(pol%dSFreqGrid,(pol%nSFreq))

      tmpFreq = pol%dInitSFreq
      ifreqCounter = 0
      freqStep = pol%dDeltaSFreq
      do while (tmpFreq .le. pol%dSFreqCutoff2)
        ifreqCounter = ifreqCounter+1
        pol%dSFreqGrid(ifreqCounter)=tmpFreq

        if (tmpFreq .lt. pol%dSFreqCutoff1) then
          tmpFreq=tmpFreq+pol%dDeltaSFreq
        else
          freqstep = freqstep + pol%dSFreqStepIncrease
          tmpFreq=tmpFreq+freqStep
        endif
      enddo
    endif
  else if ((pol%freq_dep .eq. 2) .and. abs(pol%dDeltaFreq) .le. TOL_Zero) then
    call die("Illegal value for Delta Frequency in full frequency calculation")
  else
    pol%nFreq=1
    SAFE_ALLOCATE(pol%dFreqGrid,(pol%nFreq))
    SAFE_ALLOCATE(pol%dFreqBrd,(pol%nFreq))  
    pol%dFreqGrid = 0D0
    pol%dFreqBrd = 0D0
  endif

  if (peinf%inode==0 .and. pol%nq>0) then
    write(0,'(/,a)') 'WARNING: the `number_qpoints` flag is deprecated. The code now'
    write(0,'(a,/)') 'automatically figures out the number of q-points from the input.'
  endif
  pol%nq = nqpt_read

  pol%nq1 = pol%nq - pol%nq0
  if (.not.pol%subsample.and.pol%nq0>1) then
    call die('Can only have one q->0 point', only_root_writes=.true.)
  endif
  
  ! SIB: allocate polarizability q-points array pol%qpt(3,1:pol:nq)
  ! and copy what was read from qpt_read into it
  SAFE_ALLOCATE(pol%qpt, (3,pol%nq))
  pol%qpt(1:3,1:pol%nq)=qpt_read(1:3,1:pol%nq)
  
  ! SIB: check for bad input
  if(abs(pol%ecuts).lt.TOL_Zero) then
    call die("The epsilon_cutoff keyword could not be found.", only_root_writes = .true.)
  endif

  ! FHJ: Make sure the user entered a valid number of bands
  call check_consistency_nbands(cwfn%nband, .true.)

! SIB: Allocate cwfn%band_index(cwfn%nband), which is ALL the bands,
! and initialize it.  By the end of this, band_index() will have indices
! pointing to the bands with the order being first all valence states
! (occupancy 1.0) and then all conduction states (occupancy 0.0)

  SAFE_ALLOCATE(cwfn%band_index, (cwfn%nband))
  vwfn%nband=0
  jj=0
  do ii=1,cwfn%nband
    vwfn%nband=vwfn%nband+band_occ(ii)
    if(band_occ(ii).eq.1) then
      jj=jj+1
      cwfn%band_index(ii)=jj
    endif
  enddo
  if(vwfn%nband == 0 .and. pol%ncrit == 0) &
    call die("There are no occupied or partially occupied bands.", only_root_writes = .true.)
  if(vwfn%nband == cwfn%nband) &
    call die("There are no unoccupied bands.", only_root_writes = .true.)
  do ii=1,cwfn%nband
    if(band_occ(ii).eq.0) then
      jj=jj+1
      cwfn%band_index(ii)=jj
    endif
  enddo

  funny_occs = .false.
  do ii=1,cwfn%nband
    if(cwfn%band_index(ii) /= ii) funny_occs = .true.
  enddo
  if(funny_occs) call die("Non-equilibrium occupations not working.", only_root_writes = .true.)
  ! The problem is that the v-c pairs are weighted based on inferred occupations from E_F in mtxel, ignoring these. --DAS

!  if(funny_occs .and. peinf%inode == 0) then
!    write(0,'(a)') 'WARNING: degeneracy check will be incorrect if non-equilibrium occupations are used.'
!  endif

  if(pol%fullConvLog.lt.-1.or.pol%fullConvLog.gt.2) then
    call die('Invalid full_chi_conv_log', only_root_writes = .true.)
  endif
  
  if(pol%skip_epsilon .and. pol%skip_chi) then
    call die('Cannot skip_epsilon and skip_chi', only_root_writes = .true.)
  endif

  ! FHJ: will we need to read at least one q-point from WFNq?
  pol%need_WFNq = (pol%nq0>0.and.pol%valueq0==1.and.pol%iqexactlyzero==0).or.pol%patched_sampling

! gsm: What frequency dependence we are using?

  if(peinf%inode.eq.0) then
    if(pol%freq_dep.eq.0) then
      write(6,700)
      write(7,700)
    elseif((pol%freq_dep.eq.2).and.(pol%freq_dep_method.eq.0)) then
      write(6,702)
      write(7,702)
    elseif((pol%freq_dep.eq.2).and.(pol%freq_dep_method.eq.1)) then
      write(6,703)
      write(7,703)
    elseif(pol%freq_dep.eq.3) then
      write(6,704)
      write(7,704)
    else
      call die('Need to specify frequency dependence', only_root_writes = .true.)
    endif
  endif
700 format(1x,'Computing the static inverse dielectric matrix',/)
702 format(1x,'Computing the full frequency-dependent inverse dielectric matrix (Adler-Wiser formulation)',/)
703 format(1x,'Computing the full frequency-dependent inverse dielectric matrix (Shishkin-Kresse formulation)',/)
704 format(1x,'Computing the inverse dielectric matrix for two purely imaginary frequencies (Godby-Needs formulation)',/)

! JRD: What Communication Scheme are we using?
! We should make a better default choice based on 
! whether nv*nc*nk(reduced) > nmtx*nfreq
  
  if (peinf%inode.eq.0) then
    if (pol%gcomm .eq. -1) then
      write(6,801)
      write(7,801)
    else
      write(6,802)
      write(7,802)
    endif
  endif
801 format(1x,'We are using matrix communication scheme',/)
802 format(1x,'We are using element communication scheme',/)

  if(peinf%inode == 0) then
    if(pol%iwriteint == 1) then ! comm_mpi
      if(peinf%npes > 1) then
        write(6,803)
        write(7,803)
      else
        write(6,805)
        write(7,805)
      endif
    else ! comm_disk
      write(6,804)
      write(7,804)
    endif
  endif
803 format(1x,'We are communicating via MPI',/)
804 format(1x,'We are communicating via disk',/)
805 format(1x,'We are not communicating',/)

! JRD: What truncation scheme are we using?

  if(peinf%inode.eq.0) then
    if(pol%icutv.eq.0) then
      write(6,900)
      write(7,900)
    elseif(pol%icutv.eq.2) then
      write(6,902)
      write(7,902)
      write(6,912)pol%truncval(1)
      write(7,912)pol%truncval(1)
    elseif(pol%icutv.eq.4) then
      write(6,904)
      write(7,904)
    elseif(pol%icutv.eq.5) then
      write(6,905)
      write(7,905)
    elseif(pol%icutv.eq.6) then
      write(6,906)
      write(7,906)
    endif
  endif
  900 format(1x,'We are using no truncation',/)
  902 format(1x,'We are using a truncated Coulomb interaction (spherical)')
  904 format(1x,'We are using a truncated Coulomb interaction: Cell Wire',/)
  905 format(1x,'We are using a truncated Coulomb interaction: Cell Box',/)
  906 format(1x,'We are using a truncated Coulomb interaction: Cell Slab',/)
  912 format(1x,'r_cut = ',f10.6,' Bohr'/)
  921 format(1x,'Doing q exactly zero',/)

! JRD: Do we have partially occupied bands?

  if(peinf%inode.eq.0) then
    if(pol%ncrit.gt.0) then
      write(6,951)pol%ncrit
      write(7,951)pol%ncrit
951   format(1x,'We have partially occupied bands',/, &
        1x,'number_partial_occup (ncrit) =',i3,/)
    endif
  endif

  if (peinf%inode==0 .and. pol%intraband_flag/=0) then
    if (pol%intraband_flag<0 .or. pol%intraband_flag>2) &
      call die('Invalid value for flag `intraband_flag`', only_root_writes=.true.)
    if (pol%intraband_overlap_min<0d0 .or. pol%intraband_overlap_min>1d0) &
      call die('Invalid value for flag `intraband_overlap_min`', only_root_writes=.true.)
    if (pol%intraband_flag==1) then
      str_intra = 'intraband'
    else
      str_intra = 'interband'
    endif
    write(6,'(/,1x,3a,f9.6,/)') 'Calculating only ', str_intra, &
      ' transitions, intraband_overlap_min = ', pol%intraband_overlap_min
    if (.not.pol%skip_epsilon) then
      write(0,'(/1x,3a)') 'WARNING: you are only calculating ', str_intra, ' transitions, but'
      write(0,'(1x,a)') 'you are not skipping epsilon. Remember that inverse dielectric matrices cannot'
      write(0,'(1x,a,/)') 'not be added toghether, only the polarizability!'
    endif
  endif

#ifdef MPI
  ! root lets the others go after it is done reading (see beginning of function)
  if(peinf%inode == 0) call MPI_Barrier(MPI_COMM_WORLD, mpierr)
#endif

  POP_SUB(inread)
  
  return
  
105 write(errmsg,'(3a)') 'The end of the file was reached while reading elements of the ', &
      trim(blockword),' block.'
  call die(errmsg, only_root_writes = .true.)
  
110 write(errmsg,'(3a)') 'Unexpected characters were found while reading the value for the keyword ', &
      trim(keyword), '. '
  call die(errmsg, only_root_writes = .true.)
  
end subroutine inread
