/*******************************************************************
*  Copyright (c) 1994-2008 Jetico, Inc., Finland
*  All rights reserved.
*
*  File:          wipe.c
*
*  Description:   implementation of BestCrypt wipe utility
*
*  Author:        Vitaliy Zolotarev
*
*  Created:       20-Nov-1999
*
*  Revision:      $Id: wipe.c 217 2008-08-28 07:09:58Z pav $
*
*
*******************************************************************/

#include <stdio.h>
#include <limits.h>
#include <stdlib.h>
#include <sys/types.h>
#include <dirent.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <string.h>
#include <utime.h>
#include <sys/ioctl.h>
#include <time.h>
#include <ctype.h>

#if (OS_AIX)
#define NAME_MAX FILENAME_MAX
#else
/*#include <sys/mount.h>*/
#endif

#include <time.h>
#include "wipe.h"
#include "options.h"
#include "schemes.h"
#include "sha1random.h"

#if (OS_NetBSD || OS_OpenBSD || OS_FreeBSD || OS_Darwin)
#define statvfs statfs
#define f_frsize f_bsize
#define lseek64 lseek
#else
#include <sys/statvfs.h>

#ifdef __USE_LARGEFILE64
#define statvfs statvfs64
#endif

#endif

#ifndef NAME_MAX
#ifdef MAXNAMELEN
#define NAME_MAX MAXNAMELEN
#endif 
#endif 

#ifndef PATH_MAX
#ifdef MAXPATHLEN
#define PATH_MAX MAXPATHLEN
#endif 
#endif 


#ifndef S_BLKSIZE
#define S_BLKSIZE   512
#endif    

#ifndef _LARGEFILE64_SOURCE
#define lseek64 lseek
#endif 

#ifdef OS_OSF1
#define lseek64 lseek
#endif

#if _HPUX_SOURCE
#include <sys/diskio.h>
#endif

static void *sRandom;

unsigned long long int total_size_of_slacks=0;

typedef enum _WipeFileType
{
	TYPE_EXEPTION = 0,
	TYPE_FILE,
	TYPE_DIR,
	TYPE_BLOCK_DEV,
	TYPE_CHAR_DEV,
	TYPE_LINK,
	TYPE_SOCKET,
	TYPE_PIPE
} WipeFileType;

typedef struct _WipeFileEntry
{
	char            *FileName;
	WipeFileType    Type;
	long long int   FileSize;
	long long int   SlackSize;
	int             Skip;

	/*
	// random generator settings for verification
	*/
	time_t lastTime;
	int lastPass;

	struct _WipeFileEntry   *Next;

} WipeFileEntry;

WipeFileEntry   *FileListHead=NULL;
WipeFileEntry   *FileListTail=NULL;


/* ---------------------------------------------------------------------------------------------- */

int fill_buff(char *buff, size_t size, int n, off_t s);
int ask_y_n(char *string);
long long int get_size_of_device( int fd );
int wipe_filename( char *filename );
int wipe_file( char *filename );

/*int wipe_blkdev( char *filename );*/

int wipe_dirname(char *dirname);


int nas_wiping(int argc, char *argv[]);

int make_list(int argc, char *argv[]);

int add_file_to_list(char *FileName);
int add_dir_to_list(char *DirName);

int remove_from_list(WipeFileEntry *Elem);
int free_list();

int wipe_files_in_list( int pass, int verify );
int wipe_filenames_in_list();
int wipe_file_nas(WipeFileEntry *Entry, int pass, int verify );
int wipe_file_slack_nas(WipeFileEntry *Entry, int pass);

int nas_wiping(int argc, char *argv[])
{
	int res, i;
	int save;
	int verifyPass;

	if ( o_verbose ) {
		fprintf (stdout, "Wiping in NAS mode\n" );
	}

	res = make_list( argc, argv );
	if (OK != res)
	{
		fprintf(stderr,"Can not build list of files for wiping\n");
		return res;
	}

	save = o_dont_delete;

	o_dont_delete = TRUE;

	verifyPass = o_verify_last_pass ? 1 : 0;

	for ( i=0; i < o_pas_num + verifyPass ; i++ )
	{
		int verify = ( i == o_pas_num );
		wipe_files_in_list( i, verify );
	}

	o_dont_delete = save;

	if ( !o_wipe_dev )
		wipe_filenames_in_list();

	free_list();

	return OK;
}

/*
int print_list(WipeFileEntry *ListHead)
{
while (NULL != ListHead)
{
printf("%s   type=%d size=%Ld slack=%Ld                                 \n",
ListHead->FileName,ListHead->Type,ListHead->FileSize,ListHead->SlackSize);
ListHead=ListHead->Next;
}

return OK;
}
*/

int make_list(int argc, char *argv[])
{
	int i; 
	for ( i = argc; i; i-- )
	{
		add_file_to_list( argv[i-1] );
	}
	return OK;
}

int add_file_to_list(char *FileName)
{
	struct stat buff;
	WipeFileEntry *Entry;
	int isBlockDevice;
	const char *szEntryType = "unknown type";

	if (NULL == FileName)
	{
		return ERROR;
	}

	if ( OK != lstat(FileName,&buff) )
	{
		fprintf (stderr, "Skipping %s - %s\n", FileName, strerror(errno));
		return ERROR;
	}

	isBlockDevice = S_ISBLK( buff.st_mode ) || S_ISCHR( buff.st_mode );

	/*
	//	v. 1.7-2 Solaris does not recognize link as block device.
	//	So we try to forse block device reading if o_wipe_dev fla is assigned
	*/
	if ( !isBlockDevice && S_ISLNK( buff.st_mode ) && o_wipe_dev )
		isBlockDevice = TRUE;


	if ( isBlockDevice )
	{
		/*
		//	Get stat instead of lstat.
		*/
		if ( ERROR == stat( FileName, &buff ) )
		{
			fprintf (stderr, "Skipping %s - %s\n", FileName, strerror(errno));
			return ERROR;
		}
		
		if ( !S_ISBLK( buff.st_mode ) )
		{
			fprintf(stderr,"%s: is not a block device\t - Skipping\n", FileName );
			return ERROR;	
		}
	}

	Entry = (WipeFileEntry *)malloc(sizeof(WipeFileEntry));
	if (NULL == Entry)
	{
		fprintf (stderr, "Skipping %s - %s\n",FileName,strerror(errno));
		return ERROR;
	}

	memset( Entry, 0, sizeof(WipeFileEntry) );

	if ( S_ISDIR(buff.st_mode) )  /*directory */
	{
		if (OK != add_dir_to_list(FileName))
		{
			free(Entry);
			return ERROR;
		}
		Entry->Type = TYPE_DIR;
		szEntryType = "directory";

	}
	else if ( S_ISREG(buff.st_mode) )   /*regular file */
	{
		Entry->FileSize  = buff.st_size;
		Entry->SlackSize = buff.st_blksize - (buff.st_size % buff.st_blksize);
		Entry->Type      = TYPE_FILE;
		szEntryType = "file";

	}
	else if ( S_ISLNK(buff.st_mode) )   /*link */
	{
		Entry->Type      = TYPE_LINK;
		szEntryType = "link";

	}
	else if ( S_ISCHR(buff.st_mode) )   /*character device */
	{
		Entry->Type      = TYPE_CHAR_DEV;
		szEntryType = "char device";

	}
	else if ( S_ISBLK(buff.st_mode) )   /*block device */
	{
		Entry->Type      = TYPE_BLOCK_DEV;
		szEntryType = "block device";

	}
	else if ( S_ISFIFO(buff.st_mode) )  /*named pipe (fifo) */
	{
		Entry->Type      = TYPE_PIPE;
		szEntryType = "pipe";

	}
	else if ( S_ISSOCK(buff.st_mode) )  /*socket */
	{
		Entry->Type      = TYPE_SOCKET;
		szEntryType = "socket";

	}
	else                                /*something else .... */
	{
		fprintf(stderr,"Unknown type of file: %s\n",FileName);
		free(Entry);
		return ERROR;
	}

	if ( o_interactive )
	{
		fprintf(stderr,"Add %s '%s' to wiping list ", szEntryType, FileName);
		if ( !ask_y_n("(y/[n]/a)?") ) 
		{
			free(Entry);
			return ERROR;
		}
	}

	Entry->FileName = strdup(FileName);
	if (NULL == Entry->FileName)
	{
		fprintf (stderr, "Skipping %s - %s\n",FileName,strerror(errno));
		free(Entry);
		return ERROR;
	}

	if (NULL == FileListHead && NULL == FileListTail)
	{
		FileListHead = FileListTail = Entry;
	}else
	{
		FileListTail->Next=Entry;
		FileListTail=Entry;
	}

	return OK;
}


int add_dir_to_list(char *DirName)
{
	struct dirent *d;
	DIR *fd;
	char fn[NAME_MAX+PATH_MAX];

	if ( NULL==DirName ) return ERROR;

	if ( FALSE == o_recurse )
	{
		fprintf(stderr,"%s is a directory  \t - Skipping\n",DirName);
		return ERROR;
	}

	/* check directory access */
	if ( !o_wipe_slacks && OK != access(DirName,X_OK|R_OK|W_OK))
	{
		if ( !o_force )
		{
			fprintf(stderr,"Cannot enter to directory %s : %s  \t - Skipping\n",DirName,strerror(errno));
			return ERROR;
		}
		if ( OK != chmod(DirName,S_IRUSR|S_IWUSR|S_IXUSR) )
		{
			fprintf(stderr,"Cannot chmod directory %s: %s  \t - Skipping\n",DirName,strerror(errno));
			return ERROR;
		}
	}

	fd=opendir(DirName);
	if ( NULL == fd )
	{
		fprintf(stderr,"Cannot open directory %s: %s  \t - Skipping\n",DirName,strerror(errno));
		return ERROR;
	}


	while ( NULL != (d=readdir(fd)) )
	{
		sprintf(fn,"%s/%s",DirName,d->d_name);

		if ( 0==strcmp(d->d_name,".") || 0==strcmp(d->d_name,"..") )
			continue;

		add_file_to_list(fn);
	}

	closedir(fd);

	return OK;

}

int free_list()
{
	WipeFileEntry *Entry;

	while (NULL != FileListHead)
	{
		Entry=FileListHead;
		FileListHead = Entry->Next;
		free(Entry->FileName);
		free(Entry);
	}
	FileListTail = NULL;
	return OK;
}


int wipe_files_in_list( int pass, int verify )
{
	WipeFileEntry *Entry = FileListHead;

	while (NULL != Entry)
	{
		if (OK != wipe_file_nas( Entry, pass, verify ))
		{
			Entry->Skip = TRUE;;
		}
		Entry = Entry->Next;
	}
	return OK;
}

/*
// Verification process requires same random byte sequence
// The function inites all random generators to start from same initial value
// here a time varable is used.
*/
static void restartRandom( time_t t )
{
    srand( t );
		
    SHA1RandomFinalize( sRandom );
    SHA1RandomInitialize( sRandom );
    SHA1RandomRandomize( sRandom, (byte*)&t, sizeof(t));
}

int wipe_file_nas( WipeFileEntry *Entry, int pass, int verify )
{
	char buff[BUFFSIZE];
	int fd, isBlockDevice, isDone;
	unsigned int   pr,opr;
	ssize_t j;
	time_t t,ot,bt;
	long long int s=0;
	long long int os=0;
	long long int size;

	if (o_wipe_slacks)
		return wipe_file_slack_nas( Entry, pass );

	if (Entry->Skip)
	{
		return OK;
	}

	if (TYPE_DIR == Entry->Type)
	{
		return OK;
	}

	isBlockDevice = ( ( TYPE_BLOCK_DEV == Entry->Type ) || ( TYPE_CHAR_DEV == Entry->Type ) );

	if ( ( 0 == Entry->FileSize ) && !isBlockDevice )
		return OK;

	if ( o_verbose )
	{
		if ( isBlockDevice )
			fprintf (stdout, "Device name %s\n", Entry->FileName);
		else
			fprintf (stdout, "File name   %s\n", Entry->FileName);
	}

	if ( OK != access(Entry->FileName,W_OK|R_OK) )
	{
		if ( !o_force )
		{
			fprintf(stderr,"Wiping file %s: %s  \t - Skipping\n",Entry->FileName,strerror(errno));
			return ERROR;
		}
		if ( OK != chmod(Entry->FileName,S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH|S_IWOTH) )
		{
			fprintf(stderr,"Chmod file %s: %s  \t - Skipping\n",Entry->FileName,strerror(errno));
			return ERROR;
		}
	}

	/*
	#ifdef O_SYNC
	fd = open(Entry->FileName,O_RDWR|O_SYNC);
	#else
	fd = open(Entry->FileName,O_RDWR);
	#endif
	*/
	fd = open(Entry->FileName,O_RDWR);

	if ( ERROR == fd )
	{
		fprintf(stderr,"Opening file %s: %s  \t - Skipping\n",Entry->FileName,strerror(errno));
		return ERROR;
	}

	if ( isBlockDevice )
	{
		if ( 0 == Entry->FileSize )
		{
			Entry->FileSize = get_size_of_device(fd);
		}
	}

	size = Entry->FileSize + Entry->SlackSize;

	/*
	// In verification process the last saved settings will be used.
	*/
	if ( !verify )
	{
		Entry->lastTime = time( NULL );
		Entry->lastPass = pass;
	}

	restartRandom( Entry->lastTime );

	s=os=0;
	pr=opr=0;
	ot=bt=time(NULL);
	lseek64(fd,0,SEEK_SET);

	while ( s < size )
	{
		j = size - s < BUFFSIZE ? size - s : BUFFSIZE;
		fill_buff(buff, j, Entry->lastPass, s );

		if ( verify )
		{
			char verifyBuff[BUFFSIZE];

			if ( ERROR == ( j = read( fd, (void *)verifyBuff, j ) ) )
			{
				fprintf(stderr,"\nReading of %s: %s\n", Entry->FileName, strerror(errno) );
				close(fd);
				return ERROR;
			}

			if ( memcmp( verifyBuff, buff, j ) != 0 )
			{
				fprintf(stderr,"\nError in verifying last wiping pass result\n" );
				close(fd);
				return ERROR;
			}
		}
		else
		{
			if ( ERROR == (j=write(fd,(const void *)buff,j)) )
			{
				if ( ( ENOSPC == errno ) && isBlockDevice )
				{
					Entry->FileSize = s;
					j = 0;
				}
				else
				{
					fprintf(stderr,"Writing to %s: %s\n",Entry->FileName,strerror(errno));
					close(fd);
					return ERROR;
				}
			}
		}

		s+=j;

		/*
		#ifndef O_SYNC
		(void)fsync(fd);
		#endif
		*/

		isDone = ( j != BUFFSIZE );

		if ( !o_verbose )
		{
			if ( isDone )
			{
			    /* it is not required to sleep after verification */
			    if ( !verify )
					sleep(o_nas_delay);
	
			    break;
			}
		}
		else /* o_verbose */
		{
			char outStr[ 256 ];
			int len;
			
			t=time(NULL);
			pr=(int)(s*100/size);
			
			if ( verify )
				sprintf( outStr, "\rverify pass %-2d ",Entry->lastPass+1 );
			else	sprintf( outStr, "\rwipe pass %2d/%-2d", pass+1,o_pas_num );
			
			len = strlen( outStr );

			sprintf( outStr + len, ":%10ld/%ld Kb (%3d%%)", (long int)(s>>10),(long int)(size>>10),pr );
			fprintf( stdout, outStr );

			if ( isDone )
			{
			    /* it is not required to sleep after verification */
			    if ( !verify )
			    {
			        fprintf( stdout, "   Sleeping %d seconds", o_nas_delay );

			        sleep(o_nas_delay);
			    }

			    fprintf( stdout, "%s - Done                \n", outStr );

			    break;
			}
			else
			if ( t != ot )
			{
				if ( bt != t )
					fprintf(stdout,"   Rate: %ld Kb/s          ",(long int)((s/(long long int)(t-bt))>>10));
				else	fprintf(stdout,"                        ");
				ot=t;
				opr=pr;
			}
			else if ( pr != opr )
			{
				if ( bt != t )
						fprintf(stdout,"   Rate: %ld Kb/s          ",(long int)((s/(long long int)(t-bt))>>10));
				else	fprintf(stdout,"                        " );
				opr=pr;
			}
		}
	}

	if ( !verify )
		fsync(fd);

	/*
	//
	//	Version 1.6-5
	//		sets o_dont_delete flag as TRUE for all passes excluding the last only ( see nas_wiping )
	//	The trick allows to overwrite file in same location.
	//
	*/
	if ( !o_dont_delete )
		ftruncate( fd, 0 );

	close(fd);

	return OK;

}

int remove_from_list(WipeFileEntry *Elem)
{
	WipeFileEntry *tmp;

	if (NULL == Elem || NULL == FileListHead)
	{
		return ERROR;
	}
	tmp = FileListHead;

	while (NULL != tmp)
	{
		if (Elem == tmp->Next)
		{
			tmp->Next = Elem->Next;
			if (Elem == FileListHead)
			{
				FileListHead = Elem->Next;
			}
			if (Elem == FileListTail)
			{
				FileListTail = tmp;
			}
			break;
		}

		tmp = tmp->Next;
	}

	if (NULL != Elem->FileName)
	{
		free(Elem->FileName);
	}
	free(Elem);

	return OK;
}

int wipe_filenames_in_list()
{
	WipeFileEntry *Entry = FileListHead;

	while (NULL != Entry)
	{
		if (!Entry->Skip)
		{
			if (TYPE_DIR == Entry->Type)
			{
				wipe_dirname(Entry->FileName);
			}else
			{
				wipe_filename(Entry->FileName);
			}
		}

		Entry = Entry->Next;
	}
	return OK;
}

int wipe_file_slack_nas(WipeFileEntry *Entry, int pass)
{
	char *buff;
	int fd;

	if (Entry->Skip)
	{
		return OK;
	}

	if (TYPE_FILE != Entry->Type)
	{
		return OK;
	}

	if ( o_verbose )
	{
		fprintf (stdout, "Wiping slack of  %s\npass %2d/%d ",Entry->FileName,pass+1,o_pas_num);
	}

	if (0 == Entry->SlackSize)
	{
		if ( o_verbose )
			fprintf(stdout," - Done \n");
		return OK;
	}


	if ( OK != access(Entry->FileName,W_OK|R_OK) )
	{
		if (o_verbose)
			fprintf(stderr,"\n");
		fprintf (stdout, "%s: %s  \t - Skipping\n",Entry->FileName,strerror(errno));
		return ERROR;
	}

	buff = (char *)malloc(Entry->SlackSize);
	if (NULL == buff)
	{
		if (o_verbose)
			fprintf(stderr,"\n");
		fprintf(stderr,"Allocating buffer error: %s  \t - Skipping\n",strerror(errno));
		return ERROR;
	}


#ifdef O_SYNC
	fd = open(Entry->FileName,O_RDWR|O_SYNC);
#else
	fd = open(Entry->FileName,O_RDWR);
#endif
	if ( ERROR == fd )
	{
		if (o_verbose)
			fprintf(stderr,"\n");
		fprintf(stderr,"Opening file %s: %s  \t - Skipping\n",Entry->FileName,strerror(errno));
		free(buff);
		return ERROR;
	}


	if (ERROR == lseek64(fd,Entry->FileSize,SEEK_SET))
	{
		fprintf(stderr,"Seek to end of file %s: %s  \t - Skipping\n",Entry->FileName,strerror(errno));
		ftruncate(fd,Entry->FileSize);
		close(fd);
		free(buff);
		return ERROR;
	}
	fill_buff(buff, Entry->SlackSize, pass,0);
	if (ERROR == write(fd,(const void *)buff,Entry->SlackSize))
	{
		if (o_verbose)
			fprintf(stderr,"\n");
		fprintf(stderr,"Writing to %s: %s\n",Entry->FileName,strerror(errno));
		ftruncate(fd,Entry->FileSize);
		close(fd);
		free(buff);
		return ERROR;
	}

#ifndef O_SYNC
	(void)fsync(fd);
#endif

	if ( o_verbose )
		fprintf(stdout," - Done \n");

	ftruncate(fd,Entry->FileSize);

	close(fd);

	free(buff);

	return OK;
}

int wipe_dirname(char *dirname)
{
	struct utimbuf utb;

	if ( o_dont_delete || o_wipe_slacks) return OK;

	if ( o_verbose )
	{
		fprintf (stdout, "Removing directory %s ",dirname);
	}

	if ( !o_dont_wipe_fn )
	{
		utb.actime=utb.modtime=0;
		utime(dirname,&utb);         /* set access and modification time to 0 */
	}

	if ( OK != rmdir(dirname) )
	{
		if ( o_verbose )
			fprintf (stderr, "\t - %s\n",strerror(errno));
		else
			fprintf (stderr, "Removing %s - %s\n",dirname,strerror(errno));
		return ERROR;
	}
	if ( o_verbose )
		fprintf (stdout, "\t - Done\n");
	return OK;

}

/* ---------------------------------------------------------------------------------------------- */

/*ask_y_n(char *string) */
/*asks for yes or no */
/*return value: */
/*TRUE if 'y' or 'a' entered */
/*FALSE in other cases */
int ask_y_n(char *string)
{
	char c;
	fprintf(stderr,string);
	c = tolower(fgetc(stdin));
	if ( '\n' == c ) return FALSE;
	while ( '\n' != fgetc(stdin) );
	if ( 'y' == c ) return TRUE;
	if ( 'a' == c )
	{
		o_interactive=FALSE;
		return TRUE;
	}

	return FALSE;
}


/*  wipe_filename(char *filename) */
/*  wipe filename by renaming */
/*  return value: */
/*  ERROR   on error */
/*  OK      on success */
int wipe_filename(char *filename)
{
	struct utimbuf utb;

	if ( o_dont_delete )return OK;

	if ( o_verbose )
	{
		fprintf (stdout, "Removing file      %s ",filename);
	}
	if ( o_dont_wipe_fn )
	{
		if ( OK != unlink(filename) )
		{
			if ( o_verbose )
				fprintf (stderr, "\t - %s\n",strerror(errno));
			else
				fprintf (stderr, "Removing %s - %s\n",filename,strerror(errno));
			return ERROR;
		}

		if ( o_verbose )
		{
			fprintf (stdout, "\t - Done\n");
		}
		return OK;
	}

	truncate( filename, 0 );         /* set filesize to 0 */
	utb.actime = utb.modtime = 0;
	utime( filename, &utb );         /* set access and modification time to 0 */

	if ( OK != unlink(filename) )
	{
		if ( o_verbose )
			fprintf (stderr, "\t - %s\n",strerror(errno));
		else
			fprintf (stderr, "Removing %s - %s\n",filename,strerror(errno));
		return ERROR;
	}
	if ( o_verbose )
	{
		fprintf (stdout, "\t - Done\n");
	}
	return OK;
}


/*
// fill_buff(char *buff, size_t size, int n) 
// fills buffer buff coresponding pass number n 
*/
int fill_buff(char *buff, size_t size, int n, off_t s)
{
	int i,j;

	if (o_use_zero)    
	{
		if ( 0 == s )
		{
			memset( buff, 0, size );
		}
		return OK;
	}

	if (o_use_sector_number)
	{
		memset( buff, 0, size );
		for (i=0;i<size;i+=512)
		{
			buff[i+0]=((s+i)/512)>>24 & 0xff;
			buff[i+1]=((s+i)/512)>>16 & 0xff;
			buff[i+2]=((s+i)/512)>> 8 & 0xff;
			buff[i+3]=((s+i)/512)     & 0xff;
		}
		return OK;
	}      

	if ( n == o_pas_num - 1 )
	{
		if ( s && o_use_buff ) return OK;
		if ( o_use_rand )
		{
		    for ( i = 0; i < size; i++ ) buff[i]=(char)(rand() & 0xff);
		}
		else
		{
		    SHA1RandomGetRandomBytes( sRandom, (byte*)buff, size );
		}
		
		return OK;
	}

	if ( o_use_DoD )
	{
		if (QPASSES != o_pas_num)
		{
			if ( s ) return OK;
			j = buff[0];
			buff[0] = (char)(rand() & 0xff);
			for ( i = 0; i < size; i++ )
				if ( 1 == (n % 2) ) buff[i] = ~j;
				else buff[i] = buff[0];
				return OK;
		}else
		{
			if ( 0 == dod_pass[n].len )
			{
				if ( s && o_use_buff ) return OK;
				if ( o_use_rand )
					for ( i = 0; i < size; i++ ) buff[i]=(char)(rand() & 0xff);
				else
				{
					SHA1RandomGetRandomBytes(sRandom,(byte*)buff,size);
				}
			}
			else if ( 0 == s )
				for ( i = 0, j = 0; i < size; i++ )
					buff[i] = dod_pass[n].pat[j++ % dod_pass[n].len];

			return OK;
		}
	}
	if ( 0 == pg_pass[n].len )
	{
		if ( s && o_use_buff ) return OK;
		if ( o_use_rand )
			for ( i = 0; i < size; i++ ) buff[i]=(char)(rand() & 0xff);
		else
		{
			SHA1RandomGetRandomBytes(sRandom,(byte*)buff,size);
		}
	}
	else if ( 0 == s )
		for ( i = 0, j = 0; i < size; i++ )
			buff[i] = pg_pass[n].pat[j++ % pg_pass[n].len];

	return OK;
}

/*off_t find_size_by_reading(int fd) */
/*return value: */
/*0       on filure */
/*size    on success */
long long int find_size_by_reading(int fd)
{
	off_t limit,pos,lPos,prev,b_pos=0;
	char buf[64];
	int i;

	for ( limit=0x7f,i=sizeof(limit);--i; )
		limit = (limit <<8)|0xff;

	pos=prev=limit;
	while ( pos != b_pos && pos <= limit )
	{
		/*        fprintf(stdout,"pos=%20ld   b_pos=%12ld   buf=%x\n",(long int)pos,(long int)b_pos,buf[0]&0xff); */
		
		/*
		//    Version 1.7-7 Aug 07 2008
		//    Treats the lseek error as it cannot set file pointer 
		//	beyond device size instead of sending the error. 
		//	( AIX 5.2 reports here 'Invalid argumment' error )
		//
		//if ( pos != lseek64(fd,pos,SEEK_SET) )
		//{
		//	fprintf(stderr,"lseek error: %s\n",strerror(errno));
		//	return 0;
		//}
		*/
		
		lPos = lseek64( fd, pos, SEEK_SET );
		
		if ( ( lPos != pos ) || ( 1 != read(fd,buf,1) ) )
		{
			prev=pos;
			pos=(pos+b_pos)>>1;
		}
		else
		{
			b_pos=pos;
			pos=(pos+prev)>>1;
		}
	}

	if ( pos >= limit )
		return 0;

	return lseek64(fd,0,SEEK_CUR);
}

long long int get_size_of_device(int fd)
{
	long long size=0;
	int i;
	int hist = 0;

#ifdef BLKGETSIZE
	long l;
	if ( !ioctl(fd,BLKGETSIZE,&l) )
		size=(long long)l*S_BLKSIZE;

	hist = 1;
#else
#ifdef DIOC_CAPACITY
	capacity_type ct;
	if ( !ioctl( fd, DIOC_CAPACITY, &(ct.lba) ) )
		size=(long long)ct.lba * DEV_BSIZE;

	hist |= 2;

#endif
#endif    

	if ( 0 == size || ERROR == size )
		size=lseek64(fd,0,SEEK_END);

	if ( 0 == size || ERROR == size )
	{
		size = find_size_by_reading(fd);
		hist |= 4;
	}

	if ( 0 == size || ERROR == size )
	{
		for ( size=0x7f,i=sizeof(size);--i; )
			size = (size <<8)|0xff;

		hist |= 8;
	}

	if ( o_verbose || o_verbose_light )
	{
		char byteStr[100];
		int byteSize = (int)( size & 0x3FF );

		byteStr[0] = 0;

		if ( byteSize != 0 )
			sprintf( byteStr, " %d bytes", byteSize );

		fprintf (stdout, "Device size %ld Kb%s, (%d)\n",
			(long int)(size>>10),
			byteStr,
			hist );
	}

	return size;
}

int wipe_file_slack( char *filename )
{
	char *buff;
	int fd;
	unsigned int   i;
	ssize_t size_of_slack;
	struct stat st;
	uid_t uid;
	struct utimbuf time_buf;

	if ( ERROR == lstat(filename,&st) )
	{
		fprintf(stderr,"Stating file %s: %s  \t - Skipping\n",filename,strerror(errno));
		return ERROR;
	}

	if ( !S_ISREG(st.st_mode))
		return OK;

	uid = geteuid();

	if (0 != uid && st.st_uid != uid)
	{
		fprintf(stderr,"You are not owner of %s  \t - Skipping\n",filename);
		return OK;
	}
	if ( o_verbose )
	{
		fprintf (stdout, "Wiping slack of      %s",filename);
	}

	size_of_slack = st.st_blksize - (st.st_size % st.st_blksize);
	if (st.st_blksize == size_of_slack)
	{
		if ( o_verbose )
			fprintf(stdout," - Done \n");
		return OK;
	}

	if ( OK != access(filename,W_OK|R_OK) )
	{
		if ( OK != chmod(filename,S_IRUSR|S_IWUSR|st.st_mode) )
		{
			fprintf(stderr,"Chmod file %s: %s  \t - Skipping\n",filename,strerror(errno));
			return ERROR;
		}
	}

	buff = (char *)malloc(st.st_blksize);
	if (NULL == buff)
	{
		fprintf(stderr,"Allocating buffer error: %s  \t - Skipping\n",strerror(errno));
		return OK;
	}

#ifdef O_SYNC
	fd = open(filename,O_RDWR|O_SYNC);
#else
	fd = open(filename,O_RDWR);
#endif
	if ( ERROR == fd )
	{
		fprintf(stderr,"Opening file %s: %s  \t - Skipping\n",filename,strerror(errno));
		free(buff);
		return ERROR;
	}


	for ( i=0; i < o_pas_num; i++ )
	{
		if (ERROR == lseek64(fd,st.st_size,SEEK_SET))
		{
			fprintf(stderr,"Seek to end of file %s: %s  \t - Skipping\n",filename,strerror(errno));
			ftruncate(fd,st.st_size);
			close(fd);
			free(buff);
			return ERROR;
		}
		fill_buff(buff, st.st_blksize, i,0);
		if (ERROR == write(fd,(const void *)buff,size_of_slack))
		{
			if (o_verbose)
				fprintf(stderr,"\n");
			fprintf(stderr,"Writing to %s: %s\n",filename,strerror(errno));
			ftruncate(fd,st.st_size);
			close(fd);
			free(buff);
			return ERROR;
		}

#ifndef O_SYNC
		(void)fsync(fd);
#endif
	}

	if ( o_verbose )
		fprintf(stdout," - Done \n");

	ftruncate(fd,st.st_size);

	close(fd);

	free(buff);

	total_size_of_slacks +=size_of_slack;
	if ( OK != chmod(filename,st.st_mode) )
	{
		fprintf(stderr,"Restore mode of file %s: %s  \n",filename,strerror(errno));
		return ERROR;
	}

	time_buf.actime  = st.st_atime; 
	time_buf.modtime = st.st_mtime;
	if ( OK != utime(filename,&time_buf) )
	{
		fprintf(stderr,"Restore times of file %s: %s  \n",filename,strerror(errno));
		return ERROR;
	}

	return OK;
}

static void showBuffer( const char *bfName, char *bf, int size )
{
    int col,i;

    printf( "Buffer %s: \n", bfName );
    
    for( col = 0, i = 0; i < size; i++, col++ )
    {
	if ( col >= 16 )
	{
	    printf( "\n" );
	    col = 0;
	}
	
	if ( col == 0 )
	    printf( "%08d ", i );
	
	unsigned int ch = (unsigned char)bf[i];
	printf( "%X%X ", (ch>>4)&0xF, (ch)&0xF );
    }
    
    printf("\n");
}

/* 
//	wipe_file (char *filename)
//		returns value:
//			ERROR on error
//			OK    on success
*/
int wipe_file( char *filename )
{
	char buff[BUFFSIZE];
	int fd, isBlockDevice, verifyPass;
	unsigned int   i,op,pr,opr;
	ssize_t j;
	time_t t,ot,bt;
	long long int s=0;
	long long int os=0;
	long long int size;
	struct stat st;
	const char *szOperation;
	const char *szDevType;
	time_t lastRandomTime = 0;
	int lastWritePass = 0;

	if (o_wipe_slacks)
		return wipe_file_slack( filename );

	if ( ERROR == lstat( filename, &st ) )
	{
		fprintf(stderr,"Stating file %s: %s  \t - Skipping\n", filename, strerror(errno) );
		return ERROR;
	}

	isBlockDevice = S_ISBLK( st.st_mode ) || S_ISCHR( st.st_mode );

	/*
	//	v. 1.7-2 Solaris does not recognize link as block device.
	//	So we try to forse block device reading if o_wipe_dev fla is assigned
	*/
	if ( !isBlockDevice && S_ISLNK( st.st_mode ) && o_wipe_dev )
	{
		isBlockDevice = TRUE;
	}

	if ( isBlockDevice )
	{
		/*
		//	Get stat instead of lstat.
		*/
		if ( ERROR == stat( filename, &st ) )
		{
			fprintf(stderr,"Stating file %s: %s  \t - Skipping\n", filename, strerror(errno) );
			return ERROR;
		}
		
		if ( !S_ISBLK( st.st_mode ) )
		{
			fprintf(stderr,"%s: is not a block device\t - Skipping\n", filename );
			return ERROR;	
		}
	}

	/*
	//if ( S_ISREG(st.st_mode))
	//{
	//	size = (long long int)st.st_blocks * st.st_blksize;
	//}else
	//	size = st.st_size;
	*/

	size = st.st_size;

	if ( o_interactive )
	{
		fprintf(stderr,"Wipe %s ",filename);
		if ( !ask_y_n("(y/[n]/a)?") ) return ERROR;
	}

	szOperation = "Wiping";
	szDevType = ( isBlockDevice ? "device" : "file" );

	if ( o_verbose || o_verbose_light )
		fprintf (stdout, "%s \t%s \t%s\n", szOperation, szDevType, filename);

	if ( 0 == st.st_size )
	{
		if ( isBlockDevice )
		{
			/* block device size will calculated after device opening */
		}
		else /* zero size regular file */
			return OK;
	}

	if ( OK != access( filename, W_OK | R_OK ) )
	{
		if ( !o_force )
		{
			fprintf(stderr,"%s %s %s: %s  \t - Skipping\n",szOperation, szDevType, filename, strerror(errno) );
			return ERROR;
		}
		if ( OK != chmod( filename, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH ) )
		{
			fprintf( stderr, "Chmod %s %s: %s  \t - Skipping\n", szDevType, filename, strerror( errno ) );
			return ERROR;
		}
	}

	fd = open(filename,O_RDWR);

	if ( ERROR == fd )
	{
		fprintf(stderr,"Opening %s %s: %s  \t - Skipping\n",szDevType, filename,strerror(errno));
		return ERROR;
	}

	if ( isBlockDevice )
		size = get_size_of_device( fd );

	verifyPass = o_verify_last_pass ? 1 : 0;

	/*
	//	Verification is an additinal pass after all passes.
	//	We saves here last variables for buffer filling in verification pass
	*/

	for ( i=0, op=0 ; i < ( o_pas_num + verifyPass ); i++ )
	{
		int verify = ( i == o_pas_num );

		if ( !verify )
		{
		    lastRandomTime = time( NULL );
		    lastWritePass = i;
		}

		if ( o_verbose_light )
		{
			if ( verify )
				fprintf( stdout, "\rverify pass %-2d       ", lastWritePass+1 );
			else
				fprintf( stdout, "\rwipe pass %2d/%-2d     ", i+1, o_pas_num );
		}

		restartRandom( lastRandomTime );
		
		if ( s && (S_ISBLK(st.st_mode) || S_ISCHR(st.st_mode)) ) size=s;

		s=os=0;
		pr=opr=0;
		ot=bt=time(NULL);
		lseek64(fd,0,SEEK_SET);

		while ( s < size )
		{
			j = size - s < BUFFSIZE ? size - s : BUFFSIZE;

			fill_buff( buff, j, lastWritePass, s );
			
			if ( verify )
			{
				char verifyBuff[BUFFSIZE];

				if ( ERROR == ( j = read( fd, (void *)verifyBuff, j ) ) )
				{
					fprintf(stderr,"\nReading of %s: %s\n", filename, strerror(errno) );
					close(fd);
					return ERROR;
				}

				if ( memcmp( verifyBuff, buff, j ) != 0 )
				{
					fprintf(stderr,"\nError in verifying last wiping pass result\n" );
					close(fd);
					return ERROR;
				}
			}
			else
			{
				if ( ERROR == (j=write(fd,(const void *)buff,j)) )
				{
					if ( ENOSPC == errno && (S_ISBLK(st.st_mode) || S_ISCHR(st.st_mode)) ) break;
					fprintf(stderr,"Writing to %s: %s\n",filename,strerror(errno));
					close(fd);
					return ERROR;
				}
			}

			s+=j;

			if ( o_verbose )
			{
				t=time(NULL);
				pr=(int)(s*100/size);
				
				if ( verify )
				{
					fprintf( stdout, "\rverify pass %-2d :%10ld/%ld Kb (%3d%%)   ",lastWritePass+1, (long int)(s>>10),(long int)(size>>10),pr );
				}
				else	fprintf( stdout, "\rwipe pass %2d/%-2d:%10ld/%ld Kb (%3d%%)   ",i+1,o_pas_num,(long int)(s>>10),(long int)(size>>10),pr );

				if ( pr == 100 )
				{
				}
				else
				if ( t != ot )
				{
					if ( ( op == i ) && ( bt != t ) )
						fprintf(stdout,"Rate: %ld Kb/s          ",(long int)((s/(long long int)(t-bt))>>10));
					else	fprintf(stdout,"                        ");
					ot=t;
					op=i;
					opr=pr;
				}
				else if ( pr != opr )
				{
					if ( bt != t ) fprintf(stdout,"Rate: %ld Kb/s          ",(long int)((s/(long long int)(t-bt))>>10));
					else           fprintf(stdout,"                        " );
					opr=pr;
				}
			}

			if ( j != BUFFSIZE )
				break;
		}

		if ( !verify )
			(void)fsync(fd);

		if ( o_verbose || o_verbose_light )
		    fprintf( stdout," - Done                    \n");
	}
/*
	if ( o_verbose )
		fprintf(stdout," - Done                    \n");
*/
	if ( !o_dont_delete )
		ftruncate(fd,0);

	close(fd);

	return OK;
}

int wipe_dir( char *dirname )
{
	struct dirent *d;
	DIR *fd;
	struct stat buff;
	char fn[NAME_MAX+PATH_MAX];

	if ( NULL==dirname ) return ERROR;

	if ( o_interactive )
	{
		fprintf(stderr,"Enter directory %s ",dirname);
		if ( !ask_y_n("(y/[n]/a)?") )
			return ERROR;
	}

	/* check directory access */
	if ( !o_wipe_slacks && OK != access(dirname,X_OK|R_OK|W_OK))
	{
		if ( !o_force )
		{
			fprintf(stderr,"Cannot enter to directory %s : %s  \t - Skipping\n",dirname,strerror(errno));
			return ERROR;
		}
		if ( OK != chmod(dirname,S_IRUSR|S_IWUSR|S_IXUSR) )
		{
			fprintf(stderr,"Cannot chmod directory %s: %s  \t - Skipping\n",dirname,strerror(errno));
			return ERROR;
		}
	}

	fd=opendir(dirname);

	if ( NULL == fd )
	{
		fprintf(stderr,"Cannot open directory %s: %s  \t - Skipping\n",dirname,strerror(errno));
		return ERROR;
	}

	if ( o_verbose )
	{
		fprintf (stdout, "Entering directory %s\n",dirname);
	}

	while ( NULL != (d=readdir(fd)) )
	{
		sprintf(fn,"%s/%s",dirname,d->d_name);
		if ( OK != lstat(fn,&buff) )
		{
			fprintf(stderr,"Cannot stat %s: %s  \t - Skipping\n",fn,strerror(errno));
			continue;
		}

		if ( S_ISDIR(buff.st_mode) )    /*directory */
		{
			if ( 0==strcmp(d->d_name,".") || 0==strcmp(d->d_name,"..") )
				continue;

			wipe_dir(fn);
		}
		else if ( S_ISREG(buff.st_mode) )   /*regular file */
		{
			int res = wipe_file( fn );

			if ( res == OK )
				wipe_filename( fn );
		}
		else if ( S_ISLNK(buff.st_mode) )   /*link */
		{
			wipe_filename(fn);
		}
		else if ( S_ISCHR(buff.st_mode) )   /*character device */
		{
			if ( o_wipe_dev )
				wipe_file(fn);
			else
				wipe_filename(fn);
		}
		else if ( S_ISBLK(buff.st_mode) )   /*block device */
		{
			if ( o_wipe_dev )
				wipe_file( fn ); /*wipe_blkdev(fn);*/
			else
				wipe_filename(fn);
		}
		else if ( S_ISFIFO(buff.st_mode) )  /*named pipe (fifo) */
		{
			wipe_filename(fn);
		}
		else if ( S_ISSOCK(buff.st_mode) )  /*socket */
		{
			wipe_filename(fn);
		}
		else                                /*somthing else .... */
		{
			fprintf(stderr,"Unknown type of file: %s\n",fn);
		}

	}

	closedir(fd);

	if ( o_dont_delete )
		return OK;

	if ( o_interactive )
	{
		fprintf(stderr,"Remove directory %s ",dirname);
		if ( !ask_y_n("(y/[n]/a)?") )
			return ERROR;
	}

	if ( o_verbose || o_verbose_light )
	{
		fprintf (stdout, "Removing directory %s ",dirname);
	}

	if ( OK != rmdir(dirname) )
	{
		if ( o_verbose )
			fprintf (stderr, "\t - %s\n",strerror(errno));
		else
			fprintf (stderr, "Removing %s - %s\n",dirname,strerror(errno));
		return ERROR;
	}
	if ( o_verbose || o_verbose_light )
		fprintf (stdout, "\t - Done\n");

	return OK;
}



/* -------------- wipe free space routines -------------- */

off_t get_free_space(char *path)
{
	struct statvfs sfs;
	int res;

	res = statvfs(path,&sfs);
	if (res != 0)
	{
		fprintf(stderr,"statvfs(%s): %s\n",path,strerror(errno));
		return 0;
	}

	return (off_t)sfs.f_bfree * sfs.f_frsize;

}


/*
//	struct _sFSList
//		Saves list of the files are created for free space wiping.
//		Saves unique time setting for random buffer filling
//		Initialises random with the value.
//		As result, the random generators are reinitialised  by same value at verification process.
*/
typedef struct _sFSList
{
	char m_fileName[ PATH_MAX + NAME_MAX ];

	/*
	// random generator settings for verification
	*/
	time_t m_lastTime;
	int m_lastPass;

	struct _sFSList   *m_next;

}sFSList;

static void freeFSList( sFSList *fdHead )
{
	while( fdHead != NULL )
	{
		sFSList *tmp = fdHead;

		fdHead = tmp->m_next;

		memset( tmp, 0, sizeof( sFSList ) );

		free( tmp );
	}
}

static sFSList *addFSList( sFSList **pFDHead, char *fileName, int pass )
{
	sFSList *fsl = (sFSList *)malloc( sizeof( sFSList ) );

	if ( fsl == NULL )
		return NULL;

	strncpy( fsl->m_fileName, fileName, sizeof( fsl->m_fileName ) );
	fsl->m_lastTime = time( NULL );
	fsl->m_lastPass = pass;
	fsl->m_next = *pFDHead;
	*pFDHead = fsl;

	restartRandom( fsl->m_lastTime );

	return fsl;
}

#define MAX_FILE_SIZE   1024*1024*1024

int fs_createFiles( sFSList **ppFSList, char *path, int pass, off_t *pSize )
{
	char buff[BUFFSIZE];
	char tmp[PATH_MAX+NAME_MAX];
	int fd,res;
	off_t s,os;
	off_t size;
	unsigned int   pr,opr;
	time_t t,ot,bt;
	int cnt;

	s = 0;

	size = get_free_space(path);
	
	if ( pSize != NULL )
	    *pSize = size;

	if ( 0 == size )
	{
		if (1 != pass)
		{
			fprintf(stderr,"No free space available on this filesystem\n");
			return ERROR;
		}
		
		return OK;
	}
	s=os=0;
	pr=opr=0;
	ot=bt=time(NULL);

	/*
	//	The is a strange report about HPUX 11.23.
	//	We are running it on HPUX 11.23 with the option vIF and what happens is
	//	that a directory is created with a 5 digit random number appended and in
	//	it is created up to 26 1GB files, each one prefixed by a letter a-z.
	//	Once z is reached it stops, removes the files and then starts on the
	//	next pass.
	//
	//	Version 1.7-3
	//	The 'cnt' is a name of the temporary file and extention is random.
	*/
	cnt = 0;

	while ( TRUE )
	{
		int ss;
		ss = 0;

		snprintf( tmp, PATH_MAX+NAME_MAX, "%s/%03x.XXXXXX", path, cnt++ );

		fd = mkstemp( tmp );

		if ( ERROR == fd )
		{
			if ( ENOSPC == errno )
			{
			    res = 0;
			    break;
			}

			fprintf( stderr, "mkstemp(%s): %s\n", tmp, strerror( errno ) );
			return -1; 
		}

		if ( addFSList( ppFSList, tmp, pass ) == NULL )
		{
			close( fd );
			fprintf( stderr, "Error in allocating memory\n" );
			return -1; 
		}

		do
		{
			fill_buff( buff, BUFFSIZE, pass, s );
			
			res = write( fd, buff, BUFFSIZE );
			
			if ( ERROR != res )
			{
				s  += res;
				ss += res;
			}

			if ( o_verbose )
			{
				t=time(NULL);
				pr=(int)(s*100/size);
				                
				fprintf(stdout,"\rwipe pass %2d/%-2d:%10ld/%ld Kb (%3d%%)   ", pass+1,o_pas_num,(long int)(s>>10),(long int)(size>>10),pr );

				if ( t != ot )
				{
					fprintf( stdout, "Rate: %ld Kb/s          ", (long int)((s/(long long int)(t-bt))>>10) );
					ot=t;
					opr=pr;
				}
				else if ( pr != opr )
				{
					if ( bt != t ) fprintf( stdout, "Rate: %ld Kb/s          ", (long int)((s/(long long int)(t-bt))>>10));
					else           fprintf( stdout, "                        " );
					opr=pr;
				}
			}

			if ( ERROR == res )
			{
				if (ENOSPC == errno)
				    res = ENOSPC;
				    
				break;
			}
			
			if ( MAX_FILE_SIZE <= ss )
			{
				res = 0;
				break;
			}

		} while ( ERROR != res );

		(void)fsync(fd);

		close( fd );
		
		if ( ENOSPC == res )
		{
		    res = 0;
		    break;
		}
	}

	return res;
}

static int fs_verifyFiles( sFSList *FSList, off_t size )
{
	char buff[BUFFSIZE];
	char verifyBuff[BUFFSIZE];

	int fd,res;
	off_t s,os;
	unsigned int   pr,opr;
	time_t t,ot,bt;
	sFSList *cur;

	s=os=0;
	pr=opr=0;
	ot=bt=time(NULL);

	for( cur = FSList; cur != NULL; cur = cur->m_next )
	{
		restartRandom( cur->m_lastTime );
		fd = open( cur->m_fileName, O_RDONLY );
	
		if ( ERROR == fd )
		{
			fprintf(stderr,"\nError (%s) in opening file %s for verification\n", strerror(errno), cur->m_fileName );
			return ERROR;
		}

		do
		{
			res = read( fd, buff, BUFFSIZE );
			
			if ( ERROR == res )
			{
				fprintf(stderr,"\nError in reading file %s for verification\n", cur->m_fileName, strerror(errno) );
				close( fd );
				return ERROR;
			}

			fill_buff( verifyBuff, res, cur->m_lastPass, s );
				
			if ( memcmp( buff, verifyBuff, res ) != 0 )
			{
				fprintf( stderr,"\nError in verifying last wiping pass result\n" );
				close( fd );
				return ERROR;
			}

			s += res;

			if ( o_verbose )
			{
				t=time(NULL);
				pr=(int)(s*100/size);

				fprintf( stdout, "\rverify pass %-2d :%10ld/%ld Kb (%3d%%)   ", cur->m_lastPass+1,(long int)(s>>10),(long int)(size>>10),pr );

				if ( t != ot )
				{
					fprintf( stdout, "Rate: %ld Kb/s          ", (long int)((s/(long long int)(t-bt))>>10) );
					ot=t;
					opr=pr;
				}
				else if ( pr != opr )
				{
					if ( bt != t ) fprintf( stdout, "Rate: %ld Kb/s          ", (long int)((s/(long long int)(t-bt))>>10));
					else           fprintf( stdout, "                        " );
					opr=pr;
				}
			}

		}
		while( res > 0 );

		close( fd );
	}
}

int allocate_free_space( char *path, int pass, int verify )
{
	/*
	//	List of allocated files.
	//	The list will be freed after verification.
	*/
	sFSList *FSList = NULL;
	
	off_t size;

	if ( o_verbose || o_verbose_light )
		fprintf(stdout,"\rwipe pass %2d/%-2d  ", pass+1,o_pas_num );

	int res = fs_createFiles( &FSList, path, pass, &size );

	if ( ( res == OK ) && verify )
	{
		if ( o_verbose || o_verbose_light )
			fprintf( stdout, " - Done                    \n");

		/*
		//
		//	Version 1.6-5
		//		performs the nas delay before temporary files deleting
		//		instead of prev version sleels between passes
		//
		*/
		if ( o_nas_delay )
		{
			if ( o_verbose )
			{
				fprintf( stdout, "Sleeping %d seconds\n", o_nas_delay );
			}

			sleep(o_nas_delay);
		}

		if ( o_verbose || o_verbose_light )
			fprintf(stdout,"\rverify pass %2d/%-2d  ", pass+1,o_pas_num );

		res = fs_verifyFiles( FSList, size );
	}
	
	if ( ( res == OK ) && ( o_verbose ) || o_verbose_light )
	    fprintf( stdout, " - Done                    \n");

	if ( ( res == OK ) && o_nas_delay )
	{
		if ( o_verbose )
		{
			fprintf( stdout, "Sleeping %d seconds\n", o_nas_delay );
		}

		sleep(o_nas_delay);
	}


	freeFSList( FSList );

	return res;
}

int delete_temp_dir(char *dirname)
{
	struct dirent *d;
	DIR *fd;
	struct stat buff;
	char fn[NAME_MAX+PATH_MAX];

	if ( NULL==dirname ) return ERROR;

	fd=opendir(dirname);
	if ( NULL == fd )
	{
		fprintf(stderr,"opendir(%s): %s \n",dirname,strerror(errno));
		return ERROR;
	}

	while ( NULL != (d=readdir(fd)) )
	{
		sprintf(fn,"%s/%s",dirname,d->d_name);
		if ( OK != lstat(fn,&buff) )
		{
			fprintf(stderr,"lstat(%s): %s\n",fn,strerror(errno));
			continue;
		}
		if ( S_ISDIR(buff.st_mode) )    /*directory */
		{
			if ( 0==strcmp(d->d_name,".") || 0==strcmp(d->d_name,"..") )
				continue;
	
			delete_temp_dir(fn);
		}
		else
		{
			if ( 0 != unlink(fn))
			{
				fprintf(stderr,"unlink(%s): %s\n",fn,strerror(errno));
				return ERROR;
			}
		}

	}
	closedir(fd);

	if ( OK != rmdir(dirname) )
	{
		fprintf(stderr,"rmdir(%s): %s\n",dirname,strerror(errno));
		return ERROR;
	}
	sync();
	return OK;
}

char *make_temp_dir( char *path,char *result )
{
	int fd;

	snprintf( result, PATH_MAX, "%s/bcwipe-wiping_free_space-XXXXXX", path );
/*	GCC reports warning 'mktemp is dangerous, better use mkstemp'
	if ( NULL == mktemp( result ) )
	{
		fprintf(stderr,"mktemp(%s): %s\n",result,strerror(errno));
		return NULL;
	}
*/
	fd = mkstemp( result );
	if ( fd == -1 )
	{
		fprintf(stderr,"mktemp(%s): %s\n",result,strerror(errno));
		return NULL;
	}

	close(fd);
	unlink( result );

	if (OK != mkdir(result,0700))
	{
		fprintf(stderr,"mkdir(%s): %s\n",result,strerror(errno));
		return NULL;
	}

	return result;
}

int check_permissions(char *path)
{
	if ( OK != access(path,W_OK|R_OK|X_OK) )
	{
		return ERROR;
	}
	return OK;
}

char *get_mount_point(char *path)
{
	return path;
}

int can_wipe_all_free_space(char *path)
{
	struct statvfs sfs;
	int res;

	res = statvfs(path,&sfs);
	if (res != 0)
	{
		fprintf(stderr,"statvfs(%s): %s\n",path,strerror(errno));
		return 0;
	}

	return (sfs.f_bfree == sfs.f_bavail || geteuid() == 0);

}

/*
//	wipe_free_space
//	1. Creates new files and fills tmen by wiping data for all free space available.
//	All creates file names are stored in list ( see sFSList structure ).
//	2. The last pass free space allocation verifies the new files by the list.
//	3. Removes the new directory.
*/
int wipe_free_space(char *path)
{
	int i,res;
	char tmp_dir[PATH_MAX];
	char *tmp;

	if (o_interactive)
	{
		fprintf(stderr,"Wipe free space in %s ",get_mount_point(path));
		if (!ask_y_n("(y/[n])?"))
		{
			return OK;
		}
	}

	res = check_permissions(path);
	if (OK != res)
	{
		fprintf(stderr,"Can not wipe free space in %s: %s\n",path, strerror(errno));
		return res;
	}

	if (!can_wipe_all_free_space(path) && o_interactive)
	{
		fprintf(stderr,"You can not wipe ALL free space on %s.\n",get_mount_point(path));
		if (!ask_y_n("Continue (y/[n])?"))
		{
			return OK;
		}
	}

	if ( o_verbose || o_verbose_light )
	{
		fprintf(stdout,"Wiping free space on %s\n", get_mount_point( path ) );
	}

	for (i = 0 ; i < o_pas_num; i++)
	{
		int verify = 0;

		/*
		//Verifies last pass only.
		*/
		if ( o_verify_last_pass )
			verify = ( i == o_pas_num - 1 );

		tmp = make_temp_dir(path,tmp_dir);
		if (NULL == tmp)
		{
			return ERROR;
		}

		res = allocate_free_space( tmp_dir, i, verify );
		if (OK != res)
		{
			printf("\n");
			return res;
		}

		res = delete_temp_dir(tmp_dir);
		if (OK != res)
		{
			printf("\n");
			return res;
		}
	}
/*
//	if ( o_verbose )
//		fprintf( stdout, " - Done                    \n");
*/		
	return OK;
}

/* -------------- end of wipe free space routines -------------- */


int main(int argc, char* argv[])
{
	time_t t;
	int result; 
	struct stat buff;
	int i;

	setvbuf(stderr,(char*)NULL,_IONBF,0);
	setvbuf(stdout,(char*)NULL,_IONBF,0);
	t=time(&t);
	srand(t);

	result = SHA1RandomAllocate( &sRandom );
	if ( result != 0 )
	{
		fprintf(stderr,"Can not Allocate memory for SHA1 random generator\n");
		return result;
	}

	result = SHA1RandomInitialize( sRandom );
	if ( result != 0 )
	{
		fprintf(stderr,"Can not Initialize SHA1 random generator\n");
		return result;
	}

	result = SHA1RandomRandomize( sRandom, (byte*)&t, sizeof(t));
	if ( result != 0 )
	{
		fprintf(stderr,"Can not Randomize SHA1 random generator\n");
		return result;
	}

	result = parseOptions( &argc, &argv );

	if ( result != 0 )
	{
		return result;
	}

	/*
	// Version 1.6-8 
	//	allows wiping block devices in NAS mode
	//if ( o_nas_wiping && o_wipe_dev )
	//{
	//	fprintf( stderr, "Options \"-b\" and \"-n\" are incompatible\n");
	//	return -1;
	//}
	*/

	if ( o_force ) o_interactive=FALSE;
	if ( o_wipe_slacks ) o_dont_delete = TRUE;

	if (o_wipe_free_space)
	{
		if (1 != argc)
		{
			fprintf( stderr,
				"Invalid number of parameters!\n"
				"Usage example: bcwipe -v -F /mount/point\n");
			return ERROR;
		}

		if (OK == wipe_free_space(argv[0]))
		{
			if (o_wipe_slacks)
			{
				o_recurse = TRUE;
			}
			else
				return OK;

		}
		else
		{
			return ERROR;
		}
	}

	if (o_nas_wiping)
	{
		nas_wiping(argc,argv);
	}
	else
	{
		for ( i = argc;i;i-- )
		{
			if ( OK != lstat(argv[i-1],&buff) )
			{
				fprintf(stderr,"Cannot stat %s: %s  \t - Skipping\n",argv[i-1],strerror(errno));
				continue;
			}
			if ( S_ISDIR(buff.st_mode) )  /*directory */
			{
				if ( o_recurse ) wipe_dir(argv[i-1]);
				else
				{
					fprintf(stderr,"%s is directory  \t - Skipping\n",argv[i-1]);
				}
			}
			else if ( S_ISREG(buff.st_mode) )   /*regular file */
			{
				(void) (wipe_file(argv[i-1]) || wipe_filename(argv[i-1]));
			}
			else if ( S_ISLNK(buff.st_mode) )   /*link */
			{
				if ( o_wipe_dev ) 
				    (void) wipe_file( argv[i-1] ); /* wipe_blkdev(argv[i-1]); */
				else 
				    (void) wipe_filename(argv[i-1]);
			}
			else if ( S_ISCHR(buff.st_mode) )   /*character device */
			{
				if ( o_wipe_dev ) (void) wipe_file(argv[i-1]);
				else (void) wipe_filename(argv[i-1]);
			}
			else if ( S_ISBLK(buff.st_mode) )   /*block device */
			{
				if ( o_wipe_dev ) 
					(void) wipe_file( argv[i-1] ); /*wipe_blkdev(argv[i-1]);*/
				else (void) 
					wipe_filename(argv[i-1]);
			}
			else if ( S_ISFIFO(buff.st_mode) )  /*named pipe (fifo) */
			{
				(void) wipe_filename(argv[i-1]);
			}
			else if ( S_ISSOCK(buff.st_mode) )  /*socket */
			{
				(void) wipe_filename(argv[i-1]);
			}
			else                                /*something else .... */
			{
				fprintf(stderr,"Unknown type of file: %s\n",argv[i-1]);
			}
		}
		if (o_wipe_slacks)
		{
			fprintf(stdout,"Wiped %d Kb of file slacks\n",(unsigned int)(total_size_of_slacks/1024));
		}
	}


	SHA1RandomFinalize( sRandom );
	SHA1RandomFree( &sRandom );

	return 0;
}


