/* cmd-sync-tree.c:
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "po/gettext.h"
#include "hackerlab/bugs/exception.h"
#include "hackerlab/cmd/main.h"
#include "hackerlab/fs/file-names.h"
#include "hackerlab/fs/cwd.h"
#include "hackerlab/vu/vu-dash.h"
#include "libfsutils/tmp-files.h"
#include "libfsutils/rmrf.h"
#include "libfsutils/copy-file.h"
#include "libfsutils/dir-as-cwd.h"
#include "libarch/namespace.h"
#include "libarch/project-tree.h"
#include "libarch/my.h"
#include "libarch/copy-project-tree.h"
#include "libarch/sync-tree.h"
#include "libarch/chatter.h"
#include "commands/cmd.h"
#include "commands/cmdutils.h"
#include "commands/sync-tree.h"
#include "commands/version.h"



static t_uchar * usage = N_("[options] revision");

#define OPTS(OP) \
  OP (opt_help_msg, "h", "help", 0, \
      N_("Display a help message and exit.")) \
  OP (opt_long_help, "H", 0, 0, \
      N_("Display a verbose help message and exit.")) \
  OP (opt_version, "V", "version", 0, \
      N_("Display a release identifier string\n" \
      "and exit.")) \
  OP (opt_dir, "d", "dir DIR", 1, \
      N_("Operate on project tree in DIR (default `.')")) \
  OP (opt_dest, 0, "dest DEST", 1, \
      N_("Instead of modifying the project tree in-place,\n" \
      "make a copy of it to DEST and apply the result to that")) \
  OP (opt_unescaped, 0, "unescaped", 0, \
      N_("show filenames in unescaped form"))


t_uchar arch_cmd_sync_tree_help[] = N_(
        "unify a project tree's patch-log with a given revision\n"
        "The new project tree is formed by getting the REVISION and adding \n"
        "all patch-log entries from REVISION.  No actual merging is "
        "performed -- only the patch-log is changed.\n");

enum options
{
  OPTS (OPT_ENUM)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_sync_tree (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  t_uchar * upon = NULL;
  t_uchar * dest = NULL;
  int escape_classes = arch_escape_classes;

  safe_buffer_fd (1, 0, O_WRONLY, 0);
  upon = str_save (0, ".");

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, libarch_version_string, arch_cmd_sync_tree_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;

        case opt_dir:
          {
            lim_free (0, upon);
            upon = str_save (0, option->arg_string);
            break;
          }

        case opt_dest:
          {
            lim_free (0, dest);
            dest = str_save (0, option->arg_string);
            break;
          }
        
        case opt_unescaped:
          {
            escape_classes = 0;
            break;
          }
        }
    }

  if (argc != 2)
    goto usage_error;

  {
    t_uchar * revspec = NULL;
    t_uchar * archive = NULL;
    t_uchar * revision = NULL;
    struct arch_archive * arch = NULL;
    arch_project_tree_t * orig_tree;
    arch_project_tree_t * dest_tree;

    revspec = argv[1];

    orig_tree = arch_project_tree_new (talloc_context, upon);

    if (!orig_tree->root)
      {
        safe_printfmt (2, _("%s: dir not in a project tree (%s)\n"),
                       argv[0], upon);
        exit (1);
      }

    revision = arch_determine_revision (&arch, NULL, revspec, argv[0]);

    arch_chatter (1, _("* making sync tree with %s/%s\n"), arch->official_name, revision);
    if (dest)
      arch_chatter (1, _("** source %s\n** dest %s\n"), orig_tree->root, dest);
    else
      arch_chatter (1, _("** directory %s\n"), orig_tree->root);

    if ( !dest)
      dest_tree = arch_project_tree_new (talloc_context, orig_tree -> root);
    else
      {
        safe_printfmt (1, _("* copying %s to %s\n"), orig_tree->root, dest);
        safe_flush (1);
        arch_copy_project_tree (orig_tree, dest, 1, 1);
        dest_tree = arch_project_tree_new (talloc_context, dest);
      }

    arch_sync_tree (1, dest_tree, arch, arch->official_name, revision);

    lim_free (0, archive);
    arch_archive_close (arch);
    arch_project_tree_delete (orig_tree);
    arch_project_tree_delete (dest_tree);

  }

  lim_free (0, upon);
  lim_free (0, dest);

  return 0;
}




/* tag: Tom Lord Tue Jun 17 16:02:09 2003 (cmd-sync-tree.c)
 */
