/* annotate.c
 *
 * vim:smartindent ts=8:sts=2:sta:et:ai:shiftwidth=2
 ****************************************************************
 * Copyright (C) 2005 Canonical Limited
 * 	Authors: Robert Collins <robert.collins@canonical.com>
 * Portions Copyright (C) 2003 Tom Lord, Stig Brautaset
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */

#define	PACKAGE	"bazaar"

#include "config-options.h"
#include "po/gettext.h"
#include "hackerlab/bugs/exception.h"
#include "hackerlab/bugs/panic.h"
#include "hackerlab/char/str.h"
#include "hackerlab/vu/safe.h"
#include "libawk/relational.h"
#include "libfsutils/file-contents.h"
#include "libarch/annotation-builder.h"
#include "libarch/patch-parser.h"
#include "libarch/debug.h"

static void
setup_msg_env(void)
{
  t_uchar * localedir;
  t_uchar const * statement_with_no_effect_catcher;

  localedir = str_alloc_cat (0, cfg__std__prefix, "/locale");

  setlocale (LC_ALL, "");
  /* statement_with_no_effect_catcher = bindtextdomain (PACKAGE, localedir); */
  statement_with_no_effect_catcher = textdomain (PACKAGE);

  lim_free (0, localedir);
}

int
main (int argc, char * argv[])
{
  struct exception *e;
  
  Try
    {
      arch_debug_init();
      setup_msg_env();

      panic_should_throw (1);
      if (argc <= 0)
          panic ("annotate invoked with argc <= 0 ?!?");

      if (argc < 2)
        {
usage:
          safe_printfmt (2, _("usage: annotate current-file [patch label ...]\n"));
          exit (1);
        }

      if (argc % 2 != 0)
          goto usage;

        {
          arch_annotation_builder_t * builder;
          int position;
          t_uchar *currentFile = str_save (0, argv[1]);
          rel_table patches = rel_unflatten (argc - 2, 2, &argv[2]);
          builder = arch_annotation_builder_new (talloc_context);
          arch_annotation_builder_add_file (builder, currentFile, file_line_count(currentFile));
          for (position =0; position < rel_n_records (patches); ++position)
            {
              arch_patch_id aPatch;
              t_uchar *patch_content;
              patch_line_changes_list changes;
              arch_patch_id_init (&aPatch, patches[position][1]);

              arch_annotation_builder_add_patch (builder, &aPatch, "unknown");
              patch_content = file_contents (patches[position][0]);
              changes = patch_parser_parse(patch_content);
              arch_annotation_builder_process_changes(builder, changes);

              ar_free_patch_line_change(&changes);
              lim_free (0, patch_content);
              arch_patch_id_finalise (&aPatch);
            }
            {
              int line;
              rel_table lines;
              arch_annotated_file_t *aFile;
              lines=file_lines (currentFile);
              aFile = arch_annotation_builder_get_file (builder, talloc_context, currentFile);
              for (line = 0; line < rel_n_records (lines); ++line)
                {
                  if (aFile->lines[line])
                      safe_printfmt (1, "%s: %s\n", arch_patch_id_patch_id(arch_patch_patch_id(aFile->lines[line])), lines[line][0]);
                  else 
                      safe_printfmt (1, "unknown: %s\n", lines[line][0]);
                }
              talloc_free (aFile);
              rel_free_table (lines);
            }
          talloc_free (builder);
        }
    }
  Catch(e)
    {
      /* disable exception throwing if further errors occur */
      panic_should_throw (0);
      safe_printfmt (2, _("annotate: uncaught exception: %d:(%s)\n"), e->code, e->msg);
      safe_printfmt (2, _("  please report this as a bug to bazaar@lists.canonical.com\n"));
      talloc_free (e);
      exit (2);
    }
  return 0;
}
