/* unit-annotate.c
 *
 * vim:smartindent ts=8:sts=2:sta:et:ai:shiftwidth=2
 ****************************************************************
 * Copyright (C) 2005, Canonical Limited
 *               Author: Robert Collins
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "hackerlab/vu/safe-printfmt.h"
#include "hackerlab/bugs/exception.h"
#include "hackerlab/bugs/panic.h"
#include "hackerlab/char/str.h"
#include "hackerlab/os/errno.h"
#include "libarch/annotation-builder.h"
#include "libarch/annotated-file.h"
#include "libarch/debug.h"
#include "libarch/file-offset-mapper.h"



/* typedefs */



/* local methods */

/**
 * \brief  create a mapper with :
 *
 * 0 - 1
 * 1 - 2
 * 2 - 
 *   - 3
 * 3 - 4
 * 4 - 5
 */
static file_offset_mapper_t
get_double_gap_mapper (void)
{
    file_offset_mapper_t mapper;
    /* lines 0-5 */
    file_offset_mapper_init (&mapper, 6); 
    file_offset_mapper_add (&mapper, 0);
    file_offset_mapper_subtract (&mapper, 2);
    file_offset_mapper_add (&mapper, 3);
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 1);
    invariant_int_cmp (mapper.map[0].length , 2);
    invariant_int_cmp (mapper.map[1].from , 3);
    invariant_int_cmp (mapper.map[1].to , 4);
    invariant_int_cmp (mapper.map[1].length , 2);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), 2);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 3), 4);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 4), 5);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 5), -1);
    return mapper;
}

static void
check_basic_replacement (void)
{
    file_offset_mapper_t mapper;
    file_offset_mapper_init (&mapper, 3);
    /*
     * 0 - 0
     * 1 - 1
     * 2 - 2
     * -> -1
     * 0 - 0
     * 1 - deleted
     * 2 - 1
     * 3 - 2
     */
    file_offset_mapper_subtract (&mapper, 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 3), 2);
    invariant_int_cmp (ar_size_file_offset_map_entry (mapper.map) , 2);
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 0);
    invariant_int_cmp (mapper.map[0].length , 1);
    invariant_int_cmp (mapper.map[1].from , 2);
    invariant_int_cmp (mapper.map[1].to , 1);
    invariant_int_cmp (mapper.map[1].length , 2);
    /*
     * 0 - 0
     * 1 - deleted
     * 2 - 1
     * 3 - 2
     * -> +2
     * 0 - 0
     * 1 - 
     *   - 1
     * 2 - 2
     */
    file_offset_mapper_add (&mapper, 2);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), 2);
    invariant_int_cmp (ar_size_file_offset_map_entry (mapper.map) , 2);
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 0);
    invariant_int_cmp (mapper.map[0].length , 1);
    invariant_int_cmp (mapper.map[1].from , 2);
    invariant_int_cmp (mapper.map[1].to , 2);
    invariant_int_cmp (mapper.map[1].length , 1);
    /*
     * 0 - 0
     * 1 - deleted
     * mapped - 1
     * 2 - 2
     * -> +0
     *   - 0
     * 0 - 
     *   - 1
     * 1 - 2
     */
    file_offset_mapper_add (&mapper, 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), 2);
    invariant_int_cmp (ar_size_file_offset_map_entry (mapper.map) , 1);
    invariant_int_cmp (mapper.map[0].from , 1);
    invariant_int_cmp (mapper.map[0].to , 2);
    invariant_int_cmp (mapper.map[0].length , 1);
    /*
     * mapped - 0
     * 0 - deleted
     * mapped - 1
     * 1 - 2
     * -> +0
     * mapped  - 0
     * mapped  - 1
     * 0 - 2
     */
    file_offset_mapper_add (&mapper, 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 2);
    invariant_int_cmp (ar_size_file_offset_map_entry (mapper.map) , 1);
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 2);
    invariant_int_cmp (mapper.map[0].length , 1);
    /*
     * mapped - 0
     * mapped - 1
     * 0 - 2
     * -> +0
     * mapped  - 0
     * mapped  - 1
     * mapped  - 2
     */
    file_offset_mapper_add (&mapper, 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), -1);
    invariant_int_cmp (ar_size_file_offset_map_entry (mapper.map) , 0);
    file_offset_mapper_finalise (&mapper);
}

static void
check_basic_adds (void)
{
    file_offset_mapper_t mapper;
    file_offset_mapper_init (&mapper, 1);
    /*
     * 0 - 0
     * ->
     * mapped - 0
     */
    file_offset_mapper_add (&mapper, 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), -1);
    invariant_int_cmp (ar_size_file_offset_map_entry (mapper.map) , 0);
    file_offset_mapper_finalise (&mapper);
    
    file_offset_mapper_init (&mapper, 2);
    /*
     * 0 - 0
     * 1 - 1
     * -> +0
     * mapped - 0
     * 0 - 1
     */
    file_offset_mapper_add (&mapper, 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), -1);
    invariant_int_cmp (ar_size_file_offset_map_entry (mapper.map) , 1);
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 1);
    invariant_int_cmp (mapper.map[0].length , 1);
    file_offset_mapper_finalise (&mapper);
    
    file_offset_mapper_init (&mapper, 2);
    /*
     * 0 - 0
     * 1 - 1
     * -> +1
     * 0 - 0
     * mapped - 1
     */
    file_offset_mapper_add (&mapper, 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), -1);
    invariant_int_cmp (ar_size_file_offset_map_entry (mapper.map) , 1);
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 0);
    invariant_int_cmp (mapper.map[0].length , 1);
    file_offset_mapper_finalise (&mapper);
    
    file_offset_mapper_init (&mapper, 3);
    /*
     * 0 - 0
     * 1 - 1
     * 2 - 2
     * -> +1
     * 0 - 0
     * mapped - 1
     * 1 - 2
     */
    file_offset_mapper_add (&mapper, 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), 2);
    invariant_int_cmp (ar_size_file_offset_map_entry (mapper.map) , 2);
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 0);
    invariant_int_cmp (mapper.map[0].length , 1);
    file_offset_mapper_finalise (&mapper);
}

static void
check_real_world_11_10_patch (void)
{
    file_offset_mapper_t mapper;
    file_offset_mapper_init (&mapper, 79);
    /*
     * 0 - 0
     * 78 - 78
     */

    file_offset_mapper_subtract (&mapper, 3);
    file_offset_mapper_subtract (&mapper, 4);
    file_offset_mapper_add (&mapper, 5);
    file_offset_mapper_subtract (&mapper, 6);
    file_offset_mapper_subtract (&mapper, 7);
    file_offset_mapper_subtract (&mapper, 8);
    file_offset_mapper_subtract (&mapper, 9);
    file_offset_mapper_subtract (&mapper, 10);
    file_offset_mapper_subtract (&mapper, 11);
    file_offset_mapper_subtract (&mapper, 12);
    file_offset_mapper_subtract (&mapper, 13);
    file_offset_mapper_subtract (&mapper, 14);
    file_offset_mapper_subtract (&mapper, 15);
    file_offset_mapper_subtract (&mapper, 16);
    file_offset_mapper_subtract (&mapper, 17);
    file_offset_mapper_subtract (&mapper, 18);
    file_offset_mapper_subtract (&mapper, 19);
    file_offset_mapper_subtract (&mapper, 20);
    file_offset_mapper_subtract (&mapper, 21);
    
    file_offset_mapper_subtract (&mapper, 6);
    file_offset_mapper_subtract (&mapper, 7);
    file_offset_mapper_subtract (&mapper, 8);
    file_offset_mapper_subtract (&mapper, 9);
    file_offset_mapper_subtract (&mapper, 10);
    file_offset_mapper_subtract (&mapper, 11);
    file_offset_mapper_subtract (&mapper, 12);
    file_offset_mapper_subtract (&mapper, 13);
    file_offset_mapper_subtract (&mapper, 14);
    file_offset_mapper_subtract (&mapper, 15);
    file_offset_mapper_subtract (&mapper, 16);
    file_offset_mapper_subtract (&mapper, 17);
    file_offset_mapper_add (&mapper, 18);
    file_offset_mapper_add (&mapper, 18);
    file_offset_mapper_add (&mapper, 18);
    file_offset_mapper_add (&mapper, 18);
    file_offset_mapper_add (&mapper, 18);
    file_offset_mapper_add (&mapper, 18);
    file_offset_mapper_add (&mapper, 18);
    file_offset_mapper_add (&mapper, 18);
    file_offset_mapper_add (&mapper, 18);
    file_offset_mapper_add (&mapper, 18);
    file_offset_mapper_add (&mapper, 18);
    file_offset_mapper_add (&mapper, 18);
    file_offset_mapper_add (&mapper, 18);
    file_offset_mapper_add (&mapper, 18);
    file_offset_mapper_add (&mapper, 18);
    file_offset_mapper_add (&mapper, 18);
    file_offset_mapper_add (&mapper, 18);
    file_offset_mapper_add (&mapper, 18);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 5), 4);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 18), 7);

    invariant_int_cmp (ar_size_file_offset_map_entry (mapper.map) , 3);
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 0);
    invariant_int_cmp (mapper.map[0].length , 3);
    invariant_int_cmp (mapper.map[1].from , 5);
    invariant_int_cmp (mapper.map[1].to , 4);
    invariant_int_cmp (mapper.map[1].length , 1);
    invariant_int_cmp (mapper.map[2].from , 18);
    invariant_int_cmp (mapper.map[2].to , 7);
    invariant_int_cmp (mapper.map[2].length , 72);
    /*
     * 0 - 0
     * ...
     * 30 - 30
     * ->+ 0
     * mapped - 0
     * 0 - 1
     * ...
     * 29 - 30
     */
    file_offset_mapper_finalise (&mapper);
}
static void
check_real_world_2_patch (void)
{
    file_offset_mapper_t mapper;
    file_offset_mapper_init (&mapper, 31);
    /*
     * 0 - 0
     * ...
     * 30 - 30
     * ->+ 0
     * mapped - 0
     * 0 - 1
     * ...
     * 29 - 30
     */
    file_offset_mapper_add (&mapper, 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), 2);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 29), 30);
    invariant_int_cmp (ar_size_file_offset_map_entry (mapper.map) , 1);
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 1);
    invariant_int_cmp (mapper.map[0].length , 30);
    
    /*
     *   - 0
     * ...
     * 29 - 30
     * ->+ 3 * 6 
     * mapped - 0
     * 0 - 1
     * 1 - 2
     * 2 - 3
     *   - 4
     *   - 5
     *   - 6
     *   - 7
     *   - 8
     *   - 9
     * 3 - 10
     * ...
     * 23 - 30
     */
    file_offset_mapper_add (&mapper, 3);
    file_offset_mapper_add (&mapper, 3);
    file_offset_mapper_add (&mapper, 3);
    file_offset_mapper_add (&mapper, 3);
    file_offset_mapper_add (&mapper, 3);
    file_offset_mapper_add (&mapper, 3);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), 2);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), 3);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 3), 10);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 23), 30);
    invariant_int_cmp (ar_size_file_offset_map_entry (mapper.map) , 2);
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 1);
    invariant_int_cmp (mapper.map[0].length , 3);
    invariant_int_cmp (mapper.map[1].from , 3);
    invariant_int_cmp (mapper.map[1].to , 10);
    invariant_int_cmp (mapper.map[1].length , 21);
    
    /*
     * mapped - 0
     * 0 - 1
     * 1 - 2
     * 2 - 3
     *   - 4
     *   - 5
     *   - 6
     *   - 7
     *   - 8
     *   - 9
     * 3 - 10
     * ...
     * 23 - 30
     * -> -8
     * mapped - 0
     * 0 - 1
     * 1 - 2
     * 2 - 3
     *   - 4
     *   - 5
     *   - 6
     *   - 7
     *   - 8
     *   - 9
     * 3 - 10
     * ...
     * 7 - 14
     * 8 - deleted
     * 9 - 15
     * ...
     * 24 - 30
     */
    file_offset_mapper_subtract (&mapper, 8);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), 2);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), 3);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 3), 10);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 7), 14);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 8), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 9), 15);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 24), 30);
    invariant_int_cmp (ar_size_file_offset_map_entry (mapper.map) , 3);
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 1);
    invariant_int_cmp (mapper.map[0].length , 3);
    invariant_int_cmp (mapper.map[1].from , 3);
    invariant_int_cmp (mapper.map[1].to , 10);
    invariant_int_cmp (mapper.map[1].length , 5);
    invariant_int_cmp (mapper.map[2].from , 9);
    invariant_int_cmp (mapper.map[2].to , 15);
    invariant_int_cmp (mapper.map[2].length , 16);
    
    /*
     * mapped - 0
     * 0 - 1
     * 1 - 2
     * 2 - 3
     *   - 4
     *   - 5
     *   - 6
     *   - 7
     *   - 8
     *   - 9
     * 3 - 10
     * ...
     * 7 - 14
     * 8 - deleted
     * 9 - 15
     * ...
     * 24 - 30
     * -> +9
     * mapped - 0
     * 0 - 1
     * 1 - 2
     * 2 - 3
     *   - 4
     *   - 5
     *   - 6
     *   - 7
     *   - 8
     *   - 9
     * 3 - 10
     * ...
     * 7 - 14
     * 8 - deleted
     * mapped - 15
     * 9 - 16
     * ...
     * 23 - 30
     */
    file_offset_mapper_add (&mapper, 9);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), 2);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), 3);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 3), 10);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 7), 14);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 8), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 9), 16);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 23), 30);
    invariant_int_cmp (ar_size_file_offset_map_entry (mapper.map) , 3);
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 1);
    invariant_int_cmp (mapper.map[0].length , 3);
    invariant_int_cmp (mapper.map[1].from , 3);
    invariant_int_cmp (mapper.map[1].to , 10);
    invariant_int_cmp (mapper.map[1].length , 5);
    invariant_int_cmp (mapper.map[2].from , 9);
    invariant_int_cmp (mapper.map[2].to , 16);
    invariant_int_cmp (mapper.map[2].length , 15);
    
    /*
     * mapped - 0
     * 0 - 1
     * 1 - 2
     * 2 - 3
     *   - 4
     *   - 5
     *   - 6
     *   - 7
     *   - 8
     *   - 9
     * 3 - 10
     * ...
     * 7 - 14
     * 8 - deleted
     * mapped - 15
     * 9 - 16
     * ...
     * 23 - 30
     * -> - 22
     * mapped - 0
     * 0 - 1
     * 1 - 2
     * 2 - 3
     *   - 4
     *   - 5
     *   - 6
     *   - 7
     *   - 8
     *   - 9
     * 3 - 10
     * ...
     * 7 - 14
     * 8 - deleted
     * mapped - 15
     * 9 - 16
     * ...
     * 21 - 28
     * 22 - deleted
     * 23 - 29
     * 24 - 30
     */
    file_offset_mapper_subtract (&mapper, 22);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), 2);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), 3);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 3), 10);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 7), 14);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 8), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 9), 16);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 21), 28);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 22), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 23), 29);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 24), 30);
    invariant_int_cmp (ar_size_file_offset_map_entry (mapper.map) , 4);
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 1);
    invariant_int_cmp (mapper.map[0].length , 3);
    invariant_int_cmp (mapper.map[1].from , 3);
    invariant_int_cmp (mapper.map[1].to , 10);
    invariant_int_cmp (mapper.map[1].length , 5);
    invariant_int_cmp (mapper.map[2].from , 9);
    invariant_int_cmp (mapper.map[2].to , 16);
    invariant_int_cmp (mapper.map[2].length , 13);
    invariant_int_cmp (mapper.map[3].from , 23);
    invariant_int_cmp (mapper.map[3].to , 29);
    invariant_int_cmp (mapper.map[3].length , 2);
    
    /*
     * mapped - 0
     * 0 - 1
     * 1 - 2
     * 2 - 3
     *   - 4
     *   - 5
     *   - 6
     *   - 7
     *   - 8
     *   - 9
     * 3 - 10
     * ...
     * 7 - 14
     * 8 - deleted
     * mapped - 15
     * 9 - 16
     * ...
     * 21 - 28
     * 22 - deleted
     * 23 - 29
     * 24 - 30
     * -> +0 
     * mapped - 0
     * mapped - 1
     * 0 - 2
     * 1 - 3
     *   - 4
     *   - 5
     *   - 6
     *   - 7
     *   - 8
     *   - 9
     * 2 - 10
     * ...
     * 6 - 14
     * 7 - deleted
     * mapped - 15
     * 8 - 16
     * ...
     * 20 - 28
     * 21 - deleted
     * 22 - 29
     * 23 - 30
     */
    file_offset_mapper_add (&mapper, 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 2);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), 3);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), 10);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 3), 11);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 6), 14);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 7), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 8), 16);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 9), 17);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 21), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 22), 29);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 23), 30);
    invariant_int_cmp (ar_size_file_offset_map_entry (mapper.map) , 4);
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 2);
    invariant_int_cmp (mapper.map[0].length , 2);
    invariant_int_cmp (mapper.map[1].from , 2);
    invariant_int_cmp (mapper.map[1].to , 10);
    invariant_int_cmp (mapper.map[1].length , 5);
    invariant_int_cmp (mapper.map[2].from , 8);
    invariant_int_cmp (mapper.map[2].to , 16);
    invariant_int_cmp (mapper.map[2].length , 13);
    invariant_int_cmp (mapper.map[3].from , 22);
    invariant_int_cmp (mapper.map[3].to , 29);
    invariant_int_cmp (mapper.map[3].length , 2);
    file_offset_mapper_finalise (&mapper);
}


static void
check_complex_mapper_adds (void)
{
    file_offset_mapper_t mapper;

    mapper = get_double_gap_mapper ();
    file_offset_mapper_add (&mapper, 0);
    /*
     *   - 0
     * 0 - 1
     * 1 - 2
     * 2 - 
     *   - 3
     * 3 - 4
     * 4 - 5
     * add 0, makes:
     *   - 0
     *   - 1
     * 0 - 2
     * 1 -
     *   - 3
     * 2 - 4
     * 3 - 5
     * 4 - 
     */
    /* maps */
    invariant_int_cmp (file_offset_mapper_map (&mapper, -1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 2);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), 4);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 3), 5);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 4), -1);
    /* internals */
    invariant_int_cmp (ar_size_file_offset_map_entry (mapper.map) , 2);
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 2);
    invariant_int_cmp (mapper.map[0].length , 1);
    invariant_int_cmp (mapper.map[1].from , 2);
    invariant_int_cmp (mapper.map[1].to , 4);
    invariant_int_cmp (mapper.map[1].length , 2);
    file_offset_mapper_finalise (&mapper);
    
    mapper = get_double_gap_mapper ();
    file_offset_mapper_add (&mapper, 1);
    /*
     *   - 0
     * 0 - 1
     * 1 - 2
     * 2 - 
     *   - 3
     * 3 - 4
     * 4 - 5
     * add 1, makes:
     *   - 0
     * 0 - 1
     *   - 2
     * 1 - 
     *   - 3
     * 2 - 4
     * 3 - 5
     * 4 - 
     */
    /* maps */
    invariant_int_cmp (file_offset_mapper_map (&mapper, -1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), 4);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 3), 5);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 4), -1);
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 1);
    invariant_int_cmp (mapper.map[0].length , 1);
    invariant_int_cmp (mapper.map[1].from , 2);
    invariant_int_cmp (mapper.map[1].to , 4);
    invariant_int_cmp (mapper.map[1].length , 2);
    file_offset_mapper_finalise (&mapper);
    
    mapper = get_double_gap_mapper ();
    file_offset_mapper_add (&mapper, 2);
    /*
     * 0 - 1
     * 1 - 2
     * 2 - 
     *   - 3
     * 3 - 4
     * 4 - 5
     * add 2, makes:
     * 0 - 1
     * 1 - 2
     *   - 3
     * 2 - 4
     * 3 - 5
     */
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 1);
    invariant_int_cmp (mapper.map[0].length , 2);
    invariant_int_cmp (mapper.map[1].from , 2);
    invariant_int_cmp (mapper.map[1].to , 4);
    invariant_int_cmp (mapper.map[1].length , 2);
    /* maps */
    invariant_int_cmp (file_offset_mapper_map (&mapper, -1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), 2);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), 4);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 3), 5);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 4), -1);
    file_offset_mapper_finalise (&mapper);
    
    mapper = get_double_gap_mapper ();
    file_offset_mapper_add (&mapper, 3);
    /*
     * 0 - 1
     * 1 - 2
     * 2 - 
     *   - 3
     * 3 - 4
     * 4 - 5
     * add 3, makes:
     * 0 - 1
     * 1 - 2
     * 2 -
     *   - 3
     *   - 4
     * 3 - 5
     */
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 1);
    invariant_int_cmp (mapper.map[0].length , 2);
    invariant_int_cmp (mapper.map[1].from , 3);
    invariant_int_cmp (mapper.map[1].to , 5);
    invariant_int_cmp (mapper.map[1].length , 1);
    /* maps */
    invariant_int_cmp (file_offset_mapper_map (&mapper, -1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), 2);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 3), 5);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 4), -1);
    file_offset_mapper_finalise (&mapper);
    
    mapper = get_double_gap_mapper ();
    file_offset_mapper_add (&mapper, 4);
    /*
     * 0 - 1
     * 1 - 2
     * 2 - 
     *   - 3
     * 3 - 4
     * 4 - 5
     * add 4, makes:
     * 0 - 1
     * 1 - 2
     * 2 -
     *   - 3
     * 3 - 4
     *   - 5
     */
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 1);
    invariant_int_cmp (mapper.map[0].length , 2);
    invariant_int_cmp (mapper.map[1].from , 3);
    invariant_int_cmp (mapper.map[1].to , 4);
    invariant_int_cmp (mapper.map[1].length , 1);
    /* maps */
    invariant_int_cmp (file_offset_mapper_map (&mapper, -1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), 2);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 3), 4);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 4), -1);
    file_offset_mapper_finalise (&mapper);

    mapper = get_double_gap_mapper ();
    file_offset_mapper_add (&mapper, 5);
    /*
     * 0 - 1
     * 1 - 2
     * 2 - 
     *   - 3
     * 3 - 4
     * 4 - 5
     * add 4, makes:
     * 0 - 1
     * 1 - 2
     * 2 -
     *   - 3
     * 3 - 4
     * 4 - 5
     */
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 1);
    invariant_int_cmp (mapper.map[0].length , 2);
    invariant_int_cmp (mapper.map[1].from , 3);
    invariant_int_cmp (mapper.map[1].to , 4);
    invariant_int_cmp (mapper.map[1].length , 2);
    /* maps */
    invariant_int_cmp (file_offset_mapper_map (&mapper, -1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), 2);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 3), 4);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 4), 5);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 5), -1);
    file_offset_mapper_finalise (&mapper);


    /* test setting up a somewhat complex mapper */
    /* lines 0-99 */
    file_offset_mapper_init (&mapper, 100); 
    invariant_int_cmp (mapper.map[0].from, 0);
    invariant_int_cmp (mapper.map[0].to, 0);
    invariant_int_cmp (mapper.map[0].length, 100);
    invariant_int_cmp (file_offset_mapper_map (&mapper, -1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 99), 99);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 100), -1);
    file_offset_mapper_add (&mapper, 45);
    /* 44 - 44
     * 45 - 45
     * becomes
     * 44 - 44
     *    - 45
     * 45 - 46
     */
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 0);
    invariant_int_cmp (mapper.map[0].length , 45);
    invariant_int_cmp (mapper.map[1].from , 45);
    invariant_int_cmp (mapper.map[1].to , 46);
    invariant_int_cmp (mapper.map[1].length , 54);
    /* maps */
    invariant_int_cmp (file_offset_mapper_map (&mapper, -1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 44), 44);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 45), 46);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 98), 99);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 99), -1);
    
    file_offset_mapper_add (&mapper, 45);
    /* 44 - 44
     *    - 45
     * 45 - 46
     * becomes
     * 44 - 44
     *    - 45
     *    - 46
     * 45 - 47
     */
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 0);
    invariant_int_cmp (mapper.map[0].length , 45);
    invariant_int_cmp (mapper.map[1].from , 45);
    invariant_int_cmp (mapper.map[1].to , 47);
    invariant_int_cmp (mapper.map[1].length , 53);
    /* maps */
    invariant_int_cmp (file_offset_mapper_map (&mapper, -1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 44), 44);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 45), 47);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 97), 99);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 98), -1);
    
    file_offset_mapper_add (&mapper, 44);
    /* 43 - 43
     * 44 - 44
     *    - 45
     *    - 46
     * 45 - 47
     * becomes
     * 43 - 43
     *    - 44
     *    - 45
     *    - 46
     * 44 - 47
     * 45 - 48
     */
    /* maps */
    invariant_int_cmp (file_offset_mapper_map (&mapper, -1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 43), 43);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 44), 47);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 45), 48);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 96), 99);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 97), -1);
    /* internals */
    invariant_int_cmp (ar_size_file_offset_map_entry (mapper.map) , 2);
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 0);
    invariant_int_cmp (mapper.map[0].length , 44);
    invariant_int_cmp (mapper.map[1].from , 44);
    invariant_int_cmp (mapper.map[1].to , 47);
    invariant_int_cmp (mapper.map[1].length , 53);
    
    
    file_offset_mapper_add (&mapper, 45);
    /* 43 - 43
     *    - 44
     *    - 45
     *    - 46
     * 44 - 47
     * 45 - 48
     * becomes
     * 43 - 43
     *    - 44
     *    - 45
     *    - 46
     * 44 - 47
     *    - 48
     * 45 - 49
     * 46 - 50
     */
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 0);
    invariant_int_cmp (mapper.map[0].length , 44);
    invariant_int_cmp (mapper.map[1].from , 44);
    invariant_int_cmp (mapper.map[1].to , 47);
    invariant_int_cmp (mapper.map[1].length , 1);
    invariant_int_cmp (mapper.map[2].from , 45);
    invariant_int_cmp (mapper.map[2].to , 49);
    invariant_int_cmp (mapper.map[2].length , 51);
    /* maps */
    invariant_int_cmp (file_offset_mapper_map (&mapper, -1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 43), 43);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 44), 47);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 45), 49);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 95), 99);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 96), -1);
    
    file_offset_mapper_add (&mapper, 45);
    /* 43 - 43
     *    - 44
     *    - 45
     *    - 46
     * 44 - 47
     *    - 48
     * 45 - 49
     * 46 - 50
     * becomes
     * 43 - 43
     *    - 44
     *    - 45
     *    - 46
     * 44 - 47
     *    - 48
     *    - 49
     * 45 - 50
     * 46 - 51
     */
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 0);
    invariant_int_cmp (mapper.map[0].length , 44);
    invariant_int_cmp (mapper.map[1].from , 44);
    invariant_int_cmp (mapper.map[1].to , 47);
    invariant_int_cmp (mapper.map[1].length , 1);
    invariant_int_cmp (mapper.map[2].from , 45);
    invariant_int_cmp (mapper.map[2].to , 50);
    invariant_int_cmp (mapper.map[2].length , 50);
    /* maps */
    invariant_int_cmp (file_offset_mapper_map (&mapper, -1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 43), 43);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 44), 47);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 45), 50);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 94), 99);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 95), -1);
    
    file_offset_mapper_add (&mapper, 45);
    /*
     * 43 - 43
     *    - 44
     *    - 45
     *    - 46
     * 44 - 47
     *    - 48
     *    - 49
     * 45 - 50
     * 46 - 51
     * becomes
     * 43 - 43
     *    - 44
     *    - 45
     *    - 46
     * 44 - 47
     *    - 48
     *    - 49
     *    - 50
     * 45 - 51
     * 46 - 52
     */
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 0);
    invariant_int_cmp (mapper.map[0].length , 44);
    invariant_int_cmp (mapper.map[1].from , 44);
    invariant_int_cmp (mapper.map[1].to , 47);
    invariant_int_cmp (mapper.map[1].length , 1);
    invariant_int_cmp (mapper.map[2].from , 45);
    invariant_int_cmp (mapper.map[2].to , 51);
    invariant_int_cmp (mapper.map[2].length , 49);
    /* maps */
    invariant_int_cmp (file_offset_mapper_map (&mapper, -1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 43), 43);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 44), 47);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 45), 51);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 93), 99);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 94), -1);
    
    file_offset_mapper_add (&mapper, 0);
    /*
     * 43 - 43
     *    - 44
     *    - 45
     *    - 46
     * 44 - 47
     *    - 48
     *    - 49
     *    - 50
     * 45 - 51
     * 46 - 52
     * becomes
     * ...
     * 42 - 43
     *    - 44
     *    - 45
     *    - 46
     * 43 - 47
     *    - 48
     *    - 49
     *    - 50
     * 44 - 51
     * 45 - 52
     */
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 1);
    invariant_int_cmp (mapper.map[0].length , 43);
    invariant_int_cmp (mapper.map[1].from , 43);
    invariant_int_cmp (mapper.map[1].to , 47);
    invariant_int_cmp (mapper.map[1].length , 1);
    invariant_int_cmp (mapper.map[2].from , 44);
    invariant_int_cmp (mapper.map[2].to , 51);
    invariant_int_cmp (mapper.map[2].length , 49);
    /* maps */
    invariant_int_cmp (file_offset_mapper_map (&mapper, -1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 42), 43);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 43), 47);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 44), 51);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 92), 99);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 93), -1);
    
    file_offset_mapper_add (&mapper, -1);
    /*
     * no impact
     * ...
     * 42 - 43
     *    - 44
     *    - 45
     *    - 46
     * 43 - 47
     *    - 48
     *    - 49
     *    - 50
     * 44 - 51
     * 45 - 52
     */
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 1);
    invariant_int_cmp (mapper.map[0].length , 43);
    invariant_int_cmp (mapper.map[1].from , 43);
    invariant_int_cmp (mapper.map[1].to , 47);
    invariant_int_cmp (mapper.map[1].length , 1);
    invariant_int_cmp (mapper.map[2].from , 44);
    invariant_int_cmp (mapper.map[2].to , 51);
    invariant_int_cmp (mapper.map[2].length , 49);
    /* maps */
    invariant_int_cmp (file_offset_mapper_map (&mapper, -1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 42), 43);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 43), 47);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 44), 51);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 92), 99);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 93), -1);
    
    file_offset_mapper_add (&mapper, 93);
    /*
     * no impact
     * ...
     * 42 - 43
     *    - 44
     *    - 45
     *    - 46
     * 43 - 47
     *    - 48
     *    - 49
     *    - 50
     * 44 - 51
     * 45 - 52
     */
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 1);
    invariant_int_cmp (mapper.map[0].length , 43);
    invariant_int_cmp (mapper.map[1].from , 43);
    invariant_int_cmp (mapper.map[1].to , 47);
    invariant_int_cmp (mapper.map[1].length , 1);
    invariant_int_cmp (mapper.map[2].from , 44);
    invariant_int_cmp (mapper.map[2].to , 51);
    invariant_int_cmp (mapper.map[2].length , 49);
    /* maps */
    invariant_int_cmp (file_offset_mapper_map (&mapper, -1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 42), 43);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 43), 47);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 44), 51);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 92), 99);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 93), -1);
    
    file_offset_mapper_add (&mapper, 92);
    /*
     * ...
     * 42 - 43
     *    - 44
     *    - 45
     *    - 46
     * 43 - 47
     *    - 48
     *    - 49
     *    - 50
     * 44 - 51
     * 45 - 52
     * becomes
     * ...
     * 42 - 43
     *    - 44
     *    - 45
     *    - 46
     * 43 - 47
     *    - 48
     *    - 49
     *    - 50
     * 44 - 51
     * 45 - 52
     * ...
     * 91 - 98
     *    - 99
     */
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 1);
    invariant_int_cmp (mapper.map[0].length , 43);
    invariant_int_cmp (mapper.map[1].from , 43);
    invariant_int_cmp (mapper.map[1].to , 47);
    invariant_int_cmp (mapper.map[1].length , 1);
    invariant_int_cmp (mapper.map[2].from , 44);
    invariant_int_cmp (mapper.map[2].to , 51);
    invariant_int_cmp (mapper.map[2].length , 48);
    /* maps */
    invariant_int_cmp (file_offset_mapper_map (&mapper, -1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 42), 43);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 43), 47);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 44), 51);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 91), 98);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 92), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 93), -1);

    /* test offset adjustment walks up the list,
     * not just increments 
     */
    file_offset_mapper_add (&mapper, 42);
    /*
     * ...
     * 42 - 43
     *    - 44
     *    - 45
     *    - 46
     * 43 - 47
     *    - 48
     *    - 49
     *    - 50
     * 44 - 51
     * 45 - 52
     * ...
     * 91 - 98
     *    - 99
     * becomes
     * ...
     * 41 - 42
     *    - 43
     *    - 44
     *    - 45
     *    - 46
     * 42 - 47
     *    - 48
     *    - 49
     *    - 50
     * 43 - 51
     * 44 - 52
     * ...
     * 90 - 98
     *    - 99
     */
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 1);
    invariant_int_cmp (mapper.map[0].length , 42);
    invariant_int_cmp (mapper.map[1].from , 42);
    invariant_int_cmp (mapper.map[1].to , 47);
    invariant_int_cmp (mapper.map[1].length , 1);
    invariant_int_cmp (mapper.map[2].from , 43);
    invariant_int_cmp (mapper.map[2].to , 51);
    invariant_int_cmp (mapper.map[2].length , 48);
    /* maps */
    invariant_int_cmp (file_offset_mapper_map (&mapper, -1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 41), 42);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 42), 47);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 43), 51);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 44), 52);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 90), 98);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 91), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 92), -1);



    file_offset_mapper_finalise (&mapper); 

}

/**
 * \brief  create a mapper with :
 *
 * 43 - 43
 *    - 45
 *    - 46
 *    - 47
 * 44 - 48
 * 45 - 49
 *    - 50
 *    - 51
 *    - 52
 * 46 - 53
 */
static file_offset_mapper_t
get_complex_add_mapper (void)
{
    file_offset_mapper_t mapper;
    /* lines 0-99 */
    file_offset_mapper_init (&mapper, 100); 
    file_offset_mapper_add (&mapper, 45);
    file_offset_mapper_add (&mapper, 45);
    file_offset_mapper_add (&mapper, 44);
    file_offset_mapper_add (&mapper, 46);
    file_offset_mapper_add (&mapper, 46);
    file_offset_mapper_add (&mapper, 46);
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 0);
    invariant_int_cmp (mapper.map[0].length , 44);
    invariant_int_cmp (mapper.map[1].from , 44);
    invariant_int_cmp (mapper.map[1].to , 47);
    invariant_int_cmp (mapper.map[1].length , 2);
    invariant_int_cmp (mapper.map[2].from , 46);
    invariant_int_cmp (mapper.map[2].to , 52);
    invariant_int_cmp (mapper.map[2].length , 48);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 43), 43);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 44), 47);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 45), 48);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 46), 52);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 47), 53);
    return mapper;
}

static void
check_complex_mapper_subtracts (void)
{
    /* test setting up a somewhat complex mapper */
    file_offset_mapper_t mapper;
    /* start with a well-added one */
    mapper = get_complex_add_mapper();

    file_offset_mapper_subtract (&mapper, 46);
    /*
     * 43 - 43
     *    - 45
     *    - 46
     * 44 - 47
     * 45 - 48
     *    - 49
     *    - 50
     *    - 51
     * 46 - 52
     * 47 - 53
     * ->
     * 43 - 43
     *    - 45
     *    - 46
     * 44 - 47
     * 45 - 48
     *    - 49
     *    - 50
     *    - 51
     * 47 - 52
     * 48 - 53
     */
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 0);
    invariant_int_cmp (mapper.map[0].length , 44);
    invariant_int_cmp (mapper.map[1].from , 44);
    invariant_int_cmp (mapper.map[1].to , 47);
    invariant_int_cmp (mapper.map[1].length , 2);
    invariant_int_cmp (mapper.map[2].from , 47);
    invariant_int_cmp (mapper.map[2].to , 52);
    invariant_int_cmp (mapper.map[2].length , 48);
    /* maps */
    invariant_int_cmp (file_offset_mapper_map (&mapper, 43), 43);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 44), 47);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 45), 48);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 46), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 47), 52);
    file_offset_mapper_finalise (&mapper); 
    
    /* start with a well-added one */
    mapper = get_complex_add_mapper();

    file_offset_mapper_subtract (&mapper, 47);
    /*
     * 43 - 43
     *    - 45
     *    - 46
     * 44 - 47
     * 45 - 48
     *    - 49
     *    - 50
     *    - 51
     * 46 - 52
     * 47 - 53
     * ->
     * 43 - 43
     *    - 45
     *    - 46
     *    - 47
     * 44 - 48
     * 45 - 49
     *    - 50
     *    - 51
     *    - 52
     * 46 - 53
     * 48 - 54
     */
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 0);
    invariant_int_cmp (mapper.map[0].length , 44);
    invariant_int_cmp (mapper.map[1].from , 44);
    invariant_int_cmp (mapper.map[1].to , 47);
    invariant_int_cmp (mapper.map[1].length , 2);
    invariant_int_cmp (mapper.map[2].from , 46);
    invariant_int_cmp (mapper.map[2].to , 52);
    invariant_int_cmp (mapper.map[2].length , 1);
    invariant_int_cmp (mapper.map[3].from , 48);
    invariant_int_cmp (mapper.map[3].to , 53);
    invariant_int_cmp (mapper.map[3].length , 47);
    /* maps */
    invariant_int_cmp (file_offset_mapper_map (&mapper, 43), 43);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 44), 47);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 45), 48);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 46), 52);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 47), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 48), 53);
    file_offset_mapper_finalise (&mapper); 

    /* start with a well-added one */
    mapper = get_complex_add_mapper();

    file_offset_mapper_subtract (&mapper, 48);
    /*
     * 43 - 43
     *    - 45
     *    - 46
     * 44 - 47
     * 45 - 48
     *    - 49
     *    - 50
     *    - 51
     * 46 - 52
     * 47 - 53
     * ->
     * 43 - 43
     *    - 45
     *    - 46
     *    - 47
     * 44 - 48
     * 45 - 49
     *    - 50
     *    - 51
     *    - 52
     * 46 - 53
     * 47 - 54
     * 49 - 55
     */
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 0);
    invariant_int_cmp (mapper.map[0].length , 44);
    invariant_int_cmp (mapper.map[1].from , 44);
    invariant_int_cmp (mapper.map[1].to , 47);
    invariant_int_cmp (mapper.map[1].length , 2);
    invariant_int_cmp (mapper.map[2].from , 46);
    invariant_int_cmp (mapper.map[2].to , 52);
    invariant_int_cmp (mapper.map[2].length , 2);
    invariant_int_cmp (mapper.map[3].from , 49);
    invariant_int_cmp (mapper.map[3].to , 54);
    invariant_int_cmp (mapper.map[3].length , 46);
    /* maps */
    invariant_int_cmp (file_offset_mapper_map (&mapper, 43), 43);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 44), 47);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 45), 48);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 46), 52);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 47), 53);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 48), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 49), 54);
    file_offset_mapper_finalise (&mapper); 

    /* ok, now sparse subtract tests */
    /* start with a well-added one */
    mapper = get_complex_add_mapper();

    /* subtract first element of first group */
    file_offset_mapper_subtract (&mapper, 44);
    /*
     * 43 - 43
     *    - 45
     *    - 46
     * 44 - 47
     * 45 - 48
     *    - 49
     *    - 50
     *    - 51
     * 46 - 52
     * 47 - 53
     * ->
     * 43 - 43
     *    - 45
     *    - 46
     *    - 47
     * 45 - 48
     * 46 - 49
     *    - 50
     *    - 51
     *    - 52
     * 47 - 53
     */
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 0);
    invariant_int_cmp (mapper.map[0].length , 44);
    invariant_int_cmp (mapper.map[1].from , 45);
    invariant_int_cmp (mapper.map[1].to , 47);
    invariant_int_cmp (mapper.map[1].length , 2);
    invariant_int_cmp (mapper.map[2].from , 47);
    invariant_int_cmp (mapper.map[2].to , 52);
    invariant_int_cmp (mapper.map[2].length , 48);
    /* maps */
    invariant_int_cmp (file_offset_mapper_map (&mapper, 43), 43);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 44), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 45), 47);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 46), 48);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 47), 52);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 48), 53);
    file_offset_mapper_finalise (&mapper); 

    /* start with a well-added one */
    mapper = get_complex_add_mapper();

    /* subtract both elements of first group */
    file_offset_mapper_subtract (&mapper, 44);
    file_offset_mapper_subtract (&mapper, 45);
    /*
     * 43 - 43
     *    - 45
     *    - 46
     * 44 - 47
     * 45 - 48
     *    - 49
     *    - 50
     *    - 51
     * 46 - 52
     * 47 - 53
     * ->
     * 43 - 43
     *    - 45
     *    - 46
     *    - 47
     * 46 - 48
     * 47 - 49
     *    - 50
     *    - 51
     *    - 52
     * 48 - 53
     */
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 0);
    invariant_int_cmp (mapper.map[0].length , 44);
    invariant_int_cmp (mapper.map[1].from , 46);
    invariant_int_cmp (mapper.map[1].to , 47);
    invariant_int_cmp (mapper.map[1].length , 2);
    invariant_int_cmp (mapper.map[2].from , 48);
    invariant_int_cmp (mapper.map[2].to , 52);
    invariant_int_cmp (mapper.map[2].length , 48);
    /* maps */
    invariant_int_cmp (file_offset_mapper_map (&mapper, 43), 43);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 44), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 45), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 46), 47);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 47), 48);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 48), 52);
    file_offset_mapper_finalise (&mapper); 

    /* start with a well-added one */
    mapper = get_complex_add_mapper();
    /* subtract the last element of the first  group */
    file_offset_mapper_subtract (&mapper, 45);
    /*
     * 43 - 43
     *    - 45
     *    - 46
     * 44 - 47
     * 45 - 48
     *    - 49
     *    - 50
     *    - 51
     * 46 - 52
     * 47 - 53
     * ->
     * 43 - 43
     *    - 45
     *    - 46
     * 44 - 47
     * 46 - 48
     *    - 49
     *    - 50
     *    - 51
     * 47 - 52
     * 48 - 53
     */
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 0);
    invariant_int_cmp (mapper.map[0].length , 44);
    invariant_int_cmp (mapper.map[1].from , 44);
    invariant_int_cmp (mapper.map[1].to , 47);
    invariant_int_cmp (mapper.map[1].length , 1);
    invariant_int_cmp (mapper.map[2].from , 46);
    invariant_int_cmp (mapper.map[2].to , 48);
    invariant_int_cmp (mapper.map[2].length , 1);
    invariant_int_cmp (mapper.map[3].from , 47);
    invariant_int_cmp (mapper.map[3].to , 52);
    invariant_int_cmp (mapper.map[3].length , 48);
    /* maps */
    invariant_int_cmp (file_offset_mapper_map (&mapper, 43), 43);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 44), 47);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 45), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 46), 48);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 47), 52);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 48), 53);
    file_offset_mapper_finalise (&mapper); 
    
    /* start with a well-added one */
    mapper = get_complex_add_mapper();
    /* subtract the first element group */
    file_offset_mapper_subtract (&mapper, 0);
    /*
     * 43 - 43
     *    - 45
     *    - 46
     * 44 - 47
     * 45 - 48
     *    - 49
     *    - 50
     *    - 51
     * 46 - 52
     * 47 - 53
     * ->
     * 44 - 43
     *    - 45
     *    - 46
     *    - 47
     * 45 - 48
     * 46 - 49
     *    - 50
     *    - 51
     *    - 52
     * 47 - 53
     */
    invariant_int_cmp (mapper.map[0].from , 1);
    invariant_int_cmp (mapper.map[0].to , 0);
    invariant_int_cmp (mapper.map[0].length , 44);
    invariant_int_cmp (mapper.map[1].from , 45);
    invariant_int_cmp (mapper.map[1].to , 47);
    invariant_int_cmp (mapper.map[1].length , 2);
    invariant_int_cmp (mapper.map[2].from , 47);
    invariant_int_cmp (mapper.map[2].to , 52);
    invariant_int_cmp (mapper.map[2].length , 48);
    /* maps */
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 44), 43);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 45), 47);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 46), 48);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 47), 52);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 48), 53);
    file_offset_mapper_finalise (&mapper); 

    /* start with a well-added one */
    mapper = get_complex_add_mapper();
    /* subtract the last one */
    file_offset_mapper_subtract (&mapper, 93);
    /*
     * 43 - 43
     *    - 45
     *    - 46
     * 44 - 47
     * 45 - 48
     *    - 49
     *    - 50
     *    - 51
     * 46 - 52
     * 47 - 53
     * ->
     * 43 - 43
     *    - 45
     *    - 46
     *    - 47
     * 44 - 48
     * 45 - 49
     *    - 50
     *    - 51
     *    - 52
     * 46 - 53
     * ...
     * 94 - 99
     */
    invariant_int_cmp (mapper.map[0].from , 0);
    invariant_int_cmp (mapper.map[0].to , 0);
    invariant_int_cmp (mapper.map[0].length , 44);
    invariant_int_cmp (mapper.map[1].from , 44);
    invariant_int_cmp (mapper.map[1].to , 47);
    invariant_int_cmp (mapper.map[1].length , 2);
    invariant_int_cmp (mapper.map[2].from , 46);
    invariant_int_cmp (mapper.map[2].to , 52);
    invariant_int_cmp (mapper.map[2].length , 47);
    invariant_int_cmp (mapper.map[3].from , 94);
    invariant_int_cmp (mapper.map[3].to , 99); //?? FIXME how is 100 mappable ??
    invariant_int_cmp (mapper.map[3].length , 1);
    /* maps */
    invariant_int_cmp (file_offset_mapper_map (&mapper, 43), 43);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 44), 47);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 45), 48);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 46), 52);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 47), 53);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 93), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 94), 99);
    file_offset_mapper_finalise (&mapper); 

    mapper = get_complex_add_mapper();
    file_offset_mapper_finalise (&mapper); 
}

static void
check_line_offset_mapper (void)
{
    /* comes from annotated-file for now */
    file_offset_mapper_t mapper;

    check_basic_adds();
    check_basic_replacement();
    check_complex_mapper_adds();
    check_complex_mapper_subtracts();
    check_real_world_2_patch();
    check_real_world_11_10_patch();

    /* corner cases */
    file_offset_mapper_init (&mapper, 0); 
    invariant (file_offset_mapper_map (&mapper, 0) == -1);
    invariant (file_offset_mapper_map (&mapper, 1) == -1);
    file_offset_mapper_finalise (&mapper); 
    
    file_offset_mapper_init (&mapper, 1); 
    /* -0,2 +0,1 */
    invariant (file_offset_mapper_map (&mapper, 0) == 0);
    invariant (file_offset_mapper_map (&mapper, 1) == -1);
    file_offset_mapper_subtract (&mapper, 0);
    invariant (file_offset_mapper_map (&mapper, 0) == -1);
    invariant (file_offset_mapper_map (&mapper, 1) == 0);
    file_offset_mapper_subtract (&mapper, 1);
    invariant (file_offset_mapper_map (&mapper, 0) == -1);
    invariant (file_offset_mapper_map (&mapper, 1) == -1);
    invariant (file_offset_mapper_map (&mapper, 2) == 0);
    file_offset_mapper_add (&mapper, 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), -1);
    file_offset_mapper_finalise (&mapper); 
    
    /* 7 line file */
    file_offset_mapper_init (&mapper, 7); 
    file_offset_mapper_subtract (&mapper, 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 3), 2);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 4), 3);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 5), 4);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 6), 5);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 7), 6);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 8), -1);
    file_offset_mapper_subtract (&mapper, 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 3), 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 4), 2);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 5), 3);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 6), 4);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 7), 5);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 8), 6);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 9), -1);
    file_offset_mapper_subtract (&mapper, 2);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 3), 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 4), 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 5), 2);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 6), 3);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 7), 4);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 8), 5);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 9), 6);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 10), -1);
    file_offset_mapper_add (&mapper, 3); 
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 3), 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 4), 2);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 5), 3);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 6), 4);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 7), 5);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 8), 6);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 9), -1);
    file_offset_mapper_add (&mapper, 3); 
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 3), 2);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 4), 3);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 5), 4);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 6), 5);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 7), 6);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 8), -1);
    file_offset_mapper_subtract (&mapper, 3); 
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 3), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 4), 2);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 5), 3);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 6), 4);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 7), 5);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 8), 6);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 9), -1);
    file_offset_mapper_add (&mapper, 4); 
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 3), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 4), 3);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 5), 4);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 6), 5);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 7), 6);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 8), -1);
    file_offset_mapper_subtract (&mapper, 5); 
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 3), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 4), 3);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 5), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 6), 4);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 7), 5);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 8), 6);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 9), -1);
    file_offset_mapper_add (&mapper, 6); 
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 3), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 4), 3);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 5), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 6), 5);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 7), 6);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 8), -1);
    file_offset_mapper_add (&mapper, 7); 
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 3), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 4), 3);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 5), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 6), 5);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 7), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 8), -1);
    file_offset_mapper_finalise (&mapper);

    /* from 3 lines to 1 */
    file_offset_mapper_init (&mapper, 3); 
    invariant (file_offset_mapper_map (&mapper, 0) == 0);
    invariant (file_offset_mapper_map (&mapper, 1) == 1);
    invariant (file_offset_mapper_map (&mapper, 2) == 2);
    invariant (file_offset_mapper_map (&mapper, 3) == -1);
    /* first gone line */
    file_offset_mapper_subtract (&mapper, 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 3), 2);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 4), -1);
    /* second gone line */
    file_offset_mapper_subtract (&mapper, 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 3), 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 4), 2);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 5), -1);
    /* change third line */
    file_offset_mapper_subtract (&mapper, 2);
    file_offset_mapper_add (&mapper, 3);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), -1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 3), 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 4), 2);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 5), -1);
    file_offset_mapper_finalise (&mapper); 
    
    /* 10 line file */
    file_offset_mapper_init (&mapper, 10); 
    file_offset_mapper_add (&mapper, 4);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), 2);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 3), 3);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 4), 5);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 5), 6);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 6), 7);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 7), 8);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 8), 9);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 9), -1);
    file_offset_mapper_add (&mapper, 5);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), 2);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 3), 3);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 4), 5);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 5), 7);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 6), 8);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 7), 9);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 8), -1);
    file_offset_mapper_add (&mapper, 2);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 0), 0);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 1), 1);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 2), 3);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 3), 5);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 4), 7);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 5), 8);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 6), 9);
    invariant_int_cmp (file_offset_mapper_map (&mapper, 7), -1);
    file_offset_mapper_finalise (&mapper);

}

static void 
check_annotated_file_new_delete (void)
{
  off_t start = talloc_total_size(NULL);
  arch_annotated_file_t *aFile;
  aFile = arch_annotated_file_new (NULL, "x_foo.c", 42);
  invariant (aFile->lines != NULL);
  invariant (aFile->unknown_lines == 42);
  invariant (aFile->max_lines == 42);
  invariant (aFile->mapper.map != NULL);
  invariant_str_cmp (aFile->file_id, "x_foo.c");
  invariant_int_cmp (file_offset_mapper_map (&aFile->mapper, 42), -1);
  invariant_int_cmp (file_offset_mapper_map (&aFile->mapper, 41), 41);
  
  talloc_free (aFile);
  invariant (start == talloc_total_size(NULL));
}

static void 
check_annotated_file_lines (void)
{
  arch_annotated_file_t * aFile;
  arch_patch_t *aPatch;
  
  aFile = arch_annotated_file_new (NULL, "foo.c", 42);
  invariant (aFile->lines[0] == NULL);
  aPatch = arch_patch_new ("foo@example.com/foo--bar--0--base-0");
  arch_annotated_file_note_line (aFile, 0, aPatch, -1);
  arch_patch_delete (&aPatch);
  invariant (aFile->lines[0] != NULL);
  invariant (aFile->unknown_lines == 41);
  aPatch = arch_patch_new ("foo@example.com/foo--bar--0--patch-1");
  arch_annotated_file_note_line (aFile, 1, aPatch, 0);
  arch_patch_delete (&aPatch);
  invariant (aFile->lines[2] != NULL);
  invariant (aFile->unknown_lines == 41);
  talloc_free (aFile);

  /* check that adding a line that maps into a deleted line doesn't affect anythin */
  aFile = arch_annotated_file_new (NULL, "foo.c", 3);
  /* # comment
   *
   * import
   */
  aPatch = arch_patch_new ("p1");
  arch_annotated_file_note_subtract (aFile, 2);
  invariant (aFile->lines[0] == NULL);
  invariant (aFile->lines[1] == NULL);
  invariant (aFile->lines[2] == NULL);
  arch_annotated_file_note_line (aFile, 0, aPatch, -1);
  invariant (aFile->lines[0] != NULL);
  invariant (aFile->lines[1] == NULL);
  invariant (aFile->lines[2] == NULL);
  arch_annotated_file_note_line (aFile, 0, aPatch, -1);
  invariant (aFile->lines[0] != NULL);
  invariant (aFile->lines[1] != NULL);
  invariant (aFile->lines[2] == NULL);
  arch_annotated_file_note_line (aFile, 0, aPatch, -1);
  invariant (aFile->lines[0] != NULL);
  invariant (aFile->lines[1] != NULL);
  invariant (aFile->lines[2] == NULL);
  
  arch_patch_delete (&aPatch);
  talloc_free (aFile);
}

static void 
check_annotation_builder_new_free (void)
{
  off_t start = talloc_total_size(NULL);
  arch_annotation_builder_t * aBuilder;
  aBuilder = arch_annotation_builder_new (NULL);
  invariant (!aBuilder->files);
  invariant (!aBuilder->current_file);
  invariant (!aBuilder->current_patch);
  arch_annotation_builder_add_file (aBuilder, "foo.c", 42);
  invariant (!!aBuilder->current_file);
  
  talloc_free (aBuilder);
  invariant (start == talloc_total_size(NULL));
}

static void
rambling_test (void)
{
  arch_annotation_builder_t * aBuilder;
  arch_annotated_file_t *aFile;
  struct exception * e = NULL;
  arch_patch_id aPatch;
  aBuilder = arch_annotation_builder_new(NULL);
  invariant (!arch_annotation_builder_has_file (aBuilder, "x_foo.c"));
  Try
    aFile = arch_annotation_builder_get_file (aBuilder, NULL, "x_foo.c");
  Catch (e)
    {
    }
  invariant (e != NULL);
  invariant (e->code == ENOENT);
  talloc_free (e);
  e = NULL;
  arch_annotation_builder_add_file (aBuilder, "x_foo.c", 49);
  invariant (arch_annotation_builder_has_file (aBuilder, "x_foo.c"));
  arch_annotation_builder_add_file (aBuilder, "x_foo.c", -1);
  talloc_free (arch_annotation_builder_get_file (aBuilder, NULL, "x_foo.c"));
  invariant (!arch_annotation_builder_has_file (aBuilder, "x_missing.c"));
  Try
    aFile = arch_annotation_builder_get_file (aBuilder, NULL, "x_missing.c");
  Catch (e)
    {
    }
  invariant (e != NULL);
  invariant (e->code == ENOENT);
  talloc_free (e);
  e = NULL;
  
  /* record a patch */
  /* -45,3,+46,4
   * 45 45 46 
   * 46 46    - foo
   * 47   47 + bar
   * 47   48 + gam
   * 47 47 49  zap
   */
  /* new file, we 'got' it before */
  arch_annotation_builder_add_file (aBuilder, "x_foo.c", 49);
  arch_patch_id_init (&aPatch, "foo@example.com/foo--bar--0");
  arch_annotation_builder_add_patch (aBuilder, &aPatch, "foo");
  arch_annotation_builder_add_file (aBuilder, "x_foo.c", -1);
  arch_annotation_builder_subtract_line (aBuilder, 46);
  arch_annotation_builder_add_line (aBuilder, 47);
  arch_annotation_builder_add_line (aBuilder, 47);
  arch_patch_id_finalise (&aPatch);
  
  aFile = arch_annotation_builder_get_file (aBuilder, NULL, "x_foo.c");
    {
      int line;
      for (line=0; line < aFile->max_lines; ++line)
        {
          if (line < 46 || line > 47)
              invariant (aFile->lines[line] == NULL);
          else
            {
              invariant (aFile->lines[line] != NULL);
              invariant_str_cmp ("foo@example.com/foo--bar--0", arch_patch_id_patch_id(arch_patch_patch_id(aFile->lines[line])));
            }
/*          if (aFile->lines[line] == NULL)
              safe_printfmt (2, "??? did not find creation patch\n");
          else
              safe_printfmt (2, "%s %s\n", arch_patch_id_patch_id(arch_patch_patch_id(aFile->lines[line])), arch_patch_creator(aFile->lines[line]));
              */
        }
    }
  talloc_free (aFile);

  /* two patches, walking backwards */
  /* new file, we 'got' it before */
  arch_annotation_builder_add_file (aBuilder, "x_foo.c", 50);
  arch_patch_id_init (&aPatch, "foo@example.com/foo--bar--0--patch-1");
  /* record a patch */
  /* -45,3,+46,4
   * 45 45 46 
   * 46 46    - foo
   * 47   47 + bar
   * 47   48 + gam
   * 47 49  zap
   */
  arch_annotation_builder_add_patch (aBuilder, &aPatch, "foo");
  arch_annotation_builder_add_file (aBuilder, "x_foo.c", -1);
  arch_annotation_builder_subtract_line (aBuilder, 46);
  arch_annotation_builder_add_line (aBuilder, 47);
  arch_annotation_builder_add_line (aBuilder, 47);
  arch_patch_id_finalise (&aPatch);
  arch_patch_id_init (&aPatch, "foo@example.com/foo--bar--0--base-0");
  /* record a patch */
  /* -44,3,+45,3
   * 44 44 45 
   * 45 45 46 foo
   * 46 46  - mole
   * 47    47 +zap
   */
  arch_annotation_builder_add_patch (aBuilder, &aPatch, "foo");
  arch_annotation_builder_add_file (aBuilder, "x_foo.c", -1);
  arch_annotation_builder_add_line (aBuilder, 47);
  arch_patch_id_finalise (&aPatch);
  
  
  aFile = arch_annotation_builder_get_file (aBuilder, NULL, "x_foo.c");
    {
      int line;
      for (line=0; line < aFile->max_lines; ++line)
        {
          if (line < 46 || line > 48)
              invariant (aFile->lines[line] == NULL);
          else if (line < 48)
            {
              invariant (aFile->lines[line] != NULL);
              invariant_str_cmp ("foo@example.com/foo--bar--0--patch-1", arch_patch_id_patch_id(arch_patch_patch_id(aFile->lines[line])));
            }
          else if (line == 48)
            {
              invariant (aFile->lines[line] != NULL);
              invariant_str_cmp ("foo@example.com/foo--bar--0--base-0", arch_patch_id_patch_id(arch_patch_patch_id(aFile->lines[line])));
            }
        }
    }
  talloc_free (aFile);
  talloc_free (aBuilder);
}

    /* should be a fd passed in, meh. */
static t_uchar const *my_patch=
"--- orig/libarch/tests/unit-annotate.c\n"
"+++ mod/libarch/tests/unit-annotate.c\n"
"@@ -1226,6 +1226,17 @@\n"
"   arch_annotation_builder_finalise (&aBuilder);\n"
" }\n"
" \n"
"+static void\n"
"+check_parse_patch (void)\n"
"+{\n"
"+    /* should be a fd passed in, meh. */\n"
"+    t_uchar const *my_patch= \\\n"
"+      \"aasd\"\n"
"+      \"asd\";\n"
"+    patch_changes_list changes = patch_parser_parse(my_patch);\n"
"+    \n"
"+}\n"
"-\n"
"-\n"
" \n"
"+int\n"
"@@ -1236,6 +1247,7 @@\n"
"   check_annotated_file_new_delete();\n"
"   check_annotated_file_lines();\n"
"   check_annotation_builder_init_finalise();\n"
"+  check_parse_patch();\n"
"   rambling_test();\n"
"   return 0;\n"
" }\n"
"\n"
"\n"
"\n"
"\n";

static t_uchar const *new_file_patch=
"--- orig/foo.c\n"
"+++ mod/foo.c\n"
"@@ -0,0 +1,2 @@\n"
"+file-content;\n"
"+file-content2;\n";

static void
check_parse_patch (void)
{
    off_t start = talloc_total_size(NULL);
    patch_line_changes_list changes = patch_parser_parse(my_patch);
    patch_line_change_t reference_changes [] = {{1228, PATCH_ADD_LINE},
      {1228, PATCH_ADD_LINE},
      {1228, PATCH_ADD_LINE},
      {1228, PATCH_ADD_LINE},
      {1228, PATCH_ADD_LINE},
      {1228, PATCH_ADD_LINE},
      {1228, PATCH_ADD_LINE},
      {1228, PATCH_ADD_LINE},
      {1228, PATCH_ADD_LINE},
      {1228, PATCH_ADD_LINE},
      {1228, PATCH_DEL_LINE},
      {1229, PATCH_DEL_LINE},
      {1231, PATCH_ADD_LINE},
      {1238, PATCH_ADD_LINE},
          };
    patch_line_change_t new_file_reference_changes [] = {{0, PATCH_ADD_LINE},
      {0, PATCH_ADD_LINE},
          };
    int position;
    invariant_int_cmp (14, ar_size_patch_line_change (changes));
    for (position = 0; position < ar_size_patch_line_change (changes); ++position)
      {
        invariant_int_cmp (reference_changes[position].offset, changes[position].offset);
        invariant_int_cmp (reference_changes[position].operation, changes[position].operation);
      };
    ar_free_patch_line_change (&changes);

    changes = patch_parser_parse(new_file_patch);
    invariant_int_cmp (2, ar_size_patch_line_change (changes));
    for (position = 0; position < ar_size_patch_line_change (changes); ++position)
      {
        invariant_int_cmp (new_file_reference_changes[position].offset, changes[position].offset);
        invariant_int_cmp (new_file_reference_changes[position].operation, changes[position].operation);
      };
    ar_free_patch_line_change (&changes);
    invariant (start == talloc_total_size(NULL));
}

static void
check_process_patch (void)
{
  off_t start = talloc_total_size(NULL);
  arch_annotation_builder_t * aBuilder;
  arch_annotated_file_t *aFile;
  arch_patch_id aPatch;
  patch_line_changes_list changes = patch_parser_parse(my_patch);
  aBuilder = arch_annotation_builder_new (NULL);
  arch_annotation_builder_add_file (aBuilder, "x_foo.c", 1255);
  arch_patch_id_init (&aPatch, "foo@example.com/foo--bar--0--base-0");
  arch_annotation_builder_add_patch (aBuilder, &aPatch, "creator");
  arch_patch_id_finalise (&aPatch);
  arch_annotation_builder_process_changes(aBuilder, changes);
  aFile = arch_annotation_builder_get_file (aBuilder, NULL, "x_foo.c");
    {
      int line;
      for (line=0; line < aFile->max_lines; ++line)
        {
          if (line < 1228 || line == 1238 || (line > 1239 && line < 1247) || line > 1247)
              invariant (aFile->lines[line] == NULL);
          else if ((line >1227 && line < 1238) || (line == 1239) || ( line == 1247))
            {
              invariant (aFile->lines[line] != NULL);
              invariant_str_cmp ("foo@example.com/foo--bar--0--base-0", arch_patch_id_patch_id(arch_patch_patch_id(aFile->lines[line])));
            }
        }
    }
  ar_free_patch_line_change (&changes);
  talloc_free (aFile);
  talloc_free (aBuilder);
  invariant (start == talloc_total_size(NULL));
}



int
main (int argc, char * argv[])
{
  off_t start;
  talloc_enable_leak_report ();
  start = talloc_total_size(NULL);
  arch_debug_init();
  check_line_offset_mapper();
  check_annotated_file_new_delete();
  check_annotated_file_lines();
  check_annotation_builder_new_free ();
  check_parse_patch();
  check_process_patch();
  rambling_test();
  invariant (start == talloc_total_size(NULL));
  return 0;
}

