/*
   Copyright (C) 2004-2009 Benjamin Redelings

This file is part of BAli-Phy.

BAli-Phy is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free
Software Foundation; either version 2, or (at your option) any later
version.

BAli-Phy is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with BAli-Phy; see the file COPYING.  If not see
<http://www.gnu.org/licenses/>.  */

/**
 * @file util.C
 *
 * @brief This file contains a number of utility routines.
 *
 */

#include "util.H"
#include "io.H"

using std::vector;
using std::string;

#include <iostream>
using std::cerr;
using std::endl;

int log_verbose = 0;

/// \brief Compute the number of 'true' elements in bitvector \a v.
///
/// \param v The bitvector.
/// 
int n_elements(const vector<bool>& v) {
  int count = 0;
  for(int i=0;i<v.size() ;i++)  
    if (v[i]) count++;
  return count;
}


/// \brief Combine vector of strings \a v into one string, separating entries with the character \a c.
///
/// \param v The strings to combine.
/// \param c The separator.
///
string join(const vector<string>& v,char c) {
  string s;
  if (v.size())
    s = v[0];
  for(int i=1;i<v.size();i++) {
    s += string(1,c);
    s += v[i];
  }
  return s;
}

/// \brief Combine vector of strings \a v into one string, separating entries with the string \a token.
///
/// \param v The strings to combine.
/// \param token The separator.
///
string join(const vector<string>& v,const string& token) {
  string s;
  if (v.size())
    s = v[0];
  for(int i=1;i<v.size();i++) {
    s += token;
    s += v[i];
  }
  return s;
}


/// \brief Divide a string \a s into several strings by break it at the character \a c
///
/// \param s The string to split.
/// \param c The separator.
///
vector<string> split(const string& s, char c) 
{
  vector<string> strings;
  int length=0;
  for(int i=0;i<s.size();i++)
    if (s[i] == c) {
      strings.push_back(s.substr(i-length,length));
      length = 0;
    }
    else
      length++;
  
  strings.push_back(s.substr(s.size()-length,length));
  return strings;
}

/// \brief Divide a string \a s into several strings by break it at the string \a token.
///
/// \param s The string to split.
/// \param token The separator.
///
vector<string> split(const string& s, const string& token)
{
  vector<string> strings;

  int p1 = 0;
  do {
    int p2 = s.find(token,p1);
    if (p2 == -1)
    {
      strings.push_back(s.substr(p1));
      break;
    }
    else 
    {
      assert(p2 < s.size());

      strings.push_back(s.substr(p1,p2-p1));
      p1 = p2 + token.size();

      assert(p1 <= s.size());
    }
  } while (true);
  return strings;
}

/// \brief Remove the character \a c from the string \a s.
///
/// \param s The string to strip.
/// \param c The character to remove.
/// \return the stripped string.
///
string strip(const string& s,char c) {
  string s2;
  for(int i=0;i<s.size();i++)
    if (s[i] != c)
      s2 += s[i];

  return s2;
}


/// \brief Remove all characters in string \a chars from the string \a s.
///
/// \param s The string to strip.
/// \param chars The string containing characters to remove.
/// \return the stripped string.
///
string strip(const string& s,const string& chars) {
  string s2;
  for(int i=0;i<s.size();i++) {
    bool found = false;
    for(int j=0;j<chars.size() and not found;j++) {
      if (s[i] == chars[j])
	found = true;
    }
    if (not found)
      s2 += s[i];
  }

  return s2;
}


vector<int> invert(const vector<int>& mapping) 
{
  vector<int> imapping(mapping.size(),-1);

  for(int i=0;i<imapping.size();i++)
    imapping[mapping[i]] = i;

  return imapping;
}

vector<int> compose(const vector<int>& mapping1,const vector<int>& mapping2) {
  assert(mapping1.size() == mapping2.size());

  vector<int> mapping(mapping1.size());

  for(int i=0;i<mapping.size();i++)
    mapping[i] = mapping2[mapping1[i]];

  return mapping;
}

/// \brief Check if \a mapping[i] == i
///
/// \param mapping The mapping.
///
bool is_identity(const std::vector<int>& mapping)
{
  for(int i=0;i<mapping.size();i++)
    if (mapping[i] != i)
      return false;
  return true;
}

/// \brief Check if a string \a s contains a character \a c.
///
/// \param s The string.
/// \param c The character.
/// \return true if \a s contains \a c.
/// 
bool contains_char(const string& s,char c) {
  for(int i=0;i<s.size();i++)
    if (s[i] == c)
      return true;
  return false;
}

bool get_word(string& word, int& i, const string& s,
	      const string& delimiters,const string& whitespace) 
{
  if (i >= s.size()) 
    return false;

  while(contains_char(whitespace,s[i])) {
    i++;
    if (i >= s.size()) 
      return false;
  }

  int start = i;
  if (contains_char(delimiters,s[i])) {
    word = s.substr(i,1);
    i++;
    return true;
  }

  do { i++; }
  while(not contains_char(delimiters,s[i]) and not contains_char(whitespace,s[i])
	and i < s.size());

  word = s.substr(start,i-start);

  return true;
}

/// \brief Parse a range of the form <begin>-<end> which should be a subset of [1,L]
///
/// \param range The string to parse.
/// \param L The upper bound.
/// \param begin Parameter for passing back the beginning of the range.
/// \param end Parameter for passing back the end of the range.
///
void parse_simple_range(const string& range,int L,int& begin,int& end)
{
  vector<string> R = split(range,'-');

  if (R.size() == 1) {
    begin = end = convertTo<int>(range)-1;
  }
  else if (R.size() != 2)
    throw myexception()<<"Malformed range '"<<range<<"'";
  else {
    begin = 0;
    if (R[0].size())
      begin = convertTo<int>(R[0])-1;

    end = L-1;
    if (R[1].size())
      end = convertTo<int>(R[1])-1;
  }
    
  if (begin < 0)
    throw myexception()<<"Bad range '"<<range<<"': begins before 1.";
    
  if (begin > L-1)
    throw myexception()<<"Bad range '"<<range<<"': begins after end of sequence (L="<<L<<").";
    
  if (end < 0)
    throw myexception()<<"Bad range '"<<range<<"': ends before 1!";
    
  if (end > L-1)
    throw myexception()<<"Bad range '"<<range<<"': ends after end of sequence (L="<<L<<").";
    
  if (end < begin)
    throw myexception()<<"Bad range '"<<range<<"': begins after end!";
}

/// \brief Parse a range of the form <begin>-<end>/<step> which should be a subset of [1,L]
///
/// \param range The string to parse.
/// \param L The upper bound.
/// \param begin Parameter for passing back the beginning of the range.
/// \param end Parameter for passing back the end of the range.
/// \param end Parameter for passing back the step size.
///
void parse_modulo_range(const string& range,int L,int& begin, int& end, int& step)
{
  vector<string> R = split(range,'/');

  if (R.size() == 1) 
    step = 1;
  else if (R.size() == 2) {
    try {
      step = convertTo<int>(R[1]);
    }
    catch (...) {
      throw myexception()<<"Malformed step size '"<<R[1]<<"' in range '"<<range<<"'";
    }
    if (step < 1)
      throw myexception()<<"Step is not positive in range '"<<range<<"'";
  }
  else
    throw myexception()<<"Malformed range '"<<range<<"'";

  parse_simple_range(R[0],L,begin,end);
}

/// \brief Parse a comma-separated list of ranges <begin>-<end>/<step> and construct an ordered list.
///
/// \param range The string to parse.
/// \param L The upper bound.
/// \return On ordered list constructed by concatenating the elements in the individual ranges.
///
vector<int> parse_multi_range(const string& range,int L)
{
  vector<string> ranges = split(range,',');

  vector<int> columns;
  for(int i=0;i<ranges.size();i++) 
  {
    int begin = -1;
    int end = -1;
    int step = -1;

    parse_modulo_range(ranges[i], L, begin, end, step);
    
    for(int c=begin;c<=end;c++)
      if ((c-begin)%step == 0)
	columns.push_back(c);
  }
  return columns;
}

vector<string> get_arguments(string& s,char begin, char end)
{
  if (not s.size() or s[s.size()-1] != end) 
    return vector<string>();
  
  int loc = s.find(begin);
  string args = s.substr(loc+1, s.size()-loc-2);
  s = s.substr(0,loc);

  if (loc == -1) 
    return vector<string>();
  
  return split(args,',');
}
