/*
   Copyright (C) 2010 Benjamin Redelings

This file is part of BAli-Phy.

BAli-Phy is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free
Software Foundation; either version 2, or (at your option) any later
version.

BAli-Phy is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with BAli-Phy; see the file COPYING.  If not see
<http://www.gnu.org/licenses/>.  */

/**
 * @file timer_stack.H
 */

/*
 * A timer stack contains a collection of tokens (strings) identifying
 * various code regions.  The contexts are nested, with the top of the
 * stack being most deeply nested. Elapsed CPU time is credited to each
 * token that is on the stack.
 *
 * Usage: When we enter a code region which we wish to profile, we call
 * push_timer( ) to start charging CPU time to that token.  When we leave
 * the region, we call pop_timer().
 *
 * A report can be generated by calling report().
 */

#ifndef TIME_STACK_H
#define TIME_STACK_H

#include <string>
#include <map>
#include <vector>

typedef double time_point_t;
typedef double duration_t;

time_point_t total_cpu_time();

std::string duration(time_t);

struct region_profile 
{
  duration_t duration;
  long int n_calls;
  region_profile():duration(0),n_calls(0) {}
};

class timer_stack
{
public:
  typedef std::map<std::string,region_profile> container_t;

private:
  std::vector<container_t::iterator> record_stack;
  std::vector<time_point_t> start_time_stack;

  container_t::iterator lookup_profile(const std::string&);
  
public:
  container_t total_times;

  void credit_active_timers();
  void push_timer(const std::string& s);
  void pop_timer();
  const std::string& current_timer();
  //  const std::vector<std::string>& active_timers() const {return name_stack;}
  int n_active_timers() const {return record_stack.size();}

  std::string report();
};

extern timer_stack default_timer_stack;

#endif /* TIME_STACK_H */
