/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#include "gametime.h"
#include "settings.h"
#include "server.h"

STimeManager TimeManager;

void
GameTime::time_str (char *buf, int len) const
{
	snprintf (buf, len, "%d:%02d %s",
			hour == 0 ? 12 : (hour <= 12 ? hour : hour - 12),
			ticks_in_hour * 60 / TICKS_PER_HOUR,
			(hour < 12) ? "am" : "pm");
}

String
GameTime::time_str (void) const
{
	char buffer[32];
	time_str(buffer, sizeof(buffer));
	return buffer;
}

void
GameTime::date_str (char *buf, int len) const
{
	// base date string
	size_t add = snprintf (buf, len, "%s, %d%s of %s, %d",
			TimeManager.calendar.weekdays[TimeManager.calendar.get_weekday (*this)].get(),
			day,
			get_num_suffix (day),
			TimeManager.calendar.months[month - 1].name.get(),
			year);
	// append holiday if we have one
	String holiday = TimeManager.calendar.get_holiday(*this);
	if (!holiday.empty()) {
		snprintf (buf + add, len - add, " (%s)", holiday.get());
	}
}

String
GameTime::date_str (void) const
{
	char buffer[256];
	date_str(buffer, sizeof(buffer));
	return buffer;
}

void
GameTime::update (uint ticks)
{
	ticks_in_hour += ticks;
	clip_time ();
}

void
GameTime::clip_time (void)
{
	while (ticks_in_hour >= TICKS_PER_HOUR)
	{
		ticks_in_hour -= TICKS_PER_HOUR;
		hour ++;
	}
	while (hour >= 24)
	{
		hour -= 24;
		day ++;
	}
	while (day > TimeManager.calendar.days_in_month (*this))
	{
		day -= TimeManager.calendar.days_in_month (*this);
		month ++;
	}
	if (day < 1)
		day = 1;
	while (month > TimeManager.calendar.months.size ())
	{
		month -= TimeManager.calendar.months.size ();
		year ++;
	}
	if (month < 1)
		month = 1;
}

int
GameTime::load (File::Reader& reader)
{
	File::Node node;
	FO_READ_BEGIN
		FO_ATTR_NAME("year")
			FO_GET_INT(year);
		FO_ATTR_NAME("month")
			FO_GET_INT(month);
		FO_ATTR_NAME("day")
			FO_GET_INT(day);
		FO_ATTR_NAME("hour")
			FO_GET_INT(hour);
		FO_ATTR_NAME("ticks")
			FO_GET_INT(ticks_in_hour);
	FO_READ_ERROR
		return -1;
	FO_READ_END

	clip_time();

	return 0;
}

void
GameTime::save (File::Writer& writer) const
{
	writer.attr ("year", year);
	writer.attr ("month", month);
	writer.attr ("day", day);
	writer.attr ("hour", hour);
	writer.attr ("ticks", ticks_in_hour);
}

int
STimeManager::initialize (void)
{
	// initialize calendar
	Log::Info << "Loading calendar";
	if (calendar.load())
		return 1;

	// initialize time
	Log::Info << "Loading game time";

	File::Reader reader(settings::get_path("world", "data") + "/" + "time");
	if (reader.is_open()) {
		time.load(reader);
		reader.close();
	} else {
		return 1;
	}

	// all set
	return 0;
}

int
STimeManager::save (void)
{
	// open
	File::Writer writer(settings::get_path("world", "data") + "/" + "time");
	if (!writer.is_open())
		return 1;

	// save
	time.save(writer);
	writer.close();

	return 0;
}

void
STimeManager::shutdown (void)
{
}
