/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#define TELCMDS
#define TELOPTS
#define SLC_NAMES

#include <stdio.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <arpa/telnet.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <ctype.h>
#include <stdarg.h>

#include "error.h"
#include "server.h"
#include "network.h"
#include "parse.h"
#include "player.h"
#include "command.h"
#include "room.h"
#include "zmp.h"
#include "streams.h"
#include "color.h"
#include "message.h"
#include "telnet.h"

STelnetManager TelnetManager;

#define TELOPT_MCCP2 86

// otput spacing in put()
#define OUTPUT_INDENT() \
	if (cur_col < margin) { \
		while (margin - cur_col >= 16) { \
			cur_col += 16; \
			add_output("                ", 16); \
		} \
		add_output("                ", margin - cur_col); \
		cur_col = margin; \
	}


// ---- BEGIN COLOURS ----

// color names
const char *color_value_names[] = {
	"normal",
	"black",
	"red",
	"green",
	"brown",
	"blue",
	"magenta",
	"cyan",
	"grey",
	"lightblack",
	"lightred",
	"lightgreen",
	"yellow",
	"lightblue",
	"lightmagenta",
	"lightcyan",
	"white",
	"darkred",
	"darkgreen",
	"darkyellow",
	"darkblue",
	"darkmagenta",
	"darkcyan",
	"darkgrey",
	NULL
};
// colour ansi values
const char *color_values[] = {
	ANSI_NORMAL,
	ANSI_BLACK,
	ANSI_RED,
	ANSI_GREEN,
	ANSI_BROWN,
	ANSI_BLUE,
	ANSI_MAGENTA,
	ANSI_CYAN,
	ANSI_GREY,
	ANSI_LIGHTBLACK,
	ANSI_LIGHTRED,
	ANSI_LIGHTGREEN,
	ANSI_YELLOW,
	ANSI_LIGHTBLUE,
	ANSI_LIGHTMAGENTA,
	ANSI_LIGHTCYAN,
	ANSI_WHITE,
	ANSI_DARKRED,
	ANSI_DARKGREEN,
	ANSI_DARKYELLOW,
	ANSI_DARKBLUE,
	ANSI_DARKMAGENTA,
	ANSI_DARKCYAN,
	ANSI_DARKGREY,
};
// colour type names
const char *color_type_names[] = {
	"normal",
	"title",
	"desc",
	"player",
	"npc",
	"item",
	"special",
	"admin",
	"exit",
	"stat",
	"statvbad",
	"statbad",
	"statgood",
	"statvgood",
	"bold",
	"talk",
	NULL
};
// default colour type mappings
const int color_type_defaults[] = {
	COLOR_NORMAL,
	COLOR_GREEN,
	COLOR_NORMAL,
	COLOR_MAGENTA,
	COLOR_BROWN,
	COLOR_LIGHTBLUE,
	COLOR_BROWN,
	COLOR_RED,
	COLOR_CYAN,
	COLOR_GREY,
	COLOR_LIGHTRED,
	COLOR_YELLOW,
	COLOR_LIGHTCYAN,
	COLOR_LIGHTGREEN,
	COLOR_BROWN,
	COLOR_CYAN
};

// ---- END COLOURS ----

TextBufferList TextBuffer::lists[] = {
	TextBufferList(64),
	TextBufferList(128),
	TextBufferList(256),
	TextBufferList(512),
	TextBufferList(1024),
	TextBufferList(2048),
	TextBufferList(4096),
	TextBufferList(8192),
	TextBufferList(16384),
};

char*
TextBufferList::alloc (void) {
	++ allocs;
	++ out;
	if (!list.empty()) {
		char* buf = list.back();
		list.pop_back();
		return buf;
	} else {
		char* buf = new char[size];
		return buf;
	}
}

int
TextBuffer::alloc (SizeType size)
{
	if (size == EMPTY)
		return -1;

	release();

	// do allocation
	bdata = lists[size].alloc();
	if (!bdata)
		return -1;

	bsize = size;
	return 0;
}
int
TextBuffer::grow (void)
{
	if (bsize == EMPTY) {
		// base allocation
		return alloc(SIZE_64);
	} else if (bsize < COUNT - 1) {
		// upgrade
		char* bdatanew = lists[bsize + 1].alloc();
		if (bdatanew == NULL) {
			return -1;
		}

		// copy
		memcpy(bdatanew, bdata, size());
		lists[bsize].release(bdata);
		bsize = (SizeType)(bsize + 1);
		bdata = bdatanew;
		return 0;
	} else {
		// can't grow any more
		return -1;
	}
}
void
TextBuffer::release (void)
{
	if (bdata != NULL) {
		lists[bsize].release(bdata);
		bsize = EMPTY;
		bdata = NULL;
	}
}

TelnetHandler::TelnetHandler (void) : Scriptable (AweMUD_TelnetType)
{
}

TelnetHandler::~TelnetHandler (void)
{
	// disconnect
	if (sock.alive())
		sock.close();
}

int
TelnetHandler::net_connect (const Socket& conn)
{
	// set connection
	if (sock.alive())
		net_disconnect();
	sock = conn;

	// add to telnet handler
	TelnetManager.telnet_list.push_back(this);

	// various state settings
	in_cnt = out_cnt = sb_cnt = outchunk_cnt = esc_cnt = 0;
	istate = ISTATE_TEXT;
	ostate = OSTATE_TEXT;
	margin = 0;
	width = 70; // good default?
	chunk_size = 0;
	cur_col = 0;
	memset(&io_flags, 0, sizeof(IOFlags));

	// initial telnet options
	io_flags.want_echo = true;
	io_flags.do_echo = false;
	io_flags.do_eor = false;
	io_flags.use_ansi = true;

	// send our initial telnet state and support options
	send_iac (2, WONT, TELOPT_ECHO);
	send_iac (2, WILL, TELOPT_EOR);
	send_iac (2, WILL, TELOPT_ZMP);
	send_iac (2, DO, TELOPT_NEW_ENVIRON);
	send_iac (2, DO, TELOPT_TTYPE);
	send_iac (2, DO, TELOPT_NAWS);
	send_iac (2, DO, TELOPT_LFLOW);

	// have MCCP support?
#ifdef HAVE_LIBZ
	zstate = NULL;
	send_iac (2, WILL, TELOPT_MCCP2);
#endif // HAVE_LIBZ

	// colors
	for (int i = 0; i < NUM_CTYPES; ++ i) {
		color_set[i] = -1;
	}

	// in stamp
	time (&in_stamp);

	return 0;
}

// ----- COMPRESSION -----
#ifdef HAVE_LIBZ

// initialize compression
bool
TelnetHandler::begin_mccp (void)
{
	if (!zstate) {
		// allocte
		zstate = new z_stream;
		if (zstate == NULL) {
			Log::Error << "Failed to allocate z_stream";
			return false;
		}

		// initialize
		memset(zstate, 0, sizeof(z_stream));
		if (deflateInit(zstate, Z_DEFAULT_COMPRESSION) != Z_OK) {
			Log::Error << "Failed to initialize z_stream: " << zstate->msg;
			return false;
		}

		// send the "all data will be compressed now message"
		// yes, just send now - why not? :P
		static char begin_compress_msg[] = {
			IAC,
			SB,
			TELOPT_MCCP2,
			IAC,
			SE
		};
		sock.send(begin_compress_msg, 5);

		return true;
	} else  {
		return false;
	}
}

// end compressed stream
void
TelnetHandler::end_mccp (void)
{
	if (zstate) {
		// free
		deflateEnd(zstate);
		delete zstate;
		zstate = NULL;
	}
}

#endif // HAVE_LIBZ

// send output
void
TelnetHandler::send_bytes (char* bytes, size_t len)
{
	if (len == 0)
		return;

#ifdef HAVE_LIBZ
	if (zstate) {
		char buffer[512]; // good enough for now
		// setup
		zstate->next_in = (Bytef*)bytes;
		zstate->avail_in = len;
		zstate->next_out = (Bytef*)buffer;
		zstate->avail_out = sizeof(buffer);
		
		// keep compressing until we have no more input
		do {
			deflate(zstate, Z_NO_FLUSH);

			// out of output?
			if (!zstate->avail_out) {
				sock.send(buffer, sizeof(buffer));
				zstate->next_out = (Bytef*)buffer;
				zstate->avail_out = sizeof(buffer);
			}
		} while (zstate->avail_in);

		// have we any remaining output?
		deflate(zstate, Z_SYNC_FLUSH);
		if (zstate->avail_out < sizeof(buffer)) {
			sock.send(buffer, sizeof(buffer) - zstate->avail_out);
		}
	} else
#endif // HAVE_LIBZ
		sock.send(bytes, len);
}

// disconnect
void
TelnetHandler::net_disconnect (void)
{
	// already disconnected?
	if (!net_alive())
		return;

	// flush out
	io_flags.need_prompt = false;
	flush();

	// close up networking
	sock.close();

#ifdef HAVE_LIBZ
	end_mccp();
#endif // HAVE_LIBZ

	// shutdown current mode
	if (mode) {
		mode->shutdown();
		mode = NULL;
	}
}

// toggle echo
bool
TelnetHandler::toggle_echo (bool v)
{
	io_flags.want_echo = v;

	// force echoing?
	if (io_flags.force_echo) {
		io_flags.do_echo = v;
	// normal negotiation
	} else {
		if (v == false)
			io_flags.do_echo = false;
		send_iac (2, v ? WONT : WILL, TELOPT_ECHO);
	}

	return v;
}

/* output a data of text -
 * deal with formatting new-lines and such, and also
 * escaping/removing/translating AweMUD commands
 */
void
TelnetHandler::stream_put (const char *text, size_t len) 
{
	assert(text != NULL);

	// output a newline if we need one, such as after a prompt
	if (io_flags.need_newline)
	{
		add_output ("\n\r", 2);
		io_flags.soft_break = false;
		cur_col = 0;
	}
	io_flags.need_newline = false;

	// do loop
	char c;
	for (size_t ti = 0; ti < len; ++ti) {
		c = text[ti];
		switch (ostate) {
			// normal text
			case OSTATE_TEXT:
				switch (c) {
					// space?
					case ' ':
						end_chunk();

						// not soft-wrapped?
						if (!io_flags.soft_break) {
							// word wrap?
							if (width && cur_col + 1 >= width) {
								add_output("\n\r", 2);
								cur_col = 0;
								io_flags.soft_break = true;
							} else {
								OUTPUT_INDENT()
								add_output(" ", 1);
								++cur_col;
							}
						}
						break;
					// newline?
					case '\n':
						end_chunk();
						
						// not after a soft-break
						if (!io_flags.soft_break) {
							add_output("\n\r", 2);
							cur_col = 0;
						}

						// this _is_ a hard break
						io_flags.soft_break = false;
						break;
					// escape sequence?
					case '\033':
						ostate = OSTATE_ESCAPE;
						esc_buf[0] = '\033';
						esc_cnt = 1;
						break;
					// tab?
					case '\t':
						end_chunk();
						OUTPUT_INDENT()
						add_output("    ", 4 % cur_col);
						cur_col += 4 % cur_col;
						break;
					// IAC byte
					case '\255':
						add_to_chunk("\255\255", 2); // escape IAC
						++chunk_size;
						break;
					// just data
					default:
						add_to_chunk(&c, 1);
						++chunk_size;
						break;
				}
				break;
			// escape
			case OSTATE_ESCAPE:
				// awecode?
				if (c == '!') {
					// we just want data
					esc_cnt = 0;
					ostate = OSTATE_AWECODE;
				// ansi
				} else if (c == '[') {
					// we keep whole code
					esc_buf[1] = c;
					esc_cnt = 2;
					ostate = OSTATE_ANSI;
				// unsupported/invalid
				} else {
					ostate = OSTATE_TEXT;
				}
				break;
			// awecode
			case OSTATE_AWECODE:
				// end?
				if (c != '!') {
					// add data
					if (esc_cnt < MAX_ESCAPE_SIZE)
						esc_buf[esc_cnt++] = c;
					break;
				}

				// have we anything?
				if (esc_cnt == 0) {
					ostate = OSTATE_TEXT;
					break;
				}

				// process
				switch (esc_buf[0]) {
					// color command
					case 'C':
						// no color?  bah
						if (!io_flags.use_ansi)
							break;

						// search colors
						for (int i = 0; color_type_names[i] != NULL; ++ i) {
							if (!strncmp (&esc_buf[1], color_type_names[i], esc_cnt - 1)) {
								// reset color?
								if (i == 0) {
									// normalize colors
									add_to_chunk (ANSI_NORMAL, strlen(ANSI_NORMAL));

									// eat last color
									if (!colors.empty())
										colors.pop_back();

									// old color?
									if (!colors.empty())
										add_to_chunk (color_values[colors.back()], strlen (color_values[colors.back()]));
										
								// normal color
								} else {
									// put color
									int color = get_color (i);
									colors.push_back(color);
									add_to_chunk (color_values[color], strlen (color_values[color]));
								}
								break;
							}
						}
						break;
					// indent
					case 'I':
					{
						long mlen = strtol(&esc_buf[1], NULL, 10);
						if (mlen >= 0)
							set_indent(mlen);
						break;
					}
					// auto-indent
					case 'A':
						io_flags.auto_indent = (esc_buf[1] == '1');
						break;
				}

				// done
				ostate = OSTATE_TEXT;
				break;
			// ansi code
			case OSTATE_ANSI:
				// add data
				if (esc_cnt < MAX_ESCAPE_SIZE)
					esc_buf[esc_cnt++] = c;

				// end?
				if (isalpha(c)) {
					if (io_flags.use_ansi) {
						add_to_chunk (esc_buf, esc_cnt);
					}
					ostate = OSTATE_TEXT;
				}
				break;
		}
	}

	// set output needs
	io_flags.need_prompt = true;
}

// clear the screen
void
TelnetHandler::clear_scr (void)
{
	// try clear screen sequence
	if (io_flags.use_ansi) {
		// ansi code, try both ways...
		*this << "\e[2J\e[H";
	} else {
		// cheap way
		for (uint i = 0; i < height; ++i)
			*this << "\n";
	}
}

// set indentation/margin
void
TelnetHandler::set_indent (uint amount)
{
	end_chunk();
	margin = amount;
}

// draw a progress bar
void
TelnetHandler::draw_bar (uint percent)
{
	// 20 part bar
	static const char* bar = "============";
	static const char* space = "            ";

	// clip percent
	if (percent > 100)
		percent = 100;

	// draw
	int parts = 12 * percent / 100;
	*this << "[";
	if (parts > 0)
		*this << StreamChunk(bar, parts);
	if (parts < 12)
		*this << StreamChunk(space, 12 - parts);
	*this << "]";
}

// process input
void
TelnetHandler::process (void)
{
	if (!net_alive ())
		return;

	unsigned char buffer[512];

	int size = sock.recv(buffer, sizeof(buffer) - 1);

	// error?
	if (size < 0) {
		if (errno != EINTR && errno != EAGAIN) {
			Log::Error << "Network error: " << strerror(errno);
			net_disconnect();
		}
		return;
	}
	// hungup
	if (size == 0) {
		net_disconnect();
		return;
	}
	// freaky!
	if (size >= (int)sizeof(buffer)) {
		Log::Error << "recv() size larger than buffer size";
		net_disconnect();
		return;
	}

	// finish off buffer
	buffer[size] = 0;

	// deal with telnet options
	unsigned char c;
	size_t in_size = input.size();
	for (int i = 0; i < size; i ++) {
		c = buffer[i];
		switch (istate) {
			case ISTATE_CR:
				istate = ISTATE_TEXT;
				if (c == '\n' || c == '\r' || c == '\0')
					break;
				// fall thru
			case ISTATE_TEXT:
				if (c == IAC) {
					istate = ISTATE_IAC;
					break;
				}

				// only printable characters thank you
				if (c == '\n' || isprint (c)) {
					// need to grow?
					if (in_cnt + 2 >= in_size) {
						if (input.grow()) {
							// input growth failure
							break;
						}
						in_size = input.size();
					}

					// do add
					input.data()[in_cnt ++] = c;
					input.data()[in_cnt] = '\0';

					// echo back normal characters
					if (c != '\n' && io_flags.do_echo)
						send_data (1, c);
				// basic backspace support
				} else if (c == 127) {
					if (in_cnt > 0 && input.data()[in_cnt - 1] != '\n') {
						input.data()[--in_cnt] = '\0';

						if (io_flags.do_echo)
							send_data (3, 127, ' ', 127);
					}
				}

				// get a newline in?
				if (c == '\n') {
					// handle the input data
					if (io_flags.do_echo)
						send_data (2, '\r', '\n');
					io_flags.need_newline = false;
					io_flags.need_prompt = true;

					// count current lines
					uint8 lines = 0;
					for (size_t i = 0; i < in_cnt; ++i) {
						// too many lines?
						if (lines == BUFFER_LINES) {
							*this << CADMIN "You only have " << BUFFER_LINES << " type-ahead lines." CNORMAL "\n";
							input.data()[i] = '\0';
							in_cnt = i;
							break;
						}

						// increment data count
						if (input.data()[i] == '\n')
							++lines;
					}

					istate = ISTATE_CR;
				}
				break;
			case ISTATE_IAC:
				istate = ISTATE_TEXT;
				switch (c) {
					case WILL:
						istate = ISTATE_WILL;
						break;
					case WONT:
						istate = ISTATE_WONT;
						break;
					case DO:
						istate = ISTATE_DO;
						break;
					case DONT:
						istate = ISTATE_DONT;
						break;
					case IAC:
						break;
					case SB:
						istate = ISTATE_SB;
						sb_cnt = 0;
						break;
					case EC:
						break;
					case EL:
					{
						if (in_size) {
							// find last newline
							char* nl = strrchr(input.data(), '\n');
							if (nl == NULL) {
								input.release();
								in_cnt = 0;
								in_size = 0;
							} else {
								// cut data
								in_cnt = nl - input.data();
								input.data()[in_cnt] = '\0';
							}
						}
						break;
					}
				}
				break;
			case ISTATE_SB:
				if (c == IAC) {
					istate = ISTATE_SE;
				} else {
					if (sb_cnt >= subrequest.size()) {
						if (subrequest.grow()) {
							// damn, growth failure
							break;
						}
					}
					subrequest.data()[sb_cnt++] = c;
				}
				break;
			case ISTATE_SE:
				if (c == SE) {
					process_sb ();
					subrequest.release();
					sb_cnt = 0;
					istate = ISTATE_TEXT;
				} else {
					istate = ISTATE_SB;

					if (sb_cnt >= subrequest.size()) {
						if (subrequest.grow()) {
							// damn, growth failure
							break;
						}
					}
					subrequest.data()[sb_cnt++] = IAC;
				}
				break;
			case ISTATE_WILL:
				istate = ISTATE_TEXT;
				switch (c) {
					case TELOPT_NAWS:
						// ignore
						break;
					case TELOPT_TTYPE:
						send_iac (3, SB, TELOPT_TTYPE, 1); // 1 is 'SEND'
						send_iac (1, SE);
						break;
					case TELOPT_NEW_ENVIRON:
						send_iac (4, SB, TELOPT_NEW_ENVIRON, 1, 0); // 1 is 'SEND', 0 is 'VAR'
						send_data (10, 'S', 'Y', 'S', 'T', 'E', 'M', 'T', 'Y', 'P', 'E');
						send_iac (1, SE);
						break;
					default:
						send_iac (2, DONT, c);
						break;
				}
				break;
			case ISTATE_WONT:
				istate = ISTATE_TEXT;
				switch (c) {
					case TELOPT_NAWS:
						// reset to default width
						width = 70;
						break;
				}
				break;
			case ISTATE_DO:
				istate = ISTATE_TEXT;
				switch (c) {
					case TELOPT_ECHO:
						if (io_flags.want_echo)
							io_flags.do_echo = true;
						break;
					case TELOPT_EOR:
						if (!io_flags.do_eor) {
							io_flags.do_eor = true;
							send_iac (2, WILL, TELOPT_EOR);
						}
						break;
#ifdef HAVE_LIBZ
					case TELOPT_MCCP2:
						begin_mccp();
						break;
#endif // HAVE_LIBZ
					case TELOPT_ZMP: {
						// enable ZMP support
						io_flags.zmp = true;
						// send zmp.ident command
						const char* argv[4];
						argv[0] = "zmp.ident"; // command
						argv[1] = "AweMUD NG"; // name
						argv[2] = VERSION; // version
						argv[3] = "Powerful C++ MUD server software"; // about
						send_zmp(4, argv);
						// send x-awemud.name command
						argv[0] = "zmp.check";
						argv[1] = "x-awemud.";
						send_zmp(2, argv);
						break;
					}
					default:
						send_iac (2, WONT, c);
						break;
				}
				break;
			case ISTATE_DONT:
				istate = ISTATE_TEXT;
				switch (c) {
					case TELOPT_ECHO:
						if (!io_flags.force_echo)
							io_flags.do_echo = false;
						break;
					case TELOPT_EOR:
						if (io_flags.do_eor) {
							io_flags.do_eor = false;
							send_iac (2, WONT, TELOPT_EOR);
						}
						break;
					default:
						send_iac (2, WONT, c);
						break;
				}
				break;
			// NEWS negotiation
			default:
				istate = ISTATE_TEXT;
				break;
		}
	}
}

// handle entered commands
void
TelnetHandler::process_input (void)
{
	// have we any input?
	if (!in_cnt)
		return;

	// get one data of data
	char* data = input.data();
	char* nl = (char*)memchr(input.data(), '\n', in_cnt);
	if (nl == NULL)
		return;
	*nl = '\0';

	// do process
	process_command(data);

	// consume command data
	size_t len = nl - input.data() + 1;
	in_cnt -= len;
	memmove(input.data(), input.data() + len, in_cnt);
}

// handle a specific command
void
TelnetHandler::process_command (char* cbuffer)
{
	// time stamp
	time (&in_stamp);

	// force output update
	io_flags.need_prompt = true;
	io_flags.have_output = true;

	// general input?
	if (cbuffer[0] != '!' && mode) {
		mode->process(cbuffer);
	// if it starts with !, process as a telnet layer command
	} else {
		process_telnet_command(&cbuffer[1]);
	}
}

void
TelnetHandler::set_mode (ITelnetMode* new_mode)
{
	// close old mode
	if (mode)
		mode->shutdown();

	mode = new_mode;

	// initialize new mode
	if (mode && mode->initialize()) {
		mode = NULL;
		net_disconnect();
	}
}

void
TelnetHandler::process_telnet_command(char* data)
{
	char *arg = commands::get_arg (&data);
	if (str_eq (arg, "color")) {
		arg = commands::get_arg (&data);
		if (arg && str_eq (arg, "on")) {
			io_flags.use_ansi = true;
			*this << CADMIN "ANSI Color Enabled" CNORMAL "\n";
		} else if (arg && str_eq (arg, "off")) {
			io_flags.use_ansi = false;
			*this << "ANSI Color Disabled\n";
		} else {
			*this << CADMIN "ANSI Color ";
			*this << (io_flags.use_ansi ? "Enabled" : "Disabled");
			*this << CNORMAL "\n";
		}
	} else if (str_eq (arg, "width")) {
		arg = commands::get_arg (&data);
		int new_width = 0;
		if (arg)
			new_width = tolong(arg);
		if (new_width >= 20)
			width = new_width;
		*this << CADMIN "Width: " << width << CNORMAL "\n";
	} else if (str_eq (arg, "height")) {
		arg = commands::get_arg (&data);
		int new_height = 0;
		if (arg)
			new_height = tolong(arg);
		if (new_height >= 10)
			height = new_height;
		*this << CADMIN "Height: " << height << CNORMAL "\n";
	} else if (str_eq (arg, "echo")) {
		arg = commands::get_arg (&data);
		if (arg && str_eq (arg, "on")) {
			io_flags.force_echo = true;
			if (io_flags.want_echo)
				io_flags.do_echo = true;
			*this << CADMIN "Echo Enabled" CNORMAL "\n";
		} else if (arg && str_eq (arg, "off")) {
			io_flags.force_echo = false;
			if (io_flags.want_echo)
				send_iac(WONT, TELOPT_ECHO);
			*this << CADMIN "Echo Disabled" CNORMAL "\n";
		} else {
			*this << CADMIN "Echo ";
			*this << (io_flags.do_echo ? "Enabled" : "Disabled");
			*this << CNORMAL "\n";
		}
	} else if (str_eq (arg, "debug")) {
		*this << "Debug info:\n";
		*this << "  ANSI Color:  " << (io_flags.use_ansi ? "on" : "off") << "\n";
		*this << "  Width:       " << width << "\n";
		*this << "  Forced Echo: " << (io_flags.do_echo ? "on" : "off") << "\n";
		*this << "  Do EOR:      " << (io_flags.do_eor ? "on" : "off") << "\n";
	} else {
		*this << "Telnet argands:\n";
		*this << " !color <on|off>  -- Enable or disable ANSI color.\n";
		*this << " !width [value]   -- Set the column width of your display.\n";
		*this << " !height [value]  -- Set the column height of your display.\n";
		*this << " !echo <on|off>   -- Enable or disable forced server echoing.\n";
		*this << " !debug           -- Display debugging information.\n";
		*this << " !help            -- This display.\n";
	}
}

// process a telnet sub command
void
TelnetHandler::process_sb (void)
{
	char* data = subrequest.data();
	if (data == NULL || sb_cnt == 0)
		return;

	switch (data[0]) {
		// handle ZMP
		case TELOPT_ZMP:
			if (have_zmp())
				process_zmp(sb_cnt - 1, (char*)&data[1]);
			break;
		// resize of telnet window
		case TELOPT_NAWS:
			width = ntohs(*(uint16*)&data[1]);
			height = ntohs(*(uint16*)&data[3]);
			break;
		// handle terminal type
		case TELOPT_TTYPE:
			// proper input?
			if (sb_cnt > 2 && data[1] == 0) {
				// xterm?
				if (sb_cnt == 7 && !memcmp(&data[2], "XTERM", 5))
					io_flags.xterm = true;
				// ansi 
				if (sb_cnt == 6 && !memcmp(&data[2], "ANSI", 4))
					io_flags.ansi_term = true;

				// set xterm title
				if (io_flags.xterm) {
					send_data (3, '\033', ']', ';');
					add_output ("AweMUD NG", 9);
					send_data (1, '\a');
				}
			}
			break;
		// handle new environ
		case TELOPT_NEW_ENVIRON:
			// proper input - IS, VAR
			if (sb_cnt > 3 && data[1] == 0 && data[2] == 0) {
				// system type?
				if (sb_cnt >= 13 && !memcmp(&data[3], "SYSTEMTYPE", 10)) {
					// value is windows?
					if (sb_cnt >= 19 && data[13] == 1 && !memcmp(&data[14], "WIN32", 5)) {
						// we're running windows telnet, most likely
						*this << "\n---\n" CADMIN "Warning:" CNORMAL " AweMUD has detected that "
							"you are using the standard Windows telnet program.  AweMUD will "
							"enable the slower server-side echoing.  You may disable this by "
							"typing " CADMIN "!echo off" CNORMAL " at any time.\n---\n";
						io_flags.force_echo = true;
						if (io_flags.want_echo)
							io_flags.do_echo = true;
					}
				}
			}
			break;
	}
}

// flush out the output, write prompt
void
TelnetHandler::flush(void)
{
	// only if we're alive, and ready to send
	if (!net_alive ())
		return;

	// fix up color
	if (!colors.empty()) {
		if (io_flags.use_ansi)
			add_to_chunk(ANSI_NORMAL, strlen(ANSI_NORMAL));
		colors.resize(0);
	}

	// end chunk
	end_chunk();

	// if we need an update to prompt, do so
	if (io_flags.need_prompt) {
		// prompt
		if (mode)
			mode->prompt();
		else
			*this << ">";

		// clean output
		end_chunk();
		add_output (" ", 1);

		// GOAHEAD telnet command
		if (io_flags.do_eor)
			send_iac (1, EOR);
		
		io_flags.need_prompt = false;
		io_flags.need_newline = true;
		cur_col = 0;
	}

	// finished buffer
	if (out_cnt) {
		send_bytes(output.data(), out_cnt);
		output.release();
		out_cnt = 0;
	}
}

// send out a telnet command
void
TelnetHandler::send_iac (uint count, ...)
{
	va_list va;
	static unsigned char buffer[16]; // simple buffer
	uint bc = 1; // buffer index
	uint byte;
	buffer[0] = IAC; // we need to send an IAC

	va_start (va, count);
	// loop thru args
	for (uint i = 0; i < count; i ++) {
		byte = va_arg (va, uint);

		// add byte
		buffer[bc ++] = byte;
		if (bc >= 16) {
			add_output ((char *)buffer, 16);
			bc = 0;
		}

		// doube up on IAC character
		if (byte == IAC)
			buffer[bc ++] = IAC;
		if (bc >= 16) {
			add_output ((char *)buffer, 16);
			bc = 0;
		}
	}

	// write out rest of buffer
	if (bc)
		add_output ((char *)buffer, bc);
}

// send out telnet data
void
TelnetHandler::send_data (uint count, ...)
{
	va_list va;
	static unsigned char buffer[16]; // simple buffer
	uint bc = 0; // buffer index
	uint byte;

	va_start (va, count);
	// loop thru args
	for (uint i = 0; i < count; i ++) {
		byte = va_arg (va, uint);

		// add byte
		buffer[bc ++] = byte;
		if (bc >= 16) {
			add_output ((char *)buffer, 16);
			bc = 0;
		}

		// doube up on IAC character
		if (byte == IAC)
			buffer[bc ++] = IAC;
		if (bc >= 16) {
			add_output ((char *)buffer, 16);
			bc = 0;
		}
	}

	// write out rest of buffer
	if (bc)
		add_output ((char *)buffer, bc);
}

void
TelnetHandler::add_to_chunk (const char *data, size_t len) {
	// output indenting
	OUTPUT_INDENT()

	// grow chunk buffer if needed
	while (len + outchunk_cnt >= outchunk.size()) {
		// chunk buffer overflow - force dump
		if (outchunk.grow()) {
			end_chunk();
			break;
		}
	}

	// still too big? cut excess chunks
	while (len >= outchunk.size()) {
		add_output(data, outchunk.size());
		data += outchunk.size();
		len -= outchunk.size();
	}

	// append remaining data
	if (len > 0) {
		memcpy(outchunk.data() + outchunk_cnt, data, len * sizeof(char));
		outchunk_cnt += len;
	}
}

void
TelnetHandler::add_output (const char *data, size_t len) {
	// add to output list
	io_flags.have_output = true;

	// grow output buffer if needed
	while (len + out_cnt >= output.size()) {
		// output buffer overflow - force dump
		if (output.grow()) {
			send_bytes(output.data(), out_cnt);
			out_cnt = 0;
			break;
		}
	}

	// still too big? copy & send (ugh)
	while (len >= output.size()) {
		memcpy(output.data(), data, output.size());
		send_bytes(output.data(), output.size());
		data += output.size();
		len -= output.size();
	}

	// append remaining data
	if (len > 0) {
		memcpy(output.data() + out_cnt, data, len * sizeof(char));
		out_cnt += len;
	}
}

void
TelnetHandler::end_chunk (void)
{
	// only if we have data
	if (outchunk_cnt > 0) {
		// need to word-wrap?
		if (width > 0 && chunk_size + cur_col >= width) {
			add_output("\n\r", 2);
			cur_col = 0;
			OUTPUT_INDENT()
		}

		// do output
		add_output(outchunk.data(), outchunk_cnt);
		outchunk_cnt = 0;
		cur_col += chunk_size;
		chunk_size = 0;

		io_flags.soft_break = false;
	}
}

// check various timeouts
int
TelnetHandler::check_time (void)
{
	// get time diff
	time_t now;
	time (&now);
	unsigned long diff = (unsigned long)difftime(now, in_stamp);

	// get timeout
	uint timeout = TelnetManager.timeout_minutes;
	if (mode != NULL) {
		Account* account = mode->get_account();
		if (account != NULL && account->get_timeout() != 0)
			timeout = account->get_timeout();
	}

	// check time..
	if (diff >= (timeout * 60)) {
		// disconnect the dink
		*this << CADMIN "You are being disconnected for lack of activity." CNORMAL "\n";
		Log::Info << "Connection timeout (" << TelnetManager.timeout_minutes << " minutes of no input).";
		net_disconnect();
		return 1;
	}

	// no disconnections... yet
	return 0;
}

int
STelnetManager::initialize (void)
{
	// modules we require
	if (server.require(ZMPManager) != 0)
		return 0;

	Log::Info << "Initializing telnet manager";

	// get timeout minutes
	timeout_minutes = settings::get_int("network", "timeout");
	if (timeout_minutes == 0)
		timeout_minutes = 15; // default 15 minutes

	return 0;
}

void
STelnetManager::shutdown (void)
{
	// disconnect all telnets
	while (!telnet_list.empty()) {
		telnet_list.front()->net_disconnect();
		telnet_list.erase(telnet_list.begin());
	}
	telnet_list.resize(0);
}

size_t
STelnetManager::count (void)
{
	return telnet_list.size();
}

// update all telnet connections
void
STelnetManager::process (void)
{
	for (TelnetList::iterator i = telnet_list.begin(); i != telnet_list.end();) {
		// ready?
		if ((*i)->net_inready ()) {
			// process
			(*i)->process (); 
		// telnet hung up?
		} else if ((*i)->net_hungup()) {
			// disconnect
			(*i)->net_disconnect();
		}

		// check time
		(*i)->check_time();
	
		// process input
		(*i)->process_input ();

		// disconnected?
		if (!(*i)->net_alive()) {
			i = telnet_list.erase(i);
		} else {
			++i;
		}
	}
}

void
STelnetManager::flush (void)
{
	for (TelnetList::iterator i = telnet_list.begin(); i != telnet_list.end(); ++i) {
		// ready?
		if ((*i)->io_flags.have_output) {
			// flush any remaining output
			(*i)->flush(); 
		}
	}
}
