/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#include "server.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <signal.h>
#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <termios.h>
#include <pwd.h>
#include <grp.h>
#include <netdb.h>

#include <fstream>

#include "room.h"
#include "error.h"
#include "object.h"
#include "awestr.h"
#include "parse.h"
#include "rand.h"
#include "char.h"
#include "body.h"
#include "help.h"
#include "scripts.h"
#include "network.h"
#include "npc.h"
#include "player.h"
#include "settings.h"
#include "blueprint.h"
#include "command.h"
#include "social.h"
#include "weather.h"
#include "streams.h"
#include "zone.h"
#include "control.h"
#include "message.h"
#include "telnet.h"
#include "account.h"
#include "race.h"
#include "class.h"

// Our main nifty cool server object
Server server;

// Signal handlers
static void sigterm_handler (int);
static void sigint_handler (int);
static void sighup_handler (int);

// Signal flags
volatile bool signaled_shutdown = false;
volatile bool signaled_reload = false;

void
static
close_server_on_exit (void)
{
	server.finish ();
}

static
int
write_pid_file (StringArg path)
{
	// open it up
	int fd;
	if ((fd = open(path, O_WRONLY|O_CREAT, 0640)) < 0) {
		// fatal error
		Log::Error << "Failed to create or open PID file '" << path << "': " << strerror(errno);
		return -1;
	}

	// try to lock; if it's already locked, another process
	// owns the PID lock, otherwise, it's a stale PID file and
	// we can safely over-write it.
	struct flock lockinfo;
	lockinfo.l_type = F_WRLCK;
	lockinfo.l_whence = SEEK_SET;
	lockinfo.l_start = 0;
	lockinfo.l_len = 0;
	if (fcntl(fd, F_SETLK, &lockinfo) < 0) {
		// fatal error
		Log::Error << "Failed to lock PID file '" << path << "': " << strerror(errno);
		Log::Error << "AweMUD appears to already be running.";
		return -1;
	}

	// truncate file
	if (ftruncate(fd, 0) < 0) {
		Log::Error << "Failed to truncate PID file '" << path << "': " << strerror(errno);
		unlink(path);
		close(fd);
		return -1;
	}

	// write PID
	String pidstr;
	StreamControl(pidstr) << getpid() << "\n";
	if (write(fd, pidstr.c_str(), pidstr.size()) != (int)pidstr.size()) {
		Log::Error << "Failed to write to PID file '" << path << "': " << strerror(errno);
		unlink(path);
		close(fd);
		return -1;
	}

	// all good!
	return 0;
}

Server::Server(void) : tcp_players(sockets), ctrl_users(sockets), log (NULL), sockets(), running(true) {}

int
Server::init (int argc, char **argv)
{
	const char *conf_file = NULL;

	// have we a config file?
	if (argc > 1)
		conf_file = argv[1];

	// load settings
	if (settings::init (conf_file)) {
		Log::Error << "Failed to load settings";
		return 1;
	}

	// logging
	String logpath = settings::get_path("logfile", NULL);
	if (logpath)
		log = new LogFile (logpath);
	else
		log = new LogFile (NULL, "stdout");
	if (log == NULL) {
		Log::Error << "Failed to open logfile";
		return 1;
	}
	Log::Info << "Logging to " << log->get_file();

	// fork daemon
	if (settings::get_bool ("system", "daemon")) {
		if (fork ())
			_exit (0);

		// close controlling TTY
#ifdef TIOCNOTTY
		int ttyfd = open ("/dev/tty", O_RDWR);
		ioctl (ttyfd, TIOCNOTTY);
		close (ttyfd);
#endif

		// new process group
		setpgid (0, 0);

		// we don't need these at all, don't want them
		close (0); // STDIN
		close (1); // STDOUT
		close (2); // STDERR

		// legacy code/crap, /dev/null as our stdin/stdout/stderr - yay...
		int base_fd = open ("/dev/null", O_RDWR); // stdin
		dup (base_fd); // stdout
		dup (base_fd); // stderr

		Log::Info << "Forked daemon";
	}

	// set time
	::time (&start_time);

	// write PID
	String pid_path = settings::get_path("pidfile", "awemud.pid");
	if (write_pid_file(pid_path))
		return 1;
	Log::Info << "Wrote PID file '" << pid_path << "'";

	// read group/user info
	struct group *grp = NULL;
	const char *group_name = settings::get_str ("system", "group");
	if (group_name && !str_is_number (group_name)) {
		if (str_is_number (group_name))
			grp = getgrgid (tolong(group_name));
		else
			grp = getgrnam (group_name);
		if (grp == NULL) {
			Log::Error << "Couldn't find group '" << group_name << "': " << strerror(errno);
			return 1;
		}
	}
	struct passwd *usr = NULL;
	const char *user_name = settings::get_str ("system", "user");
	if (user_name) {
		if (str_is_number (user_name))
			usr = getpwuid (tolong(user_name));
		else
			usr = getpwnam (user_name);
		if (usr == NULL) {
			Log::Error << "Couldn't find user '" << user_name << "': " << strerror(errno);
			return 1;
		}
	}

	// set our hostname
	host = settings::get_str("network", "host");
	if (!host) {
		char host_buffer[256];
		if (gethostname(host_buffer, sizeof(host_buffer))) {
			Log::Error << "gethostname() failed: " << strerror(errno);
			return 1;
		}
		struct hostent* host_info = gethostbyname(host_buffer);
		if (host_info == NULL) {
			Log::Error << "gethostbyname() failed for " << host_buffer << ": " << strerror(errno);
			return 1;
		}
		host = host_info->h_name;
	}
	Log::Info << "Host is " << host;

	// do chroot jail
	const char* chroot_dir = settings::get_str("system", "chroot");
	if (chroot_dir) {
		if (chroot (chroot_dir)) {
			Log::Error << "chroot() failed: " << strerror (errno);
			return 1;
		}
		chdir ("/"); // activate chroot
		Log::Info << "Chrooted to '" << chroot_dir << "'";
	}

	// exit() function
	atexit (close_server_on_exit);

	// seting up signal handlers
	struct sigaction sa;
	memset (&sa, 0, sizeof (sa));
	sa.sa_handler = sigterm_handler;
	if (sigaction (SIGTERM, &sa, NULL)) {
		Log::Error << "sigaction() failed (SIGTERM)";
		return 1;
	}
	memset (&sa, 0, sizeof (sa));
	sa.sa_handler = sigint_handler;
	if (sigaction (SIGINT, &sa, NULL)) {
		Log::Error << "sigaction() failed (SIGINT)";
		return 1;
	}
	memset (&sa, 0, sizeof (sa));
	sa.sa_handler = SIG_IGN;
	if (sigaction (SIGPIPE, &sa, NULL)) {
		Log::Error << "sigaction() failed (SIGPIPE)";
		return 1;
	}
	memset (&sa, 0, sizeof (sa));
	sa.sa_handler = sighup_handler;
	if (sigaction (SIGHUP, &sa, NULL)) {
		Log::Error << "sigaction() failed (SIGHUP)";
		return 1;
	}

	init_random (); // random info

	// load misc
	if (require(MessageManager))
		return 1;
	if (require(EventManager))
		return 1;
	if (require(TimeManager))
		return 1;
	if (require(WeatherManager))
		return 1;
	if (require(EntityManager))
		return 1;
	if (require(AccountManager))
		return 1;
	if (require(TelnetManager))
		return 1;
	if (require(PlayerManager))
		return 1;
	if (require(HelpManager))
		return 1;
	if (require(CommandManager))
		return 1;
	if (require(Scripts))
		return 1;
	if (require(SocialManager))
		return 1;
	if (require(ObjectClassManager))
		return 1;
	if (require(RaceManager))
		return 1;
	if (require(ClassManager))
		return 1;
	if (require(AIManager))
		return 1;
	if (require(ObjectBlueprintManager))
		return 1;
	if (require(NpcBlueprintManager))
		return 1;

	// load world
	if (require(ZoneManager))
		return 1;

	// load IP block list
	Log::Info << "Reading deny list";
	{
		String path = settings::get_path("misc", "data") + "/denylist";

		std::ifstream in(path);
		if (in) {
			String denyline;
			// ready deny lines
			while (getline(in, denyline)) {
				if (!denyline.empty()) {
					// try IPv4
					if (!tcp_players.add_deny(denyline))
						Log::Warning << "Malformed line in " << path << ": '" << denyline << "'";
				}
			}
		} else {
			Log::Error << "Failed to open '" << path << "': " << strerror(errno);
			return 1;
		}
	}

	// control interface
	String ctrl_path = settings::get_path("control", NULL);
	if (ctrl_path) {
		if (ControlManager.initialize()) {
			Log::Error << "ControlManager.initialize() failed";
			return 1;
		}
		if (ctrl_users.listen(ctrl_path)) {
			Log::Error << "Failed to open " << ctrl_path << " for control interface";
			return 1;
		}
		Log::Info << "Control interface active on " << ctrl_path;
	}

	// IPv6 message
#ifdef HAVE_IPV6
	if (settings::get_bool("network", "ipv6")) {
		tcp_players.set_use_ipv6(true);
		Log::Info << "IPv6 enabled";
	}
#endif // HAVE_IPV6

	// network options
	int maxhostconns = settings::get_int ("network", "maxhostconns");
	int maxconns = settings::get_int ("network", "maxconns");

	// network server
	int accept_port = settings::get_int ("network", "port");
	if (accept_port <= 0)
		accept_port = SERV_PORT;
	if (maxhostconns)
		tcp_players.set_max_host_conns(maxhostconns);
	if (maxconns)
		tcp_players.set_max_total_conns(maxconns);
	if (tcp_players.listen(accept_port)) {
		Log::Error << "tcp_players.listen() failed";
		return 1;
	}
	Log::Info << "Listening on port " << accept_port;

	// change user/group
	if (grp) {
		if (setregid (grp->gr_gid, grp->gr_gid)) {
			Log::Error << "Couldn't set group to '" << grp->gr_name << "': " << strerror(errno);
			return 1;
		}
		Log::Info << "Set group to " << grp->gr_name << " (" << grp->gr_gid << ")";
		// drop supplemental groups
		setgroups(0, NULL);
	}
	if (usr) {
		if (setreuid (usr->pw_uid, usr->pw_uid)) {
			Log::Error << "Couldn't set user to '" << usr->pw_name << "': " << strerror(errno);
			return 1;
		}
		Log::Info << "Set user to " << usr->pw_name << " (" << usr->pw_uid << ")";
	}

	// all set
	Log::Info << "Ready";

	return 0;
}

// initialize a manager if not already handled
int
Server::require (IManager& manager)
{
	// already initialized?
	if (std::find(managers.begin(), managers.end(), &manager) != managers.end())
		return 0;

	// add to managers list
	managers.push_back(&manager);

	// try initialization
	if (manager.initialize()) {
		// remove from list
		std::vector<IManager*>::iterator miter = std::find(managers.begin(), managers.end(), &manager);
		if (miter != managers.end())
			managers.erase(miter);

		// return error
		return 1;
	}

	// all good!
	return 0;
}

int
Server::finish (void)
{
	// shutdown all managers
	while (!managers.empty()) {
		IManager* manager = managers.back();
		managers.pop_back();
		manager->shutdown();
	}

	if (log)
		delete (log);
	log = NULL;

	String pid_path = settings::get_path("pidfile", "awemud.pid");
	unlink (pid_path);

	settings::close ();

	Log::Info << "Finalizing memory objects...";
	GC_gcollect();
	Log::Info << "Done!";

	return 0;
}

int
Server::run (void)
{
	Socket socket;

	gettimeofday (&last_time, NULL);
	game_ticks = 0;
	
	while (running) {
		// do select - no player, don't timeout
		if (!PlayerManager.count())
			sockets.poll (-1);
		else
			sockets.poll (MSECS_PER_TICK);

		// tcp player connection?
		if (tcp_players.accept (socket) == 0) {
			// create a new connection
			TelnetHandler *telnet = new TelnetHandler ();

			// failure?
			if (telnet == NULL || telnet->net_connect (socket)) {
				Log::Warning << "TelnetHandler::net_connect() failed, closing connection.";
				socket.close();
			// success
			} else {
				// banner
				telnet->clear_scr();
				*telnet <<
					"\n ----===[ " ANSI_MAGENTA "AweMUD" ANSI_NORMAL " V" VERSION " ]===----\n\n"
					"AweMUD Copyright (C) 2000-2004  AwesomePlay Productions, Inc.\n"
					"Visit " ANSI_CYAN "http://www.awemud.net" ANSI_NORMAL " for more details.\n";

				// connect message
				*telnet << StreamParse(MessageManager.get("connect"));

				// init login
				telnet->set_mode(new TelnetModeLogin(telnet));

				// force output flush
				telnet->flush();
			}
		}

		// unix control process connection?
		if (ctrl_users.accept (socket) == 0) {
			// check access
			uid_t uid;
			if (socket.get_peer_uid(&uid) || uid != getuid()) {
				// rejected!
				Log::Info << "Rejected control connection from " << uid;
				socket.close();
			} else {
				// add user
				Log::Info << "New control connection";
				socket.send("+OK Welcome\n", 12);
				ControlManager.add(new ControlUser(socket));
			}
		}

		// control manager
		ControlManager.process();
		
		// process player connections
		TelnetManager.process();
		
		// update the server info
		update ();

		// handle events
		EventManager.process();

		// run Scriptix
		Scripts.update();

		// flush output
		TelnetManager.flush();

		// check for reload
		if (signaled_reload == true) {
			signaled_reload = false;
			Log::Info << "Server received a SIGHUP";
			save ();
			log_reset ();
		}

		// check for signaled_shutdown
		if (signaled_shutdown == true) {
			signaled_shutdown = false;
			Log::Info << "Server received a terminating signal";
			save ();
			shutdown ();
		}
	}

	return 0;
}

void
Server::shutdown (void)
{
	Log::Info << "Shutting down server";
	running = false;
}

void
Server::save (void) const
{
	Log::Info << "Saving...";
	TimeManager.save();
	ZoneManager.save();
}

void
Server::update (void)
{
	struct timeval cur_time;
	static uint auto_save_time = 0;
	static uint elapsed_time = 0;

	gettimeofday (&cur_time, NULL);
	uint change_time = (((cur_time.tv_sec - last_time.tv_sec) * 1000000) + (cur_time.tv_usec - last_time.tv_usec));
	elapsed_time += change_time;
	auto_save_time += change_time;
	last_time = cur_time;

	if (auto_save_time >= (15 * 60 * 1000 * 1000)) { // 15 minutes in microseconds
		auto_save_time = 0;
		save ();
	}

	if (elapsed_time/1000 >= MSECS_PER_TICK) {
		game_ticks ++;
		elapsed_time %= MSECS_PER_TICK;

		// update entities
		EntityManager.update();

		// update weather
		WeatherManager.update();

		// update time
		bool is_day = TimeManager.time.is_day ();
		TimeManager.time.update (1);
		if (is_day && !TimeManager.time.is_day ())
			if (!TimeManager.calendar.sunset_text.empty())
				ZoneManager.announce(TimeManager.calendar.sunset_text[get_random(TimeManager.calendar.sunset_text.size())], ANFL_OUTDOORS);
		else if (!is_day && TimeManager.time.is_day ())
			if (!TimeManager.calendar.sunrise_text.empty())
				ZoneManager.announce(TimeManager.calendar.sunrise_text[get_random(TimeManager.calendar.sunrise_text.size())], ANFL_OUTDOORS);
	}
}

// termination signal handler
void
sigterm_handler (int)
{
	signaled_shutdown = true;
}

// interrupt signal handler
void
sigint_handler (int)
{
	signaled_shutdown = true;
}

// hangup signal handler
void
sighup_handler (int)
{
	signaled_reload = true;
}

void
Server::log_msg (int level, const char *text)
{
	if (log)
		log->print (level, text);
	else
		fprintf (stderr, "%s", text);
}

void
Server::log_reset (void)
{
	if (log) {
		Log::Info << "Reopening log file";
		log->reset ();
	}
}
