/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "rand.h"
#include "race.h"
#include "settings.h"
#include "fileobj.h"
#include "log.h"
#include "char.h"

SRaceManager RaceManager;

int
Race::load (File::Reader& reader)
{
	// clear and/or defaults
	adj.clear();
	about = "AweMUD player race.";
	desc.clear();
	age_min = 18;
	age_max = 25;
	life_span = 90;
	body = "human";
	eye_colors.clear();
	hair_colors.clear();
	skin_colors.clear();
	skin_type = "skin";
	for (int i = 0; i < CharStatID::COUNT; ++i)
		stats[i] = 0;

	// read loop
	File::Node node;
	FO_READ_BEGIN

		// figure out what we're setting
		FO_ATTR_NAME("name")
			name = node.get_data();
		FO_ATTR_NAME("adj")
			adj = node.get_data();
		FO_ATTR_NAME("body")
			body = node.get_data();
		FO_ATTR_NAME("about")
			about = node.get_data();
		FO_ATTR_NAME("desc")
			desc = node.get_data();
		FO_ATTR_NAME("eyes")
			ColorType eye_color = ColorType::lookup(node.get_data());
			eye_colors.push_back(eye_color);
		FO_ATTR_NAME("hair")
			ColorType hair_color = ColorType::lookup(node.get_data());
			hair_colors.push_back(hair_color);
		FO_ATTR_NAME("skin")
			ColorType skin_color = ColorType::lookup(node.get_data());
			skin_colors.push_back(skin_color);
		FO_ATTR_NAME("skin_type")
			skin_type = node.get_data();
		FO_ATTR_NAME("min_age")
			if (!node.get_int_data(age_min))
				Log::Warning << "Attribute 'min_age' in race '" << name << "' at " << reader.get_filename() << ':' << node.get_line();
		FO_ATTR_NAME("max_age")
			if (!node.get_int_data(age_max))
				Log::Warning << "Attribute 'max_age' in race '" << name << "' at " << reader.get_filename() << ':' << node.get_line();
		FO_ATTR_NAME("lifespan")
			if (!node.get_int_data(life_span))
				Log::Warning << "Attribute 'life_span' in race '" << name << "' at " << reader.get_filename() << ':' << node.get_line();
		FO_ATTR_TYPE("stat") {
			// is this a stat?
			CharStatID stat = CharStatID::lookup(node.get_name());
			if (stat) {
				if (!node.get_int_data(stats[stat.get_value()]))
					Log::Warning << "Attribute '" << node.get_name() << "' in race '" << name << "' is not a valid integer at " << reader.get_filename() << ':' << node.get_line();
			} else
				Log::Warning << "Unknown attribute '" << node.get_name() << "' in race '" << name << "' at " << reader.get_filename() << ':' << node.get_line();
			}
	FO_READ_ERROR
		return -1;
	FO_READ_END

	// fixup adjective
	if (!adj)
		adj = name;

	// sanity check age/lifespan
	if ((age_min > age_max) || (age_min <= 0) || (age_max >= life_span) || (life_span <= 0)) {
		Log::Error << "Minimum/maximum ages and/or lifespan in race '"
			<< name << "' are invalid\n";
		return -1;
	}

	// ok
	return 0;
}

int
Race::get_rand_age (void) const
{
	return age_min + get_random (age_max - age_min);
}

Race *
SRaceManager::get (StringArg name)
{
	Race *race = head;
	while (race != NULL) {
		if (str_eq (race->get_name(), name))
			return race;
		race = race->get_next();
	}
	return NULL;
}

int
SRaceManager::initialize(void)
{
	Log::Info << "Loading player races";

	File::Reader reader;
	String path = settings::get_path ("misc", "data") + "/races";

	if (reader.open(path)) {
		Log::Error << "Failed to open " << path;
		return 1;
	}

	// load
	File::Node node;
	while (reader.get(node)) {
		// new object?
		if (node.is_begin()) {
			// not a race?
			if (node.get_type() != "race") {
				Log::Warning << "Unknown object '" << node.get_type() << "' at " << reader.get_filename() << ":" << node.get_line();
				reader.consume();
				continue;
			}
			// valid parameters
			if (node.get_name().empty()) {
				Log::Warning << "Unnamed race at " << reader.get_filename() << ":" << node.get_line();
				reader.consume();
				continue;
			}

			// load race
			Race *race = new Race (node.get_name(), head);
			if (!race->load (reader)) {
				head = race;
			} else {
				delete race;
			}
		// wtf?
		} else {
			Log::Warning << "Invalid input at " << reader.get_filename() << ":" << node.get_line();
		}
	}

	return 0;
}

void
SRaceManager::shutdown (void)
{
	head = NULL;
}
