/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#include <unistd.h>
#include <stdio.h>
#include <ctype.h>
#include <fcntl.h>
#include <math.h>
#include <fnmatch.h>
#include <time.h>
#include <sys/stat.h>

#include "char.h"
#include "player.h"
#include "server.h"
#include "awestr.h"
#include "rand.h"
#include "body.h"
#include "settings.h"
#include "room.h"
#include "parse.h"
#include "streams.h"
#include "color.h"
#include "ptitle.h"
#include "zone.h"
#include "message.h"
#include "telnet.h"
#include "account.h"
#include "log.h"
#include "zmp.h"
#include "eventids.h"
#include "race.h"
#include "class.h"

// manager of players
SPlayerManager PlayerManager;

// manager of player titles
SPlayerTitleManager PlayerTitleManager;

const char *exp_names[] = {
	"General",
	"Warrior",
	"Rogue",
	"Caster",
	NULL
};

Player::Player (class Account* s_account, StringArg s_name) : Character (AweMUD_PlayerType), birthday()
{
	// initialize
	account = s_account;

	health_base = 0;
	title = NULL;
	conn = NULL;

	flags.valid = false;
	flags.bvision = false;

	pdesc.eye_color = 0;
	pdesc.hair_color = 0;
	pdesc.skin_color = 0;
	pdesc.hair_style = 0;
	pdesc.build = 0;

	race = NULL;

	for (int i = 0; i < NUM_EXPS; ++ i) {
		exp[i] = 0;
	}

	// set name
	name = s_name;
	ntype = PROPER;

	// register
	PlayerManager.player_list.push_back(this);
}

Player::~Player (void)
{
	// remove
	SPlayerManager::PlayerList::iterator i = std::find(PlayerManager.player_list.begin(), PlayerManager.player_list.end(), this);
	if (i != PlayerManager.player_list.end())
		PlayerManager.player_list.erase(i);
}

void
Player::display (const StreamControl& stream, EntityArticleType, bool) const
{
	// prefix title
	if (title != NULL) {
		// female title
		if (get_gender() == GenderType::FEMALE) {
			if (title->get_female_prefix())
				stream << title->get_female_prefix() << " ";
		// male title
		} else if (get_gender() == GenderType::MALE) {
			if (title->get_male_prefix())
				stream << title->get_male_prefix() << " ";
		// neuter title
		} else {
			if (title->get_neuter_prefix())
				stream << title->get_neuter_prefix() << " ";
		}
	}

	// name
	stream << ncolor() << get_name() << CNORMAL;
}
	
void
Player::save (File::Writer& writer) const
{
	Character::save(writer);

	writer.attr("base_hp", health_base);

	if (race != NULL)
		writer.attr("race", race->get_name());

	writer.begin("birthday");
	birthday.save(writer);
	writer.end();

	if (prompt)
		writer.attr("prompt", prompt);

	if (title != NULL)
		writer.attr("title", title->get_name());
	
	if (pdesc.eye_color)
		writer.attr("eyecolor", pdesc.eye_color.get_name());
	if (pdesc.hair_color)
		writer.attr("haircolor", pdesc.hair_color.get_name());
	if (pdesc.skin_color)
		writer.attr("skincolor", pdesc.skin_color.get_name());
	writer.attr("hairstyle", pdesc.hair_style.get_name());
	writer.attr("build", pdesc.build.get_name());

	if (get_room()) 
		writer.attr("location", get_room()->get_id());

	for (int i = 0; i < NUM_EXPS; ++ i)
		writer.attr("exp", exp_names[i], exp[i]);

	for (ClassList::const_iterator i = classes.begin (); i != classes.end (); ++ i)
		writer.attr("class", i->first->get_name(), i->second);
}

void
Player::save (void) const
{
	String path = PlayerManager.path (get_name());

	// backup player file
	if (settings::get_bool ("backup", "players")) {
		// only if it exists
		struct stat st;
		if (!stat(path, &st)) {
			time_t base_t;
			time (&base_t);
			char time_buffer[15];
			strftime (time_buffer, sizeof (time_buffer), "%Y%m%d%H%M%S", localtime (&base_t));
			String backup = path + "." + time_buffer + "~";
			if (rename (path, backup)) // move file
				Log::Error << "Backup of " << path << " to " << backup << " failed: " << strerror(errno);
		}
	}

	// do save
	mode_t omask = umask(0066);
	File::Writer writer(path);
	writer.comment(String("Player file: ") + get_name());
	time_t t;
	time(&t);
	writer.comment(String("Timestamp: ") + String(ctime(&t)));
	writer.bl();
	save(writer);
	writer.close();
	umask(omask);

	// log
	Log::Info << "Saved player " << get_name();

	return;
}

int
Player::load_node (File::Reader& reader, File::Node& node)
{
	FO_NODE_BEGIN
		FO_PARENT(Character)
		FO_ATTR_NAME("base_hp")
			FO_GET_INT(health_base);
		FO_ATTR_NAME("race")
			race = RaceManager.get (node.get_data());
			if (race == NULL) {
				Log::Error << "Player has invalid race '" << node.get_data() << "' at " << reader.get_filename() << ':' << node.get_line();
				return -1;
			}
		FO_OBJECT("birthday")
			birthday.load(reader);
		FO_ATTR_NAME("prompt")
			prompt = node.get_data();
		FO_ATTR_NAME("eyecolor")
			pdesc.eye_color = ColorType::lookup(node.get_data());
		FO_ATTR_NAME("haircolor")
			pdesc.hair_color = ColorType::lookup(node.get_data());
		FO_ATTR_NAME("skincolor")
			pdesc.skin_color = ColorType::lookup(node.get_data());
		FO_ATTR_NAME("hairstyle")
			pdesc.hair_style = HairStyleType::lookup(node.get_data());
		FO_ATTR_NAME("build")
			pdesc.build = BodyBuildType::lookup(node.get_data());
		FO_ATTR_NAME("title")
			title = PlayerTitleManager.get_title(node.get_data());
			if (title == NULL)
				Log::Warning << "Undefined player title '" << node.get_data() << "' at " << reader.get_filename() << ':' << node.get_line();
		FO_ATTR_NAME("location")
			location = ZoneManager.get_room(node.get_data());
			if (location == NULL)
				Log::Error << "Unknown room '" << node.get_data() << "' at " << reader.get_filename() << ':' << node.get_line();
		FO_ATTR_TYPE("exp")
			int e = get_index_of (exp_names, node.get_name().c_str());
			if (e >= 0)
				FO_GET_INT(exp[e]);
		FO_ATTR_TYPE("class")
			const Class *klass = ClassManager.get (node.get_name());
			if (klass) {
				int value;
				FO_GET_INT(value);
				classes[klass] += value;
			} else {
				Log::Error << "Invalid class '" << node.get_name() << "' at " << reader.get_filename() << ':' << node.get_line();
				return -1;
			}
	FO_NODE_END
}

void
Player::load_init (void)
{
}

int
Player::load_finish (void)
{
	// becomes valid is loaded
	flags.valid = true;
	
	// reset name if needed
	ntype = PROPER;

	return 0;
}

// 'startup' the player session
int
Player::start (void)
{
	// login message
	clear_scr();
	*this << "\n" << StreamParse (MessageManager.get("login"), "user", this) << "\n";

	// not already active?  add to room...
	if (!is_active()) {
		if (location) {
			// try to enter room
			if (!enter(location, NULL)) {
				*this << CADMIN "Internal error: could not enter room" CNORMAL;
				Log::Error << "Player '" << get_name() << "' could not enter '" << location->get_id() << "' at login";
				return -1;
			}
		} else {
			// location/room doesn't exist - go back to 'starting' location
			if (Scripts.hook("player_start", this) == 0) {
				*this << CADMIN "Internal error: no player_start hook" CNORMAL;
				Log::Error << "Player '" << get_name() << "' could not login because there is no player_start hook";
				return -1;
			}

			// no valid location - eek!
			if (!location) {
				*this << CADMIN "Internal error: no start location given" CNORMAL;
				Log::Error << "Player '" << get_name() << "' could not login because the player_start hook did not assign a start location";
				return -1;
			}
		}
	// already active... just "refresh" room
	} else {
		do_look();
	}

	// player becomes valid
	flags.valid = true;

	// no timeout - yet
	ninfo.timeout_ticks = 0;

	return 0;
}

// 'create' the chracter
int
Player::create (void)
{
	assert(!flags.valid);

	// either run the hook, or start creation processor
	if (!Scripts.hook("create_character", this)) {
		*this << CADMIN "New character creation is not available." CNORMAL "\n";
		Log::Error << "No create_character hook available but character creation is enabled";
		return -1;
	}

	return 0;
}

// make player valid
int
Player::validate (void)
{
	// already valid?  silly dink.
	if (is_valid())
		return -1;

	// add character to account
	get_account()->add_character(get_name());
	get_account()->save();

	// set as valid
	flags.valid = true;

	// save
	save();

	// log it
	Log::Info << "New player " << get_name() << " created for account " << get_account()->get_id();

	// start the player
	return start();
}

// quit from game
void
Player::quit (void)
{
	// save player if valid; invalid needs explicit save
	if (is_valid())
		save ();

	// disengage from game world
	remove();

	// no longer active
	if (is_active())
		deactivate();

	// disconnect
	disconnect();
}

uint
Player::get_age (void) const
{
	// calculate the age in years, based on birthdate and current time
	uint years = TimeManager.time.get_year() - birthday.get_year();
	if (TimeManager.time.get_month() < birthday.get_month())
		years --;
	else if (TimeManager.time.get_month() == birthday.get_month())
		if (TimeManager.time.get_day() < birthday.get_day())
			years --;
	return years;
}

void
Player::kill (Entity *killer)
{
	// death message
	if (get_room())
		*get_room() << StreamIgnore(this) << StreamName(this, DEFINITE, true) << " has been slain!\n";
	*this << "You have been slain!\n";

	// now laying down
	pos = CharPosition::LAY;

	// event/hook
	EventManager.send(Events::ON_DEATH, get_room(), killer, this);
	Scripts.hook("player_death", this, killer);
}

void
Player::display_inventory (void)
{
	// start - worn
	*this << "You are wearing ";

	// inventory variables
	uint loc = 0;
	Object* obj;
	Object* last = NULL;
	bool didshow = false;

	// worn items
	while ((obj = get_worn_at(loc++)) != NULL) {
		// we had one already?
		if (last) {
			// prefix
			if (didshow)
				*this << ", ";
			didshow = true;
			// do show
			*this << StreamName(last, INDEFINITE);
		}
		// remember this object
		last = obj;
	}
	// spit out the left over
	if (last) {
		// prefix
		if (didshow)
			*this << " and ";
		// show it
		*this << StreamName(last, INDEFINITE);
	} else {
		*this << "nothing";
	}

	// start - helf
	*this << ".  You are holding ";

	// held items
	loc = 0;
	didshow = false;
	last = NULL;
	while ((obj = get_held_at(loc++)) != NULL) {
		// we had one already?
		if (last) {
			// prefix
			if (didshow)
				*this << ", ";
			didshow = true;
			// show
			*this << StreamName(last, INDEFINITE);
		}
		last = obj;
	}
	// show the last one
	if (last) {
		// prefix
		if (didshow)
			*this << " and ";
		// show it
		*this << StreamName(last, INDEFINITE);
	} else {
		*this << "nothing";
	}

	// coins
	*this << ".  You have " << coins << " coins.\n";
}

void
Player::grant_exp (uint type, uint amount) {
	assert ( type < NUM_EXPS );
	if (amount == 0)
		return;

	// figure out the general exp to grant - 25%
	uint general = amount / 4;
	amount -= general;

	// increase the specified and general pool accordingly
	exp[EXP_GENERAL] += general;
	exp[type] += amount;
}

void
Player::grant_class_levels (const Class *klass, uint levels)
{
	assert (klass != NULL);

	if (levels == 0)
		return;

	// calculate hp increase
	int old_health = health.max;
	for (uint i = 0; i < levels; ++ i) {
		health_base += klass->get_hp_roll ();
	}

	// up our individual class level
	classes[klass] += levels;

	// recaluate various data
	recalc ();

	if (levels == 1) {
		*this << "** You have gained a level in class '" << klass->get_name() << "' **\n";
	} else {
		*this << "** You have gained " << levels << " levels in class '" << klass->get_name() << "' **\n";
	}
	*this << "    You have gained " << health.max - old_health << " HP.\n";
}

void
Player::level_up (const Class *klass)
{
	/* experience per level is calculated as level * 10000
	 * For classes that have a secondary experience pool type,
	 * the secondary type needs 1/3 of the exp, the primary
	 * needs 2/3 of the exp.  So, we have to determine how much
	 * exp is needed, whether we hve a secondary pool to worry
	 * about, and then figure out how much of each exp is needed.
	 * This function assumes we have enough exp - if we don't, it
	 * just uses up as much as it can. */
	
	uint need = get_class_level (klass) * 10000;
	uint primary_exp, primary_need; // how much of the primary type we have/need
	uint secondary_exp, secondary_need; // how much of the secondary type we have/need
	uint general_exp; // how much general exp we have

	general_exp = get_exp (EXP_GENERAL);

	/* first, do we have a secondary exp type?
	 * get the experience counts/needs */
	primary_exp = get_exp (klass->get_pri_exp_type ());
	if (klass->get_sec_exp_type ()) {
		secondary_exp = get_exp (klass->get_sec_exp_type ());
		primary_need = need * 2 / 3; // 2/3 needed total
		secondary_need = need / 3; // 1/3 needed total
	} else {
		secondary_exp = 0;
		primary_need = need; // 100%
		secondary_need = 0; // 0%
	}

	// if we don't have enough primary, use some general
	if (primary_exp < primary_need) {
		// use up all remaining general, or just what we need?
		if (primary_need - primary_exp > general_exp)
			general_exp = 0;
		else
			general_exp -= primary_need - primary_exp;
		// zero out exp
		exp[klass->get_pri_exp_type()] = 0;
	} else {
		// use exp
		exp[klass->get_pri_exp_type()] -= primary_need;
	}
	
	// if we have secondary, and don't have enough, use general
	if (secondary_need != 0) {
		if (secondary_exp < secondary_need) {
			// use up all remaining general, or just what we need?
			if (secondary_need - secondary_exp > general_exp)
				general_exp = 0;
			else
				general_exp -= secondary_need - secondary_exp;
			// zero out exp
			exp[klass->get_sec_exp_type()] = 0;
		} else {
			// use exp
			exp[klass->get_sec_exp_type()] -= secondary_need;
		}
	}

	// update our general exp pool
	exp[EXP_GENERAL] = general_exp;

	// grant he level
	grant_class_levels (klass, 1);
}

uint
Player::get_class_level (const Class *klass) const
{
	ClassList::const_iterator i = classes.find(klass);
	if (i != classes.end())
		return i->second;
	else
		return 0;
}

uint
Player::get_level_percent (const Class* klass) const
{
	uint cur_level = get_class_level (klass);
	if (cur_level == 0)
		return 0; /* no exp for this class */

	const uint need = cur_level * 10000; // needed exp
	if (klass->get_sec_exp_type ()) { // do we deal with a secondary type?
		// primary
		uint priexp = get_exp(klass->get_pri_exp_type());
		if (priexp > need * 2 / 3)
			priexp = need * 2 / 3;
		// secondary
		uint secexp = get_exp(klass->get_sec_exp_type());
		if (secexp > need / 3)
			secexp = need / 3;
		// total
		uint curexp = priexp + secexp + get_exp(EXP_GENERAL);
		if (curexp > need)
			curexp = need;
		// calculate
		return curexp * 100 / need;
	} else {
		// just one exp type
		uint curexp = get_exp (klass->get_pri_exp_type ()) + get_exp(EXP_GENERAL);
		if (curexp > need)
			return 100;
		else
			return curexp * 100 / need;
	}
}

uint
Player::get_level (void) const
{
	uint levels = 0;
	for (ClassList::const_iterator i = classes.begin(); i != classes.end(); ++i)
		levels += i->second;
	return levels;
}

void
Player::recalc_health (void)
{
	// HP = ((Fortitude - 50) / 5) * Level + BaseHP
	int hlmod = (((get_stat (CharStatID::FORTITUDE) - 50) / 5) * get_level ());
	// never have less than 30, becasue that would suck
	if (health_base + hlmod < 30)
		health.max = 30;
	else
		health.max = health_base + hlmod;
	// cap HP
	if (health.cur > health.max)
		health.cur = health.max;
}

void
Player::recalc_stats (void)
{
	Character::recalc_stats();

	// apply racial stat modifications
	if (race) {
		for (int i = 0; i < CharStatID::COUNT; ++i)
			stats[i].mod += race->get_stat(i);
	}
}

void
Player::recalc (void)
{
	Character::recalc();

	recalc_health ();
}

void
Player::update (void) {
	if (!is_active ())
		return;

	// do character update
	Character::update ();

	// update handler
	Scripts.hook("player_update", this);

	// timeout?  then die
	if (ninfo.timeout_ticks == 1) {
		Log::Info << "Player '" << get_name() << "' has timed out.";
		quit();
	} else if (ninfo.timeout_ticks > 0) {
		--ninfo.timeout_ticks;
	}

	// x_awemud status box - health, round
	if (get_telnet() && get_telnet()->has_x_awemud()) {
		// health
		if (get_hp() != ninfo.last_hp || get_max_hp() != ninfo.last_max_hp) {
			// store
			ninfo.last_hp = get_hp();
			ninfo.last_max_hp = get_max_hp();

			// send it
			ZMPPack hp("x-awemud.status.set");
			hp.add("hp");
			hp.add(ninfo.last_hp);
			hp.add(ninfo.last_max_hp);
			hp.send(get_telnet());
		}

		// round time
		uint rts = get_rts();
		if (rts != ninfo.last_rt) {
			// store rt
			ninfo.last_rt = rts;

			// max rt is the highest rt found, reset at 0
			if (ninfo.last_rt > ninfo.last_max_rt)
				ninfo.last_max_rt = ninfo.last_rt;
			else if (ninfo.last_rt == 0)
				ninfo.last_max_rt = 0;

			// send zmp
			ZMPPack rt("x-awemud.status.set");
			rt.add("rt");
			rt.add(ninfo.last_rt);
			rt.add(ninfo.last_max_rt);
			rt.send(get_telnet());
		}
	}

}

void
Player::activate (void)
{
	Character::activate();

	if (account != NULL)
		account->inc_active();
}

void
Player::deactivate (void)
{
	if (account != NULL)
			account->dec_active();

	Character::deactivate();
}

// handle input
void
Player::process_command (char* data)
{
	// always handle quit
	if (str_eq(data, "quit")) {
		quit();
	// have we an input processor?
	} else if (!procs.empty()) {
		IProcessor* proc = procs.front();
		// use the processor
		if (proc->process (data)) {
			// processor finished
			proc->finish ();
			procs.erase(procs.begin());

			// init next, or quit if we aren't valid
			if (!procs.empty())
				procs.front()->init();
			else if (!is_valid())
				quit();
		}
	// normal character command processing
	} else {
		process_cmd (data);
	}
}

void
Player::show_prompt (void)
{
	// bail if we have no telnet
	if (!get_telnet())
		return;

	// processor prompt
	if (!procs.empty()) {
		*this << procs.front()->prompt ();
	// custom propmpt
	} else if (!prompt.empty()) {
		*this << parse::prompt (prompt, this);
	// x-awemud around?  just show >
	} else if (get_telnet()->has_x_awemud()) {
		*this << ">";
	// do the full/stock prompt
	} else {
		char prompt[128];
		snprintf (prompt, 128, "-- HP:%d/%d RT:%u >", get_hp (), get_max_hp (), get_rts ());
		*this << prompt;
	}
}

void
Player::parse_comm (const char* comm, const StreamControl& stream) const
{
	// RACE
	if (str_eq(comm, "race")) {
		if (get_race())
			stream << get_race()->get_name();
	}
	// RACE ADJECTIVE
	else if (str_eq(comm, "race-adj")) {
		if (get_race())
			stream << get_race()->get_adj();
	}
	// EYE COLOR
	else if (str_eq(comm, "eyecolor")) {
		stream << get_eye_color().get_name();
	}
	// SKIN COLOR
	else if (str_eq(comm, "skincolor")) {
		stream << get_skin_color().get_name();
	}
	// SKIN TYPE (skin, fur, carapace, etc)
	else if (str_eq(comm, "skintype")) {
		stream << get_race()->get_skin_type();
	}
	// HAIR COLOR
	else if (str_eq(comm, "haircolor")) {
		stream << get_hair_color().get_name();
	}
	// HAIR STYLE
	else if (str_eq(comm, "hairstyle")) {
		stream << get_hair_style().get_name();
	}
	// HAIR
	else if (str_eq(comm, "hair")) {
		stream << StreamParse(get_hair_style().get_desc(), "player", this);
	}
	// BUILD
	else if (str_eq(comm, "build")) {
		stream << get_build().get_name();
	}
	// default...
	else Character::parse_comm(comm, stream);
}

// add a new command processor
int
Player::add_processor (IProcessor *p)
{
	assert (p != NULL);

	// add processor
	procs.push_back(p);

	// initialize it
	if (procs.front() == p)
		p->init();

	return 0;
}

// connect to a telnet handler
void
Player::connect (TelnetHandler* handler)
{
	// set connection
	conn = handler;
	
	// reset all network info
	memset(&ninfo, 0, sizeof(ninfo));
}

// disconnect from a telnet handler
void
Player::disconnect (void)
{
	// already disconnected?
	if (!get_telnet())
		return;
	
	// clear connection
	conn = NULL;

	// begin timeout
	ninfo.timeout_ticks = TICKS_PER_SEC * 60; // 60 second timeout
}

// output text
void
Player::stream_put (const char* data, size_t len)
{
	if (get_telnet())
		get_telnet()->stream_put(data, len);
}

// toggle echo
void
Player::toggle_echo (bool value)
{
	if (get_telnet())
		get_telnet()->toggle_echo(value);
}

// set indent
void
Player::set_indent (uint level)
{
	if (get_telnet())
		get_telnet()->set_indent(level);
}

// get width of view
uint
Player::get_width (void)
{
	if (get_telnet())
		return get_telnet()->get_width();
	else
		return DEFAULT_WIDTH;
}

// clear screen
void
Player::clear_scr (void)
{
	if (get_telnet())
		get_telnet()->clear_scr();
}

// event logging
int
Player::handle_event (Event* event)
{
	// only if we have builder vision
	if (has_bvision())
		*this << CADMIN "Event[" << event->get_name() <<
			"] R:" << (event->get_room() ? event->get_room()->get_id() : "n/a") <<
			" A:" << (event->get_actor() ? event->get_actor()->get_name() : "n/a") <<
			" T:" << (event->get_target() ? event->get_target()->get_name() : "n/a") <<
			" D:" << (event->get_data() ? Scriptix::IDToName(Scriptix::Value::TypeOf(Scripts.get_engine(), event->get_data())->GetName()) : "n/a") << CNORMAL "\n";

	// propogate
	return Character::handle_event(event);
}

// show player description
void
Player::display_desc (const StreamControl& stream) const
{
	// description
	if (get_race() && get_race()->get_desc())
		stream << StreamParse(get_race()->get_desc(), "player", this, "actor", this);
	else
		stream << StreamName(this, DEFINITE, true) << " is a " << get_build().get_name() << " " << get_gender().get_name() << " with " << get_skin_color().get_name() << " skin, " << get_eye_color().get_name() << " eyes and " << StreamParse(get_hair_style().get_desc(), "player", this) << ".";
}

int
ScriptProcessorWrapper::init (void)
{
	if (core) {
		Scriptix::Function* method = core->GetType()->GetMethod(Scriptix::NameToID("init"));
		if (method)
			return Scripts.run(method, core, player);
	}
	return 0;
}

void
ScriptProcessorWrapper::finish (void)
{
	if (core) {
		Scriptix::Function* method = core->GetType()->GetMethod(Scriptix::NameToID("close"));
		if (method)
			Scripts.run(method, core, player);
	}
}

int
ScriptProcessorWrapper::process (char* line)
{
	if (core) {
		Scriptix::Function* method = core->GetType()->GetMethod(Scriptix::NameToID("update"));
		if (method == NULL)
			return -1;
		Scriptix::Value* retval;
		if (Scripts.run(method, core, player, line, &retval))
			return -1; // error; end
		if (Scriptix::Value::IsA<Scriptix::Number>(Scripts.get_engine(), retval))
			return Scriptix::Number::ToInt(retval);
		else
			return -1; // wrong type; end
	} else {
		return 1; // end
	}
}

const char*
ScriptProcessorWrapper::prompt (void)
{
	if (core && core->s_prompt)
		return core->s_prompt->GetCStr();
	else
		return NULL;
}

