/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

/* --- PARSE FORMAT ---
 * A string is given in.  Commands are found inside of curly brackets, i.e. {
 * and }.
 *
 * Commands may either be global, or an argument member.  Global commands are
 * just given as their name, while argument members must be prefixed by the
 * entity name (letters or numbers), which is followed by a period, and then
 * the member name (i.e. user.name).  If the leading number is ommitted, and
 * only a period and member name given, the default entity argument is used.
 * (The default is the first entity in the argument list.)
 *
 * The argument command may be a single letter, which is a name output.  If the
 * letter is uppercase, the name will be capitalized, else it will have default
 * capitalization.  The letter 'd' means definite form, 'i' means indefinite
 * form, and 'n' means no-article form.  Default form may be specified with
 * 'c'.
 *
 * If more than one letter is given, it is given to the entity for processing.
 * Each entity type supports different commands.  All entities support at
 * least 'name', which is the same as using 'c'.  If the command is
 * captialized, the command result will be capitalized.  (So 'Name' is the
 * same as 'C', while 'name' is equivalent to 'c'.)
 *
 * Also, commands may be if/elif/else/end statements.  These are specified by
 * using the 'if', 'elif', 'else', or 'endif' commands.  These may not be
 * prefixed by numbers or colons or semi-colons.  The 'if' and 'elif'
 * commands take 3 arguments: a subcommand, an operator, and a second
 * subcommand.  Operators are = (equals) and != (not equals).  The
 * sub-commands are evaluated.  If the whole expression is false, all
 * following commands and text are suppressed.  The 'elif' command only works
 * after an 'if' command, and only if the previous 'if' and all previous
 * 'elif' were false.  'else' may only follow 'if' or 'elif', and is true
 * if nothing previous was false.  The 'if' statements may be nested.
 *
 * Finally, if a command has quotes (either single or double) it is considered
 * plain text.
 *
 * Examples:
 * This is a test     -> This is a test
 * My name is {.name}  -> My name is Sean
 * {if player.name = 'Sean'} I'm Sean {else} I'm not Sean,
 *    I'm {player.name} {endif}  -> I'm Sean
 * {if player-count = '1'} You're alone! {endif}
 *
 * {% enables 'bold' color, and %} ends 'bold' color.
 */

#include <vector>

#include <ctype.h>

#include "scripts.h"
#include "entity.h"
#include "parse.h"
#include "char.h"
#include "server.h"
#include "streams.h"
#include "player.h"

#define PARSE_OUT_SIZE 2048
#define MAX_COMM_NAME 32
#define MAX_COMM_DATA 128

#define CHUNK_LEN(start, end) ((end)-(start)+1)

// parse value
namespace {
	// if statements
	enum if_state {
		IF_DONE = 0,
		IF_TRUE,
		IF_FALSE
	};
	enum if_type {
		IF_NONE = 0,
		IF_EQUAL,
		IF_NOTEQUAL,
		IF_MATCH,
		IF_LESSTHAN,
		IF_LESSTHANOREQUAL,
		IF_GREATERTHAN,
		IF_GREATERTHANOREQUAL
	};

	// like strpbrk(), but only to a given end-point, and handles quoting
	const char* find_ch (const char* in, size_t len, const char* find)
	{
		char quote = 0;

		// loop 'till end
		for (size_t i = 0; i < len; ++i) {
			// quoted?
			if (quote) {
				if (in[i] == quote)
					quote = 0;
			// start quote?
			} else if (in[i] == '"' || in[i] == '\'') {
				quote = in[i];
			// end char?
			} else if (strchr(find, in[i])) {
				return in + i;
			}
		}

		// end
		return NULL;
	}

	// stream out a Scriptix value
	bool stream_value (const StreamControl& stream, const Scriptix::Value* value)
	{
		// number?
		if (Scriptix::Value::IsA<Scriptix::Number>(Scripts.get_engine(), value)) {
			stream << (long)Scriptix::Value::ToInt(Scripts.get_engine(), value);
			return true;
		// string?
		} else if (Scriptix::Value::IsA<Scriptix::String>(Scripts.get_engine(), value)) {
			stream << ((Scriptix::String*)value)->GetStr();
			return true;
		// who the hell knows?
		} else {
			return false;
		}
	}

	// parse an entity value
	bool parse_entity (const char* start, const char* end, const StreamControl& stream, const Entity* entity)
	{
		// only one character?
		if (CHUNK_LEN(start, end) == 1) {
			// parse type flags
			EntityArticleType atype = INDEFINITE;
			bool cap = false;

			switch (*start) {
				case 'D':
					cap = true;
					// fall thrue
				case 'd':
					atype = DEFINITE;
					break;
				case 'I':
					cap = true;
					// fall thrue
				case 'i':
					atype = INDEFINITE;
					break;
				case 'N':
					cap = true;
					// fall thrue
				case 'n':
					atype = NONE;
					break;
				case 'C':
					cap = true;
					break;
				case 'c':
					cap = false;
					break;
				default:
					// invalid
					stream << "<ERROR: unknown entity command: " << *start << '>';
					return false;
					break;
			}

			// do name output
			stream << StreamName(entity, atype, cap);
		} else { // full command
			// flags
			bool upper = isupper(*start);

			// up to end is the command 
			if (CHUNK_LEN(start,end) > MAX_COMM_NAME) {
				// fail - too long or empty
				stream << "<ERROR: empty command or command too long: " << StreamChunk(start,end) << '>';
				return false;
			}
			char commname[MAX_COMM_NAME + 1];
			snprintf(commname, sizeof(commname), "%.*s", CHUNK_LEN(start, end), start);
			
			// run command, get output, append - weee
			if (upper) {
				BufferSink<4096> result;
				entity->parse_comm (commname, result);
				if (result[0] != 0) {
					result[0] = toupper(result[0]);
					stream << result;
				}
			} else {
				entity->parse_comm (commname, stream);
			}
		}

		return true;
	}

	// parse a command
	bool parse_value (const char* in, const char* end, const StreamControl& stream, const ParseValueList& vlist)
	{
		// skip whitespace
		while (in <= end && isspace(*in))
			++in;
		if (in > end)
			return true;
		while (end != in && isspace(*end))
			--end;

		const char* start = in;

		// quoted?
		if (*in == '"' || *in == '\'') {
			char quote = *in;
			++in;

			// read data
			while (in <= end && *in != quote)
				++in;

			// true if we're past end
			if (in == end) {
				stream << StreamChunk(start + 1, end - 1);
				return true;
			} else {
				stream << "<ERROR: invalid quote: " << StreamChunk(start,end) << '>';
				return false;
			}
		}

		// command have a dot?
		const char* dot = find_ch(in, end-in, ".");
		
		// no dot
		if (dot == NULL) {
			// parse number, if it is one
			uint index = 0;
			const char* nptr = in;
			while (isdigit (*nptr)) {
				index *= 10;
				index += *nptr - '0';
				++ nptr;
			}
			if (nptr > end) {
				// it's a number alright!
				if (index < 1 || index > vlist.size()) {
					stream << "<<ERROR: index out of range: " << StreamChunk(start,end) << '>';
					return false;
				}
				// print it out
				if (!stream_value(stream, vlist[index - 1].value)) {
					// unknown type
					stream << "<<ERROR: invalid value type: " << StreamChunk(start,end) << '>';
					return false;
				} else {
					// all good!
					return true;
				}
			}

			// value name?
			for (ParseValueList::const_iterator i = vlist.begin(); i != vlist.end(); ++i)
				if (str_eq(i->name, in, end-in))
					if (!stream_value(stream, i->value)) {
						// unknown type
						stream << "<<ERROR: invalid value type: " << StreamChunk(start,end) << '>';
						return false;
					} else {
						// all good!
						return true;
					}

			// SERVER VERSION
			if (str_eq(in, "version", end-in)) {
				stream << VERSION;
				return true;
			}
			// SERVER BUILD
			if (str_eq(in, "build", end-in)) {
				stream << __DATE__ " " __TIME__;
				return true;
			}
			// SERVER UPTIME
			if (str_eq(in, "uptime", end-in)) {
				uint ut = (::time (NULL) - server.get_start_time ()); 
				uint d = ut;
				uint seconds = d % 60;
				d /= 60;
				uint minutes = d % 60;
				d /= 60;
				uint hours = d % 24;
				d /= 24;
				uint days = d;
				if (ut >= 86400)
					stream << days << " days, ";
				if (ut >= 3600)
					stream << hours << " hours, ";
				if (ut >= 60)
					stream << minutes << " minutes, ";
				stream << seconds << " seconds";
				return true;
			}
			// SERVER USER COUNT
			if (str_eq(in, "player-count", end-in)) {
				stream << (int)PlayerManager.count ();
				return true;
			}
			// DAY OR NIGHT
			if (str_eq(in, "day-or-night", end-in)) {
				stream << (TimeManager.time.is_night() ? "night" : "day");
				return true;
			}
			// HOSTNAME
			if (str_eq(in, "hostname", end-in)) {
				stream << server.get_host();
				return true;
			}
			// GAME DATE
			if (str_eq(in, "date", end-in)) {
				stream << TimeManager.time.date_str();
				return true;
			}
			// GAME TIME 
			if (str_eq(in, "time", end-in)) {
				stream << TimeManager.time.time_str();
				return true;
			}
			// DATE YEAR
			if (str_eq(in, "date-year", end-in)) {
				stream << TimeManager.time.get_year();
				return true;
			}
			// DATE MONTH
			if (str_eq(in, "date-month", end-in)) {
				stream << TimeManager.time.get_month();
				return true;
			}
			// DATE DAY
			if (str_eq(in, "date-day", end-in)) {
				stream << TimeManager.time.get_day();
				return true;
			}
			// TIME HOURS (24 HOUR FORMAT)
			if (str_eq(in, "time-hours24", end-in)) {
				stream << TimeManager.time.get_hour();
				return true;
			}
			// TIME HOURS (12 HOUR FORMAT)
			if (str_eq(in, "time-hours12", end-in)) {
				uint hours = TimeManager.time.get_hour();
				stream << (hours == 0 ? 12 : (hours <= 12 ? hours : hours - 12));
				return true;
			}
			// TIME AM/PM
			if (str_eq(in, "time-ampm", end-in)) {
				stream << (TimeManager.time.get_hour() < 12 ? "am" : "pm");
				return true;
			}
			// TIME MINUTES
			if (str_eq(in, "time-minutes", end-in)) {
				stream << (TimeManager.time.get_minutes());
				return true;
			}

			// unknown command
			stream << "<ERROR: unknown command: " << StreamChunk(start,end) << '>';
			return false;
		// no name/number before?
		} else if (in == dot) {
			// no entities given?
			if (vlist.empty()) {
				stream << "<ERROR: index out of range: " << StreamChunk(start,end) << '>';
				return false;
			}
			
			// value is empty?
			// FIXME: we need the system since we may have multiple in future
			if (!Scriptix::Value::IsA(Scripts.get_engine(), vlist[0].value, AweMUD_EntityType))
				return true;

			// do it
			return parse_entity(in + 1, end, stream, (Entity*)vlist[0].value);
		// must lookup entity
		} else {
			const Entity* entity = NULL;

			// parse number, if it is one
			uint index = 0;
			while (isdigit (*in)) {
				index *= 10;
				index += *in - '0';
				++ in;
			}

			// number right up to dot - it's a number alright!
			if (in == dot) {
				// check range
				if (index == 0 || index > vlist.size()) {
					// invalid!
					stream << "<ERROR: index out of range: " << StreamChunk(start,end) << '>';
					return false;
				}

				// make index 0 based, not 1 based
				--index;

				// is an entity?
				// FIXME: we need the system since we may have multiple in future
				if (!Scriptix::Value::IsA(Scripts.get_engine(), vlist[index].value, AweMUD_EntityType)) {
					// invalid!
					stream << "<ERROR: value not an entity: " << StreamChunk(start,end) << '>';
					return false;
				}

				// set the entity
				entity = (Entity*)vlist[index].value;
			// not up to dot, not a whole number - a name then
			} else {
				// find the name
				bool found = false;
				for (ParseValueList::const_iterator i = vlist.begin(); i != vlist.end(); ++i)
					if (strncasecmp(start, i->name.c_str(), dot-start) == 0) {
						// is an entity?
						// FIXME: we need the system since we may have multiple in future
						if (!Scriptix::Value::IsA(Scripts.get_engine(), i->value, AweMUD_EntityType)) {
							// invalid!
							stream << "<ERROR: value not an entity: " << StreamChunk(start,end) << '>';
							return false;
						}

						entity = (Entity*)i->value;
						found = true;
						break;
					}
				// not there?  drat
				if (!found) {
					// invalid!
					stream << "<ERROR: entity name not found: " << StreamChunk(start,end) << '>';
					return false;
				}
			}

			// NULL entity?  don't bother
			if (entity == NULL)
				return true;

			// do the parse, with method just after dot
			return parse_entity(++dot, end, stream, entity);
		}
	}

	// determine whether an if statement is true or false
	bool
	parse_if (const char* in, size_t len, const ParseValueList& vlist)
	{
		// command separator
		if_type type = IF_NONE;
		const char* first_end = NULL; // end of first arg
		const char* second_start = NULL; // begin of second arg

		// loop 'till end
		char quote = 0;
		for (const char* c = in; (size_t)(c - in) < len; ++c) {
			// quoted?
			if (quote) {
				if (c[0] == quote)
					quote = 0;
			// start quote?
			} else if (c[0] == '"' || c[0] == '\'') {
				quote = *c;
			// equals?
			} else if (c[0] == '=') {
				type = IF_EQUAL;
				first_end = c - 1;
				second_start = c + 1;
				break;
			// less than or equal to?
			} else if (c[0] == '<' && c[1] == '=') {
				type = IF_LESSTHANOREQUAL;
				first_end = c - 1;
				second_start = c + 2;
				break;
			// less than?
			} else if (c[0] == '<') {
				type = IF_LESSTHAN;
				first_end = c - 1;
				second_start = c + 1;
				break;
			// greater than or equal to?
			} else if (c[0] == '>' && c[1] == '=') {
				type = IF_GREATERTHANOREQUAL;
				first_end = c - 1;
				second_start = c + 2;
				break;
			// greater than?
			} else if (c[0] == '>') {
				type = IF_GREATERTHAN;
				first_end = c - 1;
				second_start = c + 1;
				break;
			// not equals?
			} else if (c[0] == '!' && c[1] == '=') {
				type = IF_NOTEQUAL;
				first_end = c - 1;
				second_start = c + 2;
				break;
			// match?
			} else if (c[0] == '~' && c[1] == '=') {
				type = IF_MATCH;
				first_end = c - 1;
				second_start = c + 2;
				break;
			}
		}

		// get command one
		BufferSink<1024> comm1;
		if (!parse_value(in, first_end, comm1, vlist)) {
			return false;
		}

		// no operator - just testing first
		if (type == IF_NONE) {
			return !comm1.empty();
		}

		// get command two
		BufferSink<1024> comm2;
		if (!parse_value(second_start, in + len - 1, comm2, vlist)) {
			return false;
		}

		// compare
		switch (type) {
			case IF_EQUAL:
				return str_eq(comm1, comm2);
			case IF_NOTEQUAL:
				return !str_eq(comm1, comm2);
			case IF_MATCH:
				return phrase_match(comm2, comm1);
			case IF_LESSTHAN:
				return tolong(comm1.str()) < tolong(comm2.str());
			case IF_LESSTHANOREQUAL:
				return tolong(comm1.str()) <= tolong(comm2.str());
			case IF_GREATERTHAN:
				return tolong(comm1.str()) > tolong(comm2.str());
			case IF_GREATERTHANOREQUAL:
				return tolong(comm1.str()) >= tolong(comm2.str());
			default:
				return false;
		}
	}
}

// parsing
namespace parse {
	// parse text for multiple entities
	const StreamControl&
	text(const StreamControl& stream, const char *in, const ParseValueList& vlist) {
		// null input?  damn you!
		if (in == NULL) {
			return stream;
		}

		// if states
		std::vector<int> ifs;
		bool valid = true;
		bool bold = false;

		// search string
		for (; *in != '\0'; ++ in) {
			// backslash escaping
			if (*in == '\\') {
				++in;
				if (*in == '\0')
					break;
				switch(*in) {
					case '\\':
						stream << '\\';
						break;
					case 'n':
						stream << '\n';
						break;
					case 't':
						stream << '\t';
						break;
					default:
						stream << *in;
						break;
				}
			}
			// new-style escaping
			else if (*in == '{') {
				// start bold?
				if (*(in + 1) == '%') {
					// only if not already boldened
					if (!bold) {
						stream << CBOLD;
						bold = true;
					}
					++in;
					continue;
				}

				// starting position
				const char* start = in;

				// loop 'till end
				const char* end = NULL;
				char quote = 0;
				for (const char* c = in; *c != 0; ++c) {
					// quoted?
					if (quote) {
						if (c[0] == quote)
							quote = 0;
					// start quote?
					} else if (c[0] == '"' || c[0] == '\'') {
						quote = *c;
					// end char?
					} else if (c[0] == '}') {
						end = c;
						break;
					}
				}
				if (end == NULL) {
					// no end - this wasn't intended as an escape
					stream << '{';
					continue;
				}

				// eat leading whitespace
				++in;
				while (in != end && isspace(*in))
					++in;
				if (in == end)
					continue; // empty

				// find length
				size_t len = end - in;

				// eat trailing whitespace
				while (isspace(in[len - 1]))
					--len;

				// if statement?
				if (str_eq(in, "if ", 3)) {
					// if not valid, don't bother
					if (!valid) {
						ifs.push_back(IF_DONE);
						in = end;
						continue;
					}

					// do if
					in += 3;
					len -= 3;
					valid = parse_if(in, len, vlist);
					ifs.push_back(valid ? IF_TRUE : IF_FALSE);

				// elif statement?
				} else if (str_eq(in, "elif ", 5)) {
					// have an if?
					if (ifs.empty()) {
						stream << "<ERROR: 'elif' with no 'if': " << StreamChunk(start, end) << '>';
						in = end;
						continue;
					}

					// if done, don't bother
					if (ifs.back() == IF_DONE) {
						in = end;
						continue;
					}

					// previously true, then we're not not valid
					if (ifs.back() == IF_TRUE) {
						ifs.back() = IF_DONE;
						valid = false;
						in = end;
						continue;
					}

					// do elif
					in += 5;
					len -= 5;
					valid = parse_if(in, len, vlist);
					ifs.back() = valid ? IF_TRUE : IF_FALSE;

				// else statement?
				} else if (str_eq(in, "else", 4)) {
					// have an if?
					if (ifs.empty()) {
						stream << "<ERROR: 'else' with no 'if': " << StreamChunk(start, end) << '>';
						in = end;
						continue;
					}

					// if done, don't bother
					if (ifs.back() == IF_DONE) {
						in = end;
						continue;
					}

					// extra data?
					if (len != 4) {
						stream << "<ERROR: 'else' with extraneous data: " << StreamChunk(start,end) << '>';
						in = end;
						continue;
					}

					// flip state
					if (ifs.back() == IF_FALSE) {
						ifs.back() = IF_TRUE;
						valid = true;
					} else {
						ifs.back() = IF_DONE;
						valid = false;
					}

				// endif statement?
				} else if (str_eq(in, "endif", 5)) {
					// have an if?
					if (ifs.empty()) {
						stream << "<ERROR: 'endif' with no 'if'" << StreamChunk(start,end) << '>';
						in = end;
						continue;
					}

					// extra data?
					if (len != 5) {
						stream << "<ERROR: 'endif' with extraneous data: " << StreamChunk(start,end) << '>';
						in = end;
						continue;
					}

					// pop if
					ifs.pop_back();
					valid = ifs.empty() || ifs.back() == IF_TRUE;

				// command
				} else if (valid) {
					parse_value (in, in + len - 1, stream, vlist);
				}

				in = end;
			// end bold?
			} else if (bold && *in == '%' && *(in + 1) == '}') {
				++in;
				stream << CNORMAL;
				bold = false;
			// normal character
			} else if (valid) {
				stream << *in;
			}
		}

		// finish unfinished bolding
		if (bold) {
			stream << CNORMAL;
		}

		return stream;
	}

	// parse a prompt
	const char *
	prompt (const char *in, const ::Character *ch) {
		static char out[PARSE_OUT_SIZE];
		char result[40];
		char *oc = out;

		// until end
		for (; *in != '\0'; ++ in) {
			// found a %?
			if (*in == '%') {
				++ in;
				// %% = %
				if (*in == '%') {
					if (oc - out < PARSE_OUT_SIZE - 1) {
						*(oc ++) = '%';
					}
				} else {
					// append expanasion
					result[0] = '\0';
					switch (*in) {
						// our name
						case 'n':
							snprintf (result, sizeof(result), "%s", ch->get_name ().c_str());
							break;
						// current health
						case 'h':
							snprintf (result, sizeof(result), "%d", ch->get_hp ());
							break;
						// max health
						case 'H':
							snprintf (result, sizeof(result), "%d", ch->get_max_hp ());
							break;
						// round time
						case 'R':
							snprintf (result, sizeof(result), "%u", ch->get_rts ());
							break;
						// game time
						case 't':
							snprintf (result, sizeof(result), "%s", TimeManager.time.time_str().c_str());
							break;
						// game date
						case 'D':
							snprintf (result, sizeof(result), "%s", TimeManager.time.date_str().c_str());
							break;
					}
					// do append
					if (result[0] != '\0') {
						int len = strlen (result);
						if (len > PARSE_OUT_SIZE - (oc - out) - 1)
							len = PARSE_OUT_SIZE - (oc - out) - 1;
						strncpy (oc, result, len);
						oc += len;
					}
				}
			} else {
				// append character
				if (oc - out < PARSE_OUT_SIZE - 1) {
					*(oc ++) = *in;
				}
			}
		}

		// finish up
		*oc = '\0';
		return out;
	}
}
