/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */


#include <stdlib.h>
#include <stdio.h>

#include "entity.h"
#include "awestr.h"
#include "server.h"
#include "scripts.h"
#include "room.h"
#include "eventids.h"

SEventManager EventManager;
StringList EventID::registry;

EventID
EventID::lookup (StringArg name)
{
	// empty?  invalid
	if (name.empty())
		return EventID();
	// find it
	for (size_t i = 0; i < registry.size(); ++i)
		if (registry[i] == name)
			return EventID(i + 1);
	// not found, add it
	registry.push_back(String(name).lower());
	return EventID(registry.size());
}

String
EventID::get_name (void) const
{
	if (valid())
		return registry[id - 1];
	return String();
}

int
EventHandler::load (File::Reader& reader) {
	File::Node node;
	FO_READ_BEGIN
		FO_ATTR_NAME("script")
			sxfunc = NULL;
			if (node.get_data())
				sxfunc = Scripts.compile(node.get_data(), "self,event", reader.get_filename(), node.get_line());
	FO_READ_ERROR
		return -1;
	FO_READ_END

	return 0;
}

void
EventHandler::save (File::Writer& writer) const {
	if (script)
		writer.block("script", script);
}

int
Entity::handle_event (Event* event)
{
	for (EventList::const_iterator i = events.begin (); i != events.end (); ++ i)
		if (event->get_id() == (*i)->get_event ()) {
			if ((*i)->get_func())
				Scripts.run((*i)->get_func(), this, event);
		}
	return 0;
}

Event::Event (EventID s_id, Room* s_room, Entity* s_actor, Entity* s_target, Scriptix::Value* s_data) : Scriptable(AweMUD_EventType), AttributeHolder(),
	id(s_id), room(s_room), actor(s_actor), target(s_target), data(s_data)
{}

int
SEventManager::initialize (void)
{
	Log::Info << "Initializing event manager";

	count = 0;
	Events::initialize();

	return 0;
}

void
SEventManager::shutdown (void)
{
	while (!events.empty())
		events.pop();
}

int
SEventManager::send (EventID id, Room* room, Entity *actor, Entity *target, Scriptix::Value* data)
{
	assert (id.valid());

	// fetch room if we haev none given
	if (room == NULL) {
		if (CHARACTER(target))
			room = ((Character*)target)->get_room();
		else if (OBJECT(target))
			room = ((Object*)target)->get_room();
		if (room == NULL)
			return -1;
	}

	// create event
	Event* event = new Event(id, room, actor, target, data);
	if (event == NULL) {
		fatal("new Event() failed");
		return -1;
	}

	// put event onto event list
	send(event);

	return 0;
}

int
SEventManager::send (Event* event)
{
	assert(event != NULL);
	assert(event->get_id().valid());

	++count;
	events.push(event);

	return 0;
}

void
SEventManager::process (void)
{
	// remember how many events we started with
	// this way, event which trigger more events
	// won't cause an infinite loop, because we'll
	// only process the initial batch
	size_t processing = count;

	// as long as we have events...
	while (processing >= 0 && !events.empty()) {
		// get event
		Event* event = events.pop();
		--count;
		--processing;

		// DEBUG:
		/*
		Log::Info << "Event[" << event->get_name() <<
			"] P:" << (intptr_t)event <<
			" R:" << (event->get_room() ? event->get_room()->get_id() : "n/a") <<
			" A:" << (event->get_actor() ? event->get_actor()->get_name() : "n/a") <<
			" T:" << (event->get_target() ? event->get_target()->get_name() : "n/a") <<
			" D:" << (event->get_data() ? Scriptix::IDToName(event->get_data()->GetType()->GetName()) : "n/a");
		*/

		// send event
		event->get_room()->handle_event(event);
	}
}
