/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */


#include <stdlib.h>
#include <stdio.h>
#include <typeinfo>
#include <algorithm>

#include "entity.h"
#include "error.h"
#include "awestr.h"
#include "server.h"
#include "parse.h"
#include "streams.h"
#include "blueprint.h"
#include "color.h"
#include "player.h"

// ----- EntityData -----

void
EntityData::reset_name (void)
{
	// clear flag
	set_flags.name = false;

	// get parent's name or set to a default value
	const EntityData* data = get_entity_data_parent();
	if (data != NULL)
		name = data->get_name();
	else
		name = "unnamed";
}

void
EntityData::reset_ntype (void)
{
	// clear flag
	set_flags.ntype = false;

	// get parent's ntype or set to a default value
	const EntityData* data = get_entity_data_parent();
	if (data != NULL)
		ntype = data->get_ntype();
	else
		ntype = DEFAULT;
}

void
EntityData::reset_desc (void)
{
	// clear flag
	set_flags.desc = false;

	// get parent's desc or set to a default value
	const EntityData* data = get_entity_data_parent();
	if (data != NULL)
		desc = data->get_desc();
	else
		desc = "You see nothing interesting.";
}

bool
EntityData::get_int (StringArg name, int& value) const
{
	// try ours
	if (attrs.get_int(name, value))
		return true;

	// try datas
	const EntityData* data = get_entity_data_parent();
	while (data != NULL) {
		if (data->attrs.get_int(name, value))
			return true;
		data = data->get_entity_data_parent();
	}

	// fail!
	return false;
}

bool
EntityData::get_string (StringArg name, String& value) const
{
	// try ours
	if (attrs.get_string(name, value))
		return true;

	// try datas
	const EntityData* data = get_entity_data_parent();
	while (data != NULL) {
		if (data->attrs.get_string(name, value))
			return true;
		data = data->get_entity_data_parent();
	}

	// fail!
	return false;
}

bool
EntityData::get_attr (StringArg name, Attribute& value) const
{
	// try ours
	if (attrs.get_attr(name, value))
		return true;

	// try datas
	const EntityData* data = get_entity_data_parent();
	while (data != NULL) {
		if (data->attrs.get_attr(name, value))
			return true;
		data = data->get_entity_data_parent();
	}

	// fail!
	return false;
}

EventHandler*
EntityData::get_event (EventID name)
{
	// try ours
	for (EventList::iterator i = events.begin(); i != events.end(); ++i) {
		if ((*i)->get_event() == name)
			return *i;
	}

	// try blueprints
	const EntityData* data = get_entity_data_parent();
	while (data != NULL) {
		for (EventList::const_iterator i = data->get_events().begin(); i != data->get_events().end(); ++i)
			if ((*i)->get_event() == name)
				return *i;
		data = data->get_entity_data_parent();
	}

	// nope
	return NULL;
}

void
EntityData::update_entity_data (void)
{
	if (!set_flags.name)
		reset_name();
	if (!set_flags.ntype)
		reset_ntype();
	if (!set_flags.desc)
		reset_desc();
}

int
EntityData::load_node (File::Reader& reader, File::Node& node)
{
	FO_NODE_BEGIN
		// our primary name
		FO_ATTR_NAME("name")
			set_name(node.get_data());
		// alternate names/keywords
		FO_ATTR_NAME("altname")
			if (find(alt_names.begin(), alt_names.end(), node.get_data()) == alt_names.end())
				alt_names.push_back(node.get_data());
		// name type (article)
		FO_ATTR_NAME("ntype")
			if (node.get_data() == "proper")
				set_ntype(PROPER);
			else if (node.get_data() == "unique")
				set_ntype(UNIQUE);
			else if (node.get_data() == "plural")
				set_ntype(PLURAL);
			else if (node.get_data() == "vowel")
				set_ntype(VOWEL);
			else if (node.get_data() == "normal")
				set_ntype(NORMAL);
			else
				set_ntype(DEFAULT);
		// description
		FO_ATTR_NAME("desc")
			set_desc(node.get_data());
		// try attributes
		FO_ATTR_TYPE("int")
			int value;
			FO_GET_INT(value);
			attrs.set_int(node.get_name(), value);
		FO_ATTR_TYPE("string")
			attrs.set_string(node.get_name(), node.get_data());
		FO_OBJECT("event")
			if (node.get_name().empty())
				Log::Warning << "Blueprint event with no event name at " << reader.get_filename() << ':' << node.get_line();
			EventHandler* event = new EventHandler(EventID(node.get_name()));
			if (!event->load(reader))
					events.push_back(event);
	FO_NODE_END
}

void
EntityData::save (File::Writer& writer) const
{
	// our lead name
	if (set_flags.name)
		writer.attr("name", name);

	// type of name
	if (set_flags.ntype) {
		switch (ntype) {
			case NORMAL:
				writer.attr("ntype", "normal");
				break;
			case PROPER:
				writer.attr("ntype", "proper");
				break;
			case UNIQUE:
				writer.attr("ntype", "unique");
				break;
			case PLURAL:
				writer.attr("ntype", "plural");
				break;
			case VOWEL:
				writer.attr("ntype", "vowel");
				break;
			default:
				// do nothing
				break;
		}
	}

	// write out name list
	for (StringList::const_iterator i = alt_names.begin(); i != alt_names.end(); i ++)
		writer.attr("altname", i->get());

	// save our description
	if (set_flags.desc)
		writer.attr("desc", desc);

	// event handler list
	for (EventList::const_iterator i = events.begin (); i != events.end (); i ++) {
		writer.begin("event", (*i)->get_event().get_name());
		(*i)->save(writer);
		writer.end();
	}

	// arbitrary attributes
	attrs.save(writer);
}

// ----- EntityBlueprint -----

int
EntityBlueprint::load_node (File::Reader& reader, File::Node& node)
{
	FO_NODE_BEGIN
		FO_PARENT(EntityData)
	FO_NODE_END
}

// ----- Entity -----

Entity::Entity (const Scriptix::Type* type) : Scriptable (type)
{
	flags.active = false;
}

Entity::~Entity (void)
{
#ifndef NDEBUG
	if (is_active()) {
		Log::Error << "Entity " << (void*)this << " (" << name << ") is active during call to destructor";
		deactivate();
	}
#endif
}

void
Entity::activate (void)
{
	// must not already be active
	assert(!is_active());

	// add to manager's big list
	EntityManager.elist.push_back(this);
	eself = --EntityManager.elist.end();

	// register tags
	for (StringList::iterator i = tags.begin(); i != tags.end(); ++i) 
		EntityManager.tag_map.insert(std::pair<String, Entity*> (*i, this));

	flags.active = true;
}

void
Entity::deactivate (void)
{
	// must be active
	assert(is_active());

	// BIG LIST
	if (eself == EntityManager.ecur)
		EntityManager.ecur = EntityManager.elist.erase(eself);
	else
		EntityManager.elist.erase(eself);

	// TAG MAP
	for (StringList::iterator i = tags.begin(); i != tags.end(); ++i) {
		std::pair<TagMap::iterator, TagMap::iterator> mi = EntityManager.tag_map.equal_range(*i);
		while (mi.first != mi.second) {
			if (mi.first->second == this)
				EntityManager.tag_map.erase(mi.first++);
			else
				++mi.first;
		}
	}

	flags.active = false;
}

void
Entity::remove (void)
{
	release();
	if (is_active())
		deactivate();
}

bool
Entity::name_match (StringArg name) const
{
	if (phrase_match (get_name(), name))
		return true;

	// try alt names
	for (StringList::const_iterator i = alt_names.begin(); i != alt_names.end(); i ++)
		if (phrase_match (i->get(), name))
			return true;

	// try blueprint alt names
	EntityBlueprint* blueprint = get_blueprint();
	while (blueprint != NULL) {
		for (StringList::const_iterator i = blueprint->get_alt_names().begin(); i != blueprint->get_alt_names().end(); i ++)
			if (phrase_match (i->get(), name))
				return true;
		blueprint = blueprint->get_parent();
	}

	// no match
	return false;
}


// display
void
Entity::display_name (const StreamControl& stream, EntityArticleType atype, bool capitalize) const
{
	StringArg name = get_name();
	EntityNameType ntype = get_ntype();

	// proper names, no articles
	if (ntype == PROPER || atype == NONE) {
		// specialize output
		stream << ncolor();
		if (capitalize && name) {
			stream << (char)toupper(name[0]) << name.c_str() + 1;
		} else {
			stream << name;
		}
		stream << CNORMAL;
		return;
	// definite articles - uniques
	} else if (atype == DEFINITE || ntype == UNIQUE) {
		if (capitalize)
			stream << "The ";
		else
			stream << "the ";
	// pluralized name
	} else if (ntype == PLURAL) {
		if (capitalize)
			stream << "Some ";
		else
			stream << "some ";
	// starts with a vowel-sound
	} else if (ntype == VOWEL) {
		if (capitalize)
			stream << "An ";
		else
			stream << "an ";
	// normal-type name, nifty.
	} else {
		if (capitalize)
			stream << "A ";
		else
			stream << "a ";
	}
	stream << ncolor() << name << CNORMAL;
}

void
Entity::display_desc (const StreamControl& stream) const
{
	stream << StreamParse(desc, "self", this);
}

void
Entity::save (File::Writer& writer) const
{
	EntityData::save(writer);

	// save tags
	for (StringList::const_iterator i = tags.begin(); i != tags.end(); ++i)
		writer.attr("tag", *i);
}

int
Entity::load_node (File::Reader& reader, File::Node& node)
{
	FO_NODE_BEGIN
		FO_PARENT(EntityData)
		// unique tag
		FO_ATTR_NAME("tag")
			add_tag(node.get_data());
		// event handler
		FO_OBJECT("event")
			if (node.get_name().empty())
				Log::Warning << "Entity event with no event name at " << reader.get_filename() << ':' << node.get_line();
			EventHandler* event = new EventHandler(EventID(node.get_name()));
			if (!event->load(reader))
					events.push_back(event);
	FO_NODE_END
}

void
Entity::parse_comm (const char* comm, const StreamControl& stream) const
{
	// ENTITY's NAME
	if (str_eq(comm, "name")) {
		stream << ncolor() << get_name() << CNORMAL;
		return;
	// ENTITY'S DESC
	} else if (str_eq(comm, "desc")) {
		display_desc(stream);
	}
}

bool
Entity::is_blueprint (StringArg name) const
{
	// search blueprints
	EntityBlueprint* blueprint = get_blueprint();
	while (blueprint != NULL) {
		// match?  yay
		if (blueprint->get_id() == name)
			return true;
		blueprint = blueprint->get_parent();
	}

	// nope, no match
	return false;
}

bool
Entity::has_tag (StringArg tag) const
{
	return std::find(tags.begin(), tags.end(), tag) != tags.end();
}

int
Entity::add_tag (StringArg tag)
{
	// no duplicates
	if (has_tag(tag))
		return 1;

	// add tag
	tags.push_back(tag);

	// register with entity manager
	// FIXME: check for error, maybe?
	if (is_active())
		EntityManager.tag_map.insert(std::pair<String, Entity*> (tag, this));

	return 0;
}

int
Entity::remove_tag (StringArg tag)
{
	// find
	StringList::iterator ti = std::find(tags.begin(), tags.end(), tag);
	if (ti == tags.end())
		return 1;

	// remove
	tags.erase(ti);

	// unregister with entity manager
	if (is_active()) {
		std::pair<TagMap::iterator, TagMap::iterator> mi = EntityManager.tag_map.equal_range(tag);
		while (mi.first != mi.second) {
			if (mi.first->second == this) {
				EntityManager.tag_map.erase(mi.first);
				return 0;
			}
			++mi.first;
		}
		return 2; // failed to find in manager
	} else {
		// no active - no need to unregister
		return 0;
	}
}

bool
Entity::operator< (const Entity& ent) const
{
	return strcasecmp(ent.name.c_str(), ent.name.c_str()) < -1;
}

// ----- SEntityManager -----

SEntityManager EntityManager;

SEntityManager::SEntityManager (void) : elist(), ecur()
{
}

SEntityManager::~SEntityManager (void)
{
}

int
SEntityManager::initialize (void)
{
	Log::Info << "Initializing entity manager";

	return 0; // no error
}

void
SEntityManager::shutdown (void)
{
	elist.clear();
	tag_map.clear();
}

void
SEntityManager::update (void)
{
	// begin
	ecur = elist.begin();

	// loop
	while (ecur != elist.end()) {
		// get current
		Entity* cur = *ecur;

		// move ptr
		++ecur;

		// update entity
		cur->update();
	}
}

size_t
SEntityManager::tag_count (StringArg tag) const
{
	return tag_map.count(tag);
}
