/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "rand.h"
#include "class.h"
#include "player.h"
#include "settings.h"

SClassManager ClassManager;

int
Class::load (File::Reader& reader)
{
	// setup
	pri_exp_type = EXP_GENERAL;
	sec_exp_type = EXP_GENERAL;
	diff_mod = 100;
	about = "AweMUD player class.";
	min_hp = 5;
	max_hp = 20;
	combat_mod = 100;
	for (int i = 0; i < CharStatID::COUNT; ++i)
		stats[i] = 0;

	// read in 
	File::Node node;
	FO_READ_BEGIN
		// set attributes
		FO_ATTR_NAME("about")
			about = node.get_data();
		FO_ATTR_NAME("primary")
			int type = get_index_of(exp_names, node.get_data().c_str());
			if (type < 0) {
					Log::Warning << "Unknown exp type '" << node.get_data() << "' in class '" << name << "' at " << reader.get_filename() << ':' << node.get_line();
			} else {
				pri_exp_type = type;
			}
		FO_ATTR_NAME("secondary")
			int type = get_index_of(exp_names, node.get_data().c_str());
			if (type < 0) {
					Log::Warning << "Unknown exp type '" << node.get_data() << "' in class '" << name << "' at " << reader.get_filename() << ':' << node.get_line();
			} else {
				sec_exp_type = type;
			}
		FO_ATTR_NAME("difficulty")
			int diff;
			if (!node.get_int_data(diff)) {
				Log::Warning << "Difficulty not an integer in class '" << name << "' at " << reader.get_filename() << ':' << node.get_line();
			} else if (diff < 10 || diff > 300) {
				Log::Warning << "Difficulty out of range in class '" << name << "' at " << reader.get_filename() << ':' << node.get_line();
			} else
				diff_mod = diff;
		FO_ATTR_NAME("attack")
			int attack;
			if (!node.get_int_data(attack)) {
				Log::Warning << "Attack skill not an integer in class '" << name << "' at " << reader.get_filename() << ':' << node.get_line();
			} else {
				combat_mod = attack;
			}
		FO_ATTR_TYPE("stat")
			// is this a stat?
			CharStatID stat = CharStatID::lookup(node.get_name());
			if (stat) {
				if (!node.get_int_data(stats[stat.get_value()]))
					Log::Warning << "Attribute '" << node.get_name() << "' in class '" << name << "' is not a valid integer at " << reader.get_filename() << ':' << node.get_line();
			} else {
				Log::Warning << "Unknown attribute '" << node.get_name() << "' in class '" << name << "' at " << reader.get_filename() << ':' << node.get_line();
			}
	FO_READ_ERROR
		Log::Error << error.get_what() << "\n";
		return -1;
	FO_READ_END

	return 0;
}

Class*
SClassManager::get (StringArg name)
{
	Class *klass = head;
	while (klass != NULL) {
		if (str_eq (klass->get_name(), name))
			return klass;
		klass = klass->get_next();
	}
	return NULL;
}

int
SClassManager::initialize (void)
{
	Log::Info << "Loading player classes";

	File::Reader reader;
	String path = settings::get_path ("misc", "data") + "/classes";

	if (reader.open(path)) {
		Log::Error << "Failed to open " << path;
		return 1;
	}

	// load classes
	File::Node node;
	FO_READ_BEGIN
		FO_OBJECT("class")
			// valid parameters
			if (node.get_name().empty()) {
				Log::Warning << "Unnamed class at " << reader.get_filename() << ":" << node.get_line();
				reader.consume();
				continue;
			}

			// load class
			Class *klass = new Class (node.get_name(), head);
			if (klass->load (reader))
				return -1;
			head = klass;
	FO_READ_ERROR
		return -1;
	FO_READ_END

	return 0;
}

void
SClassManager::shutdown (void)
{
	head = NULL;
}
