/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>

#include "entity.h"
#include "char.h"
#include "error.h"
#include "server.h"
#include "awestr.h"
#include "body.h"
#include "room.h"
#include "rand.h"
#include "player.h"
#include "parse.h"
#include "streams.h"
#include "eventids.h"

// ----- CharStatID -----

String CharStatID::names[CharStatID::COUNT] = {
	"Strength",
	"Agility",
	"Fortitude",
	"Intellect",
	"Spirit",
	"Willpower",
};

CharStatID
CharStatID::lookup (StringArg name)
{
	for (uint i = 0; i < COUNT; ++i)
		if (str_eq(name, names[i]))
			return i;
	return NONE;
}

const char *stat_levels[] = {
	"Wretched",
	"Horrible",
	"Bad",
	"Poor",
	"Average",
	"Fair",
	"Good",
	"Excellent",
	"Awesome",
	NULL
};

const char *
get_stat_level (uint stat) {
	if (stat <= 15)
		return stat_levels[0];
	else if (stat <= 25)
		return stat_levels[1];
	else if (stat <= 35)
		return stat_levels[2];
	else if (stat <= 45)
		return stat_levels[3];
	else if (stat <= 55)
		return stat_levels[4];
	else if (stat <= 65)
		return stat_levels[5];
	else if (stat <= 75)
		return stat_levels[6];
	else if (stat <= 85)
		return stat_levels[7];
	else
		return stat_levels[8];
}

const char *
get_stat_color (uint stat) {
	if (stat <= 35)
		return CSTAT_BAD2;
	else if (stat <= 45)
		return CSTAT_BAD1;
	else if (stat <= 55)
		return CSTAT;
	else if (stat <= 65)
		return CSTAT_GOOD1;
	else
		return CSTAT_GOOD2;
}

// ----- CharPosition -----

String CharPosition::names[CharPosition::COUNT] = {
	"stand",
	"sit",
	"lay",
	"kneel",
};
String CharPosition::verbs[CharPosition::COUNT] = {
	"stand up",
	"sit down",
	"lay down",
	"kneel",
};
String CharPosition::sverbs[CharPosition::COUNT] = {
	"stands up",
	"sits down",
	"lays down",
	"kneels",
};
String CharPosition::verbings[CharPosition::COUNT] = {
	"standing",
	"sitting",
	"laying down",
	"kneeling",
};

CharPosition
CharPosition::lookup (StringArg name)
{
	for (uint i = 0; i < COUNT; ++i)
		if (str_eq(name, names[i]))
			return i;
	return STAND;
}

// ----- CharacterData -----

CharacterData::CharacterData (void)
{
}

void
CharacterData::reset_gender (void)
{
	// reset
	gender = GenderType::NONE;
	set_flags.gender = false;

	// get parent value
	const CharacterData* data = get_character_data_parent();
	if (data != NULL)
		gender = data->get_gender();
}

void
CharacterData::reset_alignment (void)
{
	// reset
	alignment = 0;
	set_flags.alignment = false;

	// get parent value
	const CharacterData* data = get_character_data_parent();
	if (data != NULL)
		alignment = data->get_alignment();
}

void
CharacterData::update_character_data (void)
{
	if (!set_flags.gender)
		reset_gender();
	if (!set_flags.alignment)
		reset_alignment();
}

int
CharacterData::load_node (File::Reader& reader, File::Node& node)
{
	FO_NODE_BEGIN
		FO_ATTR_NAME("gender")
			set_gender(GenderType::lookup(node.get_data()));
		FO_ATTR_NAME("alignment")
			int value;
			FO_GET_INT(value);
			set_alignment(value);
	FO_NODE_END
}

void
CharacterData::save (File::Writer& writer) const
{
	if (set_flags.alignment)
		writer.attr("alignment", alignment);
	if (set_flags.gender)
		writer.attr("gender", gender.get_name());
}

// ----- Character -----

void
Character::save (File::Writer& writer) const
{
	Entity::save(writer);
	CharacterData::save(writer);

	if (dead)
		writer.attr("dead", "yes");

	writer.attr("position", pos.get_name());

	if (round_time)
		writer.attr("roundtime", round_time);
	if (coins)
		writer.attr("coins", coins);

	writer.attr("hp", health.cur);
	if (health.max)
		writer.attr("maxhp", health.max);
		
	for (int i = 0; i < CharStatID::COUNT; ++ i)
		if (stats[i].base)
			writer.attr("stat", CharStatID(i).get_name(), stats[i].base);

	if (body.right_held) {
		writer.begin("object", "right_held");
		body.right_held->save(writer);
		writer.end();
	}
	if (body.left_held) {
		writer.begin("object", "left_held");
		body.left_held->save(writer);
		writer.end();
	}
	if (body.body_worn) {
		writer.begin("object", "body_worn");
		body.body_worn->save(writer);
		writer.end();
	}
	if (body.back_worn) {
		writer.begin("object", "back_worn");
		body.back_worn->save(writer);
		writer.end();
	}
	if (body.waist_worn) {
		writer.begin("object", "waist_worn");
		body.waist_worn->save(writer);
		writer.end();
	}
}

int
Character::load_node (File::Reader& reader, File::Node& node)
{
	FO_NODE_BEGIN
		FO_PARENT(Entity)
		FO_PARENT(CharacterData)
		FO_ATTR_NAME("dead")
			dead = node.get_data();
		FO_ATTR_NAME("position")
			pos = CharPosition::lookup(node.get_data());
		FO_ATTR_NAME("roundtime")
			FO_GET_INT(round_time);
		FO_ATTR_NAME("coins")
			FO_GET_INT(coins);
		FO_ATTR_NAME("hp")
			FO_GET_INT(health.cur);
		FO_ATTR_NAME("maxhp")
			FO_GET_INT(health.max);
		FO_ATTR_TYPE("stat")
			CharStatID stat = CharStatID::lookup(node.get_name());
			if (stat)
				FO_GET_INT(stats[stat.get_value()].base);
			else
				Log::Error << "Unknown stat '" << node.get_name() << "' at " << reader.get_filename() << ':' << node.get_line();
		FO_OBJECT("object")
			String loc = node.get_name();
			if (!loc) {
				Log::Error << "Body has object in no location at " << reader.get_filename() << ':' << node.get_line();
			} else {
				// do load
				Object* object = new Object();
				if (object->load (reader))
					throw "failed to load object";

				// set body loc
				if (node.get_name() == "right_held") {
					body.right_held = object;
					object->set_parent(this);
				} else if (node.get_name() == "left_held") {
					body.left_held = object;
					object->set_parent(this);
				} else if (node.get_name() == "body_worn") {
					body.body_worn = object;
					object->set_parent(this);
				} else if (node.get_name() == "back_worn") {
					body.back_worn = object;
					object->set_parent(this);
				} else if (node.get_name() == "waist_worn") {
					body.waist_worn = object;
					object->set_parent(this);
				} else {
					Log::Error << "Character has object in unknown location '" << node.get_name() << "' at " << reader.get_filename() << ':' << node.get_line();
				}
			}
	FO_NODE_END
}

int
Character::load_finish (void)
{
	return 0;
}

/* Character */
Character::Character (const Scriptix::Type* type) : Entity (type)
{
	pos = CharPosition::STAND;
	location = NULL;
	health.cur = health.max = 0;
	round_time = 0;
	coins = 0;
	dead = false;

	body.right_held = NULL;
	body.left_held = NULL;
	body.body_worn = NULL;
	body.waist_worn = NULL;
	body.back_worn = NULL;

	for (int i = 0; i < CharStatID::COUNT; ++ i) {
		stats[i].base = 0;
		stats[i].mod = 0;
	}
}

Character::~Character (void)
{
}

void
Character::release (void)
{
	// remove
	if (location)
		location->chars.remove (this);

	// note: don't set location to NULL, we might need that ifnromation
	//  later; this could just be a temporary operation.
}

/* round time */
uint
Character::get_rts (void) const {
	uint rt = round_time / (1000 / MSECS_PER_TICK);
	uint rem = round_time % (1000 / MSECS_PER_TICK);
	if (rem)
		return rt + 1;
	else
		return rt;
}

bool
Character::check_alive (void) {
	if (is_dead()) {
		*this << "You are only a ghost.\n";
		return false;
	}
	return true;
}

bool
Character::check_move (void) {
	if (!can_move()) {
		*this << "You cannot move.\n";
		return false;
	}
	return true;
}

bool
Character::check_see (void) {
	if (!can_see()) {
		*this << "You cannot see.\n";
		return false;
	}
	return true;
}

bool
Character::check_rt (void) {
	if (get_rt()) {
		uint rts = get_rts();
		*this << "You must wait " << rts << " second" << (rts > 1 ? "s" : "") << " for your roundtime to expire.\n";
		return false;
	}
	return true;
}

/* -------------------- FUNCS ---------------------- */

// move into a new room
bool
Character::enter (Room *new_room, RoomExit *old_exit)
{
	assert (new_room != NULL);

	// already here
	if (new_room->chars.has(this)) {
		location = new_room;
		return false;
	}

	// entering exit
	RoomExit* enter_exit = NULL;

	// did we go thru an exit?
	if (old_exit) {
		// "You go..." message
		*this << StreamParse(old_exit->get_go()).add("actor", this).add("exit", old_exit) << "\n";

		// "So-and-so leaves thru..." message
		if (get_room())
			*get_room() << StreamIgnore(this) << StreamParse(old_exit->get_leaves()).add("actor", this).add( "exit", old_exit) << "\n";

		// have we a target/linked exit?
		if (old_exit->get_target_id ())
			enter_exit = new_room->get_exit_by_id (old_exit->get_target_id ());

		// otherwise, have we an opposite exit?
		if (!enter_exit)
			enter_exit = new_room->get_exit_by_dir(old_exit->get_dir().get_opposite());
	}

	// valid exit?
	if (enter_exit)
		*new_room << StreamParse(enter_exit->get_enters()).add("actor", this).add("exit", enter_exit) << "\n";
	else
		*new_room << StreamName(this, INDEFINITE, true) << " arrives.\n";

	// move, look, event
	EventManager.send(Events::ON_LEAVE, get_room(), this, old_exit);
	new_room->add_character (this);
	do_look();
	EventManager.send(Events::ON_ENTER, new_room, this, enter_exit);

	return true;
}

void
Character::heal (uint amount)
{
	bool was_dead = is_dead();
	health.cur += amount;
	int max = get_max_hp();
	if (health.cur > max)
		health.cur = max;
	// have we been resurrected?
	if (health.cur > 0 && was_dead) {
		dead = false;
		EventManager.send(Events::ON_RESURRECT, get_room(), this, NULL);
	}
}

bool
Character::damage (uint amount, Entity *trigger) {
	// already dead?  no reason to continue
	if (is_dead())
		return false;
	// do damage and event
	health.cur -= amount;
	EventManager.send(Events::ON_DAMAGE, this->get_room(), trigger, this, Scripts.tovalue(amount));
	// caused death?
	if (health.cur <= 0 && !is_dead ()) {
		dead = true;
		kill (trigger);
		return true;
	}
	// still alive
	return false;
}

uint
Character::give_coins (uint amount)
{
	uint space = UINT_MAX - coins;
	if (space < amount)
		return coins += space;
	else
		return coins += amount;
}

uint
Character::take_coins (uint amount)
{
	if (amount > coins)
		return coins = 0;
	else
		return coins -= amount;
}

void
Character::update (void)
{
	/* update action timer */
	if (round_time > 0)
		-- round_time;

	/* healing */
	if (!is_dead() && (server.get_ticks() % (50 - get_stat(CharStatID::FORTITUDE) / 5)) == 0) {
		heal (1);
	}

	// update handler
	Scripts.hook("character_update", this);
}

void
Character::activate (void)
{
	Entity::activate();

	Object* obj;
	for (int i = 0; (obj = get_equip_at(i)) != NULL; ++i)
		obj->activate();
}

void
Character::deactivate (void)
{
	Object* obj;
	for (int i = 0; (obj = get_equip_at(i)) != NULL; ++i)
		obj->deactivate();

	Entity::deactivate();
}

void
Character::parse_comm (const char* comm, const StreamControl& stream) const
{
	// HE / SHE
	if (str_eq(comm, "he")) {
		stream << get_gender().get_heshe();
	}
	// HIM / HER
	else if (str_eq(comm, "him")) {
		stream << get_gender().get_himher();
	}
	// HIS / HER
	else if (str_eq(comm, "his")) {
		stream << get_gender().get_hisher();
	}
	// HIS / HERS
	else if (str_eq(comm, "hers")) {
		stream << get_gender().get_hishers();
	}
	// MAN / WOMAN
	else if (str_eq(comm, "man")) {
		stream << get_gender().get_manwoman();
		stream << (get_gender () == GenderType::MALE ? "man" : (get_gender () == GenderType::FEMALE ? "woman" : "thing"));
	}
	// MALE / FEMALE
	else if (str_eq(comm, "male")) {
		stream << get_gender().get_malefemale();
	}
	// ALIVE / DEAD
	else if (str_eq(comm, "alive")) {
		if (is_dead())
			stream << "dead";
		else
			stream << "alive";
	}
	// POSITION
	else if (str_eq(comm, "position")) {
		stream << get_pos().get_verbing();
	}
	// LEVEL
	else if (str_eq(comm, "level")) {
		stream << get_level();
	}
	// default...
	else Entity::parse_comm(comm, stream);
}

// recalc stats
void
Character::recalc_stats (void)
{
	for (int i = 0; i < CharStatID::COUNT; ++i)
		stats[i].mod = 0;
}

// recalculate various stuff
void
Character::recalc (void)
{
	recalc_stats();
}

void
Character::display_equip (const StreamControl& stream) const
{
	// inventory variables
	uint loc = 0;
	Object* obj;
	Object* last = NULL;
	bool didshow = false;

	// worn items
	while ((obj = get_worn_at(loc++)) != NULL) {
		// hidden?  skip
		if (obj->is_hidden())
			continue;
		// we had one already?
		if (last) {
			// prefix
			if (didshow) {
				stream << ", ";
			} else {
				stream << StreamParse("  {.He} is wearing ", this);
				didshow = true;
			}
			// do show
			stream << StreamName(last, INDEFINITE);
		}
		// remember this object
		last = obj;
	}
	// spit out the left over
	if (last) {
		// prefix
		if (didshow) {
			stream << " and ";
		} else {
			stream << StreamParse("  {.He} is wearing ", this);
			didshow = true;
		}
		// show it
		stream << StreamName(last, INDEFINITE) << ".";
	}

	// held items
	loc = 0;
	didshow = false;
	last = NULL;
	while ((obj = get_held_at(loc++)) != NULL) {
		// hidden?  skip
		if (obj->is_hidden())
			continue;
		// we had one already?
		if (last) {
			// prefix
			if (didshow) {
				stream << ", ";
			} else {
				stream << StreamParse("  {.He} is holding ", this);
				didshow = true;
			}
			// show
			stream << StreamName(last, INDEFINITE);
		}
		last = obj;
	}
	// show the last one
	if (last) {
		// prefix
		if (didshow) {
			stream << " and ";
		} else {
			stream << StreamParse("  {.He} is holding ", this);
			didshow = true;
		}
		// show it
		stream << StreamName(last, INDEFINITE) << ".";
	}

	// dead or position
	if (is_dead())
		stream << StreamParse("  {.He} is laying on the ground, dead.", this);
	else if (get_pos() != CharPosition::STAND)
		stream << StreamParse("  {.He} is {.position}.", this);

	// health
	if (!is_dead() && get_max_hp() > 0) {
		if (get_hp() * 100 / get_max_hp() <= 25)
			stream << StreamParse("  {.He} appears severely wounded.", this);
		else if (get_hp() * 100 / get_max_hp() <= 75)
			stream << StreamParse("  {.He} appears wounded.", this);
		else
			stream << StreamParse("  {.He} appears to be in good health.", this);
	}
}
