/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#include <sstream>

#include <dirent.h>
#include <fnmatch.h>

#include "entity.h"
#include "ai.h"

SAIManager AIManager;

AI::AI (StringArg s_name) : next(NULL), name(s_name), states() {} 

int
AI::load (File::Reader& reader)
{
	File::Node node;

	FO_READ_BEGIN
		FO_ATTR_NAME("load")
			if (node.get_data()) {
				// build text
				String source;
				source += "public __ai_callback(self) {";
				source += node.get_data();
				source += "}";

				// compile
				if (Scripts.get_engine()->LoadString(source, reader.get_filename(), node.get_line()) == Scriptix::SXE_OK) {
					loadf = Scripts.get_engine()->GetFunction(Scriptix::NameToID("__ai_callback"));
				} else {
					throw File::Error("Script compile failed");
				}
			}
		FO_OBJECT("state")
			AIState* state = new AIState(node.get_name());
			if (state == NULL)
				throw File::Error("new AIState failed");

			states[node.get_name()] = state;

			// load state
			FO_READ_BEGIN
				FO_ATTR_TYPE("event")
					if (node.get_data()) {
						// build text
						String source;
						source += "public __ai_callback(self, event) {";
						source += node.get_data();
						source += "}";

						// compile
						if (Scripts.get_engine()->LoadString(source, reader.get_filename(), node.get_line()) == Scriptix::SXE_OK) {
							state->events[EventID::lookup(node.get_name())] = Scripts.get_engine()->GetFunction(Scriptix::NameToID("__ai_callback"));
						}
					}
				FO_ATTR_NAME("heartbeat")
					if (node.get_data()) {
						// build text
						String source;
						source += "public __ai_callback(self) {";
						source += node.get_data();
						source += "}";

						// compile
						if (Scripts.get_engine()->LoadString(source, reader.get_filename(), node.get_line()) == Scriptix::SXE_OK) {
							state->heartbeat = Scripts.get_engine()->GetFunction(Scriptix::NameToID("__ai_callback"));
						}
					}
				FO_ATTR_NAME("enter")
					if (node.get_data()) {
						// build text
						String source;
						source += "public __ai_callback(self) {";
						source += node.get_data();
						source += "}";

						// compile
						if (Scripts.get_engine()->LoadString(source, reader.get_filename(), node.get_line()) == Scriptix::SXE_OK) {
							state->start = Scripts.get_engine()->GetFunction(Scriptix::NameToID("__ai_callback"));
						}
					}
				FO_ATTR_NAME("leave")
					if (node.get_data()) {
						// build text
						String source;
						source += "public __ai_callback(self) {";
						source += node.get_data();
						source += "}";

						// compile
						if (Scripts.get_engine()->LoadString(source, reader.get_filename(), node.get_line()) == Scriptix::SXE_OK) {
							state->leave = Scripts.get_engine()->GetFunction(Scriptix::NameToID("__ai_callback"));
						}
					}
				FO_ATTR_NAME("round")
					if (node.get_data()) {
						// build text
						String source;
						source += "public __ai_callback(self) {";
						source += node.get_data();
						source += "}";

						// compile
						if (Scripts.get_engine()->LoadString(source, reader.get_filename(), node.get_line()) == Scriptix::SXE_OK) {
							state->round = Scripts.get_engine()->GetFunction(Scriptix::NameToID("__ai_callback"));
						}
					}
			FO_READ_ERROR
				throw error;
			FO_READ_END
	FO_READ_ERROR
		return -1;
	FO_READ_END

	return 0;
}

void
SAIManager::add (AI* ai)
{
	// add to list
	ai->next = head;
	head = ai;
}

AI*
SAIManager::get (StringArg name)
{
	for (AI* ai = head; ai != NULL; ai = ai->next)
		if (ai->name == name)
			return ai;
	return NULL;
}

void
AIState::do_event (Entity* self, Event* event) const
{
	// get handler
	EventList::const_iterator i = events.find(event->get_id());
	if (i != events.end())
		Scripts.run(i->second, self, event);
}

void
AIState::do_round (Entity* self) const
{
	// call handler
	if (round != NULL)
		Scripts.run(round, self);
}

void
AIState::do_heartbeat (Entity* self) const
{
	// call handler
	if (heartbeat != NULL)
		Scripts.run(heartbeat, self);
}

void
AIState::do_start (Entity* self) const
{
	// call handler
	if (start != NULL)
		Scripts.run(start, self);
}

void
AIState::do_leave (Entity* self) const
{
	// call handler
	if (leave != NULL)
		Scripts.run(leave, self);
}

void
AI::do_load (Entity* self) const
{
	// call handler
	if (loadf != NULL)
		Scripts.run(loadf, self);
}

int
SAIManager::initialize (void)
{
	DIR *dir;
	dirent *dent;

	Log::Info << "Loading AI packs";

	// read directory
	dir = opendir (settings::get_path("ai", "scripts"));
	if (dir != NULL) {
		while ((dent = readdir (dir)) != NULL) {
			if (!fnmatch ("*.ai", dent->d_name, 0)) {
				std::ostringstream file;
				file << settings::get_path("ai", "scripts") << '/' << dent->d_name;

				// load ai
				AI* ai = new AI(dent->d_name);
				if (ai == NULL) {
					fatal("new AI failed");
					closedir(dir);
					return -1;
				}
				File::Reader reader(file.str());
				if (!reader.is_open()) {
					Log::Error << "Failed to open " << file.str();
				} else if (!ai->load(reader)) {
					add(ai);
				}
			}
		}
	}
	closedir (dir);

	return 0;
}
void
SAIManager::shutdown (void)
{
	head = NULL;
}
