/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#include <ctype.h>

#include "account.h"
#include "fileobj.h"
#include "settings.h"
#include "md5.h"

SAccountManager AccountManager;

StringList AccessID::registry;

AccessID
AccessID::lookup (StringArg name)
{
	// empty?  invalid
	if (name.empty())
		return AccessID();
	// find it
	for (size_t i = 0; i < registry.size(); ++i)
		if (registry[i] == name)
			return AccessID(i + 1);
	// not found; auto-add
	registry.push_back(String(name).lower());
	return AccessID(registry.size());
}

String
AccessID::get_name (void) const
{
	if (valid())
		return registry[id - 1];
	return String();
}

Account::Account (StringArg s_id) : id(s_id), active(0), maxchars(0), maxactive(0)
{
	flags.disabled = false;
}

Account::~Account (void)
{
	// remove from account list
	SAccountManager::AccountList::iterator i = find(AccountManager.accounts.begin(), AccountManager.accounts.end(), this);
	if (i != AccountManager.accounts.end())
		AccountManager.accounts.erase(i);
}

int
Account::save (void) const
{
	String path = settings::get_path("accounts", "data") + "/" + String(id).lower() + ".acct";
	Log::Info << "Saving account " << id;

	// open
	File::Writer writer;
	if (writer.open(path))
		return -1;

	// save it out
	writer.attr("name", name);
	writer.attr("email", email);
	writer.attr("passphrase", pass);
	for (StringList::const_iterator i = chars.begin(); i != chars.end(); ++i)
		writer.attr("character", *i);
	if (flags.disabled)
		writer.attr("disabled", "yes");
	if (maxchars > 0)
		writer.attr("maxchars", maxchars);
	if (maxactive > 0)
		writer.attr("maxactive", maxactive);
	if (timeout > 0)
		writer.attr("maxactive", timeout);
	for (AccessList::const_iterator i = access.begin(); i != access.end(); ++i)
		writer.attr("access", i->get_name());

	// done
	writer.close();
	return 0;
}

// password management
void
Account::set_passphrase (StringArg s_pass)
{
	// encrypt
	char enc_pass[MD5_BUFFER_SIZE];
	MD5::encrypt (s_pass.c_str(), enc_pass);

	// store
	pass = enc_pass;

	// force save
	save();
}

// check password
bool
Account::check_passphrase (StringArg s_pass) const
{
	// empty?  auto-fail
	if (!s_pass)
		return false;

	// do compare
	return MD5::compare (pass.c_str(), s_pass.c_str());
}

// add a new character
void
Account::add_character (StringArg name)
{
	// not already in list?
	if (find(chars.begin(), chars.end(), name) != chars.end())
		return;

	// ok then, add it
	chars.push_back(name);
}

// remove a character
void
Account::del_character (StringArg name)
{
	// find in list
	StringList::iterator i;
	if ((i = find(chars.begin(), chars.end(), name)) == chars.end())
		return;

	// ok then, remove it
	chars.erase(i);
}

// get max chars allowed
uint
Account::get_max_chars (void) const
{
	// explicit?
	if (maxchars > 0)
		return maxchars;

	// default
	return settings::get_int("accounts", "maxchars");
}

// get max active chars allowed
uint
Account::get_max_active (void) const
{
	// explicit?
	if (maxactive > 0)
		return maxactive;

	// default
	return settings::get_int("accounts", "maxactive");
}

// access
bool
Account::has_access(AccessID id) const
{
	return (find(access.begin(), access.end(), id) != access.end());
}
bool
Account::grant_access(AccessID id)
{
	// have it already?
	if (has_access(id))
		return true;
	// grant it
	access.push_back(id);
	return true;
}
bool
Account::revoke_access(AccessID id)
{
	// find it
	AccessList::iterator i = find(access.begin(), access.end(), id);
	if (i == access.end())
		return false;
	// remove it
	access.erase(i);
	return true;
}

int
SAccountManager::initialize (void)
{
	Log::Info << "Initializing account manager";

	return 0;
}

void
SAccountManager::shutdown (void)
{
	Account* account;
	while (!accounts.empty()) {
		account = accounts.back();
		account->save();
		accounts.pop_back();
	}
	accounts.resize(0);
}

bool
SAccountManager::valid_name (StringArg name)
{
	// length
	if (name.size() < ACCOUNT_NAME_MIN_LEN || name.size() > ACCOUNT_NAME_MAX_LEN)
		return false;

	// check characters
	for (uint i = 0; i < name.size(); ++i)
		if (!isalnum(name[i]))
			return false;

	// must be good
	return true;
}

bool
SAccountManager::valid_passphrase (StringArg pass)
{
	// length
	if (pass.size() < ACCOUNT_PASS_MIN_LEN)
		return false;

	// must be both letters and numbers
	bool let = false;
	bool num = false;
	for (uint i = 0; i < pass.size(); ++i)
		if (isalpha(pass[i]))
			let = true;
		else if (isdigit(pass[i]))
			num = true;

	// true if both let and num are now true
	return let && num;
}

Account*
SAccountManager::get (String name)
{
	// force lower-case
	name.lower();

	// check validity
	if (!valid_name(name))
		return NULL;

	// search loaded list
	for (AccountList::iterator i = accounts.begin(); i != accounts.end(); ++i)
		if ((*i)->id == name)
			return *i;

	// try load
	File::Reader reader;
	File::Node node;

	// open
	if (reader.open(settings::get_path("accounts", "data") + "/" + name + ".acct"))
		return NULL;

	// create
	Account* account = new Account(name);
	if (account == NULL)
		return NULL;

	// read it in
	FO_READ_BEGIN
		FO_ATTR_NAME("name")
			account->name = node.get_data();
		FO_ATTR_NAME("email")
			account->email = node.get_data();
		FO_ATTR_NAME("passphrase")
			account->pass = node.get_data();
		FO_ATTR_NAME("character")
			account->chars.push_back(node.get_data());
		FO_ATTR_NAME("maxchars")
			FO_GET_INT(account->maxchars);
		FO_ATTR_NAME("maxactive")
			FO_GET_INT(account->maxactive);
		FO_ATTR_NAME("timeout")
			FO_GET_INT(account->timeout);
		FO_ATTR_NAME("disabled")
			FO_GET_BOOL(account->flags.disabled);
		FO_ATTR_NAME("access")
			account->access.push_back(AccessID::lookup(node.get_data()));
	FO_READ_ERROR
		delete account;
		return NULL;
	FO_READ_END

	// add to list
	accounts.push_back(account);

	return account;
}

Account*
SAccountManager::create (StringArg name)
{
	// check validity
	if (!valid_name(name))
		return NULL;

	// check if account exists?
	if (get(name) != NULL)
		return NULL;

	// create
	Account* account = new Account(name);
	if (account == NULL)
		return NULL;

	// save
	account->save();

	// add to list
	accounts.push_back(account);

	return account;
}

bool
SAccountManager::exists (String name)
{
	// must be lower-case
	name.lower();

	// must be a valid name
	if (!valid_name(name))
		return false;

	// look thru list for valid and/or connected players
	for (AccountList::iterator i = accounts.begin(); i != accounts.end(); ++i) {
		if ((*i)->get_name() == name)
			return true;
	}

	// check if player file exists
	String path = settings::get_path("accounts", "data") + "/" + name + ".acct";
	struct stat st;
	int res = stat (path.c_str(), &st);
	if (res == 0)
		return true;
	if (res == -1 && errno == ENOENT)
		return false;
	Log::Error << "stat() failed for " << path << ": " << strerror(errno);	
	return true;
}
