/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#ifndef TELNET_H
#define TELNET_H

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif // HAVE_CONFIG_H

#ifdef HAVE_LIBZ
#include <zlib.h>
#endif // HAVE_LIBZ

#include <vector>

#include "types.h"
#include "gcbase.h"
#include "streams.h"
#include "network.h"
#include "color.h"

// default window size
#define DEFAULT_WIDTH 80
#define DEFAULT_HEIGHT 24

// max pre-input lines
#define BUFFER_LINES 3

// maximum escape length
#define MAX_ESCAPE_SIZE 32

class TextBufferList
{
	private:
	std::vector<char*> list;
	const size_t size;
	size_t allocs, pallocs, out;

	public:
	TextBufferList (size_t s_size) : list (), size(s_size), allocs(0), pallocs(0), out(0) {}
	~TextBufferList (void) {
		for (std::vector<char*>::iterator i = list.begin(); i != list.end(); ++i)
			if (*i)
				free (*i);
	}

	size_t get_size (void) const { return size; }

	char* alloc (void);
	void release (char* buf) {
		-- out;
		list.push_back(buf);
	}
};
class TextBuffer 
{
	public:
	typedef enum {
		EMPTY = -1,
		SIZE_64 = 0,
		SIZE_128,
		SIZE_256,
		SIZE_512,
		SIZE_1024,
		SIZE_2048,
		SIZE_4096,
		SIZE_8192,
		SIZE_16384,
		COUNT
	} SizeType;
	
	private:
	SizeType bsize;
	char* bdata;

	// the buffers
	static TextBufferList lists[COUNT];

	public:
	TextBuffer (void) : bsize(EMPTY), bdata(NULL) {}
	~TextBuffer (void) { release(); }

	size_t size(void) const { return bsize != EMPTY ? lists[bsize].get_size() : 0; }
	char *data(void) const { return bdata; }
	void release (void);
	int alloc(SizeType size);
	int grow (void);

	TextBuffer& copy (TextBuffer& buffer) {
		if (bdata != NULL)
			release();
		bsize = buffer.bsize;
		bdata = buffer.bdata;
		buffer.bsize = EMPTY;
		buffer.bdata = NULL;
		return *this;
	}
};

class ITelnetMode : public GCType::Collectable
{
	public:
	ITelnetMode (class TelnetHandler* s_handler) : handler(s_handler) {}
	virtual ~ITelnetMode (void) {}

	// basics
	virtual int initialize (void) = 0;
	virtual void prompt (void) = 0;
	virtual void process (char* line) = 0;
	virtual void shutdown (void) = 0;

	// get the current Account/Player if the mode has anything to do with one.
	inline virtual class Player* get_player (void) { return NULL; }
	inline virtual class Account* get_account (void) { return NULL; }

	// the handler this mode is connected to
	inline class TelnetHandler* get_handler (void) const { return handler; }

	private:
	class TelnetHandler* handler;
};

class TelnetHandler : public Scriptable, public IStreamSink
{
	public:
	TelnetHandler (void);

	// color info
	inline uint get_color (uint i) const { return color_set[i] < 0 ? color_type_defaults[i] : color_set[i]; }
	inline void set_color (uint i, uint v) { color_set[i] = v; }
	inline void clear_color (uint i) { color_set[i] = -1; }
	inline bool use_color (void) const { return io_flags.use_ansi; }

	// low-level IO
	int net_connect (const Socket& socket);
	void net_disconnect (void);
	inline bool net_alive (void) const { return sock.alive(); }
	inline bool net_inready (void) const { return sock.in_ready(); }
	inline bool net_hungup (void) const { return sock.hung_up(); }

	// processing IO
	void process (void);
	void flush (void);
	inline int get_width (void) const { return width; }
	bool toggle_echo (bool value);
	void process_command (char* cmd); // just as if typed in by user

	// output
	virtual void stream_put (const char*, size_t len);
	inline bool have_zmp (void) const { return io_flags.zmp; }
	inline bool has_x_awemud (void) const { return io_flags.x_awemud; } // supports the x-awemud. package?
	void send_zmp (size_t argc, const char** argv);
	inline void send_zmp (size_t argc, char** argv) { send_zmp(argc, (const char**)argv); }
	void clear_scr (void); // clear da screen
	void set_indent (uint amount);
	inline uint get_indent (void) const { return margin; }
	void draw_bar (uint percent); // draws a 14 character width progress bar
	void zmp_support (const char* pkg, bool value);

	// mode
	void set_mode (ITelnetMode* new_mode);

	// get the mode's account/player, if it uses one
	inline class Player* get_player (void) const { return mode ? mode->get_player() : NULL; }
	inline class Account* get_account (void) const { return mode ? mode->get_account() : NULL; }

	protected:
	// destructor
	~TelnetHandler (void);

	protected:
	Socket sock; // socket
	TextBuffer input; // player input buffer
	TextBuffer output; // output
	TextBuffer outchunk; // chunk of output
	TextBuffer subrequest; // telnet-subrequest input
	uint in_cnt, out_cnt, sb_cnt, outchunk_cnt; // counts for buffers
	char esc_buf[MAX_ESCAPE_SIZE]; // output escape sequences
	uint esc_cnt; // count of escape characters
	uint width, height; // terminal size
	uint cur_col; // current output column
	uint margin; // forced indent
	uint chunk_size; // size of printable chunk characters
	time_t in_stamp; // last input time
	int color_set[NUM_CTYPES]; // color codes
	std::vector<int> colors; // current color stack
#ifdef HAVE_LIBZ
	z_stream* zstate; // compression
#endif
	struct IOFlags {
		int use_ansi:1, need_prompt:1, need_newline:1, do_echo:1,
			do_eor:1, want_echo:1, xterm:1, force_echo:1, zmp:1,
			ready:1, soft_break:1, ansi_term:1, have_output:1,
			x_awemud:1, auto_indent:1;
	} io_flags;

	// input states - telnet
	enum {
		ISTATE_TEXT,
		ISTATE_IAC,
		ISTATE_WILL,
		ISTATE_WONT,
		ISTATE_DO,
		ISTATE_DONT,
		ISTATE_SB,
		ISTATE_SE,
		ISTATE_CR,
	} istate;

	// output states - formatting
	enum {
		OSTATE_TEXT,
		OSTATE_ESCAPE,
		OSTATE_ANSI,
		OSTATE_AWECODE
	} ostate;

	// current mode
	ITelnetMode* mode;

#ifdef HAVE_LIBZ
	// compression
	bool begin_mccp (void);
	void end_mccp (void);
#endif // HAVE_LIBZ

	// processing
	void process_input (void);
	void process_sb (void);
	void process_zmp (size_t size, char* chunk);

	// command handling
	void process_telnet_command (char* cmd);

	// data output
	void add_output (const char* data, size_t len);
	void add_to_chunk (const char* data, size_t len);
	void end_chunk (void);
	void send_iac (uint, ...); // build iac
	void send_data (uint, ...); // don't escape
	void send_bytes (char* bytes, size_t len); // does actual transmit over socket

	// timeout handling
	int check_time (void); // check to see if we should disconnect

	SX_TYPEDEF
	friend class STelnetManager;
};

class STelnetManager : public IManager
{
	public:
	// setup/destroy
	virtual int initialize (void);
	virtual void shutdown (void);

	// count of connected connectionss
	size_t count (void);

	// input processing of all connectionss
	void process (void);

	// flush all pending output
	void flush (void);

	private:
	// list of *connected* connectionss
	typedef GCType::list<TelnetHandler*> TelnetList;
	TelnetList telnet_list;
	uint timeout_minutes;

	// yuck - let Player class manage their own membership
	friend class TelnetHandler;
};
extern STelnetManager TelnetManager;

// LOGIN
class TelnetModeLogin : public ITelnetMode
{
	public:
	TelnetModeLogin (TelnetHandler* s_handler) : ITelnetMode (s_handler), account(NULL), pass(false), tries(0) {}

	virtual int initialize (void);
	virtual void prompt (void);
	virtual void process (char* line);
	virtual void shutdown (void);

	virtual inline class Account* get_account (void) { return account; }

	private:
	class Account* account;
	bool pass;
	int tries;
};

// indent stream
class StreamIndent
{
	public:
	inline StreamIndent(uint16 s_len) : len(s_len) {}

	inline friend
	const StreamControl&
	operator << (const StreamControl& stream, const StreamIndent& indent)
	{
		return stream << "\e!" << indent.len << "!";
	}

	private:
	uint16 len;
};

#endif
