/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

// Re-using Double Queue
// Similar to STL dqueues, but reuses memory.  Optimized for push-bask,
// pop-front usage only.  Written for usage with events and other task
// queues

#ifndef RDQUEUE_H 
#define RDQUEUE_H

#include "gcbase.h"

template <typename Value, int Size>
class RDQueue : public GCType::Simple {
	typedef Value* VPtr;

	public:
	RDQueue (void);

	// add another item to end
	void push (VPtr value);

	// pop an item from the front
	VPtr pop (void);

	// check if we're empty
	inline bool empty (void) { return head == tail && headi == taili; }

	private:
	struct Chunk : public GCType::Simple {
		Chunk* next;
		VPtr data[Size];
	};
	Chunk* head;
	Chunk* tail;
	Chunk* last;
	int headi, taili;
};

// implementations

template <typename Value, int Size>
RDQueue<Value,Size>::RDQueue<Value,Size> (void) : head(NULL), tail(NULL), last(NULL), headi(0), taili(0)
{
	head = new Chunk();
	head->next = NULL;
	tail = head;
	last = head;
}

template <typename Value, int Size>
void
RDQueue<Value,Size>::push (VPtr value)
{
	// chunk full?
	if (taili == Size) {
		// need a new chunk, we're the last one
		if (tail == last) {
			last = new Chunk();
			last->next = NULL;
			tail->next = last;
			last = tail;
		// not the last one, so just use the next one
		} else {
			tail = tail->next;
		}
		// push data
		tail->data[0] = value;
		taili = 1;
	// have room left!
	} else {
		tail->data[taili++] = value;
	}
}

template <typename Value, int Size>
typename RDQueue<Value,Size>::VPtr
RDQueue<Value,Size>::pop (void)
{
	// empty?  return NULL
	if (empty())
		return NULL;

	// store return
	VPtr value = head->data[headi];
	head->data[headi] = NULL;

	// at end?
	if (++headi == Size) {
		// did we hit end?  possible with just one chunk
		if (head == last) {
			headi = taili = 0;
		// not end; kick head to last
		} else {
			last->next = head;
			head = head->next;
			last->next->next = NULL;
			headi = 0;
		}
	}

	// return value
	return value;
}

#endif // RDQUEUE_H
