/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#ifndef PLAYER_H
#define PLAYER_H

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif // HAVE_CONFIG_H

#include <vector>
#include <algorithm>

#include "awestr.h"
#include "char.h"
#include "color.h"
#include "scripts.h"
#include "network.h"
#include "pdesc.h"
#include "gametime.h"
#include "gcbase.h"


#ifdef HAVE_LIBZ
#include <zlib.h>
#endif // HAVE_LIBZ

// player name length requirements
#define PLAYER_NAME_MIN_LEN 3
#define PLAYER_NAME_MAX_LEN 15

enum exp_spec {
	EXP_GENERAL = 0,
	EXP_WARRIOR,
	EXP_ROGUE,
	EXP_CASTER,
	NUM_EXPS
};

// input processor
class IProcessor : public GCType::Collectable
{
	protected:
	class Player* player;

	public:
	IProcessor (class Player* s_player) : gc(), player(s_player) {}
	virtual ~IProcessor(void) { player = NULL; }

	virtual int init (void) = 0;
	virtual void finish (void) = 0;
	// return true/non-zero in order to remove the IProcessor
	virtual int process (char *line) = 0;
	virtual const char *prompt (void) = 0;
};

class Player : public Character
{
	public:
	// create
	Player (class Account* s_account, StringArg s_name);

	// 'disable' setname
	// this is never implemented anywhere, so it can't be called.
	// FIXME: can still call on Character/Entity...
	void set_name (StringArg s_name);

	// display
	virtual void display (const class StreamControl& stream, EntityArticleType, bool capitalize) const;

	// save and load
	void save (File::Writer& writer) const;
	void save (void) const;

	void load_init (void);
	int load_node (File::Reader& reader, File::Node& node);
	int load_finish (void);

	// output color
	inline const char *ncolor (void) const { return CPLAYER; }

	// account
	inline class Account* get_account (void) const { return account; }

	// valid character?
	//   is to a playable character
	inline bool is_valid (void) const { return flags.valid; }

	// connected?
	//   currently in use by someone
	inline bool is_connected (void) const { return conn != NULL; }

	// enable builder-vision?
	inline bool has_bvision (void) const { return flags.bvision; }
	inline void set_bvision (bool value) { flags.bvision = value; }
	inline int handle_event (Event* event); // deal with builder-vision

	// commands
	virtual void process_cmd (const char *);

	// birthday/age
	uint get_age (void) const;
	inline GameTime get_birthday (void) const { return birthday; }
	inline void set_birthday (const GameTime& gt) { birthday = gt; }

	// misc
	void kill (Entity* killer);
	void update (void);

	// manage account active counts
	virtual void activate (void);
	virtual void deactivate (void);

	// base health
	inline uint get_base_health (void) const { return health_base; }
	inline uint set_base_health (uint new_bh) { return health_base = new_bh; }

	// race
	inline class Race* get_race (void) const { return race; }
	inline class Race* set_race (class Race* race_in) { return race = race_in; }

	// player description details
	inline ColorType get_eye_color (void) const { return pdesc.eye_color; }
	inline void set_eye_color (ColorType eyes) { pdesc.eye_color = eyes; }
	inline ColorType get_hair_color (void) const { return pdesc.hair_color; }
	inline void set_hair_color (ColorType hairs) { pdesc.hair_color = hairs; }
	inline ColorType get_skin_color (void) const { return pdesc.skin_color; }
	inline void set_skin_color (ColorType skins) { pdesc.skin_color = skins; }
	inline HairStyleType get_hair_style (void) const { return pdesc.hair_style; }
	inline void set_hair_style (HairStyleType hairs) { pdesc.hair_style = hairs; }
	inline BodyBuildType get_build (void) const { return pdesc.build; }
	inline void set_build (BodyBuildType bodys) { pdesc.build = bodys; }

	// combat
	virtual uint get_combat_dodge (void) const;
	virtual uint get_combat_attack (void) const;
	virtual uint get_combat_damage (void) const;

	// class/exp
	inline uint get_exp (uint type) const { if (type < NUM_EXPS) return exp[type]; else return 0; }
	void grant_exp (uint type, uint amount);
	uint get_class_level (const class Class*) const;
	void grant_class_levels (const class Class*, uint);
	void level_up (const class Class*);
	uint get_level_percent (const class Class*) const;
	inline bool can_level (const class Class* klass) { return get_level_percent(klass) >= 100; }
	uint get_level (void) const;

	// recalcuate everything
	void recalc_stats (void);
	void recalc_health (void);
	void recalc (void);

	// display info
	void display_inventory (void);
	virtual void display_desc (const class StreamControl& stream) const;

	// processor
	int add_processor (IProcessor *);

	// I/O
	virtual void stream_put (const char* data, size_t len = 0);
	void show_prompt (void);
	void process_command (char* cmd);
	void connect (class TelnetHandler* telnet);
	void disconnect (void);
	inline class TelnetHandler* get_telnet(void) const { return conn; }
	void toggle_echo (bool value);
	void set_indent (uint level);
	uint get_width (void);
	void clear_scr (void);

	// modify prompt
	inline void set_prompt (StringArg s_prompt) { prompt = s_prompt; }
	inline void default_prompt (void) { prompt.clear(); }

	// parsing
	virtual void parse_comm (const char* comm, const class StreamControl& output) const;

	// handling "player states"
	int start (void); // start the session
	int create (void); // create new character
	int validate (void); // make the player valid
	void quit (void); // save and exit

	// player-only actions
	void do_tell (Player* who, StringArg what);
	void do_reply (StringArg what);

	protected:
	// connection
	class TelnetHandler* conn;

	// user info
	Account* account;

	// last entered command
	String last_command;
	
	// last player given a TELL
	String last_tell;

	// current prompt
	String prompt;

	// --- PLAYER DESCRIPTION DETAILS ---
	struct PDesc {
		ColorType eye_color;
		ColorType hair_color;
		ColorType skin_color;
		HairStyleType hair_style;
		BodyBuildType build;
	} pdesc;

	// flags
	struct PlayerFlags {
		uint valid:1, bvision:1;
	} flags;

	// command processors
	typedef GCType::vector<IProcessor*> ProcessorList;
	ProcessorList procs;

	// health
	int health_base;

	// character race
	class Race *race;

	// classes
	uint exp[NUM_EXPS];
	typedef std::map<const class Class*, uint> ClassList;
	ClassList classes;

	// birthdate (for fun!)
	GameTime birthday;

	// any special titles
	const class PlayerTitle* title;

	// network stuff - also x-awemud ZMP tracking
	struct NetworkInfo {
		uint last_rt; // last reported round-time
		uint last_max_rt; // last reported max round-time
		int last_hp; // last reported hp 
		int last_max_hp; // last reported hp 
		uint timeout_ticks; // remaining ticks until timeout
	} ninfo;

#ifdef HAVE_LIBZ
	// compression
	bool begin_mccp (void);
	void end_mccp (void);
#endif // HAVE_LIBZ

	SX_TYPEDEF
	E_SUBTYPE(Player,Character);

	friend class SPlayerManager; // obvious

	protected:
	~Player (void);
};

// Wrap a ScriptProcessor - neesd *somewhere*....
class ScriptProcessorWrapper : public IProcessor
{
	private:
	ScriptProcessor* core;

	public:
	inline ScriptProcessorWrapper(Player* s_player, ScriptProcessor* s_core) : IProcessor(s_player), core(s_core) {}

	int init (void);
	void finish (void);
	int process (char*);
	const char* prompt (void);
};

// manage all players
class SPlayerManager : public IManager
{
	public:
	// list of *connected* players - do *not* use GC
	typedef std::list<Player*> PlayerList;

	// initialize the player manager
	virtual int initialize (void);

	// shutdown the player manager
	virtual void shutdown (void);

	// true if 'name' is a valid player name
	bool valid_name (String name);

	// return the path a player's file is at
	String path (StringArg name);

	// return the logged-in player with the given name
	Player* get (StringArg name);

	// load a player - from disk
	Player* load (class Account* account, StringArg name);

	// create a new player with a given name
	Player* create (class Account* account, StringArg name);

	// DESTROY a player permanently (with backup)
	int destroy (StringArg name);

	// does a valid player of this name exist?
	bool exists (String name);

	// count of connected players
	size_t count (void);

	// list all connected players
	void list (const StreamControl& stream);

	// EEEEW - return list of all players - yuck
	inline const PlayerList& get_player_list (void) { return player_list; }

	private:
	PlayerList player_list;

	// yuck - let Player class manage their own membership
	friend class Player;
};
extern SPlayerManager PlayerManager;

#define PLAYER(ent) E_CAST((ent),Player)

extern const char* exp_names[];

extern const char* get_stat_level (uint value);
extern const char* get_stat_color (uint value);

#endif
