/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#ifndef OBJECT_H
#define OBJECT_H

#include <algorithm>

#include "entity.h"
#include "awestr.h"
#include "error.h"
#include "blueprint.h"
#include "body.h"
#include "action.h"
#include "elist.h"
#include "objclass.h"
#include "container.h"

// NOTES:
//  cost is in hundredths of 1 unit of current (ex: US dollar)
//  so a cost of 2100 would possibly equal 21 US$.  In most
//  fantasy settings, gold pieces or silver pieces are the standard
//  unit of currency, so 2100 might equal 21 gold or 21 silver.

//  weight is measured in stones.  that's 14 imperial pounds to a
//  stone.

class
ObjectData
{
	public:
	ObjectData (void);
	virtual ~ObjectData (void) {}

	// weight
	inline uint get_weight (void) const { return weight; }
	inline void set_weight (uint s_weight) { weight = s_weight; set_flags.weight = true; }
	void reset_weight (void);

	// cost
	inline uint get_cost (void) const { return cost; }
	inline void set_cost (uint s_cost) { cost = s_cost; set_flags.cost = true; }
	void reset_cost (void);

	// equip location
	inline BodyPartType get_equip (void) const { return equip; }
	inline void set_equip (BodyPartType s_equip) { equip = s_equip; set_flags.equip = true; }
	void reset_equip (void);

	// object class
	inline ObjectClass* get_class (void) const { return klass; }
	inline void set_class (ObjectClass* s_class) { klass = s_class; set_flags.klass = true; }
	void reset_class (void);
	bool is_class (ObjectClass* klass) const;

	// flags
	inline bool is_hidden (void) const { return flags.hidden; }
	inline void set_hidden (bool v) { flags.hidden = v; set_flags.hidden = true; }
	void reset_hidden (void);
	inline bool is_touchable (void) const { return flags.touchable; }
	inline void set_touchable (bool v) { flags.touchable = v; set_flags.touchable = true; }
	void reset_touchable (void);
	inline bool is_gettable (void) const { return flags.gettable; }
	inline void set_gettable (bool v) { flags.gettable = v; set_flags.gettable = true; }
	void reset_gettable (void);
	inline bool is_dropable (void) const { return flags.dropable; }
	inline void set_dropable (bool v) { flags.dropable = v; set_flags.dropable = true; }
	void reset_dropable (void);

	// load
	int load_node (File::Reader& reader, File::Node& node);
	void save (File::Writer& writer) const;

	// update inherited data
	void update_object_data (void);

	protected:
	uint weight;
	uint cost;
	BodyPartType equip;
	ObjectClass* klass;

	// flags
	struct Flags {
		char hidden:1, touchable:1, gettable:1, dropable:1;
		inline Flags (void) : hidden(false), touchable(true),
			gettable(true), dropable(true) {}
	} flags;

	// set flags
	struct SetFlags {
		int	weight:1,
			cost:1,
			equip:1,
			klass:1,
			hidden:1,
			touchable:1,
			gettable:1,
			dropable:1;
		inline SetFlags (void) : weight(false), cost(false), equip(false),
			klass(false), hidden(false), touchable(false), gettable(false),
			dropable(false) {}
	} set_flags;

	virtual const ObjectData* get_object_data_parent (void) const = 0;
};

// Object blueprint
class
ObjectBlueprint : public EntityBlueprint, public ObjectData
{
	public:
	typedef std::vector<ContainerType> ContainerList;
	typedef GCType::vector<Action*> ActionList;

	ObjectBlueprint (StringArg s_id) : EntityBlueprint(AweMUD_ObjectBlueprintType, s_id), parent(NULL) {}

	// type name
	static const char* get_type_name (void) { return "object"; }
	static const char* get_type_ext (void) { return "objs"; }

	// object properties
	inline const ContainerList& get_containers (void) const { return containers; }
	inline const ActionList& get_actions (void) const { return actions; }

	// load
	virtual void load_init (void);
	virtual int load_node (File::Reader& reader, File::Node& node);
	virtual int load_finish (void);

	// parent blueprint
	virtual ObjectBlueprint* get_parent (void) const { return parent; }

	private:
	ObjectBlueprint* parent;
	
	ContainerList containers;
	ActionList actions;

	void set_parent (ObjectBlueprint* blueprint);

	protected:
	inline virtual const ObjectData* get_object_data_parent (void) const { return parent; }

	SX_TYPEDEF
};
extern BlueprintWrapper<ObjectBlueprint> ObjectBlueprintManager;

// Object control
class
Object : public Entity, public ObjectData
{
	public:
	Object (void);
	Object (ObjectBlueprint* s_blueprint);

	int action (int, Entity *, Entity *);	// source, target
	int action_count (int) const;

	// save/load
	virtual void save (File::Writer& writer) const;
	virtual void load_init (void);
	virtual int load_node (File::Reader& reader, File::Node& node);
	virtual int load_finish (void);

	// weight
	inline uint get_weight (void) const { return calc_weight + get_real_weight(); }
	inline uint get_real_weight (void) const { return ObjectData::get_weight(); }

	// parent tracking - be careful!
	inline class Entity *get_parent (void) const { return parent; }
	inline void set_parent (class Entity *par) { parent = par; }
	virtual class Room* get_room (void) const; // traces up to find a parent room
	class Character* get_owner (void) const; // traces up to find an owning character

	// actions
	Action* get_action (StringArg action);

	// name color
	virtual const char *ncolor (void) const { return CITEM; }

	// for parsing, pull a property based on a char*
	virtual void parse_comm (const char* comm, const class StreamControl& stream) const;

	// remove from parent
	virtual void release (void);

	// (de)activate children
	virtual void activate (void);
	virtual void deactivate (void);

	// update state
	void update (void);

	// blueprint information
	virtual ObjectBlueprint* get_blueprint (void) const { return blueprint; }
	void set_blueprint (ObjectBlueprint* blueprint);
	static Object* load_blueprint (StringArg name);

	// containers
	bool set_container_exist (ContainerType type, bool);
	bool has_container (ContainerType type) const;
	bool add_object (Object *sub, ContainerType type);
	void remove_object (Object *sub, ContainerType type);
	uint get_coins (ContainerType type) const;
	uint take_coins (ContainerType type, uint amount);
	uint give_coins (ContainerType type, uint amount);
	Object *find_object (const char *name, uint index, ContainerType type, uint *matches = NULL) const;
	void show_contents (class Player *player, ContainerType type) const;

	// data
	private:
	uint calc_weight; // calculated weight of children objects
	class Entity *parent; // owner, container, or room
	ObjectBlueprint* blueprint;

	// containers
	class Container : public GCType::Collectable
	{
		public:
		Container(void) : type(ContainerType::IN), list(), coins(0) {}

		ContainerType type;
		EList<Object> list;
		uint coins;
	};
	typedef GCType::vector<Container*> ContainerList;
	ContainerList containers;

	// actions
	typedef GCType::vector<Action*> ActionList;
	ActionList actions;

	// weight tracking
	void recalc_weight (void);

	protected:
	~Object (void);

	inline virtual const ObjectData* get_object_data_parent (void) const { return blueprint; }

	SX_TYPEDEF
	E_TYPE(Object)
};

#define OBJECT(ent) E_CAST(ent,Object)

#endif
