/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#ifndef NETWORK_H 
#define NETWORK_H

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>
#include <netinet/in.h>
#if defined(HAVE_POLL)
#include <sys/poll.h>
#endif
#include <sys/socket.h>

#include "types.h"
#include "awestr.h"

#if !defined(HAVE_SOCKLEN_T)
#define socklen_t int
#endif

#if !defined(HAVE_POLL)
#define POLLIN 1
#define POLLHUP 2
#define POLLNVAL 4
struct pollfd {
	int fd;
	char events;
	char revents;
};
#endif // HAVE_POLL

#include <vector>

// max name length of an address
#ifdef HAVE_IPV6
// check strlen
# ifdef INET6_ADDRSTRLEN
static const uint SOCKADDR_NAME_LEN = INET6_ADDRSTRLEN;
# else // INET6_ADDRSTRLEN
static const uint SOCKADDR_NAME_LEN = 46;
# endif // INET6_ADDRSTRLEN
#else // HAVE_IPV6
// check strlen
# ifdef INET_ADDRSTRLEN
static const uint SOCKADDR_NAME_LEN = INET_ADDRSTRLEN;
# else // INET6_ADDRSTRLEN
static const uint SOCKADDR_NAME_LEN = 16;
# endif // INET6_ADDRSTRLEN
#endif // HAVE_IPV6

// sockaddr_storage workarounds
#ifdef HAVE_SOCKADDR_STORAGE
typedef struct sockaddr_storage SockStorage;
#else // HAVE_SOCKADDR_STORAGE
typedef struct sockaddr_in SockStorage;
#define ss_family sin_family
#endif // HAVE_SOCKADDR_STORAGE

// defaults
static const uint DEFAULT_MAX_HOST_CONNS = 10;
static const uint DEFAULT_MAX_CONNS = 1000;

// socket manager class
class SocketManager {
	private:
	// select() data member
	struct pollfd* fdlist;
	size_t fdlist_len;
	size_t fdlist_size;

	public:
	SocketManager (void);
	~SocketManager (void);
	
	// modify clientlist list
	void add (int sock);
	void remove (int sock);

	// get status/flags of client
	short get_flags (int sock) const;

	// run poll loop
	int poll (long timeout);
};

// setup address type for IPv4 or IPv6
struct IPAddr {
	// ze address
	SockStorage addr;

	IPAddr(void) { memset(&addr, 0, sizeof(addr)); }
	IPAddr(const IPAddr& sa) { memcpy(&addr, &sa.addr, sizeof(addr)); }
	IPAddr(const IPAddr&, uint mask); // apply mask
	IPAddr(const SockStorage&);

	// compare
	bool operator == (const IPAddr&) const;

	// copy
	IPAddr& operator = (const IPAddr&); // copy

	// apply mask
	void apply_mask (uint mask);

	// text representation of address
	char* name_of (char* buffer, size_t len) const;
};

// manage a socket
class Socket {
	private:
	int sock;
	class ISocketServer* server;

	public:
	inline Socket (int s_sock, class ISocketServer* s_server) : sock(s_sock), server(s_server) {}
	inline Socket (void) : sock(-1), server(NULL) {}
	inline Socket (const Socket& s_socket) : sock(s_socket.sock), server(s_socket.server) {}

	Socket (Socket& s_socket);
	Socket& operator= (int s_sock);

	inline bool alive (void) const { return sock >= 0; }

	bool in_ready (void) const;
	bool hung_up (void) const;

	int recv (void* buffer, size_t size);
	int send (const void* buffer, size_t size);

	void close (void);

	// only works on AF_UNIX sockets
	// puts UID in *uid, returns 0 on success,
	// errno value on error
	int get_peer_uid (uid_t* uid) const;

	// only use if you absolutely must!
	inline int get_sock (void) const { return sock; }
	void get_addr (IPAddr& addr) const; // only for IP sockets
};

// manager any client connections
class ISocketServer {
	public:
	// virtual destructor - we need it
	virtual ~ISocketServer (void) {}

	// remove dead/closed clients
	virtual void close (Socket& sock) = 0;

	// accept incoming connections
	virtual int accept (Socket& out_socket) = 0;

	// add a preconnected socket
	virtual int add (int sockfd, Socket& out_socket) = 0;

	// the socket manager
	virtual const SocketManager& get_sockets(void) const = 0;
	virtual SocketManager& get_sockets(void) = 0;
};

// manage TCP client connections
class TCPServer : public ISocketServer {
	public:
	TCPServer (SocketManager& s_sockets) : sockets(s_sockets),
		accept_sock(-1), max_host_conns(DEFAULT_MAX_HOST_CONNS),
		max_conns(DEFAULT_MAX_CONNS), use_ipv6(false)
		{}
	virtual ~TCPServer (void) {
		if (accept_sock >= 0) {
			sockets.remove(accept_sock);
			::close (accept_sock);
		}
	}

	// settings
	inline void set_max_host_conns(uint val) { max_host_conns = val; }
	inline void set_max_total_conns(uint val) { max_conns = val; }
#ifdef HAVE_IPV6
	inline void set_use_ipv6(bool val) { use_ipv6 = val; }
#endif

	// the socket manager
	inline virtual const SocketManager& get_sockets(void) const { return sockets; }
	inline virtual SocketManager& get_sockets(void) { return sockets; }

	// remove dead/closed clients
	virtual void close (Socket& sock);

	// listen/server connection
	int listen (int port);

	// deny list admin
	bool add_deny (const char* item);
	bool remove_deny (const char* item);
	void clear_deny (void);

	// accept incoming connections
	virtual int accept (Socket& out_socket);

	// add a preconnected socket
	virtual int add (int sockfd, Socket& out_socket);

	private:
	// data
	SocketManager& sockets;
	int accept_sock;
	uint max_host_conns;
	uint max_conns;
	bool use_ipv6;

	// track ip connections
	struct IPTrack {
		IPAddr addr;
		uint conns;
	};
	std::vector<IPTrack> connections;
	uint total_conns;

	// track block lists
	struct IPDeny {
		IPAddr addr;
		uint mask;
	};
	std::vector<IPDeny> denylist;

};

// manage client connects
class UNIXServer : public ISocketServer {
	public:
	UNIXServer (SocketManager& s_sockets) : sockets(s_sockets), accept_sock(-1), path() {}
	virtual ~UNIXServer (void);

	// the socket manager
	inline virtual const SocketManager& get_sockets(void) const { return sockets; }
	inline virtual SocketManager& get_sockets(void) { return sockets; }

	// remove dead/closed clients
	virtual void close (Socket& sock);

	// listen/server connection
	int listen (StringArg path);

	// accept incoming connections
	virtual int accept (Socket& out_socket);

	// add a preconnected socket
	virtual int add (int sockfd, Socket& out_socket);

	private:
	// data
	SocketManager& sockets;
	int accept_sock;
	String path;
};

#endif
