/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#ifndef CHAR_H
#define CHAR_H

#include "entity.h"
#include "types.h"
#include "awestr.h"
#include "body.h"
#include "streams.h"

// for get_comm_obj
#define GOC_HELD (1<<0)
#define GOC_WORN (1<<1)
#define GOC_EQUIP (GOC_HELD|GOC_WORN)
#define GOC_FLOOR (1<<2)
#define GOC_SUB (1<<3)
#define GOC_ROOM (GOC_FLOOR|GOC_SUB)
#define GOC_ANY (GOC_ROOM|GOC_EQUIP)

// alignment bounds
#define ALIGN_BOUND 200
#define ALIGN_NEUTRAL 300

// Character alignment
class CharAlign {
	private:
	int align;

	public:
	inline CharAlign (int s_value) : align(s_value) {}
	inline CharAlign (void) : align(0) {}

	inline int get_value (void) const { return align; }

	inline bool is_evil (void) const { return align < -ALIGN_BOUND; }
	inline bool is_good (void) const { return align > ALIGN_BOUND; }
	inline bool is_neutral (void) const { return align > -ALIGN_NEUTRAL && align < ALIGN_NEUTRAL; }
	inline operator int (void) const { return align; }
};

// Character statistic ID
class CharStatID {
	public:
	typedef enum {
		NONE = -1,
		STRENGTH = 0,
		AGILITY,
		FORTITUDE,
		INTELLECT,
		SPIRIT,
		WILLPOWER,
		COUNT,
	} type_t;
	
	inline CharStatID (int s_value) : value((type_t)s_value) {}
	inline CharStatID (void) : value(NONE) {}

	inline StringArg get_name(void) const { return names[value]; }

	inline type_t get_value (void) const { return value; }

	static CharStatID lookup (StringArg name);

	inline bool operator == (CharStatID dir) const { return dir.value == value; }
	inline bool operator != (CharStatID dir) const { return dir.value != value; }
	inline operator bool (void) const { return value != NONE; }

	private:
	type_t value;

	static String names[];
};

// Character Position object
class CharPosition {
	public:
	typedef enum {
		STAND = 0,
		SIT,
		LAY,
		KNEEL,
		COUNT
	} type_t;
	
	inline CharPosition (int s_value) : value((type_t)s_value) {}
	inline CharPosition (void) : value(STAND) {}

	inline StringArg get_name(void) const { return names[value]; }
	inline StringArg get_verb(void) const { return verbs[value]; }
	inline StringArg get_sverb(void) const { return sverbs[value]; }
	inline StringArg get_verbing(void) const { return verbings[value]; }

	inline type_t get_value (void) const { return value; }

	static CharPosition lookup (StringArg name);

	inline bool operator == (CharPosition dir) const { return dir.value == value; }
	inline bool operator != (CharPosition dir) const { return dir.value != value; }

	private:
	type_t value;

	static String names[];
	static String verbs[];
	static String sverbs[]; // sits vs. sit, etc.
	static String verbings[]; // sitting vs. sit, etc.
};

// Character data
class
CharacterData
{
	public:
	CharacterData (void);
	inline virtual ~CharacterData (void) {}

	// gender
	inline GenderType get_gender (void) const { return gender; }
	inline void set_gender (GenderType s_gender) { gender = s_gender; set_flags.gender = true; }
	void reset_gender (void);

	// alignment
	inline CharAlign get_alignment (void) const { return alignment; }
	inline CharAlign set_alignment (int value) { return alignment = CharAlign(value); set_flags.alignment = true; }
	inline CharAlign adjust_alignment (int mod) { return alignment = CharAlign(get_alignment() + mod); set_flags.alignment = true; }
	void reset_alignment (void);

	// update all data
	void update_character_data (void);

	// save and load
	int load_node (File::Reader& reader, File::Node& node);
	void save (File::Writer& writer) const;

	protected:
	GenderType gender;
	CharAlign alignment;

	struct SetFlags {
		int	gender:1,
			alignment:1;
		inline SetFlags(void) : gender(false), alignment(false) {}
	} set_flags;

	virtual const CharacterData* get_character_data_parent (void) const = 0;
};

// Character control
class
Character : public Entity, public IStreamSink, public CharacterData
{
	public:
	Character (const Scriptix::Type* type);

	// save/load
	virtual void save (File::Writer& writer) const;
	virtual int load_node (File::Reader& reader, File::Node& node);
	virtual int load_finish (void);

	// positon
	CharPosition get_pos (void) const { return pos; }
	CharPosition set_pos (CharPosition p) { return pos = p; }

	// health
	inline int get_hp (void) const { return health.cur; }
	inline int set_hp (int new_hp) { return health.cur = new_hp; } // NOTE: avoid use of, only necessary in rare cases
	virtual inline int get_max_hp (void) const { return health.max; }
	inline int set_max_hp (int new_mhp) { return health.max = new_mhp; } // NOTE: avoid use of

	// check data
	inline bool is_dead (void) const { return dead; }

	// stats
	inline void set_stat_base (CharStatID stat, int val) { assert (stat); stats[stat.get_value()].base = val; }
	inline void set_stat_mod (CharStatID stat, int val) { assert (stat); stats[stat.get_value()].mod = val; }
	virtual inline int get_stat_base (CharStatID stat) const { assert (stat); return stats[stat.get_value()].base; }
	inline int get_stat_mod (CharStatID stat) const { assert (stat); return stats[stat.get_value()].mod; }
	inline int get_stat (CharStatID stat) const { assert (stat); return get_stat_mod(stat) + get_stat_base(stat); }

	// combat
	virtual uint get_combat_dodge (void) const = 0; // dodge skill
	virtual uint get_combat_attack (void) const = 0; // attack accuracy
	virtual uint get_combat_damage (void) const = 0; // damage factor

	// equipment
	int hold (class Object*);
	int wear (class Object*);
	int equip (class Object*);

	bool is_held (class Object*) const;
	bool is_worn (class Object*) const;
	bool is_equipped (class Object*) const;

	void drop_held (class Room*);
	void drop_all (class Room*);

	class Object* get_held_at (uint index) const;
	class Object* get_worn_at (uint index) const;
	class Object* get_equip_at (uint index) const;

	class Object* get_held_by_loc (uint loc) const;
	class Object* get_worn_by_loc (uint loc) const;
	class Object* get_equip_by_loc (uint loc) const;

	class Object* find_held (const char* name, uint count = 1, uint* matches = NULL) const;
	class Object* find_worn (const char* name, uint count = 1, uint* matches = NULL) const;
	class Object* find_equip (const char* name, uint count = 1, uint* matches = NULL) const;

	void release_object (class Object*); // *ONLY* for use by Object::release() !!!!

	// hands
	int free_hands (void) const;
	void swap_hands (void);

	// experience
	virtual uint get_level (void) const = 0;

	// currency
	inline uint get_coins (void) const { return coins; }
	inline uint set_coins (uint amount) { return coins = amount; }
	uint give_coins (uint amount);
	uint take_coins (uint amount);

	// health
	void heal (uint amount);
	bool damage (uint amount, Entity *attacker); // returns true if damage killed
	virtual void kill (Entity *killer) = 0;

	// Character abilities
	inline bool can_move (void) const { return !is_dead(); }
	inline bool can_see (void) const { return true; }
	inline bool can_talk (void) const { return true; }
	inline bool can_act (void) const { return round_time == 0 && !is_dead(); }

	// round time; tracked in ticks, add by seconds - yes, ugly
	inline void add_rt (uint seconds) { round_time += seconds * TICKS_PER_SEC; }
	inline uint get_rt (void) const { return round_time; }
	uint get_rts (void) const;

	// action checks w/ error messages
	bool check_alive (void); // must be alive
	bool check_move (void); // can move
	bool check_rt (void); // roundtime has expired
	bool check_see (void); // can see stuff

	// input/output
	virtual void stream_put (const char*, size_t len) {};
	void process_cmd (const char*);

	/* command processing utility funcs
	 * Note: these need to be able to mangle the name argument using the
	 * command::* functions.  They will return the input to its original
	 * state, however, using command::fix_arg()/restore() */
	class Object* cl_find_object (char *name, int type, bool silent = false);
	class Character* cl_find_character (char *name, bool silent = false);
	class RoomExit* cl_find_exit (char *name, bool silent = false);
	/* cl_find_any looks for a character, then an object, then an exit.
	 * Object searching is the same as using cl_find_object w/ GOC_ANY. */
	class Entity* cl_find_any (char *name, bool silent = false);

	// update the Character
	virtual void update (void);

	// must (de)activate equipment
	virtual void activate (void);
	virtual void deactivate (void);

	// Room
	inline void set_room (class Room* new_room) { location = new_room; }
	bool enter (class Room*, class RoomExit *in_exit);
	virtual inline class Room *get_room (void) const { return location; }
	virtual void release (void); // remove from room without deactivation; *ONLY* call from Room character management methods!!!

	// recalculate stuff
	virtual void recalc_stats (void);
	virtual void recalc (void);

	// parsing
	virtual void parse_comm (const char* comm, const class StreamControl& stream) const;

	// output description of character or equipment lsit
	void display_equip (const class StreamControl& stream) const;

	// == ACTIONS ==
	void do_emote (char const *text);
	void do_social (const class SocialAdverb* social, Entity* target);
	void do_say (char const *text);
	void do_sing (char const *text);

	void do_look (void);
	void do_look (Character *who);
	void do_look (const class Object *what, const class ContainerType& how);
	void do_look (class RoomExit *what);

	void do_move (int dir);

	void do_position (CharPosition);

	void do_get (class Object*, class Object*, const class ContainerType&);
	void do_put (class Object*, class Object*, const class ContainerType&);
	void do_give_coins (class Character* target, uint amount);
	void do_drop (class Object*);

	void do_wear (class Object*);
	void do_remove (class Object*);

	void do_read (class Object*);
	void do_kick(class Object*);
	void do_eat (class Object*);
	void do_drink (class Object*);
	void do_raise (class Object*);
	void do_touch (class Object*);

	void do_open (class RoomExit*);
	void do_close (class RoomExit*);
	void do_unlock (class RoomExit*);
	void do_lock (class RoomExit*);
	void do_kick (class RoomExit*);

	int do_attack (class Character*);

	void do_go (class RoomExit*);

	// == DATA ITEMS ==
	protected:
	struct Body {
		class Object* left_held;
		class Object* right_held;
		class Object* body_worn;
		class Object* back_worn;
		class Object* waist_worn;
	} body;
	struct Health {
		int cur;
		int max;
	} health; // hit points
	bool dead; // are we dead?
	CharPosition pos; // standing/stting
	struct CharStat {
		int base;
		int mod;
	} stats[CharStatID::COUNT]; // Character statistics
	uint round_time; // round_time in ticks
	uint coins; // currency
	class Room *location; // room we're in

	protected:
	~Character (void);

	virtual const CharacterData* get_character_data_parent (void) const { return NULL; }

	SX_TYPEDEF
	E_TYPE(Character)
};

// stream out character descriptions
class StreamCharDesc {
	public:
	inline StreamCharDesc (const Character* s_ch) : ch(s_ch) {}

	inline friend const StreamControl&
	operator << (const StreamControl& stream, const StreamCharDesc& desc)
	{
		desc.ch->display_desc(stream);
		return stream;
	}

	private:
	const Character* ch;
};


extern const char *stat_levels[];

const char *get_stat_level (uint);

#define CHARACTER(ent) E_CAST(ent,Character)

#endif
