/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#ifndef TEMPLATE_H
#define TEMPLATE_H

#include "scripts.h"
#include "awestr.h"
#include "fileobj.h"
#include "settings.h"
#include "log.h"
#include "server.h"

#include <dirent.h>

#include <map>
#include <vector>

class BlueprintBase : public Scriptable, public File::IObject
{
	public:
	BlueprintBase (const Scriptix::Type* type, StringArg s_id) : Scriptable(type), id(s_id) {}

	inline StringArg get_id (void) const { return id; }

	virtual void save (File::Writer& writer) const {} // we don't save...

	private:
	String id;
};

template <typename TTYPE>
class BlueprintWrapper : public IManager
{
	typedef GCType::map<String,TTYPE*> BlueprintMap;
	public:
	// get a blueprint
	TTYPE* lookup (StringArg name)
	{
		typename BlueprintMap::iterator iter = blueprints.find(name);
		if (iter == blueprints.end())
			return NULL;
		else
			return iter->second;
	}

	// init/shutdown
	int initialize (void);
	void shutdown (void) { blueprints.clear(); }

	private:
	// cache
	BlueprintMap blueprints;
};

// load all blueprints
template <typename TTYPE>
int
BlueprintWrapper<TTYPE>::initialize (void)
{
	String path = settings::get_path("blueprints", "data");
	if (!path)
		path = "./data";

	Log::Info << "Loading " << TTYPE::get_type_name() << " blueprints";
	
	dirent* d_ent;
	DIR* dir = opendir(path.c_str());
	if (!dir) {
		Log::Error << "Failed to open blueprint folder '" << path << "': " << strerror(errno);
		return -1;
	}
	while ((d_ent = readdir(dir)) != NULL) {
		// match file name
		size_t len = strlen(d_ent->d_name);
		if (len >= 6 && d_ent->d_name[0] != '.' && d_ent->d_name[len - 5] == '.' && str_eq(TTYPE::get_type_ext(), &d_ent->d_name[len - 4])) {
			// load from file
			File::Reader reader(path + "/" + d_ent->d_name);
			File::Node node;
			FO_READ_BEGIN
				FO_OBJECT(TTYPE::get_type_name())
					TTYPE* blueprint = new TTYPE(node.get_name());
					if (!blueprint->load(reader)) {
						blueprints[node.get_name()] = blueprint;
					} else {
						Log::Warning << "Failed to load blueprint '" << node.get_name() << "' from " << reader.get_filename() << " at " << node.get_line();
						return -1;
					}
			FO_READ_ERROR
				return -1;
			FO_READ_END
		}
	}
	closedir(dir);

	return 0;
}

#endif
