/*
 * Copyright (c) 1995, 1996, 1997 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/* $Id: xfs_fs.h,v 1.9.2.1 2001/01/30 02:14:02 lha Exp $ */

#ifndef _xfs_h
#define _xfs_h

#include <xfs/xfs_common.h>
#include <xfs/xfs_node.h>
#include <xfs/xfs_attr.h>

/*
 * Filesystem struct.
 *
 * LOCKS: `nodes_iter' protects so that there are only one user of the
 * next_node struct, so that only one can iterate the list (this can
 * be changed to a list of `next_nodes':s).
 *
 * `nodes_modify' need to be hold when `nodes', `next_node' and
 * `nnodes' are update/read to make sure they are atomicly
 * updated/read.
 */

struct xfs {
    u_int status;			/* Inited, opened or mounted */
#define XFS_MOUNTED	0x1
    struct vfs *vfsp;
    struct xfs_node *root;
    u_int nnodes;
    
    struct xfs_node *nodes;		/* replace with hash table */
    struct xfs_node *next_node;		/* next node in iter */
    kmutex_t nodes_iter;		/* iterating over list */
    kmutex_t nodes_modify;		/* modifing/reading nodes/next_node */
    int fd;
};

#define VFS_TO_XFS(v)      ((struct xfs *) ((v)->vfs_data))
#define XFS_TO_VFS(x)      ((x)->vfsp)

#define XFS_FROM_VNODE(vp) VFS_TO_XFS((vp)->v_vfsp)
#define XFS_FROM_XNODE(xp) XFS_FROM_VNODE(XNODE_TO_VNODE(xp))

extern struct xfs xfs[];

extern struct vnodeops xfs_vnodeops;

int xfs_unloadable (void);
void xfs_init_xfs (int instance);
void xfs_destroy_xfs (int instance);

struct xfs_node *xfs_node_find (struct xfs *, struct xfs_handle *);
struct xfs_node *new_xfs_node (struct xfs *, struct xfs_node *,
			       struct xfs_msg_node *);
void free_xfs_node (struct xfs_node *);
void free_all_xfs_nodes (struct xfs *xfsp);
struct xfs_node *xfs_node_iter_start (struct xfs *xfsp);
struct xfs_node *xfs_node_iter_next (struct xfs *xfsp);
void xfs_node_iter_stop (struct xfs *xfsp);

int 
xfs_dnlc_enter (struct vnode *, char *, struct vnode *);

struct vnode *
xfs_dnlc_lookup (struct vnode *, char *);

void
xfs_dnlc_remove(vnode_t *dvp, char *name);

void
xfs_attr2vattr (const struct xfs_attr *xa, struct vattr *va, int clear_node);

void
vattr2xfs_attr (const struct vattr *va, struct xfs_attr *xa);

int xfs_has_pag(const struct xfs_node *xn, xfs_pag_t);

#endif /* _xfs_h */
