/*
 * Copyright (c) 1995 - 2000 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <arla-version.h>

#include <xfs/xfs_locl.h>
#include <xfs/xfs_message.h>
#include <xfs/xfs_fs.h>
#include <xfs/xfs_dev.h>
#include <xfs/xfs_syscalls.h>
#include <xfs/xfs_deb.h>
#include <xfs/xfs_wrap.h>

RCSID("$Id: xfs_wrap-bsd.c,v 1.33.2.1 2002/09/30 20:38:14 lha Exp $");

int xfs_dev_major;

/*
 * Iff `dev' represents a valid xfs device.
 */

int
xfs_is_xfs_dev (dev_t dev)
{
    return major (dev) == xfs_dev_major
	&& minor(dev) >= 0 && minor(dev) < NXFS;
}

static int
xfs_uninstall(void)
{
    int err, ret = 0;

    if ((err = xfs_uninstall_filesys()) != 0)
	ret = err;
    if ((err = xfs_uninstall_device()) != 0)
	ret = err;
    if ((err = xfs_uninstall_syscalls()) != 0)
	ret = err;

    return ret;
}

static int
xfs_install(void)
{
    int err = 0;

    if ((err = xfs_install_device()) ||
	(err = xfs_install_syscalls()) ||
	(err = xfs_install_filesys()))
	xfs_uninstall();
    return err;
}

static int
xfs_stat(void)
{
    int err = 0;

    if ((err = xfs_stat_device()) != 0)
	return err;
    else if ((err = xfs_stat_syscalls()) != 0)
	return err;
    else if ((err = xfs_stat_filesys()) != 0)
	return err;
    return err;
}

extern struct cdevsw xfs_dev;

/*
 * This is to build a kld module (FreeBSD3.0)
 */

#if KLD_MODULE

static void
make_devices (struct cdevsw *devsw)
{
#ifdef HAVE_KERNEL_MAKE_DEV
    int i;

    for (i = 0; i < NXFS; ++i)
	make_dev (devsw, i, UID_ROOT, GID_WHEEL, 0600, "xfs%d", i);
#endif /* HAVE_KERNEL_MAKE_DEV */
}

static void
remove_devices (struct cdevsw *devsw)
{
#ifdef HAVE_KERNEL_MAKE_DEV
    int i;

    for (i = 0; i < NXFS; ++i)
	destroy_dev (makedev (devsw->d_maj, i));
#endif /* HAVE_KERNEL_MAKE_DEV */
}

/*
 *
 */

static int
xfs_load(struct module *mod, int cmd, void *arg)
{
    int ret;

    XFSDEB(XDEBLKM, ("xfs_load\n"));

    switch (cmd) {
    case MOD_LOAD :
	ret = xfs_install ();
	if (ret == 0) {
	    make_devices (&xfs_dev);
	    xfs_dev_major = xfs_dev.d_maj;
	    printf ("xfs: cdev: %d, syscall: %d\n",
		    xfs_dev_major, xfs_syscall_num);
	}
	break;
    case MOD_UNLOAD :
	ret = xfs_uninstall ();
	if (ret == 0) {
	    remove_devices (&xfs_dev);
	}
	break;
    default :
	ret = EINVAL;
	break;
    }
    XFSDEB(XDEBLKM, ("xfs_load = %d\n", ret));
    return ret;
}

extern struct vfsops xfs_vfsops;

extern struct sysent xfs_syscallent;

VFS_SET(xfs_vfsops, xfs, 0);

/*
 * DEV_MODULE is something that apperad around fbsd4.0+some month
 * at the same time CDEV_MODULE seams to have vanished.
 */

#ifdef DEV_MODULE

DEV_MODULE(xfsdev, xfs_load, NULL);

#else /* DEV_MODULE */

CDEV_MODULE(xfs_dev, NODEV, xfs_dev, xfs_load, NULL);

#endif /* DECLARE_MODULE */

#ifdef SYSCALL_MODULE
#define AFS_SYSCALL 210 /* XXX */
int xfs_syscall_num = AFS_SYSCALL;
SYSCALL_MODULE(xfs_syscall, &xfs_syscall_num, &xfs_syscallent, NULL, NULL);
#endif /* SYSCALL_MODULE */

#else /* KLD_MODULE */

/*
 * An ordinary lkm-module
 */

#if defined(__FreeBSD__) && __FreeBSD_version < 300000
static struct lkm_dev _module = {
    LM_DEV,
    LKM_VERSION,
    "xfs_mod",
    -1,
    LM_DT_CHAR,
    {(void *)&xfs_dev}
};
#elif defined(__FreeBSD__)
MOD_DEV(xfs,LM_DT_CHAR,-1,&xfs_dev);
#else
#if __NetBSD_Version__ >= 106080000
MOD_DEV("xfs_mod","xfs_mod", NULL, -1, &xfs_dev, -1)
#elif !defined(__APPLE__)
MOD_DEV("xfs_mod",LM_DT_CHAR,-1,&xfs_dev)
#endif
#endif

/*
 *
 */

#if defined(__APPLE__)
__private_extern__ kern_return_t
xfs_modload(kmod_info_t *ki, void *data)
#else
static int
xfs_modload(struct lkm_table *lkmtp, int cmd)
#endif
{
    int error = 0;

    XFSDEB(XDEBLKM, ("xfs_modload\n"));

    error = xfs_install();
    if (error == 0)
	xfs_stat();
    return error;
}


/*
 *
 */

#if defined(__APPLE__)
__private_extern__ kern_return_t
xfs_modunload(kmod_info_t *ki, void *data)
#else
static int
xfs_modunload(struct lkm_table * lkmtp, int cmd)
#endif
{
    int error = 0;

    XFSDEB(XDEBLKM, ("xfs_modunload\n"));

    error = xfs_uninstall();
    if (!error)
	XFSDEB(XDEBLKM, ("xfs_modunload: successful\n"));
    else
	XFSDEB(XDEBLKM, ("xfs_modunload: unsuccessful, system unstable\n"));
    return error;
}

/*
 *
 */

#if !defined(__APPLE__)
static int
xfs_modstat(struct lkm_table * lkmtp, int cmd)
{
    int error = 0;

    XFSDEB(XDEBLKM, ("xfs_modstat\n"));

    error = xfs_stat();
    return error;
}

int
xfs_mod(struct lkm_table * lkmtp, int cmd, int ver);

int
xfs_mod(struct lkm_table * lkmtp,
	int cmd,
	int ver)
{
    int ret;
    if (ver != LKM_VERSION)						
	return EINVAL;	/* version mismatch */			
    switch (cmd) {							
	int	error;							
    case LKM_E_LOAD:
	lkmtp->private.lkm_any = (struct lkm_any *)&_module;
	if ((error = xfs_modload(lkmtp, cmd)) != 0)
	    return error;
	break;
    case LKM_E_UNLOAD:
	if ((error = xfs_modunload(lkmtp, cmd)) != 0)
	    return error;
	break;
    case LKM_E_STAT:
	if ((error = xfs_modstat(lkmtp, cmd)) != 0)
	    return error;
	break;
    }
    ret = lkmdispatch(lkmtp, cmd);
    if(cmd == LKM_E_LOAD) {
#if __NetBSD_Version__ >= 106080000
	xfs_dev_major = _module.lkm_cdevmaj;
#else
 	xfs_dev_major = _module.lkm_offset;
#endif
	printf ("xfs (%s): "
		"using cdev: %d, syscall: %d\n",
		arla_version,
		xfs_dev_major, xfs_syscall_num);
    }
    return ret;
}
#endif
    
#endif /* KLD_MODULE */

