/*
 * Copyright (c) 1995 - 2001 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * XFS operations.
 */

#include <xfs/xfs_locl.h>
#include <vm/vm_mmap.h>
#include <xfs/xfs_message.h>
#include <xfs/xfs_common.h>
#include <xfs/xfs_fs.h>
#include <xfs/xfs_dev.h>
#include <xfs/xfs_deb.h>
#include <xfs/xfs_syscalls.h>
#include <xfs/xfs_vnodeops.h>

#include <sys/buf.h>

RCSID("$Id: xfs_vnodeops-osf.c,v 1.23.2.5 2002/04/29 17:28:24 mattiasa Exp $");

/*
 * vnode functions
 */

static int
xfs_open(struct vnode **vpp, int mode, struct ucred *cred)
{
    return xfs_open_common (*vpp, mode, cred, xfs_curproc());
}

static int
xfs_fsync(struct vnode *vp, int fflags, struct ucred *cred, int waitfor)
{
    int error=0;
    if (vp->v_type == VREG) {
	error = ubc_flush_dirty(vp->v_object, 
				waitfor & MNT_WAIT ? B_DONE : B_ASYNC);
	if(error)
	    return error;
    }

    error = xfs_fsync_common(vp, cred, waitfor, xfs_curproc());

    return error;
}

static int
xfs_close(struct vnode *vp, int fflags, struct ucred *cred)
{
    return xfs_close_common(vp, fflags, xfs_curproc(), cred);
}

static int
xfs_read(struct vnode *vp, struct uio *uio, int ioflag, struct ucred *cred)
{
    return xfs_read_common(vp, uio, ioflag, cred);
}

static int
xfs_write(struct vnode *vp, struct uio *uiop, int ioflag, struct ucred *cred)
{
    return xfs_write_common(vp, uiop, ioflag, cred);
}

static int
xfs_ioctl(struct vnode *vp, 
	  int com, 
	  caddr_t data, 
	  int fflag, 
	  struct ucred *cred)
{
    XFSDEB(XDEBVNOPS, ("xfs_ioctl\n"));
    return EOPNOTSUPP;
}

static int
xfs_getattr(struct vnode *vp, struct vattr *vap, struct ucred *cred)
{
    return xfs_getattr_common(vp, vap, cred, xfs_curproc ());
}

static int
xfs_setattr(struct vnode *vp, struct vattr *vap, struct ucred *cred)
{
    return xfs_setattr_common (vp, vap, cred, xfs_curproc ());
}

static int
xfs_access(struct vnode *vp, int mode, struct ucred *cred)
{
    return xfs_access_common(vp, mode, cred, xfs_curproc());
}

static int
xfs_lookup(struct vnode *dvp, struct nameidata *ndp)
{
    ndp->ni_dvp = dvp;
    return xfs_lookup_common(dvp, ndp, &ndp->ni_vp);
}

static int
xfs_create(struct nameidata *ndp, struct vattr *vap)
{
    struct vnode *dvp  = ndp->ni_dvp;
    const char *name   = ndp->ni_dent.d_name;
    struct ucred *cred = ndp->ni_cred;
    int error;

    XFSDEB(XDEBVNOPS, ("xfs_create: vap->va_mode = %o\n",
		       vap->va_mode));

    error = xfs_create_common(dvp,
			      name,
			      vap,
			      cred,
			      xfs_curproc());
    if(error == 0) {
	ndp->cn_nameptr = ndp->ni_dent.d_name;
	error = xfs_lookup_common (dvp, ndp, &ndp->ni_vp);
    }
    vrele(dvp);
    XFSDEB(XDEBVNOPS, ("xfs_create: error = %d\n", error));
    
    return error;
}

static int
xfs_remove(struct nameidata *ndp)
{
    int ret = xfs_remove_common(ndp->ni_dvp, 
			     ndp->ni_vp, 
			     ndp->ni_dent.d_name, 
			     ndp->ni_cred,
			     xfs_curproc());
    vrele(ndp->ni_dvp);
    vrele(ndp->ni_vp);
    return ret;
}

static int
xfs_rename(struct nameidata *from, struct nameidata *to)
{
    int error =  xfs_rename_common(from->ni_dvp, 
				   from->ni_vp, 
				   from->ni_dent.d_name, 
				   to->ni_dvp, 
				   to->ni_vp, 
				   to->ni_dent.d_name, 
				   from->ni_cred,
				   xfs_curproc());
    
    vrele(to->ni_dvp);
    if (to->ni_vp)
	vrele(to->ni_vp);
    vrele(from->ni_dvp);
    vrele(from->ni_vp);
    
    return error;
}

static int
xfs_mkdir(struct nameidata *ndp,
	  struct vattr *vap)
{
    struct vnode *dvp  = ndp->ni_dvp;
    const char *name   = ndp->ni_dent.d_name;
    struct ucred *cred = ndp->ni_cred;
    int error;

    error = xfs_mkdir_common(dvp,
			     name,
			     vap,
			     cred,
			     xfs_curproc());

    if(error == 0) {
	ndp->cn_nameptr = ndp->ni_dent.d_name;
	error = xfs_lookup_common (dvp, ndp, &ndp->ni_vp);
    }
    vrele(dvp);
    XFSDEB(XDEBVNOPS, ("xfs_mkdir: error = %d\n", error));
    
    return error;
}

static int
xfs_rmdir(struct nameidata *ndp)
{
    int ret = xfs_rmdir_common(ndp->ni_dvp, 
			    ndp->ni_vp, 
			    ndp->ni_dent.d_name, 
			    ndp->ni_cred,
			    xfs_curproc());
    vrele(ndp->ni_dvp);
    vrele(ndp->ni_vp);
    return ret;
}

static int
xfs_readdir(struct vnode *vp, 
	    struct uio *uio, 
	    struct ucred *cred, 
	    int *eofflagp)
{
    return xfs_readdir_common(vp, uio, cred, xfs_curproc(), eofflagp);
}

static int
xfs_link(struct vnode* vp, struct nameidata *ndp)
{
    int ret = xfs_link_common(ndp->ni_dvp, vp, ndp->ni_dent.d_name,
			   ndp->ni_cred, xfs_curproc());
    vrele(ndp->ni_dvp);
    return ret;
}

static int
xfs_symlink(struct nameidata *ndp, struct vattr *vap, char *target)
{
    int ret;

    ndp->cn_nameptr = ndp->ni_dent.d_name;
    ret = xfs_symlink_common(ndp->ni_dvp, &ndp->ni_vp,
			      ndp, vap, target);
    vrele(ndp->ni_dvp);
    return ret;
}

static int
xfs_readlink(struct vnode *vp, struct uio *uiop, struct ucred *cred)
{
    return xfs_readlink_common (vp, uiop, cred);
}

static int
xfs_inactive(struct vnode *vp)
{
    return xfs_inactive_common(vp, xfs_curproc());
}

static int
xfs_reclaim(struct vnode *vp)
{
    return xfs_reclaim_common (vp);
}

static int
xfs_lock(struct vnode *vp, int flag, struct ucred *cred)
{
    XFSDEB(XDEBVNOPS, ("xfs_lock(%lx, %d, %lx)\n",
		       (unsigned long)vp, flag,
		       (unsigned long)cred));
    return EOPNOTSUPP;
}

static int
xfs_unlock(struct vnode *vp, int flag, struct ucred *cred)
{
    XFSDEB(XDEBVNOPS, ("xfs_unlock(%lx, %d, %lx)\n",
		       (unsigned long)vp, flag,
		       (unsigned long)cred));
    return EOPNOTSUPP;
}

static int
xfs_abortop(struct nameidata *ndp)
{
    XFSDEB(XDEBVNOPS, ("xfs_abortop(%s)\n", ndp->ni_dirp));
    return 0;
}

static int
xfs_mmap(struct vnode *vp,
	 vm_offset_t offset,
	 vm_map_t map,
	 vm_offset_t *addrp,
	 vm_size_t len,
	 vm_prot_t prot,
	 vm_prot_t maxprot,
	 int flags,
	 struct ucred *cred)
{
    struct vp_mmap_args args;
    extern kern_return_t u_vp_create(vm_map_t, 
				     struct vm_ubc_object *, 
				     vm_offset_t);
    
    XFSDEB(XDEBVNOPS, ("xfs_mmap (%lx, %lx, %lx, %lx, %lx, %x, %x, %d, %lx)\n",
		       (unsigned long)vp, offset, map, addrp,
		       len, prot, maxprot, flags,
		       (unsigned long)cred));
    
    args.a_offset = offset;
    args.a_vaddr = addrp;
    args.a_size = len;
    args.a_prot = prot;
    args.a_maxprot = maxprot;
    args.a_flags = flags;
    return u_vp_create(map, vp->v_object, (vm_offset_t) &args);
}

static int
xfs_bmap(struct vnode *vp, 
	 daddr_t bn, 
	 struct vnode **vpp, 
	 daddr_t *bnp)
{
    XFSDEB(XDEBVNOPS, ("xfs_bmap\n"));
    return EOPNOTSUPP;
}

/*
 *      common routine for unsupported vnode ops 
 */
int
xfs_unsupp()
{
    return ENOSYS;
}

int
xfs_getproplist (struct vnode *vp, char **names, struct uio *uio, 
		 int *min_buf_size, int mask, struct ucred *cred)
{
    XFSDEB(XDEBVNOPS, ("xfs_getproplist(%lx, %lx, %lx, %lx, %d, %lx)\n",
		       (unsigned long)vp,
		       (unsigned long)names,
		       (unsigned long)uio, min_buf_size, mask,
		       (unsigned long)cred));
    return EOPNOTSUPP;
}

int
xfs_setproplist (struct vnode *vp, struct uio *uio, struct ucred *cred)
{
    XFSDEB(XDEBVNOPS, ("xfs_setproplist(%lx, %lx, %lx)\n",
		       (unsigned long)vp,
		       (unsigned long)uio,
		       (unsigned long)cred));
    return EOPNOTSUPP;
}

int
xfs_delproplist (struct vnode *vp, char **names, int mask, struct ucred *cred)
{
    XFSDEB(XDEBVNOPS, ("xfs_delproplist(%lx, %lx, %d, %lx)\n",
		       (unsigned long)vp,
		       (unsigned long)names, mask,
		       (unsigned long)cred));
    return EOPNOTSUPP;
}


int
xfs_ok()
{
    return 0;
}

struct vnodeops xfs_vnodeops = {
    xfs_lookup,		/* lookup */
    xfs_create,		/* create */
    xfs_unsupp,		/* mknod */
    xfs_open,		/* open */
    xfs_close,		/* close */
    xfs_access,		/* access */
    xfs_getattr,	/* getattr */
    xfs_setattr,	/* setattr */
    xfs_read,		/* read */
    xfs_write,		/* write */
    xfs_ioctl,		/* ioctl */
    xfs_unsupp,		/* select */
    xfs_mmap,		/* mmap */
    xfs_fsync,		/* fsync */
    xfs_ok,		/* seek */
    xfs_remove,		/* remove */
    xfs_link,		/* link */
    xfs_rename,		/* rename */
    xfs_mkdir,		/* mkdir */
    xfs_rmdir,		/* rmdir */
    xfs_symlink,	/* symlink */
    xfs_readdir,	/* readdir */
    xfs_readlink,	/* readlink */
    xfs_ok,		/* abortop */
    xfs_inactive,	/* inactive */
    xfs_reclaim,	/* reclaim */
    xfs_unsupp,		/* bmap */
    xfs_unsupp,		/* strategy */
    xfs_unsupp,		/* print */
    xfs_unsupp,		/* page_read */
    xfs_unsupp,		/* page_write */
    xfs_unsupp,		/* swap handler */
    xfs_unsupp,		/* buffer read */
    xfs_unsupp,		/* buffer release */
    xfs_unsupp,		/* file locking */
    xfs_unsupp,		/* fsync byte range */
    xfs_lock,		/* Lock a node */
    xfs_unlock,		/* Unlock a node */
    xfs_getproplist,	/* Get extended attributes */
    xfs_setproplist,	/* Set extended attributes */
    xfs_delproplist,	/* Delete extended attributes */
    vn_pathconf,		/* pathconf */		
};

int
xfs_refer(vm_ubc_object_t vop)
{
    XFSDEB(XDEBVNOPS, ("xfs_refer: vop = %lx, vp = %lx\n",
		       (unsigned long)vop,
		       (unsigned long)vop->vu_vp));
    VREF(vop->vu_vp);
    return 0;
}

int
xfs_release(vm_ubc_object_t vop)
{
    XFSDEB(XDEBVNOPS, ("xfs_release: vop = %lx, vp = %lx\n",
		       (unsigned long)vop,
		       (unsigned long)vop->vu_vp));
    vrele(vop->vu_vp);
    return 0;
}

#ifdef HAVE_SIX_ARGUMENT_UBC_LOOKUP
int
xfs_getpage(vm_ubc_object_t vop,
	    vm_offset_t offset,
	    vm_size_t len,
	    vm_prot_t *protp,
	    vm_page_t *pl,
	    int plsz,
	    vm_map_entry_t ep,
	    vm_offset_t addr,
	    int rwflg,
	    struct ucred *cred)
#else
/* OSF5 */
int
xfs_getpage(vm_ubc_object_t vop,
	    vm_offset_t offset,
	    vm_size_t len,
	    vm_prot_t *protp,
	    vm_page_t *pl,
	    int plsz,
	    struct vm_policy *vmp,
	    int rwflg,
	    struct ucred *cred)
#endif
{
    int error;
    struct vnode *vp = vop->vu_vp;
    struct vnode *t = DATA_FROM_VNODE(vp);

    XFSDEB(XDEBVNOPS, ("xfs_getpage: enter\n"));

    error = xfs_data_valid(vp, cred, xfs_curproc(), XFS_DATA_R);

    XFSDEB(XDEBVNOPS, ("xfs_getpage = %d\n", error));

    if (error == 0) {
#ifdef HAVE_SIX_ARGUMENT_UBC_LOOKUP
	FSOP_GETPAGE(t->v_object,offset,len,protp,pl,plsz,ep,
		     addr,rwflg,cred,error);
#else
	FSOP_GETPAGE(t->v_object,offset,len,protp,pl,plsz,
		     vmp,rwflg,cred,error);
#endif
    }

    XFSDEB(XDEBVNOPS, ("xfs_getpage = %d\n", error));
    return error;
}

int
xfs_putpage(vm_ubc_object_t vop,
	    vm_page_t *pl,
	    int pcnt,
	    int flags,
	    struct ucred *cred)
{
    int error;
    struct vnode *vp = vop->vu_vp;
    struct vnode *t = DATA_FROM_VNODE(vp);

    XFSDEB(XDEBVNOPS, ("xfs_putpage: enter\n"));

    error = xfs_data_valid(vp, cred, xfs_curproc(), XFS_DATA_W);

    XFSDEB(XDEBVNOPS, ("xfs_putpage = %d\n", error));

    if (error == 0)
	FSOP_PUTPAGE(t->v_object,pl,pcnt,flags,cred,error);

    XFSDEB(XDEBVNOPS, ("xfs_putpage = %d\n", error));
    return error;
}

int
xfs_write_check(vm_ubc_object_t vpp, vm_page_t pp)
{
    XFSDEB(XDEBVNOPS, ("xfs_write_check: vpp = %lx, vp = %lx, pp = %lx\n", 
		       (unsigned long)vpp,
		       (unsigned long)vpp->vu_vp,
		       (unsigned long)pp));
    return TRUE;
}

struct vfs_ubcops xfs_ubcops = {
    xfs_refer,              /* refer vnode */
    xfs_release,            /* release vnode */
    xfs_getpage,            /* get page */
    xfs_putpage,            /* put page */
    xfs_write_check         /* check writability */
};
