/*
 * Copyright (c) 1995 - 2000 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <xfs/xfs_locl.h>

RCSID("$Id: xfs_vfsops-netbsd.c,v 1.18.2.1 2001/12/20 16:53:22 lha Exp $");

#include <xfs/xfs_common.h>
#include <xfs/xfs_message.h>
#include <xfs/xfs_fs.h>
#include <xfs/xfs_dev.h>
#include <xfs/xfs_deb.h>
#include <xfs/xfs_vfsops.h>
#include <xfs/xfs_vfsops-bsd.h>
#include <xfs/xfs_vnodeops.h>
#include <miscfs/genfs/genfs.h>

static vop_t **xfs_dead_vnodeop_p;

int
make_dead_vnode(struct mount *mp, struct vnode **vpp)
{
    XFSDEB(XDEBNODE, ("make_dead_vnode mp = %lx\n",
		      (unsigned long)mp));

    return getnewvnode(VT_NON, mp, xfs_dead_vnodeop_p, vpp);
}

static struct vnodeopv_entry_desc xfs_dead_vnodeop_entries[] = {
    {&vop_default_desc, (vop_t *) xfs_eopnotsupp},
    {&vop_lookup_desc,	(vop_t *) xfs_dead_lookup},
    {&vop_reclaim_desc, (vop_t *) xfs_returnzero},
    {&vop_lock_desc,	(vop_t *) genfs_nolock},
    {&vop_unlock_desc,	(vop_t *) genfs_nounlock},
    {&vop_islocked_desc,(vop_t *) genfs_noislocked},
    {NULL, NULL}};

static struct vnodeopv_desc xfs_dead_vnodeop_opv_desc =
{&xfs_dead_vnodeop_p, xfs_dead_vnodeop_entries};

extern struct vnodeopv_desc xfs_vnodeop_opv_desc;

struct vnodeopv_desc *xfs_vnodeopv_descs[] = {
    &xfs_vnodeop_opv_desc,
    NULL,
};

struct vnodeopv_desc *xfs_dead_vnodeopv_descs[] = {
    &xfs_dead_vnodeop_opv_desc,
    NULL
};


/*
 * Provide prototypes for vfs_opv_init_{explicit,default}
 * so we dont need to shot our head of more times then nessary
 */

#ifndef HAVE_STRUCT_VFSOPS_VFS_OPV_DESCS
void vfs_opv_init_explicit __P((struct vnodeopv_desc *));
void vfs_opv_init_default __P((struct vnodeopv_desc *));
#endif

/*
 * If the vfs_opv_descs wasn't included in `struct vfsops' it couldn't
 * get initialized by vfs_attach and we need to do it here.
 */


static void
xfs_init(void)
{
    XFSDEB(XDEBVFOPS, ("xfs_init\n"));
#ifndef HAVE_STRUCT_VFSOPS_VFS_OPV_DESCS
    vfs_opv_init_explicit(&xfs_vnodeop_opv_desc);
    vfs_opv_init_default(&xfs_vnodeop_opv_desc);
    vfs_opv_init_explicit(&xfs_dead_vnodeop_opv_desc);
    vfs_opv_init_default(&xfs_dead_vnodeop_opv_desc);
#else
    vfs_opv_init (xfs_dead_vnodeopv_descs);
#endif
}

#ifdef HAVE_STRUCT_VFSOPS_VFS_REINIT
static void
xfs_reinit(void)
{
    XFSDEB(XDEBVFOPS, ("xfs_reinit\n"));
}
#endif

#ifdef HAVE_STRUCT_VFSOPS_VFS_DONE
static void
xfs_done(void)
{
    XFSDEB(XDEBVFOPS, ("xfs_done\n"));
}
#endif

static struct vfsops
xfs_vfsops = {
    "xfs",
    xfs_mount,
    xfs_start,
    xfs_unmount,
    xfs_root,
    xfs_quotactl,
    xfs_statfs,
    xfs_sync,
    xfs_vget,
    xfs_fhtovp,
    xfs_vptofh,
    xfs_init,
#ifdef HAVE_STRUCT_VFSOPS_VFS_REINIT /* NetBSD 1.5Y */
    xfs_reinit,
#endif
#ifdef HAVE_STRUCT_VFSOPS_VFS_DONE
    xfs_done,
#endif
    NULL,			/* sysctl */
    NULL,			/* mountroot */
#ifdef HAVE_STRUCT_VFSOPS_VFS_CHECKEXP
    xfs_checkexp,		/* checkexp */
#endif
#ifdef HAVE_STRUCT_VFSOPS_VFS_OPV_DESCS
    xfs_vnodeopv_descs
#endif
};

#ifndef HAVE_KERNEL_VFS_ATTACH

int
vfs_attach (struct vfsops *ops)
{
    int i;

    for (i = 0; i < nvfssw; ++i) 
	if (vfssw[i] != NULL
	    && strcmp (vfssw[i]->vfs_name, ops->vfs_name) == 0)
	    return EEXIST;

    for (i = nvfssw - 1; i >= 0; i--)
	if (vfssw[i] == NULL)
	    break;
    if (i < 0)
	return EINVAL;

    vfssw[i] = ops;
    vfssw[i]->vfs_refcount = 0;

    if (vfssw[i]->vfs_init != NULL)
	(*(vfssw[i]->vfs_init)) ();

    return 0;
}

int
vfs_detach (struct vfsops *ops)
{
    int i;

    if (ops->vfs_refcount != 0)
	return EBUSY;

    for (i = 0; i < nvfssw; ++i)
	if (vfssw[i] == ops)
	    break;

    if (i == nvfssw)
	return ENOENT;

    vfssw[i] = NULL;
    return 0;
}

#endif /* HAVE_VFS_ATTACH */

int
xfs_install_filesys(void)
{
    return vfs_attach(&xfs_vfsops);
}

int
xfs_uninstall_filesys(void)
{
    return vfs_detach(&xfs_vfsops);
}

int
xfs_stat_filesys (void)
{
    return 0;
}
