/*
 * Copyright (c) 2001 - 2002 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "themis.h"
#include "macosx.h"

RCSID("$Id: themis.c,v 1.101.2.3 2002/12/02 03:40:17 lha Exp $");

static char *output_file = NULL;
static int help_flag = 0;
static int verbose_level = 0;
static int chown_flag = 1;
static int check_owner_flag = 1;
static int would_reboot = 0;
static int progress_flag = 0;
static int macosx_flag = 0;

static int nnodes = 0;
static int nupdated = 0;
static int exitcode = 0;

#define FALSE 0

#define BUFSIZE 1024
#define NARGS 20

#define EXITCODE_REBOOT 4

#define FLAG_ABS 0x01
#define FLAG_CLEAN 0x02
#define FLAG_REBOOT 0x04
#define FLAG_NOOVERWRITE 0x10
#define FLAG_RENAMEOLD 0x08
#define FLAG_PERMISSIONCOPY 0x20

#define FLAGS_FILE (FLAG_REBOOT|FLAG_NOOVERWRITE|FLAG_RENAMEOLD|FLAG_PERMISSIONCOPY)

enum filetype {
    TYPE_NONE,
    TYPE_B,
    TYPE_C,
    TYPE_D,
    TYPE_F,
    TYPE_L,
    TYPE_M,
    TYPE_N,
    TYPE_P,
    TYPE_T,
    TYPE_X
};

struct treenode;

struct traversal {
    int lineno;
    int typeflags;
    struct treenode *src;
    struct traversal *next;
};

struct srcnode {
    struct stat *stat;
    int stat_error;
    struct macosx_metadata metadata;
    int metadatap;
};

struct fsnode {
    struct traversal *firsttraversal;
    enum filetype type;
    int typeflags;
    struct treenode *src;
    char *linkdata;
    uid_t uid;
    gid_t gid;
    mode_t mode;
    int modevalidp;
    int uidvalidp;
    int gidvalidp;
    struct macosx_metadata metadata;
    int metadatap;
    int major;
    int minor;
};

struct treenode {
    union {
	struct fsnode *fs;
	struct srcnode *src;
    } data;
    struct treenode *firstchild;
    struct treenode *nextsibling;
    struct treenode *parent;
    char *name;
};

struct treenode_func_node {
    struct treenode_func_node *next;
    int (*func) (struct treenode *, void *arg);
    void *arg;
};

static int
parseflags(char * flagstring, int lineno)
{
    char * ptr;
    
    int flags = 0;

    assert(flagstring);

    ptr = flagstring;

    while(*ptr) {

	switch(*ptr) {
	case 'A':
	    flags |= FLAG_ABS;
	    break;
	case 'I':
	    flags |= FLAG_NOOVERWRITE;
	    break;
	case 'O':
	    flags |= FLAG_RENAMEOLD;
	    break;
	case 'Q':
	    flags |= FLAG_REBOOT;
	    break;
	case 'R':
	    flags |= FLAG_CLEAN;
	    break;
	case 'P':
	    flags |= FLAG_PERMISSIONCOPY;
	    break;
	case 'X':
	    /* We don't handle this */
	    break;
	default:
	    errx(1,"Unknown flag: %c at line %d", *ptr, lineno);
	}
	ptr++;
    }

    return flags;
}

static int
char2type(char * typestring) {
    switch(typestring[0]) {
    case 'B':
	return TYPE_B;
    case 'C':
	return TYPE_C;
    case 'D':
	return TYPE_D;
    case 'F':
	return TYPE_F;
    case 'L':
	return TYPE_L;
    case 'M':
	return TYPE_M;
    case 'N':
	return TYPE_N;
    case 'P':
	return TYPE_P;
    case 'T':
	return TYPE_T;
    case 'X':
	return TYPE_X;
    default:
	errx(1,"Unknown type: %c", typestring[0]);
    }
}

static char
type2char(enum filetype type) {
    switch (type) {
    case TYPE_B:
	return 'B';
    case TYPE_C:
	return 'C';
    case TYPE_D:
	return 'D';
    case TYPE_F:
	return 'F';
    case TYPE_L:
	return 'L';
    case TYPE_M:
	return 'M';
    case TYPE_N:
	return 'N';
    case TYPE_P:
	return 'P';
    case TYPE_T:
	return 'T';
    case TYPE_X:
	return 'X';
    default:
	errx(1,"Unknown type: %d", type);
    }
}

static void
typeflags2string(int typeflags, char *str) {
    int i=0;
    if (typeflags & FLAG_ABS)
	str[i++] = 'A';
    if (typeflags & FLAG_CLEAN)
	str[i++] = 'R';
    if (typeflags & FLAG_REBOOT)
	str[i++] = 'Q';
    if (typeflags & FLAG_NOOVERWRITE)
	str[i++] = 'I';
    if (typeflags & FLAG_RENAMEOLD)
	str[i++] = 'O';
    if (typeflags & FLAG_PERMISSIONCOPY)
	str[i++] = 'P';
    str[i] = '\0';
}

static int
change_flags(const char *path)
{
    if (!update_flag)
	return 0;

#if defined(HAVE_LCHFLAGS)
    return lchflags(path, 0);
#elif defined(HAVE_CHFLAGS)
    {
	struct stat sb;
	if (lstat(path, &sb) < 0)
	    err(1, "lstat: %s", path);
	if (S_ISLNK(sb.st_mode))
	    return 0;
    }
    return chflags(path, 0);
#else
    return 0;
#endif
}

static struct treenode *
newtreenode(struct treenode *parent, const char *name, int len)
{
    struct treenode *node;

    node = malloc(sizeof(struct treenode));
    assert(node);

    memset(node,'\0',sizeof(struct treenode));

    node->parent = parent;
    node->name = malloc(len+1);
    assert(node->name);
    
    memcpy(node->name,name,len);
    node->name[len] = '\0';

    node->firstchild = NULL;
    node->nextsibling = NULL;

    return node;
}

static struct treenode *
lookup_nocreate(struct treenode *root, const char *name, int len)
{
    struct treenode *child;

    child = root->firstchild;

    while(child) {
	if (strncmp(child->name, name, len) == 0 &&
	    child->name[len] == '\0')
	    return child;
	child = child->nextsibling;
    }

    return child;
}

/* 
 * Takes a directory node and a name of an entry in that
 * directory. Returns the node refering to that name. If the
 * node doesn't exist, it is created.
 */

static struct treenode *
lookup(struct treenode *root, const char *name, int len)
{
    struct treenode *child;

    child = lookup_nocreate(root, name, len);

    if (child)
	return child;

    child = newtreenode(root,name,len);
    nnodes++;
    child->nextsibling = root->firstchild;
    root->firstchild = child;

    return child;
}

/*
 * This is a function which takes the node of the root of a tree and a
 * pathname relative to that root as arguments, and returns the node
 * representing the last element of the path
 */

static struct treenode *
namei(struct treenode *root, const char *name)
{
    int len;
    char *next;
    struct treenode *nextnode;

    next = strchr(name, '/');

    if(next) {
	len = next - name;

	while(*(next+1) == '/')
	    next++;

	nextnode = lookup(root, name, len);
	return namei(nextnode, next+1);
    } else {
	len = strlen(name);
	if(len == 0)
	    return root;
	else 
	    return lookup(root, name, len);
    }
}

/* 
 * Returns the full path to a given node. The path is returned in
 * argument "name".
 */

void
getnodename(struct treenode *node, char *name, int *pos)
{
    int len;

    /* XXX no length checking */

    if (node->parent == NULL) {
	name[(*pos)] = '/';
	name[(*pos)+1] = '\0';
    } else {
	getnodename(node->parent, name, pos);
	len = strlen(node->name);
	name[(*pos)++] = '/';
	memcpy(name+(*pos), node->name, len);
	(*pos) += len;
	name[*pos] = '\0';
    }
}

/*
 * Makes sure the stat-information for the node is up to date. If the
 * argument follow_link is set, then an stat() is done, otherwise an
 * lstat.
 */

static void
getsrcstatinfo(struct treenode *node, int follow_link)
{
    char name[BUFSIZE];
    int pos;
    struct stat *sb;
    int ret;

    if (node->data.src->stat_error ||
	node->data.src->stat)
	return;

    pos = 0;
    getnodename(node, name, &pos);

    sb = malloc(sizeof(struct stat));
    assert(sb);

    if (follow_link)
	ret = stat(name, sb);
    else
	ret = lstat(name, sb);
    if (ret == -1) {
	node->data.src->stat_error = errno;
	free(sb);
	return;
    }
    node->data.src->stat = sb;

    if (macosx_flag && node->data.src->metadatap == 0) {
	ret = macosx_get_metadata(name, &node->data.src->metadata, 0);
	if (ret == 0)
	    node->data.src->metadatap = 1;
    }
}

enum tokenstate {
    TOKSTAT_NORM,
    TOKSTAT_QUOTE,
    TOKSTAT_BS,
    TOKSTAT_BSQUOTE,
    TOKSTAT_STOP,
    TOKSTAT_START,
    TOKSTAT_COMMENT
};

/*
 * Takes a line and returns is as an array of tokens.
 */

static int
tokenize(char *line, char **args, int nargs, int *retargs)
{
    int i = 0;
    int len = 0;
    int curarg = 0;
    int destpos = 0;
    char * arg;
    enum tokenstate state = TOKSTAT_START;

    *retargs = 0;

    len = strlen(line);

    arg = malloc(len*sizeof(char));
    assert(arg);

    for(i=0; i < len; i++) {
	switch(state) {
	case TOKSTAT_START:
	case TOKSTAT_NORM:
	    if(line[i] == '"') {
		state = TOKSTAT_QUOTE;
	    } else if(line[i] == '\\') {
		state = TOKSTAT_BS;
	    } else if(isspace((unsigned char)line[i])) {
		if(state != TOKSTAT_START)
		    state = TOKSTAT_STOP;
	    } else if(line[i] == '#') {
		if(state == TOKSTAT_START)
		    goto out;
		state = TOKSTAT_COMMENT;
	    } else {
		arg[destpos++] = line[i];
		state = TOKSTAT_NORM;
	    }
	    break;
	case TOKSTAT_QUOTE:
	    if(line[i] == '"') {
		state = TOKSTAT_NORM;
	    } else if(line[i] == '\\') {
		state = TOKSTAT_BSQUOTE;
	    } else {
		arg[destpos++] = line[i];
	    } 
	    break;
	case TOKSTAT_BS:
	    arg[destpos++] = line[i];
	    state = TOKSTAT_NORM;
	    break;
	case TOKSTAT_BSQUOTE:
	    arg[destpos++] = line[i];
	    state = TOKSTAT_QUOTE;
	    break;
	default:
	    errx(1,"Internal error, unknown state.");
	}
	if(state == TOKSTAT_COMMENT)
	    break;

	if(state == TOKSTAT_STOP) {
	    assert(curarg < nargs);

	    arg[destpos] = '\0';
	    args[curarg] = strdup(arg);
	    assert(args[curarg]);

	    curarg++;

	    destpos = 0;
	    state = TOKSTAT_START;
	}
    }

    if(state == TOKSTAT_COMMENT || state == TOKSTAT_NORM) {
	assert(curarg < nargs);
	
	arg[destpos] = '\0';
	args[curarg] = strdup(arg);
	assert(args[curarg]);
	curarg++;
    }

 out:
    assert(state == TOKSTAT_COMMENT || 
	   state == TOKSTAT_NORM || 
	   state == TOKSTAT_START);

    assert(curarg < nargs);

    args[curarg] = NULL;
    *retargs = curarg;
    return 0;
}

/*
 * Wrapper funtion for namei, returns the node refered to by path
 */

static struct treenode *
gettreenode(struct treenode *root, char *path)
{
    assert(path);

    if (path[0] != '/')
	return NULL;

    if(path[1] == '\0') {
	return root;
    } else {
	return namei(root,path+1);
    }
}

/*
 * Returns (and possibly allocates) an fsnode given a treenode.
 */

static struct treenode *
getfsnode(struct treenode *treenode)
{
    if (treenode == NULL)
	return NULL;

    if(!treenode->data.fs) {
	treenode->data.fs = malloc(sizeof(struct fsnode));
	assert(treenode->data.fs);

	treenode->data.fs->firsttraversal = NULL;
	treenode->data.fs->type = TYPE_NONE;
	treenode->data.fs->typeflags = 0;
	treenode->data.fs->src = NULL;
	treenode->data.fs->linkdata = NULL;
	treenode->data.fs->uid = 0;
	treenode->data.fs->uidvalidp = 0;
	treenode->data.fs->gid = 0;
	treenode->data.fs->gidvalidp = 0;
	treenode->data.fs->mode = 0;
	treenode->data.fs->modevalidp = 0;
	treenode->data.fs->metadatap = 0;
    }

    return treenode;
}

/*
 * Returns (and possibly allocates) an srcnode given a treenode.
 */

static struct treenode *
getsrcnode(struct treenode *treenode) 
{
    if (treenode == NULL)
	return NULL;

    if(!treenode->data.src) {
	treenode->data.src = malloc(sizeof(struct srcnode));
	assert(treenode->data.src);

	treenode->data.src->stat = NULL;
	treenode->data.src->stat_error = 0;
	treenode->data.src->metadatap = 0;
    }

    return treenode;
}

static void
fssettype(struct treenode *node, enum filetype type) 
{
    node->data.fs->type = type;
}

static void
fssettypeflags(struct treenode *node, int typeflags) 
{
    node->data.fs->typeflags = typeflags;
}

static void
fssetsrc(struct treenode *node, struct treenode *src) 
{
    node->data.fs->src = src;
}

static void
fssetlinkdata(struct treenode *node, char *path) 
{
    node->data.fs->linkdata = path;
}

static void
fssetuid(struct treenode *node, char * owner) 
{
    struct passwd * pwd;
    assert(owner);
    pwd = getpwnam(owner);
    assert(pwd);
    node->data.fs->uid = pwd->pw_uid;
    node->data.fs->uidvalidp = 1;
}

static void
fssetgroup(struct treenode *node, char * group) 
{
    struct group *grp;
    assert(group);
    if (strcmp(group, "wheel") == 0) {
	node->data.fs->gid = 0;
	node->data.fs->gidvalidp = 1;
    } else {
	grp = getgrnam(group);
	assert(grp);
	node->data.fs->gid = grp->gr_gid;
	node->data.fs->gidvalidp = 1;
    }
}

static void
fssetmode(struct treenode *node, char * mode) 
{
    int octmode;
    octmode = strtol(mode,NULL,8);
    node->data.fs->mode = octmode;
    node->data.fs->modevalidp = 1;
}

static void
fsclearogm(struct treenode *node)
{
    node->data.fs->uid = 0;
    node->data.fs->gid = 0;
    node->data.fs->mode = 0;
    node->data.fs->uidvalidp = 0;
    node->data.fs->gidvalidp = 0;
    node->data.fs->modevalidp = 0;
    node->data.fs->major = -1;
    node->data.fs->minor = -1;
}

/*
 * Set the major and minor for the node
 */

static void
fssetdev(struct treenode *node, char **args)
{
    assert(node->data.fs->type == TYPE_B || node->data.fs->type == TYPE_C);
    node->data.fs->major = strtol(args[0], NULL, 0);
    node->data.fs->minor = strtol(args[1], NULL, 0);
}

/*
 * Parsefunction for owner, group and mode. Takes an array of
 * owner,group,mode.  They can come in basicly any order, as log as an
 * owner is alway followed by a mode. 
 */

static int
fssetogm(struct treenode *node, char **args)
{
    char *ptr = NULL;
    int modeindex = -1;
    int ownerindex = -1;
    int groupindex = -1;

    assert(args);

    if(!args[0])
	return 0;

    strtol(args[0],&ptr,8);
    if(!*ptr) {
	modeindex = 0;
	if(args[1]) {
	    ownerindex=1;
	    if(args[2]) 
		groupindex=2;
	    else
		return EINVAL;
	}
    } else {
	ownerindex = 0;
	if(args[1])
	    groupindex = 1;
	else
	    return EINVAL;
	if(args[2]) {
	    ptr=NULL;
	    strtol(args[2],&ptr,8);
	    if(!*ptr)
		modeindex = 2;
	    else
		return EINVAL;
	}
    }
    if(modeindex != -1)
	fssetmode(node,args[modeindex]);

    if(ownerindex != -1)
	fssetuid(node, args[ownerindex]);
	
    if(groupindex != -1)
	fssetgroup(node,args[groupindex]);

    return 0;
}

/*
 * Reads the directory represented by node and adds it to dest and
 * srctree
 */

static int
srcreaddir_func(const char *path, const char *name, void *ptr)
{
    struct treenode *node = ptr;
    struct treenode *child;
    
    child = lookup(node, name, strlen(name));
    getsrcnode(child);

    return 0;
}


static void
srcreaddir(struct treenode *node)
{
    struct stat *sb;
    int pos;
    char dirname[MAXPATHLEN];

    /* XXX if (node->data.src->readdir_done) return; */
    getsrcstatinfo(node, 0);
    pos = 0;
    getnodename(node, dirname, &pos);
    if (node->data.src->stat_error) {
	errx(1, "couldn't read directory %s", dirname);
	return;
    }
    sb = node->data.src->stat;

    assert(S_ISDIR(sb->st_mode));

    themis_readdir(dirname, THEMIS_READDIR_EXCLUDE, node,
		   srcreaddir_func);
}

/*
 * Adds a traversal object to the node dest.
 */

static void
addtraversal(struct treenode *dest, int lineno,
	     int typeflags, struct treenode *src)
{
    struct traversal *traversal;

    traversal = malloc(sizeof(struct traversal));
    assert(traversal);

    traversal->lineno = lineno;
    traversal->typeflags = typeflags;
    traversal->src = src;
    traversal->next = dest->data.fs->firsttraversal;

    dest->data.fs->firsttraversal = traversal;
}

/*
 * Removes (frees) any children of the node.
 */

static void
removechildren(struct treenode *node)
{
    struct traversal *traversal;
    struct traversal *next;
    struct treenode *child;

    traversal = node->data.fs->firsttraversal;

    while (traversal) {
	next = traversal->next;
	free(traversal);
	traversal = next;
    }
    node->data.fs->firsttraversal = NULL;
    while (node->firstchild) {
	child = node->firstchild;
	removechildren(child);
	node->firstchild = node->firstchild->nextsibling;
	nnodes--;
	free(child);
    }
}

static void
create_path(struct treenode *node)
{
    if (node->parent) {
	create_path(node->parent);
	getfsnode(node->parent);
	/* XXX What to do if we have:
	 *  F /foo /blah
	 *  F /foo/bar /blah
	 * Below, /foo becomes a dir.
	 */
	if (node->parent->data.fs->type == TYPE_NONE ||
	    node->parent->data.fs->type == TYPE_F ||
	    node->parent->data.fs->type == TYPE_L ||
	    node->parent->data.fs->type == TYPE_B ||
	    node->parent->data.fs->type == TYPE_C ||
	    node->parent->data.fs->type == TYPE_P) {
	    node->parent->data.fs->type = TYPE_D;
	}
	if (node->parent->data.fs->type == TYPE_N) {
	    node->parent->data.fs->type = TYPE_D;
	    node->parent->data.fs->typeflags = node->parent->data.fs->typeflags & (~(FLAG_CLEAN));
	}
	if (node->parent->data.fs->type == TYPE_X) {
	    node->parent->data.fs->type = TYPE_D;
	    node->parent->data.fs->typeflags |= FLAG_CLEAN;
	}
	/* printf("%s %c\n", node->parent->name, type2char(node->parent->data.fs->type)); */
	assert(node->parent->data.fs->type == TYPE_D);
    }
}

/*
 * Begins by reading the directory (on disk) and then loops trough all
 * children and updates their state.
 */

static void
traverse(struct treenode *dest, struct traversal *traversal)
{
    struct treenode *src = traversal->src;
    struct treenode *childsrc;
    struct treenode *childfs;
    mode_t mode;

    srcreaddir(src);
    childsrc = src->firstchild;
    while (childsrc) {
	int typeflags = 0;
	getsrcstatinfo(childsrc, 0);
	if (childsrc->data.src->stat_error) {
	    char srcpath[MAXPATHLEN];
	    int pos = 0;
	    getnodename(childsrc,srcpath,&pos);
	    errx(1, "couldn't stat file %s: %s", srcpath, 
		 strerror(childsrc->data.src->stat_error));
	}

	childfs = lookup(dest, childsrc->name, strlen(childsrc->name));
	getfsnode(childfs);
	assert(childsrc->data.src->stat);
	mode = childsrc->data.src->stat->st_mode;
	fsclearogm(childfs);
	if (S_ISDIR(mode)) {
	    addtraversal(childfs, traversal->lineno,
			 traversal->typeflags, childsrc);
	    fssettype(childfs, TYPE_D);
	    typeflags |= traversal->typeflags & FLAG_CLEAN;
	    if (macosx_flag) {
		childfs->data.fs->metadatap = childsrc->data.src->metadatap;
		childfs->data.fs->metadata = childsrc->data.src->metadata;
	    }
	} else if (S_ISLNK(mode)) {
	    fssettype(childfs, TYPE_L);
	    typeflags |= traversal->typeflags & FLAGS_FILE;
	    typeflags |= FLAG_ABS;
	    removechildren(childfs);
	} else if (S_ISREG(mode)) {
	    fssettype(childfs, TYPE_F);
	    typeflags |= traversal->typeflags & FLAGS_FILE;
	    typeflags |= FLAG_ABS;
	    removechildren(childfs);
	}
	fssettypeflags(childfs, typeflags);
	/* Get rid of old data because linkdata
	   has higher prio than src. */
	fssetlinkdata(childfs, NULL);
	fssetsrc(childfs, childsrc);
	childsrc = childsrc->nextsibling;
    }
}

/* 
 * + Reverse the list of traversals.
 * + Iterate over the traversals and expand them.
 */

static void
expandtraversal(struct treenode *node)
{
    struct traversal *stack=NULL;
    struct traversal *remember=NULL;

    if (node->data.fs == NULL)
	return;

    /* First we need to reverse the stack */
    while (node->data.fs->firsttraversal) {
	remember=node->data.fs->firsttraversal;
	/* pop */
	node->data.fs->firsttraversal = node->data.fs->firsttraversal->next;

	/* push */
	remember->next = stack;
	stack = remember;
    }

    while(stack) {
	traverse(node, stack);
	stack = stack->next;

	/* XXX Free mem */
    }
}

static void
expandtraversalpath(struct treenode *node) {
    if (node->parent)
	expandtraversalpath(node->parent);
    expandtraversal(node);
}

static char *
absolute_srcpath(char *dest, char *src, struct treenode *node)
{
    char *newsrc;
    int srclen, destlen;

    if (node->data.fs->typeflags & FLAG_ABS) {
	newsrc = strdup(src);
	assert(newsrc);
    } else {
	node->data.fs->typeflags |= FLAG_ABS;
	srclen  = strlen(src);
	destlen = strlen(dest);
	newsrc = malloc(srclen + destlen + 2);
	assert(newsrc);
	
	memcpy(newsrc,src,srclen);

	memcpy(newsrc+srclen,dest,destlen);
	newsrc[srclen+destlen] = '\0';
    }

    return newsrc;
}


/*
 * Take a tokenvector and process it. Sets most of the internal
 * information in the node and adds it to the tree.
 */


static int
process(int nargs, char ** args, struct treenode *dest, 
	struct treenode *src, int lineno)
{
    int flags;
    struct treenode *node, *tmpnode;
    struct treenode * srcnode;
    enum filetype type;
    char * newsrc;

    if (args[0] == NULL)
	return 0;
    
    if (strcmp(args[0], "exclude") == 0) {
	add_to_exclude_list(args[1]);
	return 0;
    }

    assert(args[0][0]);

    type = char2type(args[0]);
    flags = parseflags(args[0]+1, lineno);
    node = gettreenode(dest,args[1]);
    if (node == NULL)
	errx(1, "invalid path %s at line %d", args[1], lineno);

    tmpnode = node;
    while (tmpnode) {
	if (!(tmpnode->data.fs)) {
	    getfsnode(tmpnode);
	}
	tmpnode = tmpnode->parent;
    }

    expandtraversalpath(node);

    if (type == TYPE_M) {
	fssettypeflags(node, flags);
	if(args[2]) {
	    fssetogm(node, args+2);
	}
	return 0;
    }

    fssettype(node, type);
    fssettypeflags(node, flags);
    fsclearogm(node);
    fssetsrc(node, NULL);
    fssetlinkdata(node, NULL);

    switch(type) {
    case TYPE_F:
    case TYPE_T:
	create_path(node);
	if (args[2] == NULL) {
	    errx(1, "invalid line %d: %c node requires a source path",
		 lineno, args[0][0]);
	}
	newsrc = absolute_srcpath(args[1], args[2], node);

	srcnode = getsrcnode(gettreenode(src,newsrc));
	if (srcnode == NULL)
	    errx(1, "invalid path %s at line %d", newsrc, lineno);

	fssetsrc(node, srcnode);

	if(args[3]) {
	    fssetogm(node, args+3);
	}

	if (type == TYPE_T) {
	    addtraversal(node, lineno, flags, srcnode);
	    type = TYPE_D;
	    fssettype(node, type);
	}

	break;

    case TYPE_L:
	create_path(node);
	if (nargs < 2)
	    errx(1, "two few arguments for L at line %d", lineno);
	    
	assert(args[2]);
	newsrc = absolute_srcpath(args[1], args[2], node);
	fssetlinkdata(node, newsrc);

	break;

    case TYPE_D:
	create_path(node);
	if(args[2]) {
	    fssetogm(node, args+2);
	}
	break;

    case TYPE_N:
    case TYPE_X:
	/* I don't think this can happen. */
	assert (node->parent->data.fs != NULL);
	/* Nothing to be done */
	break;

    case TYPE_B:
    case TYPE_C:
	create_path(node);
	
	if (nargs < 7)
	    errx(1, "to few arguments to %c at line %d", 
		 type2char(type), lineno);

	fssetogm(node, args+2);
	fssetdev(node, args+5);	

	break;
    case TYPE_P:
	/* Ignored */
	break;

    default:
	errx(1,"Internal error, incorrect type %d.", type);
    }
    if (type != TYPE_D)
	removechildren(node);

    return 0;
}

/*
 * Iterate over the tree and call func once for each node.
 */


static int
tree_iter(struct treenode *node,
	  int (*prefunc) (struct treenode *, void *arg),
	  void *prearg,
	  int (*postfunc) (struct treenode *, void *arg),
	  void *postarg)
{
    struct treenode *child;
    int error;

    if (node == NULL)
	return 0;

    if (prefunc) {
	error = (*prefunc)(node, prearg);
	if (error) {
	    return error;
	}
    }
    
    child = node->firstchild;
    while (child) {
	error = tree_iter(child, prefunc, prearg, postfunc, postarg);
	if (error)
	    return error;
	child = child->nextsibling;
    }

    if (postfunc) {
	error = (*postfunc)(node, postarg);
	if (error) {
	    return error;
	}
    }
    
    return error;
}


/*
 * Check if a string might need to be quoted.
 *  Should be replaced with a print-string-maybe-quoted().
 */

static int
needs_quotes(char *string) {
    int i;
    char c;

    for(i=0; (c=string[i]) != '\0'; i++) {
	if (!(isalnum((int)c) || isspace((int)c) || c=='_' || c=='-' || c=='/' || c=='+' || c=='.'))
	    return 1;
    }
    
    return 0;
}

/*
 * Print the node, with package syntax.
 */

static int
package_print(struct treenode *node, void *arg)
{
    char flagstring[BUFSIZE];
    char dstname[BUFSIZE], srcname[BUFSIZE];
    char linkdata[BUFSIZE];
    char *srcstring;
    int pos;
    FILE *outfile;
    int flags;
    int ret;
    int quote_dst, quote_src;

    srcstring = NULL;

    outfile = (FILE *)arg;

    expandtraversal(node);

    getfsnode(node);

    if (node->data.fs->type == TYPE_NONE)
	return 0;

    pos = 0;
    getnodename(node, dstname, &pos);

    flags = node->data.fs->typeflags;

    if(node->data.fs->type != TYPE_D) {
	struct treenode *src = node->data.fs->src;

	if (node->data.fs->type == TYPE_F) {
	    pos = 0;
	    getnodename(src, srcname, &pos);
	    srcstring = srcname;
	}
	if (node->data.fs->type == TYPE_L) {
	    assert(flags & FLAG_ABS);
	    if (flags & FLAG_ABS) {
		if (node->data.fs->linkdata) {
		    /* Have absolute link data. Great. */
		    strncpy(linkdata, node->data.fs->linkdata, sizeof(linkdata)-1);
		    srcstring = linkdata;
		} else if (src) {
		    /* Can easily get absolute link data. Ok. */
		    /* XXX Should break this out into a function. */
		    /* XXX Should set linkdata when this is done. */
		    /* XXX Should probably have a linkdata in srcnodes. */
		    pos = 0;
		    getnodename(src,srcname,&pos);
		    ret = readlink(srcname, linkdata, sizeof(linkdata) - 1);
		    if (ret == -1) {
			errx(1, "%s: cannot read link", srcname);
		    }
		    linkdata[ret] = '\0';
		    srcstring = linkdata;
		}
#if 0
	    } else {
		if (node->data.fs->linkdata) {
		    /* Real link data is $linkdata/$srcname */
		    /* Can pass pointer without copying.
		       Data will not be changed. */
		    srcstring = node->data.fs->linkdata;
		} else if (src) {
		    /* I refuse to interpret this. This is wrong! */
		    errx(1,"Internal error, inconsistent state.");
		}
#endif
	    }
	}

	/* This block is not really meaningful, but
	   it makes the output look more like the output
	   of Transarc package. If you remove this, you
	   can use 'name' instead of 'dstname' and 'srcname'. */
	if ((node->data.fs->type == TYPE_F ||
	     node->data.fs->type == TYPE_L) &&
	    flags & FLAG_ABS) {
	    if (strcmp(&srcstring[strlen(srcstring)-strlen(dstname)],dstname) == 0) {
		srcstring[strlen(srcstring)-strlen(dstname)] = '\0';
		flags &= ~FLAG_ABS;
	    }
	}
    }

    quote_dst = needs_quotes(dstname);

    if (srcstring == NULL)
	flags &= ~FLAG_ABS;

    typeflags2string(flags, flagstring);

    fprintf(outfile, "%c%s\t%s%s%s",
	    type2char(node->data.fs->type),
	    flagstring,
	    quote_dst ? "\"" : "", dstname, quote_dst ? "\"" : "");

    if (srcstring) {
	quote_src = needs_quotes(srcstring);
	fprintf(outfile, "\t%s%s%s",
		quote_src ? "\"" : "", srcstring, quote_src ? "\"" : "");
    }

    if (node->data.fs->type == TYPE_D ||
	node->data.fs->type == TYPE_F ||
	node->data.fs->type == TYPE_L) {
	struct passwd *p;
	struct group *g;
	p = getpwuid(node->data.fs->uid);
	if (p == NULL) {
	    err(1, "getpwgid %d", (int)node->data.fs->uid);
	}
	g = getgrgid(node->data.fs->gid);
	if (g == NULL) {
	    err(1, "getgrgid %d", (int)node->data.fs->gid);
	}
	fprintf(outfile, "\t%s\t%s\t%o",
		p->pw_name, g->gr_name,
		(int)node->data.fs->mode);
    }

    fprintf(outfile, "\n");

    return 0;
}

static int
lstat_rel(const char *path, struct stat *buf)
{
    char *realpath;
    int ret=0;

    asprintf(&realpath, "%s/%s", fs_root ? fs_root : "", path);
    ret = lstat(realpath, buf);
    free(realpath);
    return ret;
}

static int
utime_rel(const char *path, const struct utimbuf *buf)
{
    char *realpath;
    int ret=0;
    struct tm tm;
    char timestr[BUFSIZE];

    asprintf(&realpath, "%s/%s", fs_root ? fs_root : "", path);

    localtime_r(&buf->modtime, &tm);
    strftime(timestr, sizeof(timestr), "%Y-%m-%d %H:%M:%S", &tm);

    if (verbose_level > 0)
	printf("utime %s %s\n", timestr, realpath);

    if(update_flag)
	ret = utime(realpath, buf);

    free(realpath);
    return ret;
}

static int
chmod_rel(const char *path, mode_t mode)
{
    char *realpath;
    int ret=0;

    asprintf(&realpath, "%s/%s", fs_root ? fs_root : "", path);

    if (verbose_level > 0)
	printf("chmod %o %s\n", (int)mode, realpath);

    if(update_flag)
	ret = chmod(realpath, mode);

    free(realpath);
    return ret;
}

static int
chown_rel(const char *path, uid_t owner, gid_t group)
{
    char *realpath;
    int ret=0;

    asprintf(&realpath, "%s/%s", fs_root ? fs_root : "", path);

    if (verbose_level > 0 && owner != -1)
	printf("chown %d %s\n", (int)owner, realpath);

    if (verbose_level > 0 && group != -1)
	printf("chgrp %d %s\n", (int)group, realpath);

    if(update_flag)
	ret = chown(realpath, owner, group);

    free(realpath);
    return ret;
}

static int
mkdir_rel(const char *path, mode_t mode)
{
    char *realpath;
    int ret=0;

    asprintf(&realpath, "%s/%s", fs_root ? fs_root : "", path);

    if (verbose_level > 0)
	printf("mkdir %s\n", realpath);

    if(update_flag)
	ret = mkdir(realpath, mode);

    free(realpath);
    return ret;
}

static int
symlink_rel(const char *name, const char *path)
{
    char *realpath;
    int ret=0;

    asprintf(&realpath, "%s/%s", fs_root ? fs_root : "", path);

    if (verbose_level > 0)
	printf("ln -s %s %s\n", name, realpath);

    if(update_flag)
	ret = symlink(name, realpath);

    free(realpath);
    return ret;
}

static int
rename_rel(const char *old, const char *new)
{
    char *realold;
    char *realnew;
    int ret=0;

    asprintf(&realold, "%s/%s", fs_root ? fs_root : "", old);
    asprintf(&realnew, "%s/%s", fs_root ? fs_root : "", new);

    if (verbose_level > 0)
	printf("mv %s %s\n", realold, realnew);

    if(update_flag)
	ret = change_flags(realnew);
    if(ret == 0 && update_flag)
	ret = rename(realold, realnew);
    if (ret != 0)
	unlink(realold);

    free(realold);
    free(realnew);
    return ret;
}

/*
 * Opens the file and returns a descriptor to it. Note, this function
 * doesn't check update_flag.
 */

static int
open_rel(const char *path, int oflag, mode_t mode)
{
    char *realpath;
    int ret;

    asprintf(&realpath, "%s/%s", fs_root ? fs_root : "", path);
    ret = open(realpath, oflag, mode);
    free(realpath);
    return ret;
}


static int
unlink_rel(const char *path)
{
    char *realpath;
    int ret = 0;

    asprintf(&realpath, "%s/%s", fs_root ? fs_root : "", path);

    if (verbose_level > 0)
	printf("rm %s\n", realpath);

    if(update_flag)
	ret = change_flags(realpath);
    if(ret == 0 && update_flag)
	ret = unlink(realpath);

    free(realpath);
    return ret;
}


static int
rmdir_rel(const char *path)
{
    char *realpath;
    int ret=0;

    asprintf(&realpath, "%s/%s", fs_root ? fs_root : "", path);

    if (verbose_level > 0)
	printf("rmdir %s\n", realpath);

    if(update_flag)
	ret = rmdir(realpath);

    free(realpath);
    return ret;
}


static int
readlink_rel(const char *path, char *buf, size_t bufsiz)
{
    char *realpath;
    int ret;

    asprintf(&realpath, "%s/%s", fs_root ? fs_root : "", path);
    ret = readlink(realpath, buf, bufsiz - 1);
    free(realpath);
    if (ret > 0)
	buf[ret] = '\0';
    return ret;
}

static int
samefile(struct treenode *node, struct stat *sbdest, struct stat *sbsrc,
	 char *destpath, char *srcpath)
{
    struct macosx_metadata src_metadata;
    struct macosx_metadata dest_metadata;
    int ret;

    assert(sbdest);
    assert(sbsrc);
    assert(node);

    if (sbdest->st_size != sbsrc->st_size) {
	if (verbose_level > 2)
	    printf("size differs: dest %lu bytes, source %lu bytes\n",
		   (unsigned long) sbdest->st_size,
		   (unsigned long) sbsrc->st_size);
	return 0;
    }

    if (macosx_flag) {
	ret = macosx_get_metadata(destpath, &dest_metadata, 1);
	if (ret)
	    errx(1, "macosx_get_metadata %s: %s", destpath, strerror(ret));

	ret = macosx_get_metadata(srcpath, &src_metadata, 0);
	if (ret)
	    errx(1, "macosx_get_metadata %s: %s", srcpath, strerror(ret));

	if (dest_metadata.rsrc_len != src_metadata.rsrc_len) {
	    if (verbose_level > 2)
		printf("size of resource fork differs\n");
	    return 0;
	}
	if (dest_metadata.type != src_metadata.type) {
	    if (verbose_level > 2)
		printf("type differs\n");
	    return 0;
	}
	if (dest_metadata.creator != src_metadata.creator) {
	    if (verbose_level > 2)
		printf("creator differs\n");
	    return 0;
	}
	if (dest_metadata.flags != src_metadata.flags) {
	    if (verbose_level > 2)
		printf("flags differs\n");
	    return 0;
	}
    }

    if (sbdest->st_mtime != sbsrc->st_mtime) {
	struct tm tm_dest;
	struct tm tm_src;
	char time_dest[BUFSIZE];
	char time_src[BUFSIZE];

	if (verbose_level > 2) {
	    localtime_r(&sbdest->st_mtime, &tm_dest);
	    localtime_r(&sbsrc->st_mtime, &tm_src);
	    if (strftime(time_dest, sizeof(time_dest), "%Y-%m-%d %H:%M:%S",
			 &tm_dest) == 0)
		errx(1, "strftime: time string too large");
	    if (strftime(time_src, sizeof(time_src), "%Y-%m-%d %H:%M:%S",
			 &tm_src) == 0)
		errx(1, "strftime: time string too large");
	    printf("time differs: dest modified %s, source modified %s\n",
		   time_dest, time_src);
	}

	return 0;
    }

    if (check_owner_flag &&
	(sbdest->st_uid != node->data.fs->uid ||
	 sbdest->st_gid != node->data.fs->gid))
	return 0;

    if (sbdest->st_mode != (node->data.fs->mode | S_IFREG))
	return 0;

    return 1;
}

static int
samelink(struct stat *sbdest, char *name, char *linksrc)
{
    char linkdest[BUFSIZE];
    int ret;

    assert(sbdest);

    if (!S_ISLNK(sbdest->st_mode))
	return 0;

    ret = readlink_rel(name, linkdest, sizeof(linkdest));
    if (ret == -1) {
	err(1, "%s: cannot read link", name);
	return 0;
    }

    if(strcmp(linkdest, linksrc) != 0) {
	if(verbose_level > 2)
	    printf("link differs: dest %s, source %s\n",
		   linkdest, linksrc);
	return FALSE;
    }

    return TRUE;
}

/*
 * Removes a file on disk. If it is a directory, all its children are
 * removed too.
 */

static void
removefile(char *path, struct stat *sb);

static int
removefile_func(const char *path, const char *name, void *ptr)
{
    char childpath[MAXPATHLEN];

    snprintf(childpath, sizeof(childpath), "%s/%s",
	     path, name);
    removefile(childpath, NULL);
    return 0;
}

static void
removefile(char *path, struct stat *sb)
{
    int isdir;
    int ret;

    if (sb == NULL) {
	struct stat sb1;

	ret = lstat_rel(path, &sb1);
	if (ret == -1)
	    err(1, "removefile couldn't find %s", path);
	isdir = S_ISDIR(sb1.st_mode);
    } else {
	isdir = S_ISDIR(sb->st_mode);
    }

    if (isdir) {
	themis_readdir(path, THEMIS_READDIR_RELATIVE, 
		       NULL, removefile_func);

	ret = rmdir_rel(path);
	if (ret == -1)
	    err(1, "rmdir %s", path);
    } else {
	ret = unlink_rel(path);
	if (ret == -1)
	    err(1, "unlink %s", path);
    }
}

static void
copyfile_real(char *destname, char *srcname, mode_t mode)
{
    int dest;
    int src;
    void *data;
    struct stat sb;
    int ret;

    if (verbose_level > 0)
	printf("copy %s %s\n", srcname, destname);

    if (!update_flag)
	return;

    dest = open_rel(destname, O_CREAT|O_EXCL|O_TRUNC|O_WRONLY, mode);
    if (dest == -1)
	err(1, "open %s", destname);

    src = open(srcname, O_RDONLY, mode);
    if (src == -1)
	err(1, "open %s", srcname);

    ret = fstat(src, &sb);
    if (ret == -1)
	err(1, "fstat %s", srcname);

    if (sb.st_size) {
	data = mmap(NULL, sb.st_size, PROT_READ, MAP_SHARED, src, 0);
	if (data == MAP_FAILED)
	    err(1, "mmap %s", srcname);
	
	ret = write(dest, data, sb.st_size);
	if (ret == -1)
	    err(1, "write %s", destname);
	
	if (ret != sb.st_size)
	    errx(1, "short write %s(%d bytes)", destname, ret);
	
	ret = munmap(data, sb.st_size);
	if (ret == -1)
	err(1, "munmap %s", srcname);
    }

    close(dest);
    close(src);
}

static void
copyfile(char *dest, char *src, struct stat *sb, mode_t mode)
{
    char newpath[MAXPATHLEN];

    if (sb == NULL) {
	copyfile_real(dest, src, mode);
	if (macosx_flag)
	    copyfile_macosx(dest, src);
    } else if (S_ISDIR(sb->st_mode)) {
	removefile(dest, sb);
	copyfile_real(dest, src, mode);
	if (macosx_flag)
	    copyfile_macosx(dest, src);
    } else {
	snprintf(newpath, sizeof(newpath), "%s.new", dest);
	copyfile_real(newpath, src, mode);
	if (macosx_flag)
	    copyfile_macosx(newpath, src);
	if (rename_rel(newpath, dest) == -1)
	    err(1, "rename %s %s", newpath, dest);
    }
}

static void
createlink(char *dest, char *src)
{
    if (symlink_rel(src, dest) == -1)
	err(1, "symlink %s %s", src, dest);
}

static void
makedir(char *dest, mode_t mode)
{
    if (mkdir_rel(dest, mode) == -1)
	err(1, "mkdir %s", dest);
}

/*
 * Update owner group and mode information about a file on disk.
 */

static void
setogm(char *dest, struct stat *sbdest, uid_t uid, gid_t gid, mode_t mode)
{
    int ret;
    int update_uid;
    int update_gid;
    int update_mode;

    update_uid = (sbdest == NULL) || (sbdest->st_uid != uid);
    update_gid = (sbdest == NULL) || (sbdest->st_gid != gid);
    update_mode = (sbdest == NULL) || ((sbdest->st_mode & 07777) != mode);

    if (check_owner_flag && chown_flag) {
	if (update_uid || update_gid) {
	    ret = chown_rel(dest, 
			    (update_uid) ? uid : -1,
			    (update_gid) ? gid : -1);
	    if (ret == -1)
		warn("chown %s", dest);
	}
    }
    
    if (update_mode) {
	ret = chmod_rel(dest, mode);
	if (ret == -1)
	    warn("chmod %s", dest);
    }
}

static void
setutime(char *dest, time_t mtime)
{
    struct utimbuf timbuf;
    int ret;

    timbuf.actime = mtime;
    timbuf.modtime = mtime;
    ret = utime_rel(dest, &timbuf);
    if (ret == -1)
	err(1, "utime %s", dest);
}

static int
getstatinfo(char *destpath, int *existp, struct stat *sb)
{
    int ret;

    ret = lstat_rel(destpath, sb);
    if (ret == -1) {
	if (errno == ENOENT) {
	    *existp = 0;
	} else if (errno == ENOTDIR) {
	    errx(1, "%s: File is in the way", destpath);
	    return ENOTDIR;
	} else {
	    err(1, "Failed to stat %s", destpath);
	}
    } else {
	*existp = 1;
    }

    return 0;
}

/*
 *
 */

static void update_exitcode(struct treenode * node) {

    if(node->data.fs->typeflags & FLAG_REBOOT)
	exitcode = EXITCODE_REBOOT;
}

static void
print_type_creator(uint32_t value)
{
    char s[5];

    memcpy(s, &value, 4);
    s[4] = '\0';
    printf("%s", s);
}

static void
copy_metadata(struct treenode *node, char *dest)
{
    struct macosx_metadata metadata_dest;
    int ret;

    if (!node->data.fs->metadatap)
	return;

    ret = macosx_get_metadata(dest, &metadata_dest, 1);
    if (ret == 0) {
	if (node->data.fs->metadata.type == metadata_dest.type &&
	    node->data.fs->metadata.creator == metadata_dest.creator &&
	    node->data.fs->metadata.flags == metadata_dest.flags)
	    return;
    }
    if (verbose_level > 0) {
	printf("metadata type ");
	print_type_creator(node->data.fs->metadata.type);
	printf(" creator ");
	print_type_creator(node->data.fs->metadata.creator);
	printf(" %s\n", dest);
    }
    ret = macosx_put_metadata(dest, &node->data.fs->metadata, 1);
    if (ret)
	errx(1, "macosx_put_metadata %s: %s", dest, strerror(ret));
}

/*
 * Updates a file on disk according to the state of the node passed as
 * argument.
 */

static int
update(struct treenode *node, void *arg) 
{
    /* XXX should handle that (typeflags & FLAG_ABS) */
    struct treenode *src;
    char destpath[MAXPATHLEN];
    char srcpath[MAXPATHLEN];
    char linkdata[MAXPATHLEN];
    int pos;
    struct stat sb;
    int existp;
    int ret;

    assert(node);

    if(would_reboot) {
	if(!(node->data.fs->typeflags & FLAG_REBOOT))
	    return 0;
    }

    pos = 0;
    getnodename(node,destpath,&pos);

    if(!node->data.fs) {
	errx(1,"Incomplete %s", destpath);
    }

    expandtraversal(node);

    switch(node->data.fs->type) {
    case TYPE_B:
    case TYPE_C:
	if (getstatinfo(destpath, &existp, &sb))
	    break;

	if (!node->data.fs->modevalidp) {
	    node->data.fs->mode = 0700;
	    node->data.fs->modevalidp = 1;
	}

	if (!node->data.fs->uidvalidp) {
	    node->data.fs->uid = 0;
	    node->data.fs->uidvalidp = 1;
	}

	if (!node->data.fs->gidvalidp) {
	    node->data.fs->gid = 0;
	    node->data.fs->gidvalidp = 1;
	}


	if ((node->data.fs->typeflags & FLAG_NOOVERWRITE) == 0) {
	    if (existp)
		removefile(destpath, &sb);
	    existp = 0;
	}

	if (existp == 0) {
	    mode_t m;

	    if (node->data.fs->type == TYPE_B)
		m = S_IFBLK;
	    else
		m = S_IFCHR;
	    
	    ret = mknod(destpath, node->data.fs->mode | m, 
			makedev(node->data.fs->major, node->data.fs->minor));
	    if (ret != 0)
		err(1, "%s: cannot create dev node", destpath);
	}


	break;
    case TYPE_F:
	src = node->data.fs->src;
	assert(src);

	getsrcstatinfo(src, 1);

	pos = 0;
	getnodename(src,srcpath,&pos);

	if (src->data.src->stat_error) {
	    errx(1, "couldn't stat file %s", srcpath);
	    break;
	}

	if (node->data.fs->typeflags & FLAG_PERMISSIONCOPY) {
	    if (!node->data.fs->modevalidp) {
		node->data.fs->mode = src->data.src->stat->st_mode & 07777;
		node->data.fs->modevalidp = 1;
	    }
	    if (!node->data.fs->uidvalidp) {
		node->data.fs->uid = src->data.src->stat->st_uid;
		node->data.fs->uidvalidp = 1;
	    }
	    if (!node->data.fs->gidvalidp) {
		node->data.fs->gid = src->data.src->stat->st_gid;
		node->data.fs->gidvalidp = 1;
	    }
	}

	if (getstatinfo(destpath, &existp, &sb))
	    break;

	if (!node->data.fs->modevalidp) {
	    if (access(srcpath, X_OK) == 0) {
		node->data.fs->mode = 0755;
	    } else {
		node->data.fs->mode = 0644;
	    }
	    node->data.fs->modevalidp = 1;
	}

	if (!node->data.fs->uidvalidp) {
	    node->data.fs->uid = 0;
	    node->data.fs->uidvalidp = 1;
	}

	if (!node->data.fs->gidvalidp) {
	    node->data.fs->gid = 0;
	    node->data.fs->gidvalidp = 1;
	}

	assert(src->data.src->stat);
	if (existp) {
	    if ((node->data.fs->typeflags & FLAG_NOOVERWRITE) == 0 &&
		!samefile(node, &sb, src->data.src->stat, destpath, srcpath)) {
		update_exitcode(node);
		copyfile(destpath, srcpath, &sb, node->data.fs->mode);
		setutime(destpath, src->data.src->stat->st_mtime);
		setogm(destpath, NULL,
		       node->data.fs->uid,
		       node->data.fs->gid,
		       node->data.fs->mode);
	    } else {
		setogm(destpath, &sb,
		       node->data.fs->uid,
		       node->data.fs->gid,
		       node->data.fs->mode);
	    }
	} else {
	    update_exitcode(node);
	    copyfile(destpath, srcpath, NULL, node->data.fs->mode);
	    setutime(destpath, src->data.src->stat->st_mtime);
	    setogm(destpath, NULL,
		   node->data.fs->uid,
		   node->data.fs->gid,
		   node->data.fs->mode);
	}
	break;
    case TYPE_L:
	assert(node->data.fs->typeflags & FLAG_ABS);
	src = node->data.fs->src;
	if (node->data.fs->linkdata) {
	    strncpy(linkdata, node->data.fs->linkdata, sizeof(linkdata)-1);
	} else if (src) {
	    /* XXX Should break this out into a function. */
	    /* XXX Should set linkdata when this is done. */
	    /* XXX Should probably have a linkdata in srcnodes. */
	    pos = 0;
	    getnodename(src,srcpath,&pos);
	    ret = readlink(srcpath, linkdata, sizeof(linkdata) - 1);
	    if (ret == -1)
		err(1, "%s: cannot read link", srcpath);
	    linkdata[ret] = '\0';
	}

	if (getstatinfo(destpath, &existp, &sb))
	    break;

	if (existp) {
	    if (!samelink(&sb, destpath, linkdata)) {
		update_exitcode(node);
		removefile(destpath, &sb);
		createlink(destpath, linkdata);
	    }
	} else {
	    update_exitcode(node);
	    createlink(destpath, linkdata);
	}
	break;
    case TYPE_D:
	if (getstatinfo(destpath, &existp, &sb))
	    break;
	
	if (!node->data.fs->modevalidp) {
	    node->data.fs->mode = 0755;
	    node->data.fs->modevalidp = 1;
	}

	if (!node->data.fs->uidvalidp) {
	    node->data.fs->uid = 0;
	    node->data.fs->uidvalidp = 1;
	}

	if (!node->data.fs->gidvalidp) {
	    node->data.fs->gid = 0;
	    node->data.fs->gidvalidp = 1;
	}

	if (existp) {
	    if (!S_ISDIR(sb.st_mode)) {
		removefile(destpath, &sb);
		makedir(destpath, node->data.fs->mode);
		setogm(destpath, NULL,
		       node->data.fs->uid,
		       node->data.fs->gid,
		       node->data.fs->mode);
	    } else {
		setogm(destpath, &sb,
		       node->data.fs->uid,
		       node->data.fs->gid,
		       node->data.fs->mode);
	    }
	} else {
	    makedir(destpath, node->data.fs->mode);
	    setogm(destpath, NULL,
		   node->data.fs->uid,
		   node->data.fs->gid,
		   node->data.fs->mode);
	}	
	if (macosx_flag)
	    copy_metadata(node, destpath);
	break;
    case TYPE_X:
	if (getstatinfo(destpath, &existp, &sb))
	    break;

	if (existp) {
	    update_exitcode(node);
	    removefile(destpath, &sb);
	}
	break;
    case TYPE_N:
	if (getstatinfo(destpath, &existp, &sb))
	    errx(1, "%s: Couldn't stat.", destpath);

	if (existp) {
	    /* Change type of some TYPE_NONE nodes to make sure
	       cleandir doesn't remove this node. */
	    create_path(node);
	}
	break;
    default:
	break;
    }

    nupdated++;
    if (progress_flag) {
	fprintf(stderr, "updated %d/%d (%d%%)   \r", nupdated, nnodes,
		nupdated * 100 / nnodes);
	fflush(stdout);
    }

    if(would_reboot && exitcode)
	return exitcode;

    return 0;
}

/*
 * Cleans any unknown files from a directory marked with FLAG_CLEAN.
 */

static int
removenode_func(const char *path, const char *name, void *arg)
{
    char childpath[MAXPATHLEN];
    char destpath[MAXPATHLEN];
    struct treenode *child, *node = arg;
    int pos = 0;

    getnodename(node,destpath,&pos);

    child = lookup_nocreate(node, name, strlen(name));
    if (child == NULL ||
	child->data.fs == NULL ||
	child->data.fs->type == TYPE_NONE) {

	snprintf(childpath, sizeof(childpath), "%s/%s",
		 destpath, name);
	
	removefile(childpath, NULL);
    }
    return 0;
}

static int
cleandir(struct treenode *node, void *arg) 
{
    char destpath[MAXPATHLEN];
    int pos;

    /* XXX Should run getfsnode and all that.
       It magically works right now, but who knows... */

    /* if we test for reboot, we don't want to remove files */

    if(would_reboot)
	return 0;

    pos = 0;
    getnodename(node,destpath,&pos);

    if(!node->data.fs) {
	errx(1,"Incomplete %s", destpath);
    }

    if(node->data.fs->type != TYPE_D)
	return 0;

    if((node->data.fs->typeflags & FLAG_CLEAN) == 0)
	return 0;

    if (verbose_level > 1)
	printf("cleandir %s\n", destpath);

    return themis_readdir(destpath, THEMIS_READDIR_RELATIVE, node,
			  removenode_func);
}

static int
apply_list(struct treenode *node, struct treenode_func_node *application_list)
{
    int ret;

    ret = (application_list->func)(node, application_list->arg);
    if (ret) {
	return ret;
    } else {
	if (application_list->next) {
	    return apply_list(node, application_list->next);
	} else {
	    return 0;
	}
    }
}

static int
apply_list_void(struct treenode *node, void *arg)
{
    struct treenode_func_node *application_list;
    application_list = (struct treenode_func_node *)arg;
    return apply_list(node, application_list);
}

static int
readfile(char * filename)
{

    char buf[BUFSIZE];
    FILE * infile;
    char *args[NARGS];
    struct treenode *src, *dest;
    int i;
    int lineno = 1;
    FILE *output_fp;

    struct treenode_func_node tfn_update;
    struct treenode_func_node tfn_cleandir;
    struct treenode_func_node tfn_package_print;

    infile = fopen(filename, "r");
    if(!infile) {
	err(1, "Error opening inputfile ``%s''", filename);
    }

    src = newtreenode(NULL, "", 0);
    dest = newtreenode(NULL, "", 0);
    nnodes++;

    for (i = 0; i < NARGS; i++)
	args[i] = NULL;
    
    while(fgets(buf, sizeof(buf), infile) != NULL) {
	buf[strcspn(buf, "\n")] = '\0';

	tokenize(buf, args, sizeof(args)/sizeof(args[0]), &i);
	process(i, args, dest, src, lineno);

#if 0
	for(i = 0; args[i]; i++) {
	    printf("*%s*", args[i]);
	}
	printf("\n");
#endif
	for (i = 0; i < NARGS; i++) {
	    free(args[i]);
	    args[i] = NULL;
	}
	lineno++;
    }

    if (output_file) {
	output_fp = fopen(output_file, "w");
	if(!output_fp) {
	    err(1, "Error opening output file ``%s'' for writing",
		output_file);
	}
    }

    tfn_update.func = update;
    tfn_update.arg = NULL;
    tfn_package_print.func = package_print;
    tfn_package_print.arg = output_fp;

    /* Always run update and cleandir */
    if (output_file) {
	tfn_update.next = &tfn_package_print;
    } else {
	tfn_update.next = NULL;
    }
    tfn_package_print.next = NULL;

    tfn_cleandir.func = cleandir;
    tfn_cleandir.arg = NULL;
    tfn_cleandir.next = NULL;

    /* Doing everything in parallel makes output file
       show the progress. */
    tree_iter(dest, apply_list_void, &tfn_update, apply_list_void, &tfn_cleandir);

    if (output_file) {
	fclose(output_fp);
    }

    if (progress_flag)
	fprintf(stderr, "\n");

    return 0;
}

static struct getargs args[] = {
    {"out", 0,	arg_string,	&output_file,
     "path to output file", "file"},
    {"update",	0,	arg_negative_flag, &update_flag,
     "don't update", NULL },
    {"would-reboot",	0,	arg_flag, &would_reboot,
     "test if we would reboot", NULL },
    {"chown",	0,	arg_negative_flag, &chown_flag,
     "don't chown/chgrp", NULL },
    {"check-owner",	0,	arg_negative_flag, &check_owner_flag,
     "don't check owner/group", NULL },
    {"root",	0,	arg_string,	&fs_root,
     "root of destination tree", NULL },
    {"macosx",	0,	arg_flag,	&macosx_flag,
     "parse Mac OS X resource/finder info files", NULL},
    {"progress", 0,	arg_flag,	&progress_flag,
     "print process meter", NULL},
    {"verbose",	'v',	arg_counter,	&verbose_level,
     "print actions", NULL },
    {"help",	0,	arg_flag,	&help_flag,
     NULL, NULL}
};

static void
usage (int exit_val)
{
    arg_printusage (args, sizeof(args)/sizeof(*args), NULL, "<file name>");
    exit (exit_val);
}

int 
main(int argc, char ** argv) 
{
    char *file;
    int optind = 0;

    setprogname(argv[0]);

    if (getarg (args, sizeof(args)/sizeof(*args), argc, argv, &optind))
	usage (1);

    argc -= optind;
    argv += optind;

    /* If we test if reboot then, don't update things on disk */

    if (would_reboot)
	update_flag = 0;

    if (help_flag)
	usage (0);

    if (argc != 1)
	usage (1);

    file = *argv;
    
    if (macosx_flag)
	add_to_exclude_list("._*");

    readfile(file);

    return exitcode;
}
