/*
 * Copyright (c) 2001 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "themis.h"
#include "macosx.h"

RCSID("$Id: macosx.c,v 1.10.2.2 2002/12/02 03:40:14 lha Exp $");

struct macosx_finderinfo_disk {
    char unknown1[42];
    uint32_t finfo_len;
    uint32_t rsrc_len;
    uint32_t type;
    uint32_t creator;
    uint16_t flags;
    char unknown2[22];
} __attribute__ ((__packed__));

struct macosx_finderinfo_syscall {
    long size;
    struct {
	uint32_t type;
	uint32_t creator;
	uint16_t flags;
	uint16_t ylocation;
	uint16_t xlocation;
	uint16_t folder;
	uint16_t icon;
	uint16_t unknown1[3];
	uint8_t unknown2;
	uint8_t unknown3;
	uint16_t unknown4;
	uint32_t unknown5;
    } fndrinfo;
};


static int
open_rsrc_file(char *path, int *ret_fd, int *length, int rel, int flags)
{
    char *rsrcpath;
    char *newpath;
    char *lastslash;
    int fd;
    int ret;
    struct stat sb;

    newpath = strdup(path);
    assert(newpath);

    lastslash = strrchr(newpath, '/');
    assert(lastslash);
    *lastslash = '\0';

    if (fs_root && rel)
	asprintf(&rsrcpath, "%s%s/._%s", fs_root, newpath, lastslash + 1);
    else
	asprintf(&rsrcpath, "%s/._%s", newpath, lastslash + 1);

    free(newpath);
    
    fd = open(rsrcpath, flags);

    if (fd == -1) {
	free(rsrcpath);
	return errno;
    }

    if (length) {
	ret = fstat(fd, &sb);
	if (ret == -1)
	    err(1, "fstat %s", rsrcpath);
	
	*length = sb.st_size;
    }

    if (ret_fd)
	*ret_fd = fd;
    else
	close(fd);

    free(rsrcpath);
    return 0;
}

static int
stat_rsrc_fork(char *path, int *length, int rel)
{
    char *rsrcpath = 0;
    struct stat sb;
    int ret = -1;

    if (rel && fs_root)
	asprintf(&rsrcpath, "%s%s/rsrc", fs_root, path);
    else
	asprintf(&rsrcpath, "%s/rsrc", path);

    ret = lstat(rsrcpath, &sb); 
    if (ret) {
      free(rsrcpath);
      return errno;
    }

    if (length)
	*length = (int) sb.st_size;

    free(rsrcpath);
    return 0;
}


static int
open_rsrc_fork(char *path, int *ret_fd, int *length, int rel, int flags)
{
    char *rsrcpath;
    int fd;
    struct stat sb;
    int ret;

    if (rel && fs_root)
	asprintf(&rsrcpath, "%s%s/rsrc", fs_root, path);
    else
	asprintf(&rsrcpath, "%s/rsrc", path);
    
    fd = open(rsrcpath, flags);

    if (fd == -1) {
	free(rsrcpath);
	return errno;
    }

    if (length) {
	ret = fstat(fd, &sb);
	if (ret == -1)
	    err(1, "fstat %s", rsrcpath);
	
	*length = sb.st_size;
    }

    if (ret_fd)
	*ret_fd = fd;
    else
	close(fd);

    free(rsrcpath);
    return 0;
}

#if defined(HAVE_GETATTRLIST)
static int
macosx_get_metadata_syscall(char *path, struct macosx_metadata *metadata,
			    int rel)
{
    char *realpath;
    struct macosx_finderinfo_syscall attrdata;
    struct attrlist alist;
    int ret;
    int rsrc_length;

    alist.bitmapcount = ATTR_BIT_MAP_COUNT;
    alist.commonattr = ATTR_CMN_FNDRINFO;
    alist.volattr = 0;
    alist.dirattr = 0;
    alist.fileattr = 0;
    alist.forkattr = 0;

    if (rel && fs_root) {
	asprintf(&realpath, "%s%s", fs_root , path);
    } else {
	realpath = strdup(path);
    }

    memset(&attrdata, 0, sizeof(attrdata));
    ret = getattrlist(realpath, &alist, &attrdata,
		      sizeof(attrdata), FSOPT_NOFOLLOW);
    if (ret == -1)
	return errno;

    ret = stat_rsrc_fork(realpath, &rsrc_length, 0);
    if (ret)
	return ret;

    metadata->type = attrdata.fndrinfo.type;
    metadata->creator = attrdata.fndrinfo.creator;
    metadata->flags = attrdata.fndrinfo.flags;

    metadata->rsrc_len = rsrc_length;

    free(realpath);

    return 0;
}
#else
static int
macosx_get_metadata_syscall(char *path, struct macosx_metadata *metadata,
			 int rel)
{
    return EOPNOTSUPP;
}
#endif

static int
macosx_get_metadata_file(char *path, struct macosx_metadata *metadata, int rel)
{
    int fd;
    int ret;
    int length;
    struct macosx_finderinfo_disk finfo;

    ret = open_rsrc_file(path, &fd, &length, rel, O_RDONLY);
    if (ret)
	return ret;
    
    ret = read(fd, &finfo, sizeof(finfo));
    if (ret == -1)
	err(1, "read metadata file for %s", path);

    if (ret != sizeof(finfo))
	errx(1, "short read metadata file for %s (%d bytes)", path, ret);

    ret = close(fd);
    if (ret == -1)
	err(1, "close metadata file for %s", path);

    assert(finfo.finfo_len == sizeof(finfo));
    assert(finfo.rsrc_len == length - sizeof(finfo));

    metadata->rsrc_len = finfo.rsrc_len;
    metadata->type = finfo.type;
    metadata->creator = finfo.creator;
    metadata->flags = finfo.flags;

    return 0;
}

int
macosx_get_metadata(char *path, struct macosx_metadata *metadata, int rel)
{
    int ret;

    ret = macosx_get_metadata_syscall(path, metadata, rel);
    if (ret == EOPNOTSUPP)
	ret = macosx_get_metadata_file(path, metadata, rel);
    if (ret == ENOENT) {
	metadata->rsrc_len = 0;
	metadata->type = 0;
	metadata->creator = 0;
	metadata->flags = 0;
	ret = 0;
    }

    return ret;
}

static int
macosx_get_resource(char *path, char **buffer, int *ret_length, int rel)
{
    int ret;
    int fd;
    int offset = 0;
    int length;

    ret = open_rsrc_fork(path, &fd, &length, rel, O_RDONLY);
    if (ret == ENOTDIR) {
	ret = open_rsrc_file(path, &fd, &length, rel, O_RDONLY);
	if (ret == 0) {
	    offset = sizeof(struct macosx_finderinfo_disk);
	    length -= offset;
	}
    }
    if (ret)
	return ret;

    *buffer = malloc(length);
    assert(*buffer);

    ret = lseek(fd, offset, SEEK_SET);
    if (ret == -1)
	err(1, "macosx_get_resource lseek");

    ret = read(fd, *buffer, length);
    if (ret == -1)
	err(1, "macosx_get_resource read");
    if (ret != length)
	errx(1, "macosx_get_resource short read (%d bytes)", ret);

    *ret_length = length;
    close(fd);
    return 0;
}

static int
macosx_put_resource(char *path, char *buffer, int length, int rel)
{
    int ret;
    int fd;
    int offset = 0;

    if(!update_flag)
	return 0;

    ret = open_rsrc_fork(path, &fd, NULL, rel, O_WRONLY|O_TRUNC);
    if (ret)
	return ret;

    ret = ftruncate(fd, offset + length);
    if (ret == -1)
	err(1, "macosx_put_resource ftruncate");

    ret = lseek(fd, offset, SEEK_SET);
    if (ret == -1)
	err(1, "macosx_put_resource lseek");

    ret = write(fd, buffer, length);
    if (ret == -1)
	err(1, "macosx_put_resource write");
    if (ret != length)
	errx(1, "macosx_put_resource short write (%d bytes)", ret);

    close(fd);
    return 0;
}

#if defined(HAVE_GETATTRLIST) && defined(HAVE_SETATTRLIST)
static int
macosx_put_metadata_syscall(char *path, struct macosx_metadata *metadata,
			    int rel)
{
    char *realpath;
    struct macosx_finderinfo_syscall attrdata;
    struct attrlist alist;
    int ret;

    alist.bitmapcount = ATTR_BIT_MAP_COUNT;
    alist.commonattr = ATTR_CMN_FNDRINFO;
    alist.volattr = 0;
    alist.dirattr = 0;
    alist.fileattr = 0;
    alist.forkattr = 0;

    if (rel && fs_root) {
	asprintf(&realpath, "%s%s", fs_root , path);
    } else {
	realpath = strdup(path);
    }

    memset(&attrdata, 0, sizeof(attrdata));
    ret = getattrlist(realpath, &alist, &attrdata,
		      sizeof(attrdata), FSOPT_NOFOLLOW);
    if (ret == -1)
	return errno;

    attrdata.fndrinfo.type = metadata->type;
    attrdata.fndrinfo.creator = metadata->creator;
    attrdata.fndrinfo.flags = metadata->flags;
    ret = setattrlist(realpath, &alist, &attrdata.fndrinfo,
		      sizeof(attrdata.fndrinfo), FSOPT_NOFOLLOW);
    if (ret == -1)
	return errno;

    free(realpath);
    return 0;
}
#else
static int
macosx_put_metadata_syscall(char *path, struct macosx_metadata *metadata,
			    int rel)
{
    return EOPNOTSUPP;
}
#endif

int
macosx_put_metadata(char *path, struct macosx_metadata *metadata, int rel)
{
    int ret=0;

    if(update_flag)
	ret = macosx_put_metadata_syscall(path, metadata, rel);

    return ret;
}


void
copyfile_macosx(char *destpath, char *srcpath)
{
    int ret;
    char *buffer;
    int length;
    struct macosx_metadata metadata;

    if(!update_flag)
	return;

    ret = macosx_get_metadata(srcpath, &metadata, 0);
    if (ret)
	errx(1, "macosx_get_metadata %s: %s", srcpath, strerror(ret));

    ret = macosx_get_resource(srcpath, &buffer, &length, 0);
    if (ret == 0) {
	ret = macosx_put_resource(destpath, buffer, length, 1);
	if (ret)
	    errx(1, "macosx_put_resource %s: %s", destpath, strerror(ret));
	free(buffer);
    } else if (ret != ENOENT) {
	errx(1, "macosx_get_resource %s: %s", srcpath, strerror(ret));
    }

    ret = macosx_put_metadata(destpath, &metadata, 1);
    if (ret) {
      errx(1, "macosx_put_metadata %s: %s", destpath, strerror(ret));
    }
}

