!! Copyright (C) 2004-2014 M. Oliveira, F. Nogueira
!! Copyright (C) 2011-2012 T. Cerqueira
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

#include "global.h"

module states_m
  use global_m
  use oct_parser_m
  use io_m
  use messages_m
  use utilities_m
  use units_m
  use output_m
  use mesh_m
  use quantum_numbers_m
  use potentials_m
  use wave_equations_m
  use hartree_m
  use eigensolver_m
  use hamann_m
  use troullier_martins_m
  use multireference_m
  use gsl_interface_m
  implicit none


                    !---Interfaces---!

  interface assignment (=)
    module procedure state_copy
  end interface

  interface operator (==)
    module procedure state_equal
  end interface


                    !---Derived Data Types---!

  type state_t
    ! General information about the state
    type(qn_t)       :: qn      !< state quantum numbers
    real(R8)         :: occ     !< occupation
    real(R8)         :: ev      !< eigenvalue
    character(len=6) :: label   !< a label to identify the state
    integer          :: wave_eq !< wave-equation used to obtain the wave-functions

    logical          :: frozen      !< If set to true, the state is frozen and never updated
    real(R8)         :: frozen_ekin !< Kinetic energy of the frozen state.

    ! The wavefunctions
    integer :: np, wf_dim
    real(R8), pointer :: wf(:,:) !<  Schrodinger equation:
                                 !!   wf(:,1) -> wavefunction
                                 !!  Scalar-relativistic equation
                                 !!   wf(:,1) -> wavefunction
                                 !!  Dirac equation:
                                 !!   wf(:,1) -> spinor major component
                                 !!   wf(:,2) -> spinor minor component
                                 !!  Dirac equation with spin-polarization:
                                 !!   wf(:,1) -> spinor major component +
                                 !!   wf(:,2) -> spinor minor component +
                                 !!   wf(:,3) -> spinor major component -
                                 !!   wf(:,4) -> spinor minor component -
    real(R8), pointer :: wfp(:,:) !< Derivative of the wavefunction

    ! Some information about the wavefunctions
    real(R8) :: peak !< outermost peak position
    real(R8) :: node !< outermost node position
  end type state_t


                    !---Global Variables---!

  integer, parameter :: NONE  = 0, &
                        HAM   = 1, &
                        TM    = 2, &
                        RTM   = 3, &
                        MRPP  = 4, &
                        RMRPP = 5, &
                        MTM   = 6


                    !---Public/Private Statements---!

  private
  public :: state_t, &
            state_null, &
            state_init, &
            state_end, &
            state_save, &
            state_load, &
            assignment(=), &
            operator(==), &
            state_density, &
            state_density_grad, &
            state_density_lapl, &
            state_charge_density, &
            state_magnetization_density, &
            state_tau, &            
            state_eigenvalue, &
            state_charge, &
            state_density_moment, &
            state_kinetic_energy, &
            state_external_energy, &
            state_label, &
            state_qn, &
            state_dot_product, &
            state_update, &
            state_update_charge, &
            state_update_number_of_nodes, &
            state_freeze, &
            state_is_frozen, &
            state_ld, &
            state_dipole_matrix_element, &
            state_output_wf, &
            state_outermost_peak, &
            state_outermost_node, &
            state_default_rc, &
            state_psp_generation, &
            state_test_consistency, &
            state_kb_projector, &
            state_test_ghost, &
            state_hf_yk, &
            state_exchange_coefficients, &
            state_r_integral, &
            NONE, HAM, TM, RTM, MRPP, RMRPP, MTM

contains

  !-----------------------------------------------------------------------
  !>  Nullifies and sets to zero all the components of the state.        
  !-----------------------------------------------------------------------
  subroutine state_null(state)
    type(state_t), intent(out) :: state

    call push_sub("state_null")

    state%qn = QN_NULL
    state%ev = M_ZERO
    state%occ = M_ZERO
    state%label = ""
    state%wave_eq = 0
    state%frozen = .false.
    state%frozen_ekin = M_ZERO
    state%peak = M_ZERO
    state%node = M_ZERO
    state%np = 0
    state%wf_dim = 0
    nullify(state%wf, state%wfp)

    call pop_sub()
  end subroutine state_null

  !-----------------------------------------------------------------------
  !> Inititalizes a state. At this stage only the quantum numbers and the 
  !> occupations are set. Wavefunctions are allocated and set to a        
  !> constant.                                                            
  !-----------------------------------------------------------------------
  subroutine state_init(state, mesh, qn, occ, label)
    type(state_t),    intent(inout) :: state !< the state to be initialized
    type(mesh_t),     intent(in)    :: mesh  !< the mesh
    type(qn_t),       intent(in)    :: qn    !< the quantum numbers
    real(R8),         intent(in)    :: occ   !< the occupation
    character(len=6), intent(in)    :: label !< the lable

    call push_sub("state_init")

    state%qn = qn
    state%occ = occ
    state%label = label

    state%np = mesh%np
    state%wf_dim = qn_wf_dim(state%qn)
    allocate(state%wf(mesh%np, state%wf_dim))
    allocate(state%wfp(mesh%np, state%wf_dim))
    state%wf = M_ZERO
    state%wfp = M_ZERO

    call pop_sub()
  end subroutine state_init

  !-----------------------------------------------------------------------
  !> Frees all memory associated to a state.                              
  !-----------------------------------------------------------------------
  subroutine state_end(state)
    type(state_t), intent(inout) :: state

    call push_sub("state_end")

    state%qn = QN_NULL
    state%ev = M_ZERO
    state%occ = M_ZERO
    state%label = ""
    state%wave_eq = 0
    state%frozen = .false.
    state%frozen_ekin = M_ZERO
    state%peak = M_ZERO
    state%node = M_ZERO
    state%np = 0
    state%wf_dim = 0
    if (associated(state%wf))  deallocate(state%wf)
    if (associated(state%wfp))  deallocate(state%wfp)

    call pop_sub()
  end subroutine state_end

  !-----------------------------------------------------------------------
  !> Copies state_b to state_a.                                           
  !-----------------------------------------------------------------------
  subroutine state_copy(state_a, state_b)
    type(state_t), intent(inout) :: state_a
    type(state_t), intent(in)    :: state_b

    call push_sub("state_copy")

    call state_end(state_a)

    state_a%qn = state_b%qn
    state_a%ev = state_b%ev
    state_a%occ = state_b%occ
    state_a%label = state_b%label
    state_a%wave_eq = state_b%wave_eq
    state_a%frozen = state_b%frozen
    state_a%frozen_ekin = state_b%frozen_ekin
    state_a%peak = state_b%peak
    state_a%node = state_b%node
    state_a%np = state_b%np
    state_a%wf_dim = state_b%wf_dim
    allocate(state_a%wf(state_b%np, state_b%wf_dim))
    allocate(state_a%wfp(state_b%np, state_b%wf_dim))
    state_a%wf = state_b%wf
    state_a%wfp = state_b%wfp

    call pop_sub()
  end subroutine state_copy

  !-----------------------------------------------------------------------
  !> Writes the state information to a file.                              
  !-----------------------------------------------------------------------
  subroutine state_save(unit, state)
    integer,       intent(in) :: unit  !< file unit number
    type(state_t), intent(in) :: state !< state to be written

    integer :: i, n

    call push_sub("state_save")

    write(unit) state%qn
    write(unit) state%occ, state%ev, state%label, state%wave_eq, &
                state%frozen, state%frozen_ekin, &
                state%np, state%wf_dim, state%peak, state%node

    do i = 1, state%wf_dim
      do n = 1, state%np
        write(unit) state%wf(n, i), state%wfp(n, i)
      end do
    end do

    call pop_sub()
  end subroutine state_save

  !-----------------------------------------------------------------------
  !> Reads the state information from a file.                             
  !-----------------------------------------------------------------------
  subroutine state_load(unit, state)
    integer,       intent(in)    :: unit  !< file unit number
    type(state_t), intent(inout) :: state !< state to be read

    integer :: i, n

    call push_sub("state_load")

    read(unit) state%qn
    read(unit) state%occ, state%ev, state%label, state%wave_eq, &
               state%frozen, state%frozen_ekin, &
               state%np, state%wf_dim, state%peak, state%node

    allocate(state%wf(state%np, state%wf_dim))
    allocate(state%wfp(state%np, state%wf_dim))
    do i = 1, state%wf_dim
      do n = 1, state%np
        read(unit) state%wf(n, i), state%wfp(n, i)
      end do
    end do

    call pop_sub()
  end subroutine state_load

  !-----------------------------------------------------------------------
  !> Returns true if state_a and state_b have the same label.             
  !-----------------------------------------------------------------------
  elemental function state_equal(state_a, state_b)
    type(state_t), intent(in) :: state_a, state_b
    logical :: state_equal
    
    state_equal = state_a%label == state_b%label

  end function state_equal

  !-----------------------------------------------------------------------
  !> Computes the spin density associated with a state.                   
  !-----------------------------------------------------------------------
  function state_density(nspin, state)
    integer,       intent(in) :: nspin
    type(state_t), intent(in) :: state
    real(R8) :: state_density(state%np, nspin)

    real(R8) :: clm

    call push_sub("state_density")

    select case (nspin)
    case (1)
      state_density(:,1) = sum(state%wf**2, dim=2)
    case (2)
      if (state%qn%m == M_ZERO) then
        if (state%qn%s == -M_HALF) then
          state_density(:,1) = state%wf(:,1)**2
          state_density(:,2) = M_ZERO
        elseif(state%qn%s == M_HALF) then
          state_density(:,1) = M_ZERO
          state_density(:,2) = state%wf(:,1)**2
        else
          message(1) = "Error in state_density: invalid spin quantum number"
          call write_fatal(1)
        end if

      else
        state_density(:,1) = (M_ONE - (2*state%qn%m)/(2*state%qn%l + M_ONE))  *state%wf(:,1)**2 + &
                             (M_ONE - (2*state%qn%m)/(2*state%qn%l + M_THREE))*state%wf(:,2)**2
        state_density(:,2) = (M_ONE + (2*state%qn%m)/(2*state%qn%l + M_ONE))  *state%wf(:,1)**2 + &
                             (M_ONE + (2*state%qn%m)/(2*state%qn%l + M_THREE))*state%wf(:,2)**2

        if (abs(state%qn%m) /= state%qn%l + M_HALF) then
          clm = -sqrt( (M_TWO*state%qn%l + M_ONE)**2 - M_FOUR*state%qn%m**2  )/(M_TWO*state%qn%l + M_ONE)

          state_density(:,1) = state_density(:,1) + &
                               (M_ONE + (2*state%qn%m)/(2*state%qn%l + M_ONE))*state%wf(:,3)**2 + &
                               (M_ONE + (2*state%qn%m)/(2*state%qn%l - M_ONE))*state%wf(:,4)**2 - &
                               M_TWO*clm*state%wf(:,1)*state%wf(:,3)
          state_density(:,2) = state_density(:,2) + &
                               (M_ONE - (2*state%qn%m)/(2*state%qn%l + M_ONE))*state%wf(:,3)**2 + &
                               (M_ONE - (2*state%qn%m)/(2*state%qn%l - M_ONE))*state%wf(:,4)**2 + &
                               M_TWO*clm*state%wf(:,1)*state%wf(:,3)
        end if

        state_density = M_HALF*state_density
      end if

    end select

    state_density = state_density/(M_FOUR*M_PI)*state%occ

    call pop_sub()
  end function state_density

  !-----------------------------------------------------------------------
  !> Computes the gradient of the electronic density associated with a    
  !> state.                                                               
  !-----------------------------------------------------------------------
  function state_density_grad(nspin, state, mesh)
    integer,       intent(in) :: nspin
    type(state_t), intent(in) :: state
    type(mesh_t),  intent(in) :: mesh
    real(R8) :: state_density_grad(mesh%np, nspin)
    
    integer  :: i
    real(R8) :: clm

    call push_sub("state_density_grad")

    state_density_grad = M_ZERO
    select case(nspin)
    case(1)
      do i = 1, state%wf_dim    
        state_density_grad(:,1) = state_density_grad(:,1) + &
                                  M_TWO*(state%wf(:,i)*state%wfp(:,i))
      end do
    case(2)        
      if (state%qn%m == M_ZERO) then
        if (state%qn%s == -M_HALF) then
          do i = 1, state%wf_dim
            state_density_grad(:,1) = state_density_grad(:,1) + &
                                      M_TWO*(state%wf(:,i)*state%wfp(:,i))
          end do
        elseif (state%qn%s == M_HALF) then
          do i = 1, state%wf_dim
            state_density_grad(:,2) = state_density_grad(:,2) + &
                                      M_TWO*(state%wf(:,i)*state%wfp(:,i))
          end do
        else
          message(1) = "Error in state_density_grad: invalid spin quantum number"
          call write_fatal(1)  
        end if
      else

        state_density_grad(:,1) = &
             (M_ONE - (2*state%qn%m)/(2*state%qn%l + M_ONE))  *(state%wf(:,1)*state%wfp(:,1)) + &
             (M_ONE - (2*state%qn%m)/(2*state%qn%l + M_THREE))*(state%wf(:,2)*state%wfp(:,2))
        state_density_grad(:,2) = &
             (M_ONE + (2*state%qn%m)/(2*state%qn%l + M_ONE))  *(state%wf(:,1)*state%wfp(:,1)) + &
             (M_ONE + (2*state%qn%m)/(2*state%qn%l + M_THREE))*(state%wf(:,2)*state%wfp(:,2))

        if (abs(state%qn%m) /= state%qn%l + M_HALF) then
          clm = -sqrt( (M_TWO*state%qn%l + M_ONE)**2 - M_FOUR*state%qn%m**2  )/(M_TWO*state%qn%l + M_ONE)

          state_density_grad(:,1) = state_density_grad(:,1) + &
               (M_ONE + (2*state%qn%m)/(2*state%qn%l + M_ONE))*(state%wf(:,3)*state%wfp(:,3)) + &
               (M_ONE + (2*state%qn%m)/(2*state%qn%l - M_ONE))*(state%wf(:,4)*state%wfp(:,4)) - &
               clm*(state%wfp(:,1)*state%wf(:,3) + state%wf(:,1)*state%wfp(:,3))
          state_density_grad(:,2) = state_density_grad(:,2) + &
               (M_ONE - (2*state%qn%m)/(2*state%qn%l + M_ONE))*(state%wf(:,3)*state%wfp(:,3)) + &
               (M_ONE - (2*state%qn%m)/(2*state%qn%l - M_ONE))*(state%wf(:,4)*state%wfp(:,4)) + &
               clm*(state%wfp(:,1)*state%wf(:,3) + state%wf(:,1)*state%wfp(:,3))
        end if

      end if
    end select
    
    state_density_grad = state_density_grad/(M_FOUR*M_PI)*state%occ

    call pop_sub()
  end function state_density_grad

  !-----------------------------------------------------------------------
  !> Computes the laplacian of the electronic density associated with a   
  !> state.                                                               
  !-----------------------------------------------------------------------
  function state_density_lapl(nspin, state, mesh)
    integer,       intent(in) :: nspin
    type(state_t), intent(in) :: state
    type(mesh_t),  intent(in) :: mesh
    real(R8) :: state_density_lapl(state%np, nspin)

    integer  :: i
    real(R8) :: clm
    real(R8), allocatable :: wfpp(:,:)

    call push_sub("state_density_lapl")

    state_density_lapl = M_ZERO
    select case(nspin)
    case(1)
      do i = 1, state%wf_dim    
        state_density_lapl(:,1) = state_density_lapl(:,1) + &
             M_TWO*(M_TWO*state%wf(:,i)*state%wfp(:,i)/mesh%r + state%wfp(:,i)**2 + &
             state%wf(:,i)*mesh_derivative(mesh, state%wfp(:,i))) 
      end do
    case(2)        
      if (state%qn%m == M_ZERO) then
        if (state%qn%s == -M_HALF) then
          do i = 1, state%wf_dim
            state_density_lapl(:,1) = state_density_lapl(:,1) + &
                 M_TWO*(M_TWO*state%wf(:,i)*state%wfp(:,i)/mesh%r + state%wfp(:,i)**2 + &
                 state%wf(:,i)*mesh_derivative(mesh, state%wfp(:,i)))
          end do
        elseif (state%qn%s == M_HALF) then
          do i = 1, state%wf_dim
            state_density_lapl(:,2) = state_density_lapl(:,2) + &
                 M_TWO*(M_TWO*state%wf(:,i)*state%wfp(:,i)/mesh%r + state%wfp(:,i)**2 + &
                 state%wf(:,i)*mesh_derivative(mesh, state%wfp(:,i)))
          end do
        else
          message(1) = "Error in state_density_lapl: invalid spin quantum number"
          call write_fatal(1)  
        end if
      else
        allocate(wfpp(mesh%np, state%wf_dim))
        do i = 1, state%wf_dim
          wfpp(:,i) = mesh_derivative(mesh, state%wfp(:,i))
        end do

        state_density_lapl(:,1) = &
             ((M_ONE - (2*state%qn%m)/(2*state%qn%l + M_ONE))  *(state%wf(:,1)*state%wfp(:,1)) + &
              (M_ONE - (2*state%qn%m)/(2*state%qn%l + M_THREE))*(state%wf(:,2)*state%wfp(:,2)))* &
             M_TWO/mesh%r
        state_density_lapl(:,2) = &
             ((M_ONE + (2*state%qn%m)/(2*state%qn%l + M_ONE))  *(state%wf(:,1)*state%wfp(:,1)) + &
              (M_ONE + (2*state%qn%m)/(2*state%qn%l + M_THREE))*(state%wf(:,2)*state%wfp(:,2)))* &
             M_TWO/mesh%r

        state_density_lapl(:,1) = state_density_lapl(:,1) + &
             (M_ONE - (2*state%qn%m)/(2*state%qn%l + M_ONE))  *(state%wfp(:,1)**2 + state%wf(:,1)*wfpp(:,1)) + &
             (M_ONE - (2*state%qn%m)/(2*state%qn%l + M_THREE))*(state%wfp(:,2)**2 + state%wf(:,2)*wfpp(:,2))
        state_density_lapl(:,2) = state_density_lapl(:,2) + &
             (M_ONE + (2*state%qn%m)/(2*state%qn%l + M_ONE))*  (state%wfp(:,1)**2 + state%wf(:,1)*wfpp(:,1)) + &
             (M_ONE + (2*state%qn%m)/(2*state%qn%l + M_THREE))*(state%wfp(:,2)**2 + state%wf(:,2)*wfpp(:,2))

        if (abs(state%qn%m) /= state%qn%l + M_HALF) then
          clm = -sqrt( (M_TWO*state%qn%l + M_ONE)**2 - M_FOUR*state%qn%m**2  )/(M_TWO*state%qn%l + M_ONE)

          state_density_lapl(:,1) = state_density_lapl(:,1) + &
               ((M_ONE + (2*state%qn%m)/(2*state%qn%l + M_ONE))*(state%wf(:,3)*state%wfp(:,3)) + &
                (M_ONE + (2*state%qn%m)/(2*state%qn%l - M_ONE))*(state%wf(:,4)*state%wfp(:,4)) - &
                clm*(state%wfp(:,1)*state%wf(:,3)) - clm*(state%wf(:,1)*state%wfp(:,3)))* &
                M_TWO/mesh%r
          state_density_lapl(:,2) = state_density_lapl(:,2) + &
               ((M_ONE - (2*state%qn%m)/(2*state%qn%l + M_ONE))*(state%wf(:,3)*state%wfp(:,3)) + &
                (M_ONE - (2*state%qn%m)/(2*state%qn%l - M_ONE))*(state%wf(:,4)*state%wfp(:,4)) + &
                clm*(state%wfp(:,1)*state%wf(:,3)) + clm*(state%wf(:,1)*state%wfp(:,3)))* &
                M_TWO/mesh%r
          state_density_lapl(:,1) = state_density_lapl(:,1) + &
               (M_ONE + (2*state%qn%m)/(2*state%qn%l + M_ONE))*(state%wfp(:,3)**2 + state%wf(:,3)*wfpp(:,3)) + &
               (M_ONE + (2*state%qn%m)/(2*state%qn%l - M_ONE))*(state%wfp(:,4)**2 + state%wf(:,4)*wfpp(:,4)) - &
               clm*(M_TWO*state%wfp(:,1)*state%wfp(:,3) + wfpp(:,1)*state%wf(:,3) + wfpp(:,3)*state%wf(:,1))
          state_density_lapl(:,2) = state_density_lapl(:,2) + &
               (M_ONE - (2*state%qn%m)/(2*state%qn%l + M_ONE))*(state%wfp(:,3)**2 + state%wf(:,3)*wfpp(:,3)) + &
               (M_ONE - (2*state%qn%m)/(2*state%qn%l - M_ONE))*(state%wfp(:,4)**2 + state%wf(:,4)*wfpp(:,4)) + &
               clm*(M_TWO*state%wfp(:,1)*state%wfp(:,3) + wfpp(:,1)*state%wf(:,3) + wfpp(:,3)*state%wf(:,1))
        end if

        deallocate(wfpp)
      end if
    end select
    
    state_density_lapl = state_density_lapl/(M_FOUR*M_PI)*state%occ

    call pop_sub()
  end function state_density_lapl

  !-----------------------------------------------------------------------
  !> Computes the charge density associated with a state.                 
  !-----------------------------------------------------------------------
  function state_charge_density(state)
    type(state_t), intent(in) :: state
    real(R8) :: state_charge_density(state%np)

    call push_sub("state_charge_density")

    state_charge_density(:) = sum(state%wf**2, dim=2)/(M_FOUR*M_PI)*state%occ

    call pop_sub()
  end function state_charge_density

  !-----------------------------------------------------------------------
  !> Computes the magnetization density associated with a state.          
  !-----------------------------------------------------------------------
  function state_magnetization_density(state)
    type(state_t), intent(in) :: state
    real(R8) :: state_magnetization_density(state%np)

    real(R8), allocatable :: density(:,:)

    call push_sub("state_magnetization_density")

    allocate(density(state%np, 2))

    density = state_density(2, state)
    state_magnetization_density = density(:,2) - density(:,1)

    deallocate(density)

    call pop_sub()
  end function state_magnetization_density

  !-----------------------------------------------------------------------
  !> Computes the kinetic energy density associated with a state.         
  !-----------------------------------------------------------------------
  function state_tau(nspin, state, mesh)
    integer,       intent(in) :: nspin
    type(state_t), intent(in) :: state
    type(mesh_t),  intent(in) :: mesh
    real(R8) :: state_tau(mesh%np, nspin)

    real(R8) :: l

    call push_sub("state_tau")

    l = state%qn%l
    select case (nspin)
    case (1)
      if (state%qn%j == M_ZERO) then
        state_tau(:,1) = state%wfp(:,1)**2 + l*(l + M_ONE)*state%wf(:,1)**2/mesh%r**2
      else
        state_tau(:,1) = state%wfp(:,1)**2 + state%wfp(:,2)**2 + &
             M_TWO*(state%qn%k + M_ONE)/mesh%r*state%wf(:,1)*state%wfp(:,1) - &
             M_TWO*(state%qn%k - M_ONE)/mesh%r*state%wf(:,2)*state%wfp(:,2) + &
             ((state%qn%k + M_ONE)/mesh%r*state%wf(:,1))**2 + &
             ((state%qn%k - M_ONE)/mesh%r*state%wf(:,2))**2
      end if
    case (2)
      if (state%qn%m == M_ZERO) then
        if (state%qn%s == -M_HALF) then
          state_tau(:,1) = state%wfp(:,1)**2 + l*(l + M_ONE)*state%wf(:,1)**2/mesh%r**2
          state_tau(:,2) = M_ZERO
        elseif(state%qn%s == M_HALF) then
          state_tau(:,1) = M_ZERO
          state_tau(:,2) = state%wfp(:,1)**2 + l*(l + M_ONE)*state%wf(:,1)**2/mesh%r**2
        else
          message(1) = "Error in state_tau: invalid spin quantum number"
          call write_fatal(1)
        end if
      else
        !TODO
        state_tau = M_ZERO
      end if
    end select

    state_tau = state_tau/(M_FOUR*M_PI)*state%occ

    call pop_sub()
  end function state_tau

  !-----------------------------------------------------------------------
  !> Computes the density moment: \f$ \left< R(r) | r^n | R(r)\right>\f$               
  !-----------------------------------------------------------------------
  function state_density_moment(state, mesh, order)
    type(state_t), intent(in) :: state
    type(mesh_t),  intent(in) :: mesh
    integer,       intent(in) :: order
    real(R8) :: state_density_moment

    integer :: i

    call push_sub("calculate_density_moment")

    state_density_moment = state%occ*mesh_integrate(mesh, state%wf(:,1)**2*mesh%r**order)
    if (state%wave_eq == DIRAC) then
      do i = 2, state%wf_dim
        state_density_moment = state_density_moment + &
             state%occ*mesh_integrate(mesh, state%wf(:,i)**2*mesh%r**order)
      end do
    end if

    call pop_sub()
  end function state_density_moment

  !-----------------------------------------------------------------------
  !> Returns the eigenvalue of a state.                                   
  !-----------------------------------------------------------------------
  elemental function state_eigenvalue(state)
    type(state_t), intent(in) :: state
    real(R8) :: state_eigenvalue

    state_eigenvalue = state%ev

  end function state_eigenvalue

  !-----------------------------------------------------------------------
  !> Returns the number of electrons in an state.                         
  !-----------------------------------------------------------------------
  elemental function state_charge(state)
    type(state_t), intent(in) :: state
    real(R8) :: state_charge

    state_charge = state%occ

  end function state_charge

  !-----------------------------------------------------------------------
  !> Returns the kinetic energy of a state.                               
  !-----------------------------------------------------------------------
  function state_kinetic_energy(mesh, state, potential)
    type(mesh_t),      intent(in) :: mesh
    type(state_t),     intent(in) :: state
    type(potential_t), intent(in) :: potential
    real(R8) :: state_kinetic_energy

    integer :: i
    real(R8), allocatable :: rho(:), vext(:), wfpp(:)

    call push_sub("state_kinetic_energy")

    if (state%frozen) then
      state_kinetic_energy = state%frozen_ekin

    else
      !Eigenvalue
      state_kinetic_energy = state%ev*state%occ
      
      !Minus external potential energy
      allocate(rho(mesh%np), vext(mesh%np))
      rho = state_charge_density(state)
      if (state%qn%m /= M_ZERO) then
        vext = state_magnetization_density(state)
      else
        vext = M_ZERO
      end if

      do i = 1, mesh%np
        if (state%qn%m /= M_ZERO) vext(i) = vext(i)*bxc(potential, mesh%r(i)) 
        vext(i) = vext(i) + v(potential, mesh%r(i), state%qn)*rho(i)
      end do
      state_kinetic_energy = state_kinetic_energy - M_FOUR*M_PI*mesh_integrate(mesh, vext)
      deallocate(rho, vext)

      !Minus MGGA term
      allocate(vext(mesh%np))
      select case (state%wave_eq)
      case (SCHRODINGER)
        allocate(wfpp(mesh%np))
        wfpp = mesh_derivative(mesh, state%wfp(:,1))
        do i = 1, mesh%np
          vext(i) = -state%wf(i,1)*state%occ* &
               (dvtaudr(potential, mesh%r(i), state%qn)*state%wfp(i,1) + &
               vtau(potential, mesh%r(i), state%qn)*(M_TWO/mesh%r(i)*state%wfp(i,1) + wfpp(i)))
        end do
        deallocate(wfpp)
      case (SCALAR_REL, DIRAC)
        !TODO
        vext = M_ZERO
      case default
        vext = M_ZERO
      end select
      state_kinetic_energy = state_kinetic_energy - mesh_integrate(mesh, vext)
      deallocate(vext)

    end if

    call pop_sub()
  end function state_kinetic_energy

  !-----------------------------------------------------------------------
  !> Returns the electron interaction energy of a state with an external 
  !> potential.                               
  !-----------------------------------------------------------------------
  function state_external_energy(mesh, state, potential)
    type(mesh_t),      intent(in) :: mesh
    type(state_t),     intent(in) :: state
    type(potential_t), intent(in) :: potential
    real(R8) :: state_external_energy

    integer :: i
    real(R8), allocatable :: rho(:), vext(:)

    call push_sub("state_external_energy")

    allocate(rho(mesh%np), vext(mesh%np))
    rho = state_charge_density(state)
    do i = 1, mesh%np
      vext(i) = v(potential, mesh%r(i), state%qn, unscreened=.true.)
    end do
    state_external_energy = M_FOUR*M_PI*mesh_integrate(mesh, vext*rho)
    deallocate(rho, vext)

    call pop_sub()
  end function state_external_energy

  !-----------------------------------------------------------------------
  !> Returns a label identifying the state. If "full" is true the label   
  !> includes information about the spin or sigma                         
  !-----------------------------------------------------------------------
  function state_label(state, full)
    type(state_t),           intent(in) :: state
    logical,       optional, intent(in) :: full
    character(len=10) :: state_label

    logical :: full_

    state_label = state%label

    full_ = .false.
    if (present(full)) full_ = full

    if (full_ .and. (state%qn%s /= M_ZERO .or. state%qn%sg /= M_ZERO) ) then
      if (state%qn%s == M_HALF .or. state%qn%sg == M_HALF) then
        state_label = trim(state%label)//"_up"
      elseif (state%qn%s == -M_HALF .or. state%qn%sg == -M_HALF) then
        state_label = trim(state%label)//"_dn"
      end if
    end if

  end function state_label

  !-----------------------------------------------------------------------
  !> Returns the quantum numbers of the state.                            
  !-----------------------------------------------------------------------
  elemental function state_qn(state)
    type(state_t), intent(in) :: state
    type(qn_t) :: state_qn

    state_qn = state%qn

  end function state_qn

  !-----------------------------------------------------------------------
  !> Calculates the dot product between two states.
  !-----------------------------------------------------------------------
  function state_dot_product(mesh, state_a, state_b)
    type(mesh_t),  intent(in) :: mesh
    type(state_t), intent(in) :: state_a
    type(state_t), intent(in) :: state_b
    real(R8) :: state_dot_product
    
    select case (state_a%wave_eq)
    case (SCHRODINGER)
      if (qn_equal_fold(state_a%qn, state_b%qn, .false.)) then
        state_dot_product = mesh_integrate(mesh, sum(state_a%wf*state_b%wf, dim=2))
      else
        state_dot_product = M_ZERO
      end if
    case default
      message(1) = "state_dot_product only implemented for Schrodinger equation"
      call write_fatal(1)
    end select

  end function state_dot_product

  !-----------------------------------------------------------------------
  !> Gets the states eigenvalue and eigenfunctions for a given potential. 
  !-----------------------------------------------------------------------
  subroutine state_eigensolve(state, mesh, wave_eq, potential, integrator_dp, &
       integrator_sp, eigensolver)
    type(state_t),       intent(inout) :: state         !< state
    type(mesh_t),        intent(in)    :: mesh          !< mesh
    integer,             intent(in)    :: wave_eq       !< wave-equation to solve
    type(potential_t),   intent(in)    :: potential     !< potential to use in the wave-equation
    type(integrator_t),  intent(inout) :: integrator_dp !< single-precision integrator object
    type(integrator_t),  intent(inout) :: integrator_sp !< double-precision integrator object
    type(eigensolver_t), intent(in)    :: eigensolver   !< information about the eigensolver

    integer :: n_bound
    logical :: unbound, bracketed(1)
    real(R8) :: ev, bracket(2,1)
    type(qn_t) :: qn(1)

    call push_sub("state_eigensolve")

    !If state is frozen, then there is nothing to be done
    if (state%frozen) then
      call pop_sub()
      return
    end if

    !Check if state is bound
    n_bound = wavefunctions_n_bound_states(state%qn, wave_eq, mesh, potential, integrator_sp)
    if (n_bound >= state%qn%n - state%qn%l) then
      unbound = .false.
      !Bracket the eigenvalue
      qn(1) = state%qn
      call eigensolver_bracket(1, qn, wave_eq, &
                               eigensolver, potential, integrator_sp, bracket, bracketed)
    else
      unbound = .true.
    end if

    if (bracketed(1) .and. .not. unbound) then
      !Locate the eigenvalue more accuratly
      call eigensolver_find_ev(state%qn, wave_eq, eigensolver, &
                               potential, integrator_dp, bracket(:,1), ev)
    else
      if (state%occ == M_ZERO) then
        ev = M_ZERO
      else
        if (unbound) then
          write(message(1),'("State: ",A," is unbound")') &
               trim(qn_label(state%qn, .true.))
          call write_fatal(1)
        else
          message(1) = "Unable to bracket eigenvalues for state:"
          write(message(2),'(2X,A)') trim(qn_label(state%qn, .true.))
          call write_fatal(2)
        end if

      end if
    end if

    ! Updtate the wavefunctions
    call state_update(state, mesh, wave_eq, potential, integrator_dp, ev)

    call pop_sub()
  end subroutine state_eigensolve

  !-----------------------------------------------------------------------
  !> Update the wavefunctions of a given state.                           
  !-----------------------------------------------------------------------
  subroutine state_update(state, mesh, wave_eq, potential, integrator, ev)
    type(state_t),      intent(inout) :: state      !< state
    type(mesh_t),       intent(in)    :: mesh       !< mesh
    integer,            intent(in)    :: wave_eq    !< wave-equation to solve
    type(potential_t),  intent(in)    :: potential  !< potential to use in the wave-equation
    type(integrator_t), intent(inout) :: integrator !< integrator object
    real(R8),           intent(in)    :: ev         !< new eigenvalue

    integer :: k

    call push_sub("state_update")

    !If state is frozen, then there is nothing to be done
    if (state%frozen) then
      call pop_sub()
      return
    end if

    state%wave_eq = wave_eq
    state%ev = ev

    ! Compute new wavefunctions
    call wavefunctions(state%qn, state%ev, state%wave_eq, mesh, &
                       potential, integrator, state%wf, state%wfp)

    !Outermost peak
    if (state%ev /= M_ZERO) then
      state%peak = M_ZERO
      do k = 1, mesh%np - 1
        if (abs(state%wf(mesh%np-k+1,1)*mesh%r(mesh%np-k+1)) > abs(state%wf(mesh%np-k,1)*mesh%r(mesh%np-k)) &
             .and. abs(state%wf(mesh%np-k+1,1)*mesh%r(mesh%np-k+1)) > 1.0E-5_r8) then
          state%peak = mesh%r(mesh%np-k+1)
          exit
        end if
      end do
    else
      state%peak = M_ZERO
    end if

    !Outermost node
    if (state%ev /= M_ZERO) then
      state%node = M_ZERO
      do k = 1, mesh%np - 1
        if (state%wf(mesh%np-k+1,1)*state%wf(mesh%np-k,1) < M_ZERO) then
          state%node = mesh%r(mesh%np-k+1)
          exit
        end if
      end do
    else
      state%node = M_ZERO
    end if

    call pop_sub()
  end subroutine state_update

  !-----------------------------------------------------------------------
  !> Update the charge of a given state.                                  
  !-----------------------------------------------------------------------
  subroutine state_update_charge(state, charge)
    type(state_t), intent(inout) :: state  !< state
    real(R8),      intent(in)    :: charge !< new charge

    call push_sub("state_update_charge")

    state%occ = charge

    call pop_sub()
  end subroutine state_update_charge

  !-----------------------------------------------------------------------
  !> Update the wavefunctions number of nodes of a given state.
  !-----------------------------------------------------------------------
  subroutine state_update_number_of_nodes(state, nnodes)
    type(state_t), intent(inout) :: state  !< state
    integer,       intent(in)    :: nnodes !< new number of nodes

    call push_sub("state_update_number_of_nodes")

    call qn_update_number_of_nodes(state%qn, nnodes)

    call pop_sub()
  end subroutine state_update_number_of_nodes
  
  !-----------------------------------------------------------------------
  !> Freezes a state. The kinetic energy is also frozen using the given
  !> value.
  !-----------------------------------------------------------------------
  subroutine state_freeze(state, ekin)
    type(state_t), intent(inout) :: state !< state
    real(R8),      intent(in)    :: ekin  !< kinetic energy of the frozen state

    call push_sub("state_freeze")

    state%frozen = .true.
    state%frozen_ekin = ekin

    call pop_sub()
  end subroutine state_freeze

  !-----------------------------------------------------------------------
  !> Get the frozen attribute of a state.
  !-----------------------------------------------------------------------
  logical function state_is_frozen(state) result(frozen)
    type(state_t), intent(in)  :: state  !< state

    call push_sub("state_is_frozen")

    frozen = state%frozen

    call pop_sub()
  end function state_is_frozen

  !-----------------------------------------------------------------------
  !> Returns the logarithmic derivative of the wave-functions at r.       
  !-----------------------------------------------------------------------
  function state_ld(state, e, r, integrator, potential, mesh, dldde)
    type(state_t),                intent(in)    :: state      !< state
    real(R8),                     intent(in)    :: e          !< energy
    real(R8),                     intent(in)    :: r          !< radius where to compute the logarithmic derivative
    type(integrator_t),           intent(inout) :: integrator !< integrator object
    type(potential_t),            intent(in)    :: potential  !< potential to use in the wave-equation
    type(mesh_t),                 intent(in)    :: mesh       !< mesh
    real(R8),           optional, intent(out)   :: dldde      !< 
    real(R8) :: state_ld

    integer :: nnodes, wf_dim
    type(mesh_t) :: mr
    real(R8), allocatable :: wf(:,:), wfp(:,:)

    call push_sub("state_ld")

    !New mesh
    call mesh_null(mr)
    mr = mesh
    call mesh_truncate(mr, r)

    if (present(dldde)) then
      wf_dim = qn_wf_dim(state%qn)
      allocate(wf(mr%np, wf_dim), wfp(mr%np, wf_dim))

      call hamann_wavefunction(state%qn, e, state%wave_eq, mr, potential, integrator, wf, wfp)
      state_ld = wfp(mr%np, 1)/wf(mr%np, 1)
      dldde = -M_TWO/(r*wf(mr%np,1))**2*mesh_integrate(mr, sum(wf,dim=2)**2)

      deallocate(wf, wfp)
    else
      !Log derivative a la Hamann
      state_ld = hamann_ld(state%qn, e, state%wave_eq, mr, potential, integrator, nnodes)
    end if

    call mesh_end(mr)

    call pop_sub()
  end function state_ld

  !-----------------------------------------------------------------------
  !> Computes the dipole matrix element between two states                 
  !>                                                                       
  !>            \f$ M_{if} = \left< R_i(r) | r | R_f(r) \right> \f$
  !>                                                                       
  !> Notice that the angular part is not taken into  account. This is OK,  
  !> because of the spatial averaging and spherical symmetry.              
  !-----------------------------------------------------------------------
  function state_dipole_matrix_element(mesh, state_i, state_f)
    type(mesh_t),  intent(in) :: mesh    !< mesh
    type(state_t), intent(in) :: state_i !< state i
    type(state_t), intent(in) :: state_f !< state f
    real(R8) :: state_dipole_matrix_element

    integer :: i
    real(R8), allocatable :: tmp(:)

    call push_sub("state_dipole_matrix_element")
        
    if (state_i%qn%s /= state_i%qn%s .or. &
         (state_i%qn%n == state_f%qn%n .and. state_i%qn%l == state_f%qn%l .and. &
         state_i%qn%j /= state_f%qn%j) ) then
      state_dipole_matrix_element = M_ZERO
      return
    end if

    allocate(tmp(mesh%np))

    tmp = state_i%wf(:,1)*state_f%wf(:,1)*mesh%r
    if (state_i%wave_eq == DIRAC .and. state_f%wave_eq == DIRAC) then
      ASSERT(state_i%wf_dim == state_f%wf_dim)
      do i = 2, state_i%wf_dim
        tmp = tmp + state_i%wf(:,i)*state_f%wf(:,i)*mesh%r
      end do
    end if

    state_dipole_matrix_element = mesh_integrate(mesh, tmp)

    deallocate(tmp)

    call pop_sub()
  end function state_dipole_matrix_element

  !-----------------------------------------------------------------------
  !> Returns the positions of the wave-functions outermost peak.          
  !-----------------------------------------------------------------------
  function state_outermost_peak(state)
    type(state_t), intent(in) :: state
    real(R8) :: state_outermost_peak

    call push_sub("state_outermost_peak")

    state_outermost_peak = state%peak

    call pop_sub()
  end function state_outermost_peak

  !-----------------------------------------------------------------------
  !> Returns the positions of the wave-functions outermost node.          
  !-----------------------------------------------------------------------
  function state_outermost_node(state)
    type(state_t), intent(in) :: state
    real(R8) :: state_outermost_node

    call push_sub("state_outermost_node")

    state_outermost_node = state%node

    call pop_sub()
  end function state_outermost_node

  !-----------------------------------------------------------------------
  !> Returns the default cut-off radius.
  !-----------------------------------------------------------------------
  elemental function state_default_rc(state, scheme)
    type(state_t), intent(in)  :: state
    integer,       intent(in)  :: scheme
    real(R8) :: state_default_rc

    !Get default core radius
    select case (scheme)
    case (HAM)
      if (state%qn%n > state%qn%l + 1) then
        state_default_rc = 0.6_r8*state%peak
      else
        state_default_rc = 0.4_r8*state%peak
      end if
    case (TM)
      state_default_rc = M_ZERO
    case (RTM)
      state_default_rc = M_ZERO
    case (MRPP)
      state_default_rc = M_ZERO
    case (RMRPP)
      state_default_rc = M_ZERO
    case (MTM)
      state_default_rc = M_ZERO
    end select

  end function state_default_rc

  !-----------------------------------------------------------------------
  !> Generate the pseudo wave-functions and the pseudo-potential using    
  !> one of the implemented schemes.                                      
  !-----------------------------------------------------------------------
  subroutine state_psp_generation(mesh, scheme, wave_eq, tol, ae_potential, &
                                  integrator_sp, integrator_dp, ps_v, state1, &
                                  rc, state2)
    type(mesh_t),                 intent(in)    :: mesh          !< mesh
    integer,                      intent(in)    :: scheme        !< scheme used to generate the pseudo-potentials
    integer,                      intent(in)    :: wave_eq       !< wave-equation to use
    real(R8),                     intent(in)    :: tol           !< tolerance
    type(potential_t),            intent(in)    :: ae_potential  !< all-electron potential
    type(integrator_t),           intent(inout) :: integrator_sp !< single-precision integrator object
    type(integrator_t),           intent(inout) :: integrator_dp !< double-precision integrator object
    real(R8),                     intent(out)   :: ps_v(mesh%np) !< pseudo-potential on the mesh
    type(state_t),                intent(inout) :: state1        !< pseudo-state 1 (either valence or semi-core)
    real(R8),                     intent(in)    :: rc            !< core radius
    type(state_t),      optional, intent(inout) :: state2        !< valence state when state 1 is a semi-core state

    character(len=10) :: label
    character(40) :: scheme_name

    call push_sub("state_psp_generation")

    if (present(state2)) then
      ASSERT(scheme == MRPP .or. scheme == RMRPP)
    end if

    !Write information
    label = state_label(state1, full=.true.)
    write(message(1),'(2x,"State: ",a)') trim(label)
    if (present(state2)) then
      label = state_label(state2, full=.true.)
      write(message(1),'(a,2x,a)') trim(message(1)), trim(label)
    end if
    select case (scheme) 
    case (HAM) 
      scheme_name = "Hamann" 
    case (TM) 
      scheme_name = "Troullier-Martins" 
    case (RTM) 
      scheme_name = "Troullier-Martins Relativistic extension" 
    case (MRPP) 
      scheme_name = "Multireference Pseudopotentials (MRPP)" 
    case (RMRPP) 
      scheme_name = "MRPP Relativistic extension" 
    case (MTM) 
      scheme_name = "Modified Troullier-Martins" 
    end select
    write(message(2),'(4x,"Scheme: ",a)') trim(scheme_name)
    call write_info(2,20)
    call write_info(2,unit=info_unit("pp"))


    select case (scheme)
    case (HAM) !Hamann scheme
      call hamann_gen(state1%qn, state1%ev, wave_eq, tol, mesh, ae_potential, &
                      rc, integrator_sp, integrator_dp, ps_v, &
                      state1%wf, state1%wfp)

    case (TM) !Troullier-Martins scheme
      call tm_gen(state1%qn, state1%ev, wave_eq, .false., tol, mesh, &
                  ae_potential, rc, ps_v, state1%wf, state1%wfp)
      
    case (RTM) !Relativistic extension of the Troullier-Martins scheme
      call tm_gen(state1%qn, state1%ev, wave_eq, .true., tol, mesh, &
                  ae_potential, rc, ps_v, state1%wf, state1%wfp)

    case (MRPP) !Multireference Pseudopotentials
      call mrpp_gen(state1%qn, state2%qn, state1%ev, state2%ev, &
                    wave_eq, .false., tol, mesh, ae_potential, rc, &
                    integrator_dp, ps_v, state1%wf, state1%wfp, &
                    state2%wf, state2%wfp)

    case (RMRPP) !Multireference Pseudopotentials
      call mrpp_gen(state1%qn, state2%qn, state1%ev, state2%ev, &
                    wave_eq, .true., tol, mesh, ae_potential, rc, &
                    integrator_dp, ps_v, state1%wf, state1%wfp, &
                    state2%wf, state2%wfp)

    end select

    call pop_sub()
  end subroutine state_psp_generation

  !-----------------------------------------------------------------------
  !> Test the consistency of the pseudo-potential and pseudo              
  !> wave-functions. The test consists in solving the wave-equations for  
  !> the pseudo-potential and then comparing the resulting eigenvalue and 
  !> wave-functions with the all-electron ones.                           
  !-----------------------------------------------------------------------
  subroutine state_test_consistency(mesh, wave_eq, eigensolver, integrator_sp, &
                          integrator_dp, ae_potential, ps_potential, ps_state, &
                          rmatch, ev, norm, slope)
    type(mesh_t),        intent(in)    :: mesh          !< mesh
    integer,             intent(in)    :: wave_eq       !< wave-equation to use
    type(eigensolver_t), intent(in)    :: eigensolver   !< information about the eigensolver
    type(integrator_t),  intent(inout) :: integrator_sp !< single-precision integrator object
    type(integrator_t),  intent(inout) :: integrator_dp !< double-precision integrator object
    type(potential_t),   intent(in)    :: ae_potential  !< all-electron potential
    type(potential_t),   intent(in)    :: ps_potential  !< pseudo-potential
    type(state_t),       intent(in)    :: ps_state      !< pseudo-state
    real(R8),            intent(in)    :: rmatch        !< matching radius
    real(R8),            intent(out)   :: ev            !< eigenvalue
    real(R8),            intent(out)   :: norm          !< norm of the wave-function up to rc
    real(R8),            intent(out)   :: slope         !< slope of the wave-functions at rc

    type(state_t) :: state

    call push_sub("state_test_consistency")

    call state_null(state)

    state = ps_state
    call state_eigensolve(state, mesh, wave_eq, ps_potential, integrator_dp, &
                          integrator_sp, eigensolver)
    ev = state%ev
    norm = mesh_integrate(mesh, sum(state%wf**2,dim=2), b=rmatch)
    slope = mesh_extrapolate(mesh, state%wfp(:,1), rmatch)

    call state_update(state, mesh, wave_eq, ae_potential, integrator_dp, ev)
    norm = norm/mesh_integrate(mesh, sum(state%wf**2,dim=2), b=rmatch)
    slope = abs(slope/mesh_extrapolate(mesh, state%wfp(:,1), rmatch))

    call state_end(state)

    call pop_sub()
  end subroutine state_test_consistency

  !-----------------------------------------------------------------------
  !> Computes the KB projector defined in the following way:         
  !>                                                                 
  !> KB projector = \f$ \left| \hat{P}_l \right> E \left< \hat{P}_l \right| \f$
  !> 
  !> \f$ E = \frac{\left<\phi_l (v_l - v_{local}) | (v_l - v_{local}) \phi_l\right>}
  !>              {\left< \phi_l | (v_l - v_{local}) | \phi_l \right>} \f$
  !> 
  !> \f$ \left| \hat{P}_l \right> = \frac{\left| (v_l - v_{local}) \phi_l \right>}
  !>                                     {|| \left< phi_l (v_l - v_{local}) | (v_l - v_{local}) phi_l \right> ||} \f$
  !> 
  !
  !     <phi_l (v_l - v_local) | (v_l - v_local) phi_l>                   
  ! E = -----------------------------------------------                   
  !          < phi_l | (v_l - v_local) | phi_l >                          
  !                                                                       
  !                             | (v_l - v_local) phi_l>                  
  ! | proj_f > = -------------------------------------------------------  
  !              || < phi_l (v_l - v_local) | (v_l - v_local) phi_l > ||  
  !
  !-----------------------------------------------------------------------
  subroutine state_kb_projector(mesh, local_ps_potential, ps_potential, &
                                state, e, proj_f)
    type(mesh_t),      intent(in)  :: mesh               !< mesh
    type(potential_t), intent(in)  :: local_ps_potential !< local component of the KB form
    type(potential_t), intent(in)  :: ps_potential       !< pseudo-potential
    type(state_t),     intent(in)  :: state              !< pseudo-state
    real(R8),          intent(out) :: e                  !< KB energy
    real(R8),          intent(out) :: proj_f(mesh%np)    !< KB projector function

    integer :: i
    real(R8) :: int1, int2, cos
    real(R8), parameter :: delta = 1.0e-20_r8
    real(R8), allocatable :: vphi(:), phi(:)
    character(len=10) :: label

    call push_sub("state_kb_projector")

    allocate(vphi(mesh%np), phi(mesh%np))

    phi = state%wf(:,1)
    do i = 1, mesh%np
      vphi(i) = (v(ps_potential, mesh%r(i), state%qn, unscreened=.true.) &
           - v(local_ps_potential, mesh%r(i), state%qn, unscreened=.true.))*phi(i)
    end do
    int1 = mesh_integrate(mesh, vphi**2)
    int2 = mesh_integrate(mesh, vphi*phi)

    if (int1 /= M_ZERO) then
      e = int1/(int2 + delta)
      proj_f = vphi/sqrt(int1)
      cos = sqrt(int1)/e
    else
      e = M_ZERO
      proj_f = M_ZERO
      cos = M_ZERO
    end if

    if (abs(e) < 10e-12_r8) then
      e = M_ZERO
      proj_f = M_ZERO
      cos = M_ZERO
    end if

    deallocate(vphi, phi)

    !Output information
    if (e /= M_ZERO) then
      label = state_label(state)
      select case (len_trim(label))
      case (2)
        write(message(1),'(6X,A,5X,F9.4,8X,F7.4)') trim(label), e/units_out%energy%factor, cos
      case (5)
        write(message(1),'(4X,A,4X,F9.4,8X,F7.4)') trim(label), e/units_out%energy%factor, cos
      end select
      call write_info(1)
      call write_info(1,unit=info_unit("kb"))
    end if

    call pop_sub()
  end subroutine state_kb_projector

  !-----------------------------------------------------------------------
  !> Tests the KB form of the pseudo-potential for ghost states.          
  !-----------------------------------------------------------------------
  subroutine state_test_ghost(mesh, wave_eq, integrator_sp, integrator_dp, &
                              eigensolver, kb_potential, state)
    type(mesh_t),        intent(in)    :: mesh          !< mesh
    integer,             intent(in)    :: wave_eq       !< wave-equation to use
    type(integrator_t),  intent(inout) :: integrator_sp !< single-precision integrator object
    type(integrator_t),  intent(inout) :: integrator_dp !< double-precision integrator object
    type(eigensolver_t), intent(in)    :: eigensolver   !< information about the eigensolver
    type(potential_t),   intent(in)    :: kb_potential  !< the KB potential
    type(state_t),       intent(in)    :: state         !< pseudo-state

    real(R8) :: kb_e, eloc0, eloc1, u
    character(len=10) :: label
    type(state_t) :: state0, state1

    call push_sub("state_test_ghost")

    kb_e = potential_kb_energy(kb_potential, state%qn)

    if (kb_e == M_ZERO) then
      call pop_sub()
      return
    end if

    call state_null(state0)
    state0 = state
    state0%occ = M_ZERO
    call state_eigensolve(state0, mesh, wave_eq, kb_potential, integrator_sp, integrator_dp, eigensolver)
    eloc0 = state0%ev
    call state_end(state0)

    call state_null(state1)
    state1 = state
    state1%qn%n = state%qn%n + 1
    state1%occ = M_ZERO
    call state_eigensolve(state1, mesh, wave_eq, kb_potential, integrator_sp, integrator_dp, eigensolver)
    eloc1 = state1%ev
    call state_end(state1)

    label = state_label(state, full=.true.)
    u = units_out%energy%factor
    write(message(1),'(4x,"State: ",A)') trim(label)
    write(message(3),'(6X,"Local potential eigenvalues: ",F9.4," (E0)  ",F9.4," (E1)")') eloc0/u, eloc1/u
    write(message(4),'(6X,"Reference energy:            ",F9.4," (Eref)")') state%ev/u

    if (kb_e > M_ZERO) then
      if (state%ev > eloc0) then
        if (state%ev < eloc1) then
          write(message(2),'(6X,"KB energy > 0; E0 < Eref < E1  =>  No ghost states")')
        else
          if (eloc1 < M_ZERO) then
            write(message(2),'(6X,"KB energy > 0; Eref > E1       =>  Ghost state found")')
          else
            write(message(2),'(6X,"KB energy > 0; Eref = E1 = 0   =>  Unable to determine")')
          end if
        end if
      else
        write(message(2),'(6X,"KB energy > 0;  Eref < E0      =>  Illdefined")')
      end if
    elseif (kb_e < M_ZERO) then
      if (state%ev < eloc0) then
        write(message(2),'(6X,"KB energy < 0; Eref < E0       =>  No ghost states")')
      else
        if (eloc0 < M_ZERO) then
          write(message(2),'(6X,"KB energy < 0; Eref > E0       =>  Ghost state found")')
        else
          write(message(2),'(6X,"KB energy < 0; Eref = E0 = 0   =>  Unable to determine")')
        end if
      end if
    end if
    call write_info(4)
    call write_info(4, unit=info_unit("kb"))

    call pop_sub()
  end subroutine state_test_ghost

  !-----------------------------------------------------------------------
  !> Writes the wave-function to a file in a format suitable for plotting.
  !-----------------------------------------------------------------------
  subroutine state_output_wf(state, mesh, dir)
    type(state_t),    intent(in) :: state
    type(mesh_t),     intent(in) :: mesh
    character(len=*), intent(in) :: dir

    integer            :: i, j, unit
    real(R8)           :: u
    character(len=10)  :: label
    character(len=80)  :: fmt

    call push_sub("state_output_wf")

    label = state_label(state, full=.true.)
    call io_open(unit, file=trim(dir)//"/wf-"//trim(label))
    write(unit,'("# Radial wavefunctions and first derivative.")')
    write(unit,'("# Energy units: ",A)') trim(units_out%energy%name)
    write(unit,'("# Length units: ",A)') trim(units_out%length%name)
    write(unit,'("#")')
    select case (int(2*state%qn%s))
    case (0)
      write(unit,'("# State: ",A)') trim(state%label)
    case (-1)
      write(unit,'("# State: ",A," down")') trim(state%label)
    case (1)
      write(unit,'("# State: ",A," up")') trim(state%label)
    end select

    u = units_out%length%factor

    write(unit,'("# Nodes: ")',advance='no')
    do i = 2, mesh%np
      if (state%wf(i,1)*state%wf(i-1,1) < M_ZERO) &
        write(unit,'(F12.6,2X)',advance='no') (mesh%r(i)+mesh%r(i-1))/(M_TWO*u)
    end do
    write(unit,*)
    write(unit,'("# Peaks: ")',advance='no')
    do i = 2, mesh%np-1
      if ( ( abs(state%wf(i,1)*mesh%r(i)) > abs(state%wf(i-1,1)*mesh%r(i-1)) ) &
            .and. ( abs(state%wf(i,1)*mesh%r(i)) > abs(state%wf(i+1,1)*mesh%r(i+1)) ) ) &
            write(unit,'(F12.6,2X)',advance='no') mesh%r(i)/u
    end do
    write(unit,*)

    write(fmt,'(A,I1,A)') "(3X,ES14.8E2,", 2*state%wf_dim, "(3X,ES15.8E2))"
    do i = 1, mesh%np        
      write(unit,fmt) mesh%r(i)/u, (state%wf(i,j)*sqrt(u), j=1,state%wf_dim), (state%wfp(i,j)*sqrt(u), j=1,state%wf_dim)
    end do

    close(unit)

    call pop_sub()
  end subroutine state_output_wf

  !-----------------------------------------------------------------------
  !>
  !-----------------------------------------------------------------------
  subroutine state_hf_yk(mesh, state_a, state_b, k, yk)
    type(mesh_t),         intent(in)  :: mesh
    type(state_t),        intent(in)  :: state_a
    type(state_t),        intent(in)  :: state_b
    integer,              intent(in)  :: k
    real(R8),             intent(out) :: yk(mesh%np)

    call hartree_screening_function(mesh, sum(state_a%wf*state_b%wf, dim=2)*mesh%r**2, k, yk)

  end subroutine state_hf_yk

  !-----------------------------------------------------------------------
  !>
  !-----------------------------------------------------------------------
  function state_exchange_coefficients(state_a, state_b, l) result(coeff)
    type(state_t),  intent(in) :: state_a, state_b
    integer,        intent(in) :: l
    real(R8) :: coeff

    integer :: la, lb

    la = state_a%qn%l
    lb = state_b%qn%l

    if (state_a == state_b) then
      if (l /= 0) then
        coeff = M_HALF*state_a%occ*(state_a%occ - M_ONE)*(M_FOUR*la + M_TWO)/(M_FOUR*la + M_ONE)*&
             (gsl_sf_coupling_3j(2*la, 2*l, 2*la, 0, 0, 0))**M_TWO
      else
        coeff = state_a%occ
      end if
    else
      coeff = M_HALF*state_a%occ*state_b%occ*(gsl_sf_coupling_3j(2*la, 2*l, 2*lb, 0, 0, 0))**M_TWO
    end if

  end function state_exchange_coefficients

  !-----------------------------------------------------------------------
  !>
  !-----------------------------------------------------------------------
  function state_r_integral(mesh, state_a, state_b, state_c, state_d, k)
    type(mesh_t),       intent(in)  :: mesh
    type(state_t),      intent(in)  :: state_a
    type(state_t),      intent(in)  :: state_b
    type(state_t),      intent(in)  :: state_c
    type(state_t),      intent(in)  :: state_d
    integer,            intent(in)  :: k
    real(R8) :: state_r_integral
  
    real(R8), allocatable :: yk(:)

    call push_sub("state_r_integral")

    allocate(yk(mesh%np))

    !Calculate the Yk function
    call state_hf_yk(mesh, state_b, state_d, k, yk)
     
    !Integrate the functions
    state_r_integral = mesh_integrate(mesh, sum(state_a%wf*state_c%wf, dim=2)*yk, dv=mesh%r)

    !Deallocate
    deallocate(yk)
   
    call pop_sub()
  end function state_r_integral

end module states_m
