!! Copyright (C) 2004-2010 M. Oliveira, F. Nogueira
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

#include "global.h"

module multiroot_m
  use global_m
  use messages_m
  use gsl_interface_m
  use linalg_m
  implicit none


                    !---Interfaces---!

  interface
    function non_linear_system_solve(s, n, tol_x_abs, tol_x_rel, tol_f_abs, max_iter, x_guess, x, myfunc, write_info)
      use global_m
      integer(POINTER_SIZE) :: s, x_guess
      integer, intent(in)  :: n, max_iter
      real(R8), intent(in)  :: tol_x_abs, tol_x_rel, tol_f_abs
      real(R8), intent(out)  :: x(n)
      integer :: non_linear_system_solve
      interface
        subroutine myfunc(n, x, f)
          use global_m
          integer,  intent(in)  :: n
          real(R8), intent(in)  :: x(n)
          real(R8), intent(out) :: f(n)
        end subroutine myfunc
      end interface
      interface
        subroutine write_info(iter, n, x, f)
          use global_m
          integer,  intent(in) :: iter
          integer,  intent(in) :: n
          real(R8), intent(in) :: x(n)
          real(R8), intent(in) :: f(n)
        end subroutine write_info
      end interface
    end function non_linear_system_solve
  end interface


                    !---Derived Data Types---!

  type multiroot_solver_t
    integer  :: type        ! What solver to use (Hydrid, Discrete Newton, etc)
    integer  :: n           ! Dimension of the problem (number of roots)
    real(R8) :: tol_x_rel   ! Absolute tolerance for the positions x
    real(R8) :: tol_x_abs   ! Relative tolerance for the positions x
    real(R8) :: tol_f_abs   ! Absolute tolerance for the residues f
    integer  :: max_iter    ! Maximum number of iterations
  end type multiroot_solver_t


                    !---Global Variables---!

  integer, parameter :: HYBRIDS = 1, &
                        HYBRID  = 2, &
                        DNEWTON = 3, &
                        BROYDEN = 4


                    !---Public/Private Statements---!

  private
  public :: multiroot_solver_t,  &
            multiroot_solver_init,  &
            multiroot_solve_system, &
            HYBRIDS, HYBRID, DNEWTON, BROYDEN

contains

  subroutine multiroot_solver_init(n, tol_x_abs, tol_x_rel, tol_f_abs, max_iter, type, solver)
    !-----------------------------------------------------------------------!
    ! Initialize the multiroot solver object.                               !
    !                                                                       !
    !  n         - dimension of the problem                                 !
    !  tol_x_abs - absolute tolerance for the positions x                   !
    !  tol_x_rel - relative tolerance for the positions x                   !
    !  tol_f_abs - absolute tolerance for the residues f                    !
    !  max_iter  - maximum number of iterations                             !
    !  type      - what solver to use                                       !
    !-----------------------------------------------------------------------!
    integer,                  intent(in) :: n, max_iter, type
    real(R8),                 intent(in) :: tol_x_abs, tol_x_rel, tol_f_abs
    type(multiroot_solver_t), intent(out) :: solver

    call push_sub("multiroot_solver_init")

    solver%type      = type
    solver%n         = n
    solver%tol_x_abs = tol_x_abs
    solver%tol_x_rel = tol_x_rel
    solver%tol_f_abs = tol_f_abs
    solver%max_iter  = max_iter

    call pop_sub()
  end subroutine multiroot_solver_init

  subroutine multiroot_solve_system(solver, x, myfunc, write_info)
    !-----------------------------------------------------------------------!
    ! Find the roots of a set of non-linear equations.                      !
    !                                                                       !
    !  solver     - the solver to use                                       !
    !  x          - contains a guess on input and the solution on output    !
    !  myfunc     - subroutine returning the residues for a given set of    !
    !               positions.                                              !
    !  write_info - subroutine that prints relevant information at each     !
    !               iteration.                                              !
    !-----------------------------------------------------------------------!
    type(multiroot_solver_t),  intent(in) :: solver
    real(R8),                  intent(inout) :: x(solver%n)
    interface
      subroutine myfunc(n, x, f)
        use global_m
        integer,  intent(in)  :: n
        real(R8), intent(in)  :: x(n)
        real(R8), intent(out) :: f(n)
      end subroutine myfunc
    end interface
    interface
      subroutine write_info(iter, n, x, f)
        use global_m
        integer,  intent(in) :: iter
        integer,  intent(in) :: n
        real(R8), intent(in) :: x(n)
        real(R8), intent(in) :: f(n)
      end subroutine write_info
    end interface

    integer               :: ierr
    integer(POINTER_SIZE) :: gsl_solver, x_guess

    call push_sub("multiroot_solve_system")

    !Initialize gsl solver
    call gsl_multiroot_fsolver_alloc(gsl_solver, solver%type, solver%n)

    !Initialize guess
    call gsl_vector_init(solver%n, x, x_guess)

    !Solve non-linear system    
    ierr = non_linear_system_solve(gsl_solver, solver%n, solver%tol_x_abs, &
         solver%tol_x_rel, solver%tol_f_abs, solver%max_iter, &
         x_guess, x, myfunc, write_info)
    if (ierr /= 0) then
      message(1) = "Error in subtoutine multiroot_solve_system. GSL error message:"
      call gsl_strerror(ierr, message(2))
      call write_fatal(2)
    end if

    !Free memory
    call gsl_vector_free(x_guess)
    call gsl_multiroot_fsolver_free(gsl_solver)

    call pop_sub()
  end subroutine multiroot_solve_system

end module multiroot_m
