!! Copyright (C) 2004-2014 M. Oliveira, F. Nogueira
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

#include "global.h"

module kb_projectors_m
  use global_m
  use messages_m
  use io_m
  use units_m
  use splines_m
  use mesh_m
  use ps_io_m
  use quantum_numbers_m
  use loc_potentials_m
  implicit none


                    !---Interfaces---!

  interface assignment (=)
     module procedure kb_projectors_copy
  end interface


                    !---Derived Data Types---!

  type kb_projectors_t
    private
    !Local part
    integer :: l_local
    type(loc_potential_t) :: vl
    !Projectors
    integer :: nc
    type(qn_t), pointer :: qn(:)
    real(R8),   pointer :: e(:)
    real(R8),   pointer :: p(:,:)
    type(spline_t), pointer :: p_spl(:)
  end type kb_projectors_t


                    !---Public/Private Statements---!

  private
  public :: kb_projectors_t, &
            kb_projectors_null, &
            kb_projectors_init, &
            assignment(=), &
            kb_projectors_end, &
            kb_projectors_save, &
            kb_projectors_load, &
            kb_v, &
            kb_dvdr, &
            kb_d2vdr2, &
            kb_projectors_energy, &
            kb_projectors_debug, &
            kb_projectors_output, &
            kb_projectors_ps_io_set


contains

  !-----------------------------------------------------------------------
  !> Nullifies and sets to zero all the components of the KB projectors.  
  !-----------------------------------------------------------------------
  subroutine kb_projectors_null(kb_proj)
    type(kb_projectors_t), intent(out) :: kb_proj

    call push_sub("kb_projectors_null")

    call loc_potential_null(kb_proj%vl)
    kb_proj%nc = 0
    nullify(kb_proj%qn)
    nullify(kb_proj%e)
    nullify(kb_proj%p)
    nullify(kb_proj%p_spl)

    call pop_sub()
  end subroutine kb_projectors_null

  !-----------------------------------------------------------------------
  !> Initialize a non-local potential                                     
  !-----------------------------------------------------------------------
  subroutine kb_projectors_init(kb_proj, l_local, v_local, m, nc, qn, e, p)
    type(kb_projectors_t), intent(inout) :: kb_proj     !< KB projectors to be initialized
    integer,               intent(in)    :: l_local     !< which angular momentum component was used as local
    type(loc_potential_t), intent(in)    :: v_local     !< the local part of the KB projectors
    type(mesh_t),          intent(in)    :: m           !< mesh
    integer,               intent(in)    :: nc          !< number of projectors
    type(qn_t),            intent(in)    :: qn(nc)      !< quantum numbers of each projector
    real(R8),              intent(in)    :: e(nc)       !< KB energies
    real(R8),              intent(in)    :: p(m%np, nc) !< values of the projectors functios on the mesh

    integer :: i, k

    call push_sub("kb_projectors_init")

    kb_proj%l_local = l_local
    kb_proj%vl = v_local

    kb_proj%nc = nc
    allocate(kb_proj%qn(nc))
    allocate(kb_proj%e(nc))
    allocate(kb_proj%p(m%np, nc))
    allocate(kb_proj%p_spl(nc))

    do i = 1, nc
      k = qn(i)%l + int(qn(i)%j) + 1
      kb_proj%qn(k) = qn(i)
      kb_proj%e(k) = e(i)
      kb_proj%p(:, k) = p(:, i)
      call spline_null(kb_proj%p_spl(k))
      call spline_init(kb_proj%p_spl(k), m%np, m%r, kb_proj%p(:, k), 3)
    end do

    call pop_sub()
  end subroutine kb_projectors_init

  !-----------------------------------------------------------------------
  !> Copies the KB projectors kb_proj_a to KB projectors kb_proj_b.       
  !-----------------------------------------------------------------------
  subroutine kb_projectors_copy(kb_proj_a, kb_proj_b)
    type(kb_projectors_t), intent(inout) :: kb_proj_a
    type(kb_projectors_t), intent(in)    :: kb_proj_b

    integer :: i

    call push_sub("kb_projectors_copy")

    call kb_projectors_end(kb_proj_a)

    kb_proj_a%l_local = kb_proj_b%l_local
    kb_proj_a%vl = kb_proj_b%vl

    kb_proj_a%nc = kb_proj_b%nc

    allocate(kb_proj_a%qn(kb_proj_a%nc))
    kb_proj_a%qn = kb_proj_b%qn

    allocate(kb_proj_a%e(kb_proj_a%nc))
    kb_proj_a%e = kb_proj_b%e

    allocate(kb_proj_a%p(size(kb_proj_b%p, dim=1), kb_proj_a%nc))
    kb_proj_a%p = kb_proj_b%p

    allocate(kb_proj_a%p_spl(kb_proj_a%nc))
    do i = 1, kb_proj_a%nc
      call spline_null(kb_proj_a%p_spl(i))
      kb_proj_a%p_spl(i) = kb_proj_b%p_spl(i)
    end do

    call pop_sub()
  end subroutine kb_projectors_copy

  !-----------------------------------------------------------------------
  !> Frees all the memory associated to the KB projectors.                
  !-----------------------------------------------------------------------
  subroutine kb_projectors_end(kb_proj)
    type(kb_projectors_t), intent(inout) :: kb_proj

    integer :: i

    call push_sub("kb_projectors_end")

    call loc_potential_end(kb_proj%vl)

    if (associated(kb_proj%qn)) then
      deallocate (kb_proj%qn)
    end if
    if (associated(kb_proj%e)) then
      deallocate (kb_proj%e)
    end if
    if (associated(kb_proj%p)) then
      deallocate (kb_proj%p)
    end if

    if (associated(kb_proj%p_spl)) then
      do i = 1, kb_proj%nc
        call spline_end(kb_proj%p_spl(i))
      end do
      deallocate(kb_proj%p_spl)
    end if

    call pop_sub()
  end subroutine kb_projectors_end

  !-----------------------------------------------------------------------
  !> Writes the KB projectors to a file.                                  
  !-----------------------------------------------------------------------
  subroutine kb_projectors_save(unit, m, kb_proj)
    integer,               intent(in) :: unit    !< file unit number
    type(mesh_t),          intent(in) :: m       !< mesh
    type(kb_projectors_t), intent(in) :: kb_proj !< KB projectors to be written

    integer :: k, i

    call push_sub("kb_projectors_save")

    write(unit) kb_proj%l_local
    call loc_potential_save(unit, m, kb_proj%vl)

    write(unit) kb_proj%nc
    do k = 1, kb_proj%nc
      write(unit) kb_proj%qn(k)
      write(unit) kb_proj%e(k)
      do i = 1, m%np
        write(unit) kb_proj%p(i, k)
      end do
    end do

    call pop_sub()
  end subroutine kb_projectors_save

  !-----------------------------------------------------------------------
  !> Reads the KB projectors from a file.                                 
  !-----------------------------------------------------------------------
  subroutine kb_projectors_load(unit, m, kb_proj)
    integer,               intent(in)    :: unit    !< file unit number
    type(mesh_t),          intent(in)    :: m       !< mesh
    type(kb_projectors_t), intent(inout) :: kb_proj !< kb_proj to be read

    integer :: k, i

    call push_sub("kb_projectors_load")

    read(unit) kb_proj%l_local
    call loc_potential_load(unit, m, kb_proj%vl)

    read(unit) kb_proj%nc
    allocate(kb_proj%qn(kb_proj%nc))
    allocate(kb_proj%e(kb_proj%nc))
    allocate(kb_proj%p(m%np, kb_proj%nc))
    allocate(kb_proj%p_spl(kb_proj%nc))
    do k = 1, kb_proj%nc
      read(unit) kb_proj%qn(k)
      read(unit) kb_proj%e(k)
      do i = 1, m%np
        read(unit) kb_proj%p(i, k)
      end do
      call spline_null(kb_proj%p_spl(k))
      call spline_init(kb_proj%p_spl(k), m%np, m%r, kb_proj%p(:, k), 3)
    end do

    call pop_sub()
  end subroutine kb_projectors_load

  !-----------------------------------------------------------------------
  !> Returns the value of the local potential felt by an electron at      
  !> radius r.                                                            
  !-----------------------------------------------------------------------
  function kb_v(kb_proj, r)
    type(kb_projectors_t), intent(in) :: kb_proj
    real(R8),              intent(in) :: r
    real(R8) :: kb_v

    kb_v = loc_v(kb_proj%vl, r)

  end function kb_v

  !-----------------------------------------------------------------------
  !> Returns the value of the first derivative of the local potential felt
  !> by an electron at radius r.                                          
  !-----------------------------------------------------------------------
  function kb_dvdr(kb_proj, r)
    type(kb_projectors_t), intent(in) :: kb_proj
    real(R8),              intent(in) :: r
    real(R8) :: kb_dvdr

    kb_dvdr = loc_dvdr(kb_proj%vl, r)

  end function kb_dvdr

  !-----------------------------------------------------------------------
  !> Returns the value of the second derivative of the local potential    
  !> felt by an electron at radius r.                                     
  !-----------------------------------------------------------------------
  function kb_d2vdr2(kb_proj, r)
    type(kb_projectors_t), intent(in) :: kb_proj
    real(R8),              intent(in) :: r
    real(R8) :: kb_d2vdr2

    kb_d2vdr2 = loc_d2vdr2(kb_proj%vl, r)

  end function kb_d2vdr2

  !-----------------------------------------------------------------------
  !> Returns the value of the KB energy.                                  
  !-----------------------------------------------------------------------
  function kb_projectors_energy(kb_proj, qn)
    type(kb_projectors_t), intent(in) :: kb_proj
    type(qn_t),            intent(in) :: qn
    real(R8) :: kb_projectors_energy

    call push_sub("kb_projectors_energy")

    kb_projectors_energy = kb_proj%e(qn%l + int(qn%j) + 1)

    call pop_sub()
  end function kb_projectors_energy

  !-----------------------------------------------------------------------
  !> Prints debug information to the "debug_info" directory.              
  !-----------------------------------------------------------------------
  subroutine kb_projectors_debug(kb_proj)
    type(kb_projectors_t), intent(in) :: kb_proj

    call push_sub("kb_projectors_debug")

    call pop_sub()
  end subroutine kb_projectors_debug

  !-----------------------------------------------------------------------
  !> Writes the potential to a file in a format suitable for plotting.    
  !-----------------------------------------------------------------------
  subroutine kb_projectors_output(kb_proj, m, dir)
    type(kb_projectors_t), intent(in) :: kb_proj
    type(mesh_t),          intent(in) :: m
    character(len=*),      intent(in) :: dir

    integer  :: i, k, unit
    real(R8) :: ue, ul
    character(len=10) :: label

    call push_sub("kb_projectors_output")

    !Output local part
    call loc_potential_output(kb_proj%vl, m, trim(dir)//"/kb-local")


    !Output KB projectors
    ul = units_out%length%factor
    ue = units_out%energy%factor

    do k = 1, kb_proj%nc
      if (abs(kb_proj%e(k)) == M_ZERO) cycle

      label = qn_label(kb_proj%qn(k))
      call io_open(unit, file=trim(dir)//"/kb-"//trim(label(2:)))

      write(unit,'("# ")')
      write(unit,'("# Energy units: ",A)') trim(units_out%energy%name)
      write(unit,'("# Length units: ",A)') trim(units_out%length%name)
      write(unit,'("#")')
      write(unit,'("# ",35("-"))')
      write(unit,'("# |",7X,"r",7X,"|",7X,"w(r)",6X,"|")')
      write(unit,'("# ",35("-"))')
      do i = 1, m%np
        write(unit,'(3X,ES14.8E2,3X,ES15.8E2)') m%r(i)/ul, kb_proj%p(i, k)/ue
      end do

      close(unit)
    end do

    call pop_sub()    
  end subroutine kb_projectors_output

  !-----------------------------------------------------------------------
  !> Pass the information about the KB projectors to the ps_io module.    
  !-----------------------------------------------------------------------
  subroutine kb_projectors_ps_io_set(kb_proj, m)
    type(kb_projectors_t), intent(in) :: kb_proj
    type(mesh_t),          intent(in) :: m

    integer :: np, i, n
    integer,  allocatable :: l(:)
    real(R8), allocatable :: j(:), e(:), p(:,:), v_loc(:)

    call push_sub("kb_projectors_ps_io_set")

    np = count(kb_proj%e /= M_ZERO)
    allocate(l(np))
    allocate(j(np))
    allocate(v_loc(m%np))
    allocate(e(np))
    allocate(p(m%np, np))

    n = 0
    do i = 1, kb_proj%nc
      if (kb_proj%e(i) == M_ZERO) cycle
      n = n + 1

      l(n) = kb_proj%qn(i)%l
      j(n) = kb_proj%qn(i)%j
      e(n) = kb_proj%e(i)
      p(:, n) = kb_proj%p(:, i)
    end do

    do i = 1, m%np
      v_loc(i) = loc_v(kb_proj%vl, m%r(i))
    end do

    call ps_io_set_kb(m%np, kb_proj%l_local, v_loc, np, l, j, e, p)

    deallocate(l, j, v_loc, e, p)

    call pop_sub()
  end subroutine kb_projectors_ps_io_set

end module kb_projectors_m
