!! Copyright (C) 2004-2012 M. Oliveira, F. Nogueira, T. Cerqueira
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!
!! $Id: atom_ae_inc.F90 778 2013-07-11 15:49:39Z micael $

  subroutine atom_create_ae(atm, eigensolver)
    !-----------------------------------------------------------------------!
    ! Initializes the atom by reading the relevant input options and prints !
    ! some relevant informantion.                                           !
    !-----------------------------------------------------------------------!
    type(atom_t),        intent(inout) :: atm
    type(eigensolver_t), intent(in)    :: eigensolver

    real(R8), allocatable :: vh(:,:), vxc(:,:), vxctau(:,:)

    call push_sub("atom_create_ae")

                    !---Read information about the atom---!

    !Read theory level
    call oct_parse_int('TheoryLevel', DFT, atm%theory_level)
    select case (atm%theory_level)
    case (INDEPENDENT_PARTICLES, DFT)
    case default
      message(1) = "Illegal TheoryLevel."
      call write_fatal(1)
    end select
    
    !Read nuclear charge
    call oct_parse_float('NuclearCharge', M_ZERO, atm%z)
    if (atm%z <= M_ZERO) then
      message(1) = "Wrong or absent value for the nuclear charge."
      call write_fatal(1)
    end if
    atm%symbol = label(atm%z)

    !Read rel
    call  oct_parse_int('WaveEquation', SCHRODINGER, atm%wave_eq)
    select case (atm%wave_eq)
    case (SCHRODINGER, DIRAC, SCALAR_REL)
    case default
      message(1) = "Illegal WaveEquation."
      call write_fatal(1)
    end select

    !Read polarization
    call oct_parse_int('SpinMode', 1, atm%nspin)
    select case (atm%nspin)
    case (1,2)
    case default
      message(1) = "Illegal SpinMode."
      call write_fatal(1)
    end select


                    !---Initialize the data structures---!

    atm%type = ATOM_AE

    !Initialize xc
    call xc_null(atm%xc_model)
    if (atm%theory_level == DFT) call xc_init(atm%nspin, atm%xc_model)

    !Initialize mesh
    call mesh_null(atm%m)
    call mesh_init_from_input(atm%z, atm%m)

    !Initialize orbitals from file
    call atom_init_states_from_block(atm, "Orbitals")

    !Initialize potential
    allocate(vh(atm%m%np, atm%nspin), vxc(atm%m%np, atm%nspin), vxctau(atm%m%np, atm%nspin))
    select case (atm%theory_level)
    case (INDEPENDENT_PARTICLES)
      vh = M_ZERO; vxc = M_ZERO; vxctau = M_ZERO
    case (DFT)
      call hartree_potential(atm%m, states_batch_charge_density(atm%states, atm%m), &
                             states_batch_charge(atm%states), vh = vh)
      call xc_potential(atm%xc_model, atm%m, atm%states, atm%nspin, vxc = vxc, vxctau=vxctau)
    end select
    call potential_init(atm%potential, atm%m, atm%nspin, vh + vxc, vxctau, atm%z)
    deallocate(vh, vxc, vxctau)

    !Initialize integrator 
    call integrator_init(atm%integrator_sp, atm%integrator_dp)


                    !---Print information about the atom---!

    message(1) = ""
    message(2) = "General Information about the atom:"
    call write_info(2)
    call write_info(2,unit=info_unit("ae"))

    !Print info to screen and to the ae/info file
    write(message(1),'(2X,"Symbol: ",A)') trim(atm%symbol)
    select case (atm%theory_level)
    case (INDEPENDENT_PARTICLES)
      message(2) = "  Theory Level: independent particles"
    case (DFT)
      message(2) = "  Theory Level: DFT"
    end select
    select case (atm%wave_eq)
    case (SCHRODINGER)
      message(3) = "  Wave-equation: Schrodinger"
    case (SCALAR_REL)
      message(3) = "  Wave-equation: scalar-relativistic"
    case (DIRAC)
      message(3) = "  Wave-equation: Dirac"
    end select
    if (atm%nspin == 1) then
      message(4) = "  Spin mode: unpolarized"
    else
      message(4) = "  Spin mode: polarized"
    end if
    write(message(5),'(2X,"Nuclear charge: ",F6.2)') atm%z
    write(message(6),'(2X,"Total charge:   ",F6.2)') atm%z - states_batch_charge(atm%states)  
    call write_info(6, verbose_limit=20)
    call write_info(6, unit=info_unit("ae"))

    call states_batch_sort(atm%states, SORT_QN)
    call states_batch_output_configuration(atm%states, atm%nspin, verbose_limit=20)
    call states_batch_output_configuration(atm%states, atm%nspin, unit=info_unit("ae"))

    if (atm%theory_level == DFT) then
      call xc_model_output_info(atm%xc_model, verbose_limit=20)
      call xc_model_output_info(atm%xc_model, unit=info_unit("ae"))
    end if

    call mesh_output_params(atm%m, verbose_limit=20)
    call mesh_output_params(atm%m, unit=info_unit("ae"))


                    !---Solve Khon-Sham equation---!

    call atom_solve(atm, eigensolver, info_unit("ae"))
    
    call pop_sub()
  end subroutine atom_create_ae

  subroutine atom_xc_eval(atm)
    !-----------------------------------------------------------------------!
    ! Evaluates the exchange and correlation potential and energy using     !
    ! the wavefunctions of a previous run.                                  !
    !-----------------------------------------------------------------------!
    type(atom_t), intent(inout)    :: atm

    real(R8) :: exc

    call push_sub("atom_xc_eval")

    ASSERT(atm%type == ATOM_AE)

    !Initialialize xc_model from input file
    call xc_end(atm%xc_model)
    call xc_init(atm%nspin, atm%xc_model)
    call xc_model_output_info(atm%xc_model, verbose_limit=20)
    call xc_model_output_info(atm%xc_model, unit=info_unit("xc"))

    !Output energies and xc potential
    call xc_potential(atm%xc_model, atm%m, atm%states, atm%nspin, exc = exc) 
    call xc_evaluate_ked_approximation(atm%xc_model, atm%m, atm%states, atm%nspin, info_unit("xc"))
    write(message(1),'("  Exchange-correlation energy [",A,"]: ",F14.6)') &
                                trim(units_out%energy%abbrev), exc/units_out%energy%factor

    call write_info(1,20)
    call write_info(1,unit=info_unit("xc"))
    call xc_output(atm%xc_model, atm%m, atm%states, atm%nspin, "xc")

    call pop_sub()
  end subroutine atom_xc_eval
