!! Copyright (C) 2004-2012 M. Oliveira, F. Nogueira
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!
!! $Id: mesh.F90 778 2013-07-11 15:49:39Z micael $

#include "global.h"

module mesh_m
  use global_m
  use oct_parser_m
  use messages_m
  use utilities_m
  use math_m
  use splines_m
  use finite_diff_m
  use units_m
  implicit none


                    !---Interfaces---!

  interface assignment (=)
     module procedure mesh_copy
  end interface

  interface operator (==)
     module procedure equal_mesh
  end interface


                    !---Derived Data Types---!

  type mesh_t
    integer,  private          :: type !mesh type
    real(R8), public           :: a    !mesh parameters
    real(R8), public           :: b    !
    integer,  public           :: np   !mesh number of points
    real(R8), public,  pointer :: r(:) !mesh points

    integer, private :: intrp_method ! Method to interpolate functions between meshes
    integer, private :: integ_method ! Method used to calculate integrals
    integer, private :: deriv_method ! Method used to calculate derivatives
    integer, private :: interp_range
    integer, private :: fd_order
    type(fd_operator_t), private :: deriv
    type(fd_operator_t), private :: deriv2
    type(fd_operator_t), private :: deriv3
  end type mesh_t


                    !---Global Variables---!

  !Mesh types
  integer, parameter :: MESH_LINEAR  = 1, &
                        MESH_LOG1    = 2, &
                        MESH_LOG2    = 3

  !Interpolation, derivatives, and integrals methods
  integer, parameter :: MESH_CUBIC_SPLINE = 1, &
                        MESH_FINITE_DIFF  = 2


                    !---Public/Private Statements---!

  private
  public :: mesh_t, &
            mesh_null, &
            mesh_init, &
            mesh_init_from_input, &
            mesh_generation, &
            mesh_save, &
            mesh_load, &
            mesh_transfer, &
            mesh_truncate, &
            mesh_extrapolate, &
            mesh_derivative, &
            mesh_derivative2, &
            mesh_derivative3, &
            mesh_integrate, &
            mesh_primitive, &
            mesh_gradient, &
            mesh_divergence, &
            mesh_laplacian, &
            mesh_output_params, &
            mesh_end, &
            assignment(=), &
            operator(==), &
            MESH_LOG1, &
            MESH_LINEAR, &
            MESH_LOG2, &
            MESH_CUBIC_SPLINE, &
            MESH_FINITE_DIFF

contains

  subroutine mesh_null(m)
    !-----------------------------------------------------------------------!
    ! Nullifies and sets to zero all the components of mesh m.              !
    !-----------------------------------------------------------------------!
    type(mesh_t), intent(out) :: m

    call push_sub("mesh_null")

    m%type = 0
    m%a    = M_ZERO
    m%b    = M_ZERO
    m%np   = 0
    m%intrp_method = 0
    m%interp_range = 0
    m%deriv_method = 0
    m%integ_method = 0
    m%fd_order = 0
    call fd_operator_null(m%deriv)
    call fd_operator_null(m%deriv2)
    call fd_operator_null(m%deriv3)
    nullify(m%r)

    call pop_sub()
  end subroutine mesh_null

  subroutine mesh_init(m, type, deriv_method, r1, rmax, np, a, fd_order)
    !-----------------------------------------------------------------------!
    ! Initializes the calculation mesh by reading the mesh parameter from   !
    ! the input file. The default parameters depend on the nuclear charge.  !
    !                                                                       !
    !                                                                       !
    !  m - mesh object                                                      !
    !-----------------------------------------------------------------------!
    type(mesh_t), intent(inout) :: m
    integer,      intent(in)    :: type, deriv_method
    real(R8),     intent(in)    :: r1, rmax
    integer,      intent(in), optional :: np, fd_order
    real(R8),     intent(in), optional :: a

    call push_sub("mesh_init")

    ASSERT(m%type == 0)
    ASSERT(present(a) .or. present(np))
    ASSERT(deriv_method /= MESH_FINITE_DIFF .or. (deriv_method == MESH_FINITE_DIFF .and. present(fd_order)))

    !Generate mesh
    if (present(a)) then
      call mesh_generation(m, type, r1, rmax, a=a)
    else
      call mesh_generation(m, type, r1, rmax, n=np)
    end if

    m%deriv_method = deriv_method
    if (m%deriv_method == MESH_FINITE_DIFF) then
      m%fd_order = fd_order
      call fd_operator_init(m%deriv,  1, m%fd_order, m%np, m%r)
      call fd_operator_init(m%deriv2, 2, m%fd_order, m%np, m%r)
      call fd_operator_init(m%deriv3, 3, m%fd_order, m%np, m%r)
    end if
    m%integ_method = MESH_CUBIC_SPLINE
    m%intrp_method = MESH_CUBIC_SPLINE
    m%interp_range = 25

    call pop_sub()
  end subroutine mesh_init

  subroutine mesh_init_from_input(z, m)
    !-----------------------------------------------------------------------!
    ! Initializes the calculation mesh by reading the mesh parameter from   !
    ! the input file. The default parameters depend on the nuclear charge.  !
    !                                                                       !
    !  z - nuclear charge                                                   !
    !  m - mesh object                                                      !
    !-----------------------------------------------------------------------!
    real(R8),     intent(in)    :: z
    type(mesh_t), intent(inout) :: m

    logical  :: is_def_np, is_def_a
    integer  :: type, np, deriv_method, fd_order
    real(R8) :: r1, rmax, a

    call push_sub("mesh_init_from_input")

    ASSERT(m%type == 0)

    !Read input options
    call oct_parse_int('MeshType', MESH_LOG1, type)
    select case (type)
    case (MESH_LOG1, MESH_LOG2, MESH_LINEAR)
    case default
      message(1) =  "Illegal MeshType."
      call write_fatal(1)
    end select

    call oct_parse_float('MeshStartingPoint', sqrt(z)*1.0E-5, r1)
    if (r1 <= M_ZERO) then
      message(1) =  "MeshStartingPoint can not take negative values."
      call write_fatal(1)
    end if
    r1 = r1*units_in%length%factor

    call oct_parse_float('MeshOutmostPoint', sqrt(z)*M_THIRTY, rmax)
    rmax = rmax*units_in%length%factor
    if (rmax <= M_ZERO .or. rmax < r1) then
      message(1) = "MeshOutmostPoint can''t take negative values"
      message(2) = "and must be greater than the mesh start point."
      call write_fatal(2)
    end if
    
    is_def_np = oct_parse_isdef('MeshNumberOfPoints') == 1
    is_def_a  = oct_parse_isdef('MeshParameter') == 1
    if (is_def_np .and. is_def_a) then
      message(1) = "MeshNumberOfPoints and MeshParameter input options"
      message(2) = "can not be used at the same time."
      call write_fatal(2)
    end if

    fd_order = 0
    call oct_parse_int('MeshDerivMethod', MESH_CUBIC_SPLINE, deriv_method)
    select case (deriv_method)
    case (MESH_CUBIC_SPLINE)
    case (MESH_FINITE_DIFF)
      call oct_parse_int('MeshFiniteDiffOrder', 4, fd_order)
      if (fd_order < 1) then
        message(1) = "MeshFiniteDerivOrder must be greater than 0."
        call write_fatal(1)
      end if
    case default
      message(1) =  "Illegal MeshDerivMethod."
      call write_fatal(1)
    end select

    if (is_def_a) then
      call oct_parse_float('MeshParameter', M_DIME, a)
      if (a < M_ZERO) then
        message(1) = "MeshParameter can not take negative values."
        call write_fatal(1)
      end if

      call mesh_init(m, type, deriv_method, r1, rmax, a=a, fd_order=fd_order)
    else
      call oct_parse_int('MeshNumberOfPoints', int(sqrt(z))*200, np)
      if (np <= 3) then
        message(1) = "Mesh must have at least 3 points."
        call write_fatal(1)
      end if

      call mesh_init(m, type, deriv_method, r1, rmax, np=np, fd_order=fd_order)
    end if

    call pop_sub()
  end subroutine mesh_init_from_input

  subroutine mesh_generation(m, type, r1, rn, n, a)
    !-----------------------------------------------------------------------!
    ! Initializes a mesh and generates the mesh points. Note that only one  !
    ! of the optional variables n and a should be used.                     !
    !                                                                       !
    !  m    - mesh                                                          !
    !  type - mesh type                                                     !
    !  r1   - starting point                                                !
    !  rn   - ending point                                                  !
    !  n    - number of points                                              !
    !  a    - mesh parameter a                                              !
    !-----------------------------------------------------------------------!
    type(mesh_t), intent(inout) :: m
    integer,      intent(in)    :: type
    real(R8),     intent(in)    :: r1, rn
    integer,      intent(in), optional :: n
    real(R8),     intent(in), optional :: a

    integer :: i
    real(R8) :: a1, a2, n1, n2, am, nm, f1, fm

    call push_sub("mesh_generation")

    !Check optional arguments
    if (present(n) .and. present(a)) then
      message(1) = "Only one optional argument can be set in mesh_generation"
      call write_fatal(1)
    end if

    !Set the mesh type, the number of points, the first point and the parameter a
    m%type = type
    if (present(n)) then
      m%np = n
      allocate(m%r(m%np))
      m%r(1) = r1
      select case (m%type)
      case (MESH_LINEAR)
        m%a = (rn - r1)/real(n-1, R8)
      case (MESH_LOG1)
        m%a = log(rn/r1)/real(n - 1,R8)
      case (MESH_LOG2)
        a1 = 1.0e-8_r8
        f1 = func(r1, rn, real(n,R8), a1)
        a2 = M_ONE
        do
          am = (a2 + a1)*M_HALF
          fm = func(r1, rn, real(n,R8), am)
          if (M_HALF*abs(a1 - a2) < 1.0e-16) exit
          if (fm*f1 > M_ZERO) then
            a1 = am; f1 = fm
          else
            a2 = am
          end if
        end do
        m%a = am
      end select

    elseif (present(a)) then
      select case (m%type)
      case (MESH_LINEAR)
        m%np = int((rn - r1)/a)
      case (MESH_LOG1)
        m%np = int(log(rn/r1)/a) + 1
      case (MESH_LOG2)
        n1 = M_ONE
        f1 = func(r1, rn, n1, a)
        n2 = 10000.0_r8
        do
          nm = (n2 + n1)*M_HALF
          fm = func(r1, rn, nm, a)
          if (M_HALF*abs(n1 - n2) < 1.0e-12) exit
          if (fm*f1 > M_ZERO) then
            n1 = nm; f1 = fm
          else
            n2 = nm
          end if
        end do
        m%np = int(nm)
      end select
      allocate(m%r(m%np))
      m%r(1) = r1
      m%a = a

    end if

    !Set the parameter b and the remaining points
    select case (m%type)
    case (MESH_LINEAR)
      m%b = M_ZERO
      do i = 2, m%np-1
        m%r(i) = m%r(i-1) + m%a
      end do
    case (MESH_LOG1)
      m%b = r1/exp(m%a)
      do i = 2, m%np-1
        m%r(i) = exp(m%a)*m%r(i-1)
      end do
    case (MESH_LOG2)
      m%b = r1/(exp(m%a) - M_ONE)
      do i = 2, m%np-1
        m%r(i) = m%r(i-1)*exp(m%a) + r1
      end do
    end select
    m%r(m%np) = rn

    call pop_sub()
  contains

    real(R8) function func(r1, rn, n, a)
      real(R8), intent(in) :: r1, rn, a, n
      func = exp(n*a)*r1 - M_ONE*r1 - rn*exp(a) + rn*M_ONE
    end function func

  end subroutine mesh_generation

  subroutine mesh_copy(m_a, m_b)
    !-----------------------------------------------------------------------!
    ! Copies m_a mesh to m_b.                                               !
    !-----------------------------------------------------------------------!
    type(mesh_t), intent(inout) :: m_a
    type(mesh_t), intent(in)    :: m_b

    call push_sub("mesh_copy")

    call mesh_end(m_a)
    m_a%type = m_b%type
    m_a%a    = m_b%a
    m_a%b    = m_b%b
    m_a%np   = m_b%np
    m_a%intrp_method = m_b%intrp_method
    m_a%interp_range = m_b%interp_range
    m_a%deriv_method = m_b%deriv_method
    m_a%integ_method = m_b%integ_method
    allocate(m_a%r(m_a%np))
    m_a%r = m_b%r
    if (m_a%deriv_method == MESH_FINITE_DIFF) then
      m_a%fd_order = m_b%fd_order
      m_a%deriv = m_b%deriv
      m_a%deriv2 = m_b%deriv2
      m_a%deriv3 = m_b%deriv3
    end if

    call pop_sub()
  end subroutine mesh_copy

  function equal_mesh(m_a, m_b)
    !-----------------------------------------------------------------------!
    ! Returns true if m_a and m_b are the same mesh; false otherwise.       !
    !-----------------------------------------------------------------------!
    type(mesh_t), intent(in) :: m_a, m_b
    logical :: equal_mesh

    call push_sub("equal_mesh")

    equal_mesh = (m_a%type == m_b%type .and. m_a%a == m_b%a .and. &
                  m_a%b == m_b%b .and. m_a%np == m_b%np)

    call pop_sub()
  end function equal_mesh

  subroutine mesh_save(unit, m)
    !-----------------------------------------------------------------------!
    ! Writes the mesh information to a file.                                !
    !                                                                       !
    !  unit - file unit number                                              !
    !  m    - mesh to be written                                            !
    !-----------------------------------------------------------------------!
    integer,      intent(in) :: unit
    type(mesh_t), intent(in) :: m

    integer :: i

    call push_sub("mesh_save")

    ASSERT(m%type /= 0)

    write(unit) m%type, m%a, m%b, m%np, m%intrp_method, m%interp_range, m%deriv_method, &
                m%integ_method, m%fd_order
    write(unit) (m%r(i), i=1, m%np)

    call pop_sub()
  end subroutine mesh_save

  subroutine mesh_load(unit, m)
    !-----------------------------------------------------------------------!
    ! Reads the mesh information from a file.                               !
    !                                                                       !
    !  unit - file unit number                                              !
    !  m    - mesh to be read                                               !
    !-----------------------------------------------------------------------!
    integer,      intent(in)    :: unit
    type(mesh_t), intent(inout) :: m

    integer :: i

    call push_sub("mesh_load")

    ASSERT(m%type == 0)

    read(unit) m%type, m%a, m%b, m%np, m%intrp_method, m%interp_range, m%deriv_method, &
                m%integ_method, m%fd_order
    allocate(m%r(m%np))
    read(unit) (m%r(i), i=1, m%np)

    if (m%deriv_method == MESH_FINITE_DIFF) then
      call fd_operator_init(m%deriv, 1,  m%fd_order, m%np, m%r)
      call fd_operator_init(m%deriv2, 2, m%fd_order, m%np, m%r)
      call fd_operator_init(m%deriv3, 3, m%fd_order, m%np, m%r)
    end if

    call pop_sub()
  end subroutine mesh_load

  subroutine mesh_truncate(m, rt)
    !-----------------------------------------------------------------------!
    ! Truncates a mesh so that the last point is rt                         !
    !-----------------------------------------------------------------------!
    type(mesh_t), intent(inout) :: m
    real(R8),     intent(in)    :: rt

    real(R8), allocatable :: r(:)

    call push_sub("mesh_truncate")

    ASSERT(m%type /= 0)

    m%np = locate(m%r, rt, 0)
    allocate(r(m%np+1))
    r(1:m%np) = m%r(1:m%np)
    if (m%r(m%np) /= rt) then
      r(m%np + 1) = rt
      m%np = m%np + 1
    end if
    deallocate(m%r)
    allocate(m%r(m%np))
    m%r(1:m%np) = r(1:m%np)
    deallocate(r)

    if (m%deriv_method == MESH_FINITE_DIFF) then
      call fd_operator_end(m%deriv)
      call fd_operator_end(m%deriv2)
      call fd_operator_init(m%deriv, 1,  m%fd_order, m%np, m%r)
      call fd_operator_init(m%deriv2, 2, m%fd_order, m%np, m%r)
      call fd_operator_init(m%deriv3, 3, m%fd_order, m%np, m%r)
    end if

    call pop_sub()
  end subroutine mesh_truncate

  subroutine mesh_transfer(m_a, fa, m_b, fb)
    !-----------------------------------------------------------------------!
    ! Having a function on a mesh m_a, this routines returns the values of  !
    ! that function on a mesh m_b, by interpolating the function.           !
    !                                                                       !
    !  m_a         - mesh m_a                                               !
    !  fa          - values of the function on mesh A                       !
    !  x_b         - mesh m_b                                               !
    !  fa          - values of the function on mesh A                       !
    !-----------------------------------------------------------------------!
    type(mesh_t), intent(in)  :: m_a, m_b
    real(R8),     intent(in)  :: fa(m_a%np)
    real(R8),     intent(out) :: fb(m_b%np)

    call push_sub("mesh_transfer")

    ASSERT(m_a%type /= 0 .and. m_b%type /= 0)

    select case (m_a%intrp_method)
    case (MESH_CUBIC_SPLINE)
      call spline_mesh_transfer(m_a%np, m_a%r, fa, m_b%np, m_b%r, fb, 3)

    case default
      write(message(1),'("Illegal interpolation method in mesh_transfer: ",I1)') &
           m_a%intrp_method
      call write_fatal(1)
    end select

    call pop_sub()
  end subroutine mesh_transfer

  function mesh_extrapolate(m, f, r)
    !-----------------------------------------------------------------------!
    ! Returns the value of f at r                                           !
    !                                                                       !
    !  m - mesh                                                             !
    !  f - values of the function on mesh                                   !
    !  r - point where to evaluate the function                             ! 
    !-----------------------------------------------------------------------!
    type(mesh_t), intent(in)  :: m
    real(R8),     intent(in)  :: f(m%np)
    real(R8),     intent(in)  :: r
    real(R8) :: mesh_extrapolate

    integer :: i, ii, if
    type(spline_t) :: spl

    call push_sub("mesh_eval_deriv")

    select case (m%intrp_method)
    case (MESH_CUBIC_SPLINE)
      i = locate(m%r, r, 0)
      ii = max(1, i-m%interp_range)
      if = min(m%np, i+m%interp_range)
      call spline_null(spl)
      call spline_init(spl, if-ii+1, m%r(ii:if), f(ii:if), 3)
      mesh_extrapolate = spline_eval(spl, r)
      call spline_end(spl)

    case default
      write(message(1),'("Illegal interpolation method in mesh_eval: ",I1)') &
           m%intrp_method
      call write_fatal(1)
    end select

    call pop_sub()
  end function mesh_extrapolate

  function mesh_derivative(m, f)
    !-----------------------------------------------------------------------!
    ! Returns the derivate of f                                             !
    !                                                                       !
    !  m     - mesh                                                         !
    !  f     - values of the function on mesh                               !
    !-----------------------------------------------------------------------!
    type(mesh_t), intent(in)  :: m
    real(R8),     intent(in)  :: f(m%np)
    real(R8) :: mesh_derivative(m%np)

    integer :: i
    type(spline_t) :: spl

    call push_sub("mesh_derivative")

    select case (m%deriv_method)
    case (MESH_CUBIC_SPLINE)
      call spline_null(spl)
      call spline_init(spl, m%np, m%r, f, 3)
      do i = 1, m%np
        mesh_derivative(i) = spline_eval_deriv(spl, m%r(i))
      end do
      call spline_end(spl)

    case (MESH_FINITE_DIFF)
      call fd_operator_apply(m%deriv, f, mesh_derivative)

    case default
      write(message(1),'("Illegal numerical derivative method in mesh_derivative: ",I1)') &
           m%deriv_method
      call write_fatal(1)
    end select

    call pop_sub()
  end function mesh_derivative

  function mesh_derivative2(m, f)
    !-----------------------------------------------------------------------!
    ! Returns the second derivate of f                                      !
    !                                                                       !
    !  m     - mesh                                                         !
    !  f     - values of the function on mesh                               !
    !-----------------------------------------------------------------------!
    type(mesh_t), intent(in)  :: m
    real(R8),     intent(in)  :: f(m%np)
    real(R8) :: mesh_derivative2(m%np)

    integer :: i
    type(spline_t) :: spl

    call push_sub("mesh_derivative2")

    select case (m%deriv_method)
    case (MESH_CUBIC_SPLINE)
      call spline_null(spl)
      call spline_init(spl, m%np, m%r, f, 3)
      do i = 1, m%np
        mesh_derivative2(i) = spline_eval_deriv2(spl, m%r(i))
      end do
      call spline_end(spl)

    case (MESH_FINITE_DIFF)
      call fd_operator_apply(m%deriv2, f, mesh_derivative2)

    case default
      write(message(1),'("Illegal numerical derivative method in mesh_derivative2: ",I1)') &
           m%deriv_method
      call write_fatal(1)
    end select

    call pop_sub()
  end function mesh_derivative2

  function mesh_derivative3(m, f)
    !-----------------------------------------------------------------------!
    ! Returns the third derivate of f                                      !
    !                                                                       !
    !  m     - mesh                                                         !
    !  f     - values of the function on mesh                               !
    !-----------------------------------------------------------------------!
    type(mesh_t), intent(in)  :: m
    real(R8),     intent(in)  :: f(m%np)
    real(R8) :: mesh_derivative3(m%np)

    integer :: i
    type(spline_t) :: spl

    call push_sub("mesh_derivative3")

    select case (m%deriv_method)
    case (MESH_CUBIC_SPLINE)
      call spline_null(spl)
      call spline_init(spl, m%np, m%r, mesh_derivative(m, f), 3)
      do i = 1, m%np
        mesh_derivative3(i) = spline_eval_deriv2(spl, m%r(i))
      end do
      call spline_end(spl)

    case (MESH_FINITE_DIFF)
      call fd_operator_apply(m%deriv3, f, mesh_derivative3)

    case default
      write(message(1),'("Illegal numerical derivative method in mesh_derivative3: ",I1)') &
           m%deriv_method
      call write_fatal(1)
    end select

    call pop_sub()
  end function mesh_derivative3

  function mesh_integrate(m, f, a, b, dv)
    !-----------------------------------------------------------------------!
    ! Returns the integral of f between a and b. Default values for a and b !
    ! are 0 and the last point of the mesh.                                 !
    !                                                                       !
    !  m  - mesh                                                            !
    !  f  - values of the function on mesh                                  !
    !  a  - lower bound of the integration interval                         !
    !  b  - upper bound of the integration interval                         !
    !  dv - volume element for integration. Default is r**2                 !
    !-----------------------------------------------------------------------!
    type(mesh_t), intent(in) :: m
    real(R8),     intent(in) :: f(m%np)
    real(R8),     intent(in), optional :: a, b, dv(m%np)
    real(R8) :: mesh_integrate

    real(R8) :: a_, b_, dv_(m%np)
    type(spline_t) :: spl

    call push_sub("mesh_integrate")

    if (present(a)) then
      a_ = a
    else
      a_ = M_ZERO
    end if

    if (present(b)) then
      b_ = b
    else
      b_ = m%r(m%np)
    end if

    if (present(dv)) then
      dv_ = dv
    else
      dv_ = m%r**2
    end if
    
    select case (m%integ_method)
    case (MESH_CUBIC_SPLINE)      
      call spline_null(spl)
      call spline_init(spl, m%np, m%r, f*dv_, 3)
      if (a_ < m%r(1)) then
        mesh_integrate = polynomial_extrapolative_integration(4, m%r(1:4), f(1:4)*dv_(1:4), a_, m%r(1))
        a_ = m%r(1)
      else
        mesh_integrate = M_ZERO
      end if
      mesh_integrate = mesh_integrate + spline_eval_integ(spl, a_, b_)
      call spline_end(spl)

    case default
      write(message(1),'("Illegal numerical integration method in mesh_integrate: ",I1)') &
           m%integ_method
      call write_fatal(1)
    end select

    call pop_sub()
  end function mesh_integrate

  function mesh_primitive(m, f, dv)
    !-----------------------------------------------------------------------!
    ! Returns the primitive of f.                                           !
    !                                                                       !
    !  m  - mesh                                                            !
    !  f  - values of the function on mesh                                  !
    !  dv - volume element for integration. Default is r**2                 !
    !-----------------------------------------------------------------------!
    type(mesh_t), intent(in) :: m
    real(R8),     intent(in) :: f(m%np)
    real(R8),     intent(in), optional :: dv(m%np)
    real(R8) :: mesh_primitive(m%np)

    integer :: i
    real(R8) :: dv_(m%np)
    type(spline_t) :: spl

    call push_sub("mesh_primitive")

    if (present(dv)) then
      dv_ = dv
    else
      dv_ = m%r**2
    end if

    select case (m%integ_method)
    case (MESH_CUBIC_SPLINE)
      call spline_null(spl)
      call spline_init(spl, m%np, m%r, f*dv_, 3)  
      mesh_primitive(1) = polynomial_extrapolative_integration(4, m%r(1:4), f(1:4)*dv_(1:4), M_ZERO, m%r(1))    
      do i = 2, m%np
        mesh_primitive(i) = mesh_primitive(i-1) + spline_eval_integ(spl, m%r(i-1), m%r(i))
      end do
      call spline_end(spl)

    case default
      write(message(1),'("Illegal numerical integration method in mesh_primitive: ",I1)') &
           m%integ_method
      call write_fatal(1)
    end select

    call pop_sub()
  end function mesh_primitive

  function mesh_gradient(m, f)
    !-----------------------------------------------------------------------!
    ! Returns the gradient of f                                             !
    !                                                                       !
    !  m     - mesh                                                         !
    !  f     - values of the function on mesh                               !
    !-----------------------------------------------------------------------!
    type(mesh_t), intent(in)  :: m
    real(R8),     intent(in)  :: f(m%np)
    real(R8) :: mesh_gradient(m%np)

    call push_sub("mesh_gradient")

    mesh_gradient = mesh_derivative(m, f)

    call pop_sub()
  end function mesh_gradient

  function mesh_divergence(m, f)
    !-----------------------------------------------------------------------!
    ! Returns the divergence of f                                           !
    !                                                                       !
    !  m     - mesh                                                         !
    !  f     - values of the function on mesh                               !
    !-----------------------------------------------------------------------!
    type(mesh_t), intent(in)  :: m
    real(R8),     intent(in)  :: f(m%np)
    real(R8) :: mesh_divergence(m%np)

    call push_sub("mesh_divergence")

    mesh_divergence = M_TWO/m%r*f + mesh_derivative(m, f)

    call pop_sub()
  end function mesh_divergence

  function mesh_laplacian(m, f)
    !-----------------------------------------------------------------------!
    ! Returns the laplacian of f                                            !
    !                                                                       !
    !  m     - mesh                                                         !
    !  f     - values of the function on mesh                               !
    !-----------------------------------------------------------------------!
    type(mesh_t), intent(in)  :: m
    real(R8),     intent(in)  :: f(m%np)
    real(R8) :: mesh_laplacian(m%np)

    call push_sub("mesh_laplacian")

    mesh_laplacian = M_TWO/m%r*mesh_derivative(m, f) + mesh_derivative2(m, f)

    call pop_sub()
  end function mesh_laplacian

  subroutine mesh_output_params(m, unit, verbose_limit)
    !-----------------------------------------------------------------------!
    ! Writes the mesh input options and the mesh paremeters in a nice       !
    ! readable format. If a unit number is provided, it writes it to a file !
    ! otherwise it writes it to the standard output.                        !
    !-----------------------------------------------------------------------!
    type(mesh_t), intent(in) :: m
    integer,      intent(in), optional :: unit, verbose_limit

    call push_sub("mesh_output_params")

    ASSERT(m%type /= 0)

    message(1) = ""
    message(2) = "Mesh information:"
    select case (m%type)
    case (MESH_LINEAR)
      message(3) = "  Type: linear"
    case (MESH_LOG1)
      message(3) = "  Type: logarithmic [ri = b*exp(a*i)]"
    case (MESH_LOG2)
      message(3) = "  Type: logarithmic [ri = b*(exp(a*i) - 1)]"
    end select
    write(message(4), '("  Mesh starting point:   ",ES8.2E2,1X,A)') m%r(1)/units_out%length%factor, trim(units_out%length%abbrev)
    write(message(5), '("  Mesh outmost point:    ",F7.3,1X,A)') m%r(m%np)/units_out%length%factor, trim(units_out%length%abbrev)
    write(message(6), '("  Mesh parameters (a, b): ",ES12.5E2,", ",ES12.5E2)') m%a, m%b
    write(message(7), '("  Mesh number of points:  ",I5)') m%np
    select case (m%deriv_method)
    case (MESH_CUBIC_SPLINE)
      message(8) = "  Derivatives Method: Cubic slines"
    case (MESH_FINITE_DIFF)
      write(message(8), '("  Derivatives Method: Finite differences (order = ",I2,")")') m%fd_order
    end select

    if (present(unit)) then
      call write_info(8, unit=unit)
    else
      if (present(verbose_limit)) then
        call write_info(8, verbose_limit)
      else
        call write_info(8)
      end if
    end if
 
    call pop_sub()
  end subroutine mesh_output_params

  subroutine mesh_end(m)
    !-----------------------------------------------------------------------!
    ! Frees all memory associated to the mesh object m.                     !
    !-----------------------------------------------------------------------!
    type(mesh_t), intent(inout) :: m

    call push_sub("mesh_end")

    m%type = 0
    m%a    = M_ZERO
    m%b    = M_ZERO
    m%np   = 0
    m%intrp_method = 0
    m%interp_range = 0
    m%deriv_method = 0
    m%integ_method = 0
    if (associated(m%r)) then
      deallocate(m%r)
    end if
    m%fd_order = 0
    call fd_operator_end(m%deriv)
    call fd_operator_end(m%deriv2)
    call fd_operator_end(m%deriv3)

    call pop_sub()
  end subroutine mesh_end

end module mesh_m
