!! Copyright (C) 2004-2012 M. Oliveira, F. Nogueira
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!
!! $Id: hartree.F90 778 2013-07-11 15:49:39Z micael $

module hartree_m
  use global_m
  use io_m
  use units_m
  use mesh_m
  implicit none


                    !---Public/Private Statements---!

  private
  public :: hartree_potential, &
            hartree_output

contains

  subroutine hartree_potential(m, density, charge, vh, eh)
    !-----------------------------------------------------------------------!
    ! Calculates the Hartree potencial for a given density.                 !
    !                                                                       !
    ! m        - the radial mesh                                            !
    ! density  - the charge density                                         !
    ! vh       - the Hartree potential                                      !
    ! eh       - the Hartree energy                                         !
    !-----------------------------------------------------------------------!
    type(mesh_t),  intent(in)  :: m
    real(R8),      intent(in)  :: density(m%np), charge
    real(R8),      intent(out), optional :: vh(:,:)
    real(R8),      intent(out), optional :: eh

    integer :: i
    real(R8), allocatable :: vh_(:), int1(:), int2(:)
    
    !Allocate memory
    allocate(vh_(m%np), int1(m%np), int2(m%np))

    !Calculate the total density and auxiliary functions
    int1 = mesh_primitive(m, density)
    int2 = mesh_primitive(m, density, dv=m%r)

    !Calculate the potential
    do i = 1, m%np
      vh_(i) = (int1(i)/m%r(i) + int2(m%np) - int2(i))*M_FOUR*M_PI
    end do
    
    !Force potential to have the correct asymptotic limit
    if (vh_(m%np) /= M_ZERO .and. charge /= M_ZERO) then
      vh_ = vh_/vh_(m%np)*charge/m%r(m%np)
    end if

    if (present(vh)) then
      do i = 1, m%np
        vh(i,:) = vh_(i)
      end do
    end if

    if (present(eh)) then
      eh = M_TWO*M_PI*mesh_integrate(m, vh_*density)
    end if

    !Deallocate arrays
    deallocate(vh_, int1, int2)

  end subroutine hartree_potential

  subroutine hartree_output(m, density, charge, dir)
    !-----------------------------------------------------------------------!
    ! Calculates the Hartree potencial for a given density.                 !
    !                                                                       !
    ! m        - the radial mesh                                            !
    ! density  - the charge density                                         !
    !-----------------------------------------------------------------------!
    type(mesh_t),     intent(in) :: m
    real(R8),         intent(in) :: density(m%np), charge
    character(len=*), intent(in) :: dir

    integer :: i, unit
    real(R8), allocatable :: vh(:,:)

    allocate(vh(m%np,1))

    !Get hartree potential
    call hartree_potential(m, density, charge, vh=vh)

    !Open file
    call io_open(unit, file=trim(dir)//"/v_h")

    !Write header
    write(unit,'("# ")')
    write(unit,'("# Energy units: ",A)') trim(units_out%energy%name)
    write(unit,'("# Length units: ",A)') trim(units_out%length%name)
    write(unit,'("#")')
    write(unit,'("# ",36("-"))')
    write(unit,'("# |",8X,"r",7X,"|",7X,"v(r)",6X,"|")')
    write(unit,'("# ",36("-"))')

    !Ouput
    do i = 1, m%np
      write(unit,'(4(3X,ES15.8E2))') m%r(i)/units_out%length%factor, &
           vh(i,1)/units_out%energy%factor
    end do

    close(unit)
    deallocate(vh)

  end subroutine hartree_output

end module hartree_m
