/**
 * \file snd-sdl.c
 * \brief SDL sound support
 *
 * Copyright (c) 2004 Brendon Oliver <brendon.oliver@gmail.com>
 * Copyright (c) 2007 Andi Sidwell <andi@takkaria.org>
 * A large chunk of this file was taken and modified from main-ros.
 *
 * This work is free software; you can redistribute it and/or modify it
 * under the terms of either:
 *
 * a) the GNU General Public License as published by the Free Software
 *    Foundation, version 2, or
 *
 * b) the "Angband licence":
 *    This software may be copied and distributed for educational, research,
 *    and not for profit purposes provided that this copyright and statement
 *    are included in all such copies.  Other copyrights may also apply.
 */
#include "angband.h"
#include "init.h"
 
#ifdef SOUND_SDL


#include "SDL.h"
#include "SDL_mixer.h"


/**
 * Don't cache audio
 */
static bool no_cache_audio = FALSE;

/**
 * Using mp3s
 */
static bool use_mp3 = FALSE;

/**
 * Arbitary limit on number of samples per event
 */
#define MAX_SAMPLES      16

/**
 * Struct representing all data about an event sample
 */
typedef struct
{
	int num;                        /* Number of samples for this event */
	Mix_Chunk *wavs[MAX_SAMPLES];   /* Sample array */
	Mix_Music *mp3s[MAX_SAMPLES];   /* Sample array */
	char *paths[MAX_SAMPLES]; /* Relative pathnames for samples */
} sample_list;


/**
 * Just need an array of SampInfos
 */
static sample_list samples[MSG_MAX];


/**
 * Shut down the sound system and free resources.
 */
static void close_audio(void)
{
	size_t i;
	int j;

	/* Free all the sample data*/
	for (i = 0; i < MSG_MAX; i++) {
  		sample_list *smp = &samples[i];

		/* Nuke all samples */
		for (j = 0; j < smp->num; j++) {
			if (use_mp3)
				Mix_FreeMusic(smp->mp3s[j]);
			else
				Mix_FreeChunk(smp->wavs[j]);
			string_free(smp->paths[j]);
		}
	}

	/* Close the audio */
	Mix_CloseAudio();

	/* XXX This may conflict with the SDL port */
	SDL_Quit();
}


/**
 * Initialise SDL and open the mixer
 */
static bool open_audio(void)
{
	int audio_rate;
	Uint16 audio_format;
	int audio_channels;
	
	/* Initialize variables */
	audio_rate = 22050;
	audio_format = AUDIO_S16;
	audio_channels = 2;

	/* Initialize the SDL library */
	if (SDL_Init(SDL_INIT_AUDIO) < 0) {
		plog_fmt("Couldn't initialize SDL: %s", SDL_GetError());
		return FALSE;
	}

	/* Try to open the audio */
	if (Mix_OpenAudio(audio_rate, audio_format, audio_channels, 4096) < 0) {
		plog_fmt("Couldn't open mixer: %s", SDL_GetError());
		return FALSE;
	}

	/* Success */
	return TRUE;
}



/**
 * Read sound.cfg and map events to sounds; then load all the sounds into
 * memory to avoid I/O latency later.
 */
static bool sound_sdl_init(bool no_cache)
{
	char path[2048];
	char buffer[2048];
	ang_file *fff;


	/* Initialise the mixer  */
	if (!open_audio())
	    return FALSE;


	/* Find and open the config file */
	path_build(path, sizeof(path), ANGBAND_DIR_SOUNDS, "sound.cfg");
	fff = file_open(path, MODE_READ, -1);

	/* Handle errors */
	if (!fff) {
		plog_fmt("Failed to open sound config (%s):\n    %s", 
		          path, strerror(errno));
		return FALSE;
	}

	/* Parse the file */
	/* Lines are always of the form "name = sample [sample ...]" */
	while (file_getl(fff, buffer, sizeof(buffer))) {
		char *msg_name;
		char *sample_list;
		char *search;
		char *cur_token;
		char *next_token;
		int event;

		/* Skip anything not beginning with an alphabetic character */
		if (!buffer[0] || !isalpha((unsigned char)buffer[0])) continue;

		/* Split the line into two: message name, and the rest */
		search = strchr(buffer, ' ');
		sample_list = strchr(search + 1, ' ');
		if (!search) continue;
		if (!sample_list) continue;

		/* Set the message name, and terminate at first space */
		msg_name = buffer;
		search[0] = '\0';


		/* Make sure this is a valid event name */
		event = message_lookup_by_sound_name(msg_name);
		if (event < 0) continue;

		/* Advance the sample list pointer so it's at the beginning of text */
		sample_list++;
		if (!sample_list[0]) continue;

		/* Terminate the current token */
		cur_token = sample_list;
		search = strchr(cur_token, ' ');
		if (search) {
			search[0] = '\0';
			next_token = search + 1;
		} else {
			next_token = NULL;
		}

        /*
         * Now we find all the sample names and add them one by one
         */
        while (cur_token) {
			int num = samples[event].num;
			bool got_file_type = FALSE;

			/* Don't allow too many samples */
			if (num >= MAX_SAMPLES) break;

			/* Build the path to the sample */
			path_build(path, sizeof(path), ANGBAND_DIR_SOUNDS, cur_token);
			if (!file_exists(path)) goto next_token;

			if (!got_file_type) {
			        if (streq(path + strlen(path) - 3, "mp3")) {
				        use_mp3 = TRUE;
					got_file_type = TRUE;
				}
			}

			/* Don't load now if we're not caching */
			if (no_cache) {
				/* Just save the path for later */
				samples[event].paths[num] = string_make(path);
			} else {
				/* Load the file now */
				if (use_mp3) {
					samples[event].mp3s[num] = Mix_LoadMUS(path);
					if (!samples[event].mp3s[num]) {
						plog_fmt("%s: %s", SDL_GetError(), strerror(errno));
						goto next_token;
					}
				} else {
					samples[event].wavs[num] = Mix_LoadWAV(path);
					if (!samples[event].wavs[num]) {
					        plog_fmt("%s: %s", SDL_GetError(), strerror(errno));
						goto next_token;
					}
				}
			}

			/* Imcrement the sample count */
			samples[event].num++;

		next_token:

			/* Figure out next token */
			cur_token = next_token;
			if (next_token) {
				/* Try to find a space */
				search = strchr(cur_token, ' ');

				/* If we can find one, terminate, and set new "next" */
				if (search) {
					search[0] = '\0';
					next_token = search + 1;
				} else {
					/* Otherwise prevent infinite looping */
					next_token = NULL;
				}
			}
		}
	}

	/* Close the file */
	file_close(fff);


	/* Success */
	return TRUE;
}

/**
 * Play a sound of type "event".
 */
static void play_sound(game_event_type type, game_event_data *data, void *user)
{
	Mix_Chunk *wave = NULL;
	Mix_Music *mp3 = NULL;
	int s;

	int event = data->message.type;

	/* Paranoia */
	if (event < 0 || event >= MSG_MAX) return;

	/* Check there are samples for this event */
	if (!samples[event].num) return;

	/* Choose a random event */
	s = randint0(samples[event].num);
	if (use_mp3)
	        mp3 = samples[event].mp3s[s];
	else
	        wave = samples[event].wavs[s];

	/* Try loading it, if it's not cached */
	if (!(wave || mp3)) {
		/* Verify it exists */
		const char *filename = samples[event].paths[s];
		if (!file_exists(filename)) return;

		/* Load */
		if (use_mp3)
		        mp3 = Mix_LoadMUS(filename);
		else
		        wave = Mix_LoadWAV(filename);
	}

	/* Check to see if we have a sound again */
	if (!(wave || mp3)) {
		plog("SDL sound load failed.");
		return;
	}

	/* Actually play the thing */
	if (use_mp3)
	        Mix_PlayMusic(mp3, 1);
	else
	        Mix_PlayChannel(-1, wave, 0);
}


/**
 * Init the SDL sound "module".
 */
errr init_sound_sdl(int argc, char **argv)
{
	int i;

	/* Parse args */
	for (i = 1; i < argc; i++) {
		if (prefix(argv[i], "-c")) {
			no_cache_audio = TRUE;
			plog("Audio cache disabled.");
			continue;
		}
	}

	/* Load sound preferences if requested */
	if (!sound_sdl_init(no_cache_audio)) {
		plog("Failed to load sound config");

		/* Failure */
		return (1);
	}

	/* Enable sound */
	event_add_handler(EVENT_SOUND, play_sound, NULL);
	atexit(close_audio);

	/* Success */
	return (0);
}


#endif /* SOUND_SDL */
