/*
 *  hermes_info.m
 * 
 *  802.11b airport control app 020514 ragge@nada.kth.se
 */

#include <stdio.h>

#include "Apple80211.h"

#include "hermes_info.h"

typedef uint8_t u8;
typedef uint16_t u16;
typedef uint32_t u32;
#define ETH_ALEN 6
#include "linux_hermes_rid.h"

#include "hermes_more.h"

#define BUFSIZE 1000

extern WirelessContextPtr gWCtxt;

struct ltv {
  uint16_t   rid;
  char       *ridn;
  char       *desc;
  int        len;
  void       (*func)(struct ltv *, void *);
  char       *comment;
};

void ltv_short(struct ltv *l, void *data);
void ltv_shorthex(struct ltv *l, void *data);
void ltv_mac(struct ltv *l, void *data);
void ltv_lstring(struct ltv *l, void *data);
void ltv_string(struct ltv *l, void *data);
void ltv_hex(struct ltv *l, void *data);
void ltv_regdomains(struct ltv *l, void *data);
void ltv_profile(struct ltv *l, void *data);
void ltv_commsqual(struct ltv *l, void *data);
void ltv_scalethr(struct ltv *l, void *data);
void ltv_supdatarates(struct ltv *l, void *data);

/*
 * Configuration RIDs
 */
struct ltv cnf_rids[] = {
  {HERMES_RID_CNFPORTTYPE, "CNFPORTTYPE", "Port type", 2, ltv_short,
   "   (1=Basic Service Set (BSS) | 2=Wireless Distribution System (WDS) | 3=Pseudo IBSS)\n"},
  {HERMES_RID_CNFOWNMACADDR, "CNFOWNMACADDR", "Own MAC address", 6, ltv_mac, 0},
  {HERMES_RID_CNFDESIREDSSID, "CNFDESIREDSSID", "Desired network name", 34, ltv_lstring, 0},
  {HERMES_RID_CNFOWNCHANNEL, "CNFOWNCHANNEL", "Own P2P channel", 2, ltv_short, 0},
  {HERMES_RID_CNFOWNSSID, "CNFOWNSSID", "Own P2P or ESS network name", 34, ltv_lstring, 0},
  {HERMES_RID_CNFOWNATIMWINDOW, "CNFOWNATIMWINDOW", "ATIM window time for IBSS creation", 2, ltv_short, 0},
  {HERMES_RID_CNFSYSTEMSCALE, "CNFSYSTEMSCALE", "AP density", 2, ltv_short,
   "   (1=Low density | 2=Medium density | 3=High density)\n"},
  {HERMES_RID_CNFMAXDATALEN, "CNFMAXDATALEN", "Max frame body size", 2, ltv_short, 0},
  {HERMES_RID_CNFWDSADDRESS, "CNFWDSADDRESS", "Address of corr WDS Link node", 6, ltv_mac, 0},
  {HERMES_RID_CNFPMENABLED, "CNFPMENABLED", "ESS Power Management", 2, ltv_short, 0},
#if 0
  {HERMES_RID_CNFPMEPS, "CNFPMEPS", "ESS PM EPS/PS Mode", 2, ltv_short, 0},
#endif
  {HERMES_RID_CNFMULTICASTRECEIVE, "CNFMULTICASTRECEIVE", "Multicast reception", 2, ltv_short, 0},
  {HERMES_RID_CNFMAXSLEEPDURATION, "CNFMAXSLEEPDURATION", "Max sleep time", 2, ltv_short, 0},
  {HERMES_RID_CNFPMHOLDOVERDURATION, "CNFPMHOLDOVERDURATION", "Holdover time", 2, ltv_short, 0},
  {HERMES_RID_CNFOWNNAME, "CNFOWNNAME", "Name for diagnostics", 34, ltv_lstring, 0},
#if 0
  /* AP only */
  {HERMES_RID_CNFOWNDTIMPERIOD, "CNFOWNDTIMPERIOD", "Beacon intervals between successive DTIMs", 2, ltv_short, 0},
  {HERMES_RID_CNFWDSADDRESS1, "CNFWDSADDRESS1", "Port 1 address of corr WDS Link node", 6, ltv_mac, 0},
  {HERMES_RID_CNFWDSADDRESS2, "CNFWDSADDRESS2", "Port 2 address of corr WDS Link node", 6, ltv_mac, 0},
  {HERMES_RID_CNFWDSADDRESS3, "CNFWDSADDRESS3", "Port 3 address of corr WDS Link node", 6, ltv_mac, 0},
  {HERMES_RID_CNFWDSADDRESS4, "CNFWDSADDRESS4", "Port 4 address of corr WDS Link node", 6, ltv_mac, 0},
  {HERMES_RID_CNFWDSADDRESS5, "CNFWDSADDRESS5", "Port 5 address of corr WDS Link node", 6, ltv_mac, 0},
  {HERMES_RID_CNFWDSADDRESS6, "CNFWDSADDRESS6", "Port 6 address of corr WDS Link node", 6, ltv_mac, 0},
  {HERMES_RID_CNFMULTICASTPMBUFFERING, "CNFMULTICASTPMBUFFERING", "PM buffering of mcast msgs", 2, ltv_short, 0},
  {HERMES_RID_CNFWEPENABLED_AGERE, "CNFWEPENABLED_AGERE", "WEP Enabled", 2, ltv_short, 0},
  {HERMES_RID_CNFMANDATORYBSSID_SYMBOL, "CNFMANDATORYBSSID_SYMBOL", "", 2, ltv_short, 0},
#endif
#if 0
  {HERMES_RID_CNFWEPDEFAULTKEYID, "CNFWEPDEFAULTKEYID", "", 2, ltv_short, 0},
  {HERMES_RID_CNFDEFAULTKEY0, "CNFDEFAULTKEY0", "", 16, ltv_hex, 0},
  {HERMES_RID_CNFDEFAULTKEY1, "CNFDEFAULTKEY1", "", 16, ltv_hex, 0},
#endif
  {HERMES_RID_CNFMWOROBUST_AGERE, "CNFMWOROBUST_AGERE", "Microwave (Oven) Robustness", 2, ltv_short, 0},
#if 0
  {HERMES_RID_CNFDEFAULTKEY2, "CNFDEFAULTKEY2", "", 16, ltv_hex, 0},
  {HERMES_RID_CNFDEFAULTKEY3, "CNFDEFAULTKEY3", "", 16, ltv_hex, 0},
  {HERMES_RID_CNFWEPFLAGS_INTERSIL, "CNFWEPFLAGS_INTERSIL", "", 2, ltv_short, 0},
#endif
#if 0
  {HERMES_RID_CNFWEPKEYMAPPINGTABLE, "CNFWEPKEYMAPPINGTABLE", "", 2, ltv_short, 0},
  {HERMES_RID_CNFAUTHENTICATION, "CNFAUTHENTICATION", "", 2, ltv_short, 0},
  {HERMES_RID_CNFMAXASSOCSTA, "CNFMAXASSOCSTA", "", 2, ltv_short, 0},
  {HERMES_RID_CNFKEYLENGTH_SYMBOL, "CNFKEYLENGTH_SYMBOL", "", 2, ltv_short, 0},
  {HERMES_RID_CNFTXCONTROL, "CNFTXCONTROL", "", 2, ltv_short, 0},
  {HERMES_RID_CNFROAMINGMODE, "CNFROAMINGMODE", "", 2, ltv_short, 0},
  {HERMES_RID_CNFHOSTAUTHENTICATION, "CNFHOSTAUTHENTICATION", "", 2, ltv_short, 0},
  {HERMES_RID_CNFRCVCRCERROR, "CNFRCVCRCERROR", "", 2, ltv_short, 0},
  {HERMES_RID_CNFMMLIFE, "CNFMMLIFE", "", 2, ltv_short, 0},
  {HERMES_RID_CNFALTRETRYCOUNT, "CNFALTRETRYCOUNT", "", 2, ltv_short, 0},
  {HERMES_RID_CNFBEACONINT, "CNFBEACONINT", "", 2, ltv_short, 0},
  {HERMES_RID_CNFAPPCFINFO, "CNFAPPCFINFO", "", 2, ltv_short, 0},
  {HERMES_RID_CNFSTAPCFINFO, "CNFSTAPCFINFO", "", 2, ltv_short, 0},
  {HERMES_RID_CNFPRIORITYQUSAGE, "CNFPRIORITYQUSAGE", "", 2, ltv_short, 0},
  {HERMES_RID_CNFTIMCTRL, "CNFTIMCTRL", "", 2, ltv_short, 0},
  {HERMES_RID_CNFTHIRTY2TALLY, "CNFTHIRTY2TALLY", "", 2, ltv_short, 0},
  {HERMES_RID_CNFENHSECURITY, "CNFENHSECURITY", "", 2, ltv_short, 0},
#endif
  {HERMES_RID_CNFGROUPADDRESSES, "CNFGROUPADDRESSES", "Address for RX-message", 6, ltv_mac, 0},
  {HERMES_RID_CNFCREATEIBSS, "CNFCREATEIBSS", "Create P2P network", 2, ltv_short, 0},
  {HERMES_RID_CNFFRAGMENTATIONTHRESHOLD, "CNFFRAGMENTATIONTHRESHOLD", "Max unicast TX frag len", 2, ltv_short, 0},
  {HERMES_RID_CNFRTSTHRESHOLD, "CNFRTSTHRESHOLD", "Frame len threshold for RTS/CTS handshake", 2, ltv_short, 0},
  {HERMES_RID_CNFTXRATECONTROL, "CNFTXRATECONTROL", "Data rate control for TX", 2, ltv_short,
   "   (1=Fixed 1 Mbit | 2=Fixed 2 Mbit | 3=Auto Fallback)\n"},
  {HERMES_RID_CNFPROMISCUOUSMODE, "CNFPROMISCUOUSMODE", "Promiscuous mode", 2, ltv_short, 0},
#if 0
  {HERMES_RID_CNFBASICRATES_SYMBOL, "CNFBASICRATES_SYMBOL", "", 2, ltv_short, 0},
  {HERMES_RID_CNFPREAMBLE_SYMBOL, "CNFPREAMBLE_SYMBOL", "", 2, ltv_short, 0},
  /* AP only */
  {HERMES_RID_CNFFRAGMENTATIONTHRESHOLD0, "CNFFRAGMENTATIONTHRESHOLD0", "Port 0 frag len unicast TX", 2, ltv_short, 0},
  {HERMES_RID_CNFFRAGMENTATIONTHRESHOLD1, "CNFFRAGMENTATIONTHRESHOLD1", "Port 1 frag len unicast TX", 2, ltv_short, 0},
  {HERMES_RID_CNFFRAGMENTATIONTHRESHOLD2, "CNFFRAGMENTATIONTHRESHOLD2", "Port 2 frag len unicast TX", 2, ltv_short, 0},
  {HERMES_RID_CNFFRAGMENTATIONTHRESHOLD3, "CNFFRAGMENTATIONTHRESHOLD3", "Port 3 frag len unicast TX", 2, ltv_short, 0},
  {HERMES_RID_CNFFRAGMENTATIONTHRESHOLD4, "CNFFRAGMENTATIONTHRESHOLD4", "Port 4 frag len unicast TX", 2, ltv_short, 0},
  {HERMES_RID_CNFFRAGMENTATIONTHRESHOLD5, "CNFFRAGMENTATIONTHRESHOLD5", "Port 5 frag len unicast TX", 2, ltv_short, 0},
  {HERMES_RID_CNFFRAGMENTATIONTHRESHOLD6, "CNFFRAGMENTATIONTHRESHOLD6", "Port 6 frag len unicast TX", 2, ltv_short, 0},
  {HERMES_RID_CNFRTSTHRESHOLD0, "CNFRTSTHRESHOLD0", "Port 0 frame len threshold RTS/CTS hshake", 2, ltv_short, 0},
  {HERMES_RID_CNFRTSTHRESHOLD1, "CNFRTSTHRESHOLD1", "Port 1 frame len threshold RTS/CTS hshake", 2, ltv_short, 0},
  {HERMES_RID_CNFRTSTHRESHOLD2, "CNFRTSTHRESHOLD2", "Port 2 frame len threshold RTS/CTS hshake", 2, ltv_short, 0},
  {HERMES_RID_CNFRTSTHRESHOLD3, "CNFRTSTHRESHOLD3", "Port 3 frame len threshold RTS/CTS hshake", 2, ltv_short, 0},
  {HERMES_RID_CNFRTSTHRESHOLD4, "CNFRTSTHRESHOLD4", "Port 4 frame len threshold RTS/CTS hshake", 2, ltv_short, 0},
  {HERMES_RID_CNFRTSTHRESHOLD5, "CNFRTSTHRESHOLD5", "Port 5 frame len threshold RTS/CTS hshake", 2, ltv_short, 0},
  {HERMES_RID_CNFRTSTHRESHOLD6, "CNFRTSTHRESHOLD6", "Port 6 frame len threshold RTS/CTS hshake", 2, ltv_short, 0},
  {HERMES_RID_CNFTXRATECONTROL0, "CNFTXRATECONTROL0", "Port 0 TX data rate", 2, ltv_short, 0},
  {HERMES_RID_CNFTXRATECONTROL1, "CNFTXRATECONTROL1", "Port 1 TX data rate", 2, ltv_short, 0},
  {HERMES_RID_CNFTXRATECONTROL2, "CNFTXRATECONTROL2", "Port 2 TX data rate", 2, ltv_short, 0},
  {HERMES_RID_CNFTXRATECONTROL3, "CNFTXRATECONTROL3", "Port 3 TX data rate", 2, ltv_short, 0},
  {HERMES_RID_CNFTXRATECONTROL4, "CNFTXRATECONTROL4", "Port 4 TX data rate", 2, ltv_short, 0},
  {HERMES_RID_CNFTXRATECONTROL5, "CNFTXRATECONTROL5", "Port 5 TX data rate", 2, ltv_short, 0},
  {HERMES_RID_CNFTXRATECONTROL6, "CNFTXRATECONTROL6", "Port 6 TX data rate", 2, ltv_short, 0},
  {HERMES_RID_CNFSHORTPREAMBLE, "CNFSHORTPREAMBLE", "", 2, ltv_short, 0},
#endif
#if 0
  {HERMES_RID_CNFWEPKEYS_AGERE, "CNFWEPKEYS_AGERE", "Encryption keys", 100, ltv_hex, 0},
#endif
#if 0
  {HERMES_RID_CNFEXCLUDELONGPREAMBLE, "CNFEXCLUDELONGPREAMBLE", "", 2, ltv_short, 0},
#endif
#if 0
  {HERMES_RID_CNFTXKEY_AGERE, "CNFTXKEY_AGERE", "Key used for sending", 100, ltv_hex, 0},
#endif
  {HERMES_RID_CNFAUTHENTICATIONRSPTO, "CNFAUTHENTICATIONRSPTO", "", 34, ltv_lstring, 0},
#if 1
  {HERMES_RID_CNFBASICRATES, "CNFBASICRATES", "", 2, ltv_shorthex, 0},
  {HERMES_RID_CNFSUPPORTEDRATES, "CNFSUPPORTEDRATES", "", 2, ltv_shorthex, 0},
#endif
  {HERMES_RID_CNFTICKTIME, "CNFTICKTIME", "Aux timer tick interval", 2, ltv_short, 0},
#if 0
  {HERMES_RID_CNFSCANREQUEST, "CNFSCANREQUEST", "", 2, ltv_short, 0},
  {HERMES_RID_CNFJOINREQUEST, "CNFJOINREQUEST", "", 2, ltv_short, 0},
  {HERMES_RID_CNFAUTHENTICATESTATION, "CNFAUTHENTICATESTATION", "", 2, ltv_short, 0},
  {HERMES_RID_CNFCHANNELINFOREQUEST, "CNFCHANNELINFOREQUEST", "", 2, ltv_short, 0},
#endif
  {0, 0, 0, 0, 0, 0}
};

/*
 * Information RIDs
 */
struct ltv inf_rids[] = {
  {HERMES_RID_MAXLOADTIME, "RID_MAXLOADTIME", "Max resp time of download command", 2, ltv_short, 0},
  {HERMES_RID_DOWNLOADBUFFER, "RID_DOWNLOADBUFFER", "Download buffer location and size", 6, ltv_hex, 0}, /* XXX */
  {HERMES_RID_PRIID, "RID_PRIID", "Primary functions firmware ID", 8, ltv_hex, 0},
  {HERMES_RID_PRISUPRANGE, "RID_PRISUPRANGE", "Primary functions compatibility range", 10, ltv_hex, 0},
  {HERMES_RID_CFIACTRANGES, "RID_CFIACTRANGES", "Controller interface compatibility range", 10, ltv_hex, 0},
  {HERMES_RID_NICSERNUM, "RID_NICSERNUM", "Card serial number", 40, ltv_string, 0},
  {HERMES_RID_NICID, "RID_NICID", "Card ID", 8, ltv_hex, 0},
  {HERMES_RID_MFISUPRANGE, "RID_MFISUPRANGE", "Modem ifc compatibility range", 10, ltv_hex, 0},
  {HERMES_RID_CFISUPRANGE, "RID_CFISUPRANGE", "Controller ifc compatibility range", 10, ltv_hex, 0},
  {HERMES_RID_CHANNELLIST, "RID_CHANNELLIST", "Allowed channels", 2, ltv_shorthex, 0},
  {HERMES_RID_REGULATORYDOMAINS, "RID_REGULATORYDOMAINS", "List of regulatory domains", 100, ltv_regdomains, 0},
  {HERMES_RID_TEMPTYPE, "RID_TEMPTYPE", "Hardware temperature range code", 2, ltv_short, 0},
#if 0
  {HERMES_RID_CIS, "RID_CIS", "PC Card information structure", 100, ltv_hex, 0},
#endif
  {HERMES_RID_PROFILE, "RID_PROFILE", "PC Card Profile", 6, ltv_profile, 0},
  {HERMES_RID_STAID, "RID_STAID", "Station Functions firmware ID", 8, ltv_hex, 0},
  {HERMES_RID_STASUPRANGE, "RID_STASUPRANGE", "Station Funcs compatibility range", 10, ltv_hex, 0},
  {HERMES_RID_MFIACTRANGES, "RID_MFIACTRANGES", "Modem ifc compatibility ranges", 0x28, ltv_hex, 0},
  {HERMES_RID_CFIACTRANGES2, "RID_CFIACTRANGES2", "Controller ifc compatibility ranges", 10, ltv_hex, 0},
#if 0
  {HERMES_RID_SECONDARYVERSION_SYMBOL, "RID_SECONDARYVERSION_SYMBOL", "", 2, ltv_short, 0},
#endif
  {HERMES_RID_PORTSTATUS, "RID_PORTSTATUS", "Actual port status", 2, ltv_short, 
  "   (Uncertain: 1=Off | 2=Conn bad | 3=AdHoc Create | 4=BSS(Client) | 5=Out of range)\n"},
  {HERMES_RID_CURRENTSSID, "RID_CURRENTSSID", "Actual Network name", 34, ltv_lstring, 0},
  {HERMES_RID_CURRENTBSSID, "RID_CURRENTBSSID", "Base station address", 6, ltv_mac, 0},
  {HERMES_RID_COMMSQUALITY, "RID_COMMSQUALITY", "Quality of connection", 6, ltv_commsqual, 0},
  {HERMES_RID_CURRENTTXRATE, "RID_CURRENTTXRATE", "Current TX data rate", 2, ltv_short, 0},
  {HERMES_RID_CURRENTBEACONINTERVAL, "RID_CURRENTBEACONINTERVAL", "Beacon transmit interval for P2P in ms", 2, ltv_short, 0},
  {HERMES_RID_CURRENTSCALETHRESHOLDS, "RID_CURRENTSCALETHRESHOLDS", "System scale thresholds", 12, ltv_scalethr, 0},
  {HERMES_RID_PROTOCOLRSPTIME, "RID_PROTOCOLRSPTIME", "Timeout for response", 2, ltv_short, 0},
  {HERMES_RID_SHORTRETRYLIMIT, "RID_SHORTRETRYLIMIT", "Max retransmits - short frames", 2, ltv_short, 0},
  {HERMES_RID_LONGRETRYLIMIT, "RID_LONGRETRYLIMIT", "Max retransmits - long frames", 2, ltv_short, 0},
  {HERMES_RID_MAXTRANSMITLIFETIME, "RID_MAXTRANSMITLIFETIME", "Max TX frame handling time", 2, ltv_short, 0},
  {HERMES_RID_MAXRECEIVELIFETIME, "RID_MAXRECEIVELIFETIME", "Max RX frame handling time", 2, ltv_short, 0},
  {HERMES_RID_CFPOLLABLE, "RID_CFPOLLABLE", "Contention Free Pollable capability", 2, ltv_short, 0},
  {HERMES_RID_AUTHENTICATIONALGORITHMS, "RID_AUTHENTICATIONALGORITHMS", "Available auth algorithms", 4, ltv_hex, 0},
  {HERMES_RID_PRIVACYOPTIONIMPLEMENTED, "RID_PRIVACYOPTIONIMPLEMENTED", "WEP option availability", 2, ltv_short, 0},
  {HERMES_RID_CURRENTREMOTERATES, "RID_CURRENTREMOTERATES", "Current remote rates", 2, ltv_shorthex, 0},
  {HERMES_RID_CURRENTUSEDRATES, "RID_CURRENTUSEDRATES", "Current used rates", 2, ltv_shorthex, 0},
#if 0
  /* AP Only */
  {HERMES_RID_CURRENTTXRATE1, "RID_CURRENTTXRATE1", "Port 1 actual TX data rate", 2, ltv_short, 0},
  {HERMES_RID_CURRENTTXRATE2, "RID_CURRENTTXRATE2", "Port 2 actual TX data rate", 2, ltv_short, 0},
  {HERMES_RID_CURRENTTXRATE3, "RID_CURRENTTXRATE3", "Port 3 actual TX data rate", 2, ltv_short, 0},
  {HERMES_RID_CURRENTTXRATE4, "RID_CURRENTTXRATE4", "Port 4 actual TX data rate", 2, ltv_short, 0},
  {HERMES_RID_CURRENTTXRATE5, "RID_CURRENTTXRATE5", "Port 5 actual TX data rate", 2, ltv_short, 0},
  {HERMES_RID_CURRENTTXRATE6, "RID_CURRENTTXRATE6", "Port 6 actual TX data rate", 2, ltv_short, 0},
  {HERMES_RID_OWNMACADDR, "RID_OWNMACADDR", "", 6, ltv_mac, 0},
  {HERMES_RID_POINTCOORDFUNC, "CFG_PCF_INFO", "Point Coord Function capability", 10, ltv_hex, 0},
#endif
#if 0
  {HERMES_RID_SCANRESULTSTABLE, "RID_SCANRESULTSTABLE", "", 100, ltv_hex, 0},
#endif
  {HERMES_RID_PHYTYPE, "RID_PHYTYPE", "Physical layer type", 2, ltv_short, 0},
  {HERMES_RID_CURRENTCHANNEL, "RID_CURRENTCHANNEL", "Channel used", 2, ltv_short, 0},
  {HERMES_RID_CURRENTPOWERSTATE, "RID_CURRENTPOWERSTATE", "Current power state", 2, ltv_short, 0},
  {HERMES_RID_CCAMODE, "RID_CCAMODE", "Clear channel assessment mode", 2, ltv_short, 0},
  {HERMES_RID_CCATIME, "RID_CCATIME", "Clear channel assessment time", 2, ltv_short, 0},
  {HERMES_RID_MACPROCESSINGDELAY, "RID_MACPROCESSINGDELAY", "Mac processing delay time", 2, ltv_short, 0},
  {HERMES_RID_SUPPORTEDDATARATES, "RID_SUPPORTEDDATARATES", "Data rates capability", 10, ltv_supdatarates, 0},
  {HERMES_RID_BUILDSEQ, "RID_BUILDSEQ", "", 2, ltv_short, 0},
  {HERMES_RID_FWID, "RID_FWID", "", 0x28, ltv_hex, 0}, /* XXX */
  {0, 0, 0, 0, 0, 0}
};



#define S16(x) ( ((x >> 8) & 0x00FF) | \
		 ((x << 8) & 0xFF00) )
#define S32(x) ( ((x >> 24) & 0x000000FF) | \
		 ((x >> 8)  & 0x0000FF00) | \
		 ((x << 8)  & 0x00FF0000) | \
		 ((x << 24) & 0xFF000000) )


void ltv_pnam(struct ltv *l)
{
  printf("%s (%s) (0x%04x) = ", l->ridn, l->desc, l->rid);
}


void ltv_short(struct ltv *l, void *data)
{
  ltv_pnam(l);
  printf("%d\n", S16( *((short *)data) ) );
}


void ltv_shorthex(struct ltv *l, void *data)
{
  ltv_pnam(l);
  printf("0x%04x\n", S16( *((short *)data) ) );
}


void ltv_mac(struct ltv *l, void *data)
{
  unsigned char *d = (unsigned char *) data;

  ltv_pnam(l);
  printf("%02x:%02x:%02x:%02x:%02x:%02x\n",
	 d[0], d[1], d[2], d[3], d[4], d[5]);
}


void ltv_lstring(struct ltv *l, void *data)
{
  char buf[BUFSIZE+1];
  int i;

  i = S16( *((short*) data) );
  if(i > BUFSIZE)
    i = BUFSIZE;
  memcpy(buf, ((char *)data) + 2, i);
  buf[i] = '\0';

  ltv_pnam(l);
  printf("\"%s\"\n", buf);
}


void ltv_string(struct ltv *l, void *data)
{
  ltv_pnam(l);
  printf("\"%s\"\n", (char *) data);
}


void ltv_hex(struct ltv *l, void *data)
{
  unsigned char *d = (unsigned char *)data;
  int i;

  ltv_pnam(l);
  printf("\n");

  for(i = 0; i < l->len; ) {
    int i2 = i;

    printf("  %02x:", i);

    while(i < l->len) {
      if( ! (i % 4))
	printf(" ");
      printf("%02x", d[i]);
      if( ! (++i % 16) )
	break;
    }

    printf("   ");

    if( i % 16 ) {
      int space = 2 * ( 16 - (i % 16) );
      if(space >= 8) space++;
      if(space >= 16) space++;
      if(space >= 24) space++;

      while(space--)
	printf(" ");
    }

    while(i2 < l->len) {
      unsigned char c = d[i2];
      printf("%c", isgraph(c) ? c : '.');
      if( ! (++i2 % 16) )
	break;
    }
    
    printf("\n");
  }
}


void
ltv_regdomains(struct ltv *l, void *data)
{
  int count, i;
  struct hermes_regulatory_domains *d = (struct hermes_regulatory_domains *) data;

  ltv_pnam(l);

  count = S16(d->count);
  if(count > 40)
    count = 40;
  printf("%d: ", count);
  for(i = 0; i < count; i++) {
    printf("%d%s", d->d[i], i < (count-1) ? ", " : "");
  }
  
  printf("\n");
}


void
ltv_profile(struct ltv *l, void *data)
{
  struct hermes_profile *d = (struct hermes_profile *) data;

  ltv_pnam(l);

  printf("\n  Code: 0x%04x  Capabilites: 0x%04x  Allowed Data Rates: 0x%04x\n",
	 S16(d->code), S16(d->capabilities), S16(d->allowed_data_rates));
}


void
ltv_commsqual(struct ltv *l, void *data)
{
  struct hermes_comms_quality *d = (struct hermes_comms_quality *) data;

  ltv_pnam(l);

  printf("\n  Comm Quality: %d  Signal: %d  Noise: %d\n",
	 S16(d->comms_qual), S16(d->signal_lvl), S16(d->noise_lvl));
}


void
ltv_scalethr(struct ltv *l, void *data)
{
  struct hermes_scale_thresholds *d = (struct hermes_scale_thresholds *) data;

  ltv_pnam(l);

  printf("\n  RX Energy Detect: %d  RX Carrier Detect: %d  RX Defer: %d\n",
	 S16(d->energy_detect_thrh), S16(d->carrier_detect_thrh), S16(d->defer_thrh));
  printf("  Roaming Cell Search: %d  Roaming Out of Range: %d  Roaming Delta SNR: %d\n",
	 S16(d->cell_search_thrh), S16(d->out_of_range_thrh), S16(d->delta_snr));
}


void
ltv_supdatarates(struct ltv *l, void *data)
{
  int i;
  struct hermes_supported_data_rates *d = (struct hermes_supported_data_rates *) data;

  ltv_pnam(l);

  for(i = 0; i < 5; i++) {
    printf("0x%04x%s", S16(d->rates[i]), i < 4 ? ", " : "\n");
  }
}


void
dump_rids(struct ltv *rids)
{
  int i;
  unsigned char buf[BUFSIZE];
  SInt32 err;

  // clear buffer
  memset(buf, 0, BUFSIZE);

  for(i = 0; rids[i].rid != 0; i++) {

    // clear buf
    err = WirelessHCF_GetInfo(gWCtxt, HERMES_RID_SCANRESULTSTABLE, 100, buf);
    err = WirelessHCF_GetInfo(gWCtxt, 0xffff, 100, buf);
    err = WirelessHCF_GetInfo(gWCtxt, HERMES_RID_CNFOWNSSID, 100, buf);

    err = WirelessHCF_GetInfo(gWCtxt, rids[i].rid, rids[i].len, buf);
    if(err) {
      fprintf(stderr, "Error: WirelessHCF_GetInfo for \"%s\" (0x%04x) failed! (%d)\n",
	      rids[i].ridn, rids[i].rid, (int) err);
    } else {
      rids[i].func(rids + i, buf);
    }

    if(rids[i].comment)
      printf("%s", rids[i].comment);
  }
}


WIErr
wlHermesTallies(void)
{
  SInt32 err;
  struct hermes_tallies_frame_fixed t;
  uint32_t i, *p;

  /* do this twice, the counters doesn't seem to be updated when doing to call */
  err = WirelessHCF_GetInfo(gWCtxt, HERMES_INQ_TALLIES, sizeof(t), (void*) &t);
  if(!err)
    err = WirelessHCF_GetInfo(gWCtxt, HERMES_INQ_TALLIES, sizeof(t), (void*) &t);
  if(err) {
    fprintf(stderr, "Error: WirelessHCF_GetInfo for \"HERMES_INQ_TALLIES\" (0x%04x) failed! (%d)\n",
	    HERMES_INQ_TALLIES, (int) err);
    return err;
  }

  p = (uint32_t *) &p;
  for(i = 0; i < (sizeof(t) / sizeof(*p)); i++) {
    register uint32_t a, b;
    a = p[i];
    b = S32(a);
    p[i] = a;
  }

  printf("  TxUnicastFrames:         %d\n", t.TxUnicastFrames);
  printf("  TxMulticastFrames:       %d\n", t.TxMulticastFrames);
  printf("  TxFragments:             %d\n", t.TxFragments);
  printf("  TxUnicastOctets:         %d\n", t.TxUnicastOctets);
  printf("  TxMulticastOctets:       %d\n", t.TxMulticastOctets);
  printf("  TxDeferredTransmissions: %d\n", t.TxDeferredTransmissions);
  printf("  TxSingleRetryFrames:     %d\n", t.TxSingleRetryFrames);
  printf("  TxMultipleRetryFrames:   %d\n", t.TxMultipleRetryFrames);
  printf("  TxRetryLimitExceeded:    %d\n", t.TxRetryLimitExceeded);
  printf("  TxDiscards:              %d\n", t.TxDiscards);
  printf("  RxUnicastFrames:         %d\n", t.RxUnicastFrames);
  printf("  RxMulticastFrames:       %d\n", t.RxMulticastFrames);
  printf("  RxFragments:             %d\n", t.RxFragments);
  printf("  RxUnicastOctets:         %d\n", t.RxUnicastOctets);
  printf("  RxMulticastOctets:       %d\n", t.RxMulticastOctets);
  printf("  RxFCSErrors:             %d\n", t.RxFCSErrors);
  printf("  RxDiscards_NoBuffer:     %d\n", t.RxDiscards_NoBuffer);
  printf("  TxDiscardsWrongSA:       %d\n", t.TxDiscardsWrongSA);
  printf("  RxWEPUndecryptable:      %d\n", t.RxWEPUndecryptable);
  printf("  RxMsgInMsgFragments:     %d\n", t.RxMsgInMsgFragments);
  printf("  RxMsgInBadMsgFragments:  %d\n", t.RxMsgInBadMsgFragments);
  printf("  RxDiscards_WEPICVError:  %d\n", t.RxDiscards_WEPICVError);
  printf("  RxDiscards_WEPExcluded:  %d\n", t.RxDiscards_WEPExcluded);
	 
  return 0;
}


WIErr
wlHermesInfo(void)
{
  printf("Hermes chip variables - data may be misleading and/or just plain wrong.\n");
  printf("### Configuration RIDs ###\n");
  dump_rids(cnf_rids);
  printf("\n### Information RIDs ###\n");
  dump_rids(inf_rids);

  return 0;
}
