/* ==========================================================================
 * AirCtl - Airport Control: Control your Airport Base Station
 * --------------------------------------------------------------------------
 * Copyright (C) 2002  William Ahern
 * Copyright (C) 2000  John Sevy (OSU-NMS integer conversion routine)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 * ==========================================================================
 */
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <ctype.h>
#include <string.h>

#include <sys/time.h>	/* select */
#include <sys/types.h>	/* select */
#include <sys/select.h>	/* select */

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include <err.h>

#include "snprintx.h"


#define AIRPORT_PORT	192
#define AIRPORT_ADDR	"10.0.1.1"

#define AIRCTL_STS_SUCCESS	0x00
#define AIRCTL_STS_FAILURE	0x01

#define AIRCTL_STS_CONNECTED	0x00
#define AIRCTL_STS_DISCONNECTED	0x01
#define AIRCTL_STS_CONNECTING	0x02
#define AIRCTL_STS_DISABLED	0x03

#define AIRCTL_VER_SILENT	0x00
#define AIRCTL_VER_SHORT	0x01
#define AIRCTL_VER_LONG		0x02
#define AIRCTL_VER_DEBUG	0x03
#define AIRCTL_VER_DEFAULT	AIRCTL_VER_LONG

#define PACKAGE "airctl"

#define USAGE "\
Usage: airctl [options] [address] connect|disconnect|status\n\
  -0	display nothing (only exit code available)\n\
  -1	display short output\n\
  -2	display extended output (default)\n\
  -3	display debugging output\n\
  -v	display version information and exit\n\
  -h	display this help and exit\n\
\n\
Report bugs to William Ahern <william@25thandClement.com>\n"

#define VERSION "Airport Control v0.7.2\n"

#define COMMANDS {"connect","disconnect","status"}


struct options {
	short int verbosity;
} opts;


struct osunms_element {
	signed char *value;
	int tag;
	int len;
};


void usage(FILE *fp) {
	fprintf(fp,USAGE);
}


void version(FILE *fp) {
	fprintf(fp,VERSION);
}


/*
 * OSU-NMS conversion routine
 *
 * Mostly from AirportBaseStationHangup Utility Copyright (C) Jon Sevy
 * (jsevy@mcs.drexel.edu)
 *
 * NOTE: ntohs() and ntohl() seem to work in lieu of the additive loop
 */
long int osunms2int(void *buf, int len) {
	int i;
	signed char *str	= (signed char *)buf;
	long int n	= 0;

	if (len == 2) {
		n	= ntohs(*(short *)buf);	
	} else if (len == 4) {
		n	= ntohs(*(long *)buf);
	} else {
		for (i = 0; i < len; i++) {
			n =+ (n*256) + ((str[i] < 0)? (str[i]+256) : str[i]);
		}
	}
	
	return n;
}


/* OSU-NMS packet format
 * 1) 2 bytes packet type 
 * 2) 2 bytes element length (exclusive of this; whole element = 2+2+N)
 * 	a) 2 bytes tag name
 * 	b) N bytes value
 * 3) goto #2
 *
 */
struct osunms_element *osunms_element_get(struct osunms_element *element,void *buf, int size, int tag) {
	signed char *packet	= buf;
	int len;

	packet	+= 2; /* skip first two bytes; */

	while ((packet - (signed char *)buf) < (size-4)) {
		len	= ntohs(*(short *)packet);
		packet	+= 2;

		if (ntohs(*(short int *)packet) == tag) {
			element->value	= packet+2;
			element->len	= (len-2);
			element->tag	= tag;

			return element;
		}

		packet	+= len;
	}

	return NULL;
}


void airctl_write(int sd, void *packet, int len) {
        if (write(sd,packet,len) != len)
                        err(AIRCTL_STS_FAILURE,"failed to write airctl packet");
}

void airctl_read(int sd, void *packet, int len) {
	fd_set fds;
	struct timeval tv	= {3,0};
	int r;

	FD_ZERO(&fds);
	FD_SET(sd,&fds);

	switch(select(sd + 1,&fds,NULL,NULL,&tv)) {
		case 1:
		        if ((r = read(sd,packet,len)))
				break;
			goto fail;
		case 0:
			errno	= ETIMEDOUT;
		default:
			goto fail;
	}

	if (r != len && opts.verbosity == AIRCTL_VER_DEBUG)
		warnx("requested %d bytes, received packet of %d bytes",len,r);

	return;
fail:
	err(AIRCTL_STS_FAILURE,"failed to read airctl packet");
}


int airctl_status(int sd) {
	unsigned char packet[128]	= {0};
	struct osunms_element element;
	char baudtag[]	= {0x03,0x15};
	char timetag[]	= {0x03,0x16};
	char rmaintag[]	= {0x03,0x19};
	int total, hrs, min, sec;

	packet[0] = 0x08;
	packet[1] = 0x01;
	packet[2] = 0x03;
	packet[3] = 0x10;
	
	airctl_write(sd,packet,116);
	airctl_read(sd,packet,128);

	if (opts.verbosity == AIRCTL_VER_DEBUG) {
		printf("osu-nms packet:\n");
		PRINTX_STATIC(packet,128);
	}

	switch (packet[6]) {
		case 4: /* connected */
			if (opts.verbosity != AIRCTL_VER_SILENT)
				printf("connected\n");

			if (opts.verbosity >= AIRCTL_VER_LONG) {
				if (osunms_element_get(&element,packet,128,osunms2int(baudtag,2)) != NULL) {
					printf("connection speed: %ld bps\n",osunms2int(element.value,element.len));
				}

				if (osunms_element_get(&element,packet,128,osunms2int(timetag,2)) != NULL) {
					total	= osunms2int(element.value,element.len);
					hrs		= total / 3600;
                    min		= (total / 60) % 60;
                    sec		= total % 60;
					printf("connection time: %.2d:%.2d:%.2d\n",hrs,min,sec);
				}
			}

			break;
		case 5: /* modem temporarily disabled */
			if (opts.verbosity != AIRCTL_VER_SILENT)
				printf("disabled (disconnecting)\n");

			if (osunms_element_get(&element,packet,128,osunms2int(rmaintag,2)) != NULL) {
				total	= osunms2int(element.value,element.len);
				hrs		= total / 3600;
				min		= (total / 60) % 60;
				sec		= total % 60;
				printf("remaining time: %.2d:%.2d:%.2d\n",hrs,min,sec);
			}

			break;
		case 2:
		case 3: /* connecting */
			if (opts.verbosity != AIRCTL_VER_SILENT)
				printf("connecting\n");
			break;
		case 1:
		default: /* disconnected (not disabled) */
			if (opts.verbosity != AIRCTL_VER_SILENT)
				printf("disconnected\n");
	}
	
	return 1;
}


int airctl_connect(int sd) {
	unsigned char packet[116]	= {0};
	
	packet[0] = 0x07;

	airctl_write(sd,packet,116);

	return 1;
}


int airctl_disconnect(int sd) {
	unsigned char packet[116]	= {0};
	
	packet[0] = 0x06;

	airctl_write(sd,packet,116);

	return 1;
}


int main(int argc, char *argv[]) {
	int i;
	char *commands[]	= COMMANDS;
	int commands_num	= (sizeof(commands)/sizeof(char *));

	extern char *optarg;
	extern int optind;
	extern int opterr;
	int ch;

	char *address;
	char *command;
	struct hostent *resolv;
	struct sockaddr_in addr;
	int sd;

	opts.verbosity	= AIRCTL_VER_DEFAULT;

	
	/*
	 * get options
	 */
	opterr = 0;
	while((ch = getopt(argc,argv,"vh0123")) != -1) {
		switch (ch) {
			case 'v':
				version(stdout);
				goto success;
			case 'h':
				usage(stdout);
				goto success;
			case '0':
				opts.verbosity	= AIRCTL_VER_SILENT;
				break;				
			case '1':
				opts.verbosity	= AIRCTL_VER_SHORT;
				break;				
			case '2':
				opts.verbosity	= AIRCTL_VER_LONG;
				break;				
			case '3':
				opts.verbosity	= AIRCTL_VER_DEBUG;
				break;				
			case '?':
			default:
				usage(stdout);
				goto failure;
		}
	} /* end get options */


	/*
	 * get command
	 */
	if (optind >= argc) {
		usage(stderr);
		goto failure;
	}

	if ((argc-optind) == 1) {
		address	= AIRPORT_ADDR;
		command = argv[optind];
	} else {
		address	= argv[optind];
		optind++;
		command	= argv[optind];
	}

	for (i = 0; i < commands_num;i++) {
		if (strcmp(commands[i],command) == 0)
			break;
	}

	if (i == commands_num) {
		warnx("illegal command: %s",command);
		goto failure;
	}

	if (inet_aton(address,&addr.sin_addr) != 1) {
		if ((resolv = gethostbyname(address)) == NULL) {
			warn("could not resolve address[%s]",address);
			usage(stderr);
			goto failure;
		}

		memcpy(&addr.sin_addr,resolv->h_addr,sizeof(addr.sin_addr));
	}

	addr.sin_family	= AF_INET;
	addr.sin_port	= htons(AIRPORT_PORT);

	sd = socket(AF_INET,SOCK_DGRAM,IPPROTO_UDP);
	if (connect(sd,(struct sockaddr *)&addr,sizeof(addr)) != 0) {
		warn("error connecting socket");
		goto failure;
	}
	/* end get command */


	/*
	 * execute command
	 */	
	if (strcmp(command,"connect") == 0) {
		if (airctl_connect(sd) == 0) {
			goto success;
		} else {
			goto failure;
		}
	} else if (strcmp(command,"disconnect") == 0) {
		if (airctl_disconnect(sd) == 0) {
			goto success;
		} else {
			goto failure;
		}
	} else if (strcmp(command,"status") == 0) {
		if (airctl_status(sd) == 0) {
			goto success;
		} else {
			goto failure;
		}
	} else {
		warnx("command not supported: %s",command);
		goto failure;
	}
	/* end execute command */

	success:	
	return EXIT_SUCCESS;

	failure:
	return EXIT_FAILURE;
} /* end main() */
