// ===========================================================================
// File: "aidaProcs.c"
//                        Created: 2010-09-03 07:04:00
//              Last modification: 2012-03-30 17:50:24
// Author: Bernard Desgraupes
// e-mail: <bdesgraupes@users.sourceforge.net>
// (c) Copyright: Bernard Desgraupes 2010-2012
// All rights reserved.
// ===========================================================================

#include "aidaMain.h"
#include "aida.tab.h"


// ------------------------------------------------------------------------
// 
// "aida_anchorProc" --
// 
// Handle the ((a ... a)) tags. 
// 
// ------------------------------------------------------------------------
int 
aida_anchorProc(Tcl_DString * inDst, aida_parse_t * inScanParam)
{
	int			result = TCL_OK;
	Tcl_Obj 	*labelObj;
	
	if (gPrescan) {
		labelObj = aida_externalToTclObj(Tcl_DStringValue(inDst));
		Tcl_SetVar2Ex(gInterp, "aida_anchor", Tcl_GetString(labelObj), Tcl_NewIntObj(gSplitNum), TCL_GLOBAL_ONLY);
		Tcl_DStringFree(inDst);
	} else {
		result = aida_executeTextProc("aida::anchorProc", inScanParam, inDst); 	
	}
	
	return result;	
}


// ------------------------------------------------------------------------
// 
// "aida_commentProc" --
// 
// Invoke Tcl callback for comment lines. 
// 
// ------------------------------------------------------------------------
int 
aida_commentProc(Tcl_DString * inDst, aida_parse_t * inScanParam)
{
	return aida_executeTextProc("aida::commentProc", inScanParam, inDst);
}


// ------------------------------------------------------------------------
// 
// "aida_horizRuleProc" --
// 
// Handle the ((hr)) tag. 
// 
// ------------------------------------------------------------------------
int
aida_horizRuleProc(aida_parse_t * inScanParam)
{
	return aida_executeProc("aida::horizRuleProc", inScanParam, 0);
}


// ------------------------------------------------------------------------
// 
// "aida_imageProc" --
// 
// Handle the ((img  img)) tag. 
// It is possible to specify attributes:
// 		((img height=250 alt='some text'
// 		  foobar.png img))
// or just ((img foobar.png img))
// 
// ------------------------------------------------------------------------
int
aida_imageProc(Tcl_DString * inDst, Tcl_DString * inAttrDst, aida_parse_t * inScanParam)
{
	int				result;
	Tcl_Obj 		*attrObj, *txtObj;
	Tcl_DString 	*txtds;
	
	if (gPrescan) {
		result = TCL_OK;
	} else {
		// Parse the text string and put it in a Tcl_Obj
		txtds = aida_parseDString(inDst, START_TR_FRAG);
		txtObj = aida_externalToTclObj(Tcl_DStringValue(txtds));
		Tcl_DStringFree(txtds);
		
		if (inAttrDst != NULL) {
			attrObj = aida_parseAttrString(inAttrDst);
			Tcl_DStringFree(inAttrDst);
		} else {
			attrObj = Tcl_NewDictObj();
		} 

		Tcl_IncrRefCount(attrObj);
		Tcl_IncrRefCount(txtObj);
		result = aida_executeProc("aida::imageProc", inScanParam, 2, txtObj, attrObj);
		Tcl_DecrRefCount(attrObj);
		Tcl_DecrRefCount(txtObj);
	}
	
	Tcl_DStringFree(inDst);
	
	return result; 
}


// ------------------------------------------------------------------------
// 
// "aida_linkProc" --
// 
// Handle the ((lk [url] text lk)) and ((rf [label] text rf)) tags. 
// 
// ------------------------------------------------------------------------
int
aida_linkProc(Tcl_DString * inUrl, Tcl_DString * inTxt, bool isRef, aida_parse_t * inScanParam)
{
	int				result;
	Tcl_Obj 		*urlObj, *txtObj;
	Tcl_DString 	*urlds, *txtds;
	
	if (gPrescan) {
		result = TCL_OK;
	} else {
		// Parse the url string and put it in a Tcl_Obj
		urlds = aida_parseDString(inUrl, START_TR_FRAG);
		urlObj = aida_externalToTclObj(Tcl_DStringValue(urlds));
		
		// Parse the text string and put it in a Tcl_Obj
		txtds = aida_parseDString(inTxt, START_TR_FRAG);
		txtObj = aida_externalToTclObj(Tcl_DStringValue(txtds));

		Tcl_IncrRefCount(urlObj);
		Tcl_IncrRefCount(txtObj);
		if (isRef) {
			if (gSplitting) {
				Tcl_Obj *	numObj;
				int			splitnum;
				
				numObj = Tcl_GetVar2Ex(gInterp, "aida_anchor", Tcl_GetString(urlObj), TCL_GLOBAL_ONLY);
				if (numObj != NULL) {
					result = Tcl_GetIntFromObj(gInterp, numObj, &splitnum);
					if (result == TCL_OK) {
						result = aida_executeProc("aida::refProc", inScanParam, 3, txtObj, urlObj, aida_splitFileName(splitnum));
					}
				} else {
					aida_verbose(3, "can't get file num for anchor");
					result = aida_executeProc("aida::refProc", inScanParam, 2, txtObj, urlObj);
					//result = TCL_ERROR;
				}
			} else {
				result = aida_executeProc("aida::refProc", inScanParam, 2, txtObj, urlObj);
			} 
		} else {
			result = aida_executeProc("aida::linkProc", inScanParam, 2, txtObj, urlObj);
		} 
		Tcl_DecrRefCount(urlObj);
		Tcl_DecrRefCount(txtObj);
		
		// Free the DStrings
		Tcl_DStringFree(urlds);
		Tcl_DStringFree(txtds);
	}
		
	Tcl_DStringFree(inUrl);
	Tcl_DStringFree(inTxt);

	return result; 
}


// ------------------------------------------------------------------------
// 
// "aida_navBarProc" --
// 
// Handle the navigation bars in split files. 
// 
// ------------------------------------------------------------------------
int
aida_navBarProc(int inNum, int inLastNum, FILE * inDest)
{
	int				result;
	Tcl_Obj 		*currObj, *prevObj, *nextObj, *topObj;
	
	currObj = aida_splitFileName(inNum);
	if (inNum > 1) {
		prevObj = aida_splitFileName(inNum-1);
	} else {
		prevObj = Tcl_NewStringObj("", 0);
	} 
	if (inNum < inLastNum) {
		nextObj = aida_splitFileName(inNum+1);
	} else {
		nextObj = Tcl_NewStringObj("", 0);
	} 
	topObj = aida_splitFileName(1);
	
	Tcl_IncrRefCount(currObj);
	Tcl_IncrRefCount(prevObj);
	Tcl_IncrRefCount(nextObj);		
	Tcl_IncrRefCount(topObj);		
	result = aida_procToFile(Tcl_NewStringObj("aida::navBarProc", -1), inDest, 4, currObj, prevObj, nextObj, topObj);
	Tcl_DecrRefCount(currObj);
	Tcl_DecrRefCount(prevObj);
	Tcl_DecrRefCount(nextObj);
	Tcl_DecrRefCount(topObj);
	
	return result; 
}


// ------------------------------------------------------------------------
// 
// "aida_newLineProc" --
// 
// Handle the ((nl)) tag. 
// 
// ------------------------------------------------------------------------
int
aida_newLineProc(aida_parse_t * inScanParam)
{
	return aida_executeProc("aida::newLineProc", inScanParam, 0);
}


// ------------------------------------------------------------------------
// 
// "aida_printIndexProc" --
// 
// Handle the ((index)) tag. Open a fragment of type kind_proc to declare a
// proc which will be invoked when the output is reconstructed.
// 
// ------------------------------------------------------------------------
int
aida_printIndexProc(aida_parse_t * inScanParam)
{
	if (!gPrescan && inScanParam->to == to_fptr) {
		aida_newFragment(kind_proc, "aida::printIndexProc");
		inScanParam->out.fptr = aida_switchOutput();
	} 
	
	return TCL_OK;
}


// ------------------------------------------------------------------------
// 
// "aida_sectionProc" --
// 
// Handle ((s1 ... s1)), ..., ((s6 ... s6)) tags.  
// 
// ------------------------------------------------------------------------
int
aida_sectionProc(Tcl_DString * inDst, int inLevel, aida_parse_t * inScanParam)
{
	int				result;
	Tcl_Obj *		secObj;
	Tcl_DString *	dsptr;
	
	if (gPrescan) {
		if (inLevel <= gSplitLevel) {
			gSplitNum++;
		} 
		result = TCL_OK;
	} else {
		// Parse the title string
		dsptr = aida_parseDString(inDst, START_TR_FRAG);
		
		// Put it in a Tcl_Obj
		secObj = aida_externalToTclObj(Tcl_DStringValue(dsptr));
		
		// Free the returned string
		Tcl_DStringFree(dsptr);
						 
		// Execute the callback
		if (gSplitting) {		
			if (inLevel <= gSplitLevel && inScanParam->to == to_fptr) {
				aida_forceSplit(inScanParam);
			} 
			// Pass the split file name as additional argument
			result = aida_executeProc("aida::sectionProc", inScanParam, 3, secObj, Tcl_NewIntObj(inLevel), aida_splitFileName(gSplitNum));
		} else {
			result = aida_executeProc("aida::sectionProc", inScanParam, 2, secObj, Tcl_NewIntObj(inLevel));
		}
	}

	return result;
}


// ------------------------------------------------------------------------
// 
// "aida_setIndexProc" --
// 
// Handle the ((x ... x)) tags. 
// 
// ------------------------------------------------------------------------
int 
aida_setIndexProc(Tcl_DString * inDst, aida_parse_t * inScanParam)
{
	int				result;
	Tcl_Obj *		idxObj;
	
	if (gPrescan) {
		return TCL_OK;
	} else {
		idxObj = aida_externalToTclObj(Tcl_DStringValue(inDst));
		Tcl_IncrRefCount(idxObj);
		
		// Execute the callback
		if (gSplitting) {		
			// Pass the split file name as additional argument
			result = aida_executeProc("aida::setIndexProc", inScanParam, 2, idxObj, aida_splitFileName(gSplitNum));
		} else {
			result = aida_executeProc("aida::setIndexProc", inScanParam, 1, idxObj);
		}
		Tcl_DecrRefCount(idxObj);
	}
		
	// Release the Tcl_DString
	Tcl_DStringFree(inDst);

	return result;
}


// ------------------------------------------------------------------------
// 
// "aida_styleTagProc" --
// 
// Handle style tags. The 'inType' argument can be: "i", "b", "u". The
// inOpen argument tells if it is an opening or a closing tag.
// 
// ------------------------------------------------------------------------
int
aida_styleTagProc(char inType, bool inOpen, aida_parse_t * inScanParam)
{
	return aida_executeProc("aida::styleProc", inScanParam, 2, Tcl_NewStringObj(&inType, 1), Tcl_NewBooleanObj(inOpen));
}


// ------------------------------------------------------------------------
// 
// "aida_tocProc" --
// 
// Handle the ((toc)) tag. Open a fragment of type kind_proc to declare a proc
// which will be invoked when the output is reconstructed.
// 
// ------------------------------------------------------------------------
int
aida_tocProc(aida_parse_t * inScanParam)
{
	if (!gPrescan && inScanParam->to == to_fptr) {
		aida_newFragment(kind_proc, "aida::tocProc");
		inScanParam->out.fptr = aida_switchOutput();
	} 
	
	return TCL_OK;
}


// ------------------------------------------------------------------------
// 
// "aida_verbatimBlockProc" --
// 
// Handle ((| ... |)) tags.
// 
// ------------------------------------------------------------------------
int
aida_verbatimBlockProc(Tcl_DString * inDst, aida_parse_t * inScanParam)
{
	return aida_executeTextProc("aida::verbatimProc", inScanParam, inDst);
}


// ------------------------------------------------------------------------
// 
// "aida_verbProc" --
// 
// Invoke Tcl callback for ((v ... v)) tags. 
// 
// ------------------------------------------------------------------------
int 
aida_verbProc(Tcl_DString * inDst, aida_parse_t * inScanParam)
{
	return aida_executeTextProc("aida::verbProc", inScanParam, inDst);
}


// ------------------------------------------------------------------------
// 
// "aida_quoteProc" --
// 
// Handle the ((q ... q)) tags. 
// 
// ------------------------------------------------------------------------
int 
aida_quoteProc(Tcl_DString * inDst, aida_parse_t * inScanParam)
{
	int			result = TCL_OK;
	Tcl_Obj 	*txtObj;

	if (!gPrescan) {
		txtObj = aida_externalToTclObj(Tcl_DStringValue(inDst));
		Tcl_IncrRefCount(txtObj);
		aida_writeTclObj(txtObj, inScanParam);
		Tcl_DecrRefCount(txtObj);
	}

	// Release the Tcl_DString
	Tcl_DStringFree(inDst);
	
	return result;
}


