% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/time_seq_id.R
\name{time_seq_id}
\alias{time_seq_id}
\title{Generate a unique identifier for a regular time sequence with gaps}
\usage{
time_seq_id(
  x,
  time_by = NULL,
  threshold = 1,
  g = NULL,
  na_skip = TRUE,
  rolling = TRUE,
  switch_on_boundary = FALSE,
  time_type = getOption("timeplyr.time_type", "auto")
)
}
\arguments{
\item{x}{Date, datetime or numeric vector.}

\item{time_by}{Time unit. \cr
This signifies the granularity of the time data with which to measure gaps
in the sequence.
If your data is daily for example, supply \code{time_by = "days"}.
If weekly, supply \code{time_by = "week"}.
Must be one of the three:
\itemize{
\item string, specifying either the unit or the number and unit, e.g
\code{time_by = "days"} or \code{time_by = "2 weeks"}
\item named list of length one, the unit being the name, and
the number the value of the list, e.g. \code{list("days" = 7)}.
For the vectorized time functions, you can supply multiple values,
e.g. \code{list("days" = 1:10)}.
\item Numeric vector. If time_by is a numeric vector and x is not a date/datetime,
then arithmetic is used, e.g \code{time_by = 1}.
}}

\item{threshold}{Threshold such that when the time elapsed
exceeds this, the sequence ID is incremented by 1.
For example, if \code{time_by = "days"} and \code{threshold = 2},
then when 2 days have passed, a new ID is created.
Furthermore, \code{threshold} generally need not be supplied as \cr
\code{time_by = "3 days"} & \code{threshold = 1} \cr
is identical to \cr
\code{time_by = "days"} & \code{threshold = 3}. \cr}

\item{g}{Object used for grouping x.
This can for example be a vector or data frame.
\code{g} is passed directly to \code{collapse::GRP()}.}

\item{na_skip}{Should \code{NA} values be skipped? Default is \code{TRUE}.}

\item{rolling}{When this is \code{FALSE}, a new ID is created every time
a cumulative amount of time has passed. Once that amount of time has passed,
a new ID is created, the clock "resets" and we start counting from that point.}

\item{switch_on_boundary}{When an exact amount of time
(specified in \code{time_by}) has passed, should there an increment in ID?
The default is \code{FALSE}. For example, if \code{time_by = "days"} and
\code{switch_on_boundary = FALSE}, \code{>} 1 day must have passed, otherwise
\code{>=} 1 day must have passed.}

\item{time_type}{If "auto", \code{periods} are used for
the time expansion when days, weeks, months or years are specified,
and \code{durations} are used otherwise.}
}
\value{
An integer vector of \code{length(x)}.
}
\description{
A unique identifier is created every time a specified amount of
time has passed, or in the case of regular sequences, when there is a gap
in time.
}
\details{
\code{time_seq_id()} Assumes \code{x} is regular and in
ascending or descending order.
To check this condition formally, use \code{time_is_regular()}.
}
\examples{
library(dplyr)
library(timeplyr)
library(lubridate)
\dontshow{
.n_dt_threads <- data.table::getDTthreads()
.n_collapse_threads <- collapse::get_collapse()$nthreads
data.table::setDTthreads(threads = 2L)
collapse::set_collapse(nthreads = 1L)
}
# Weekly sequence, with 2 gaps in between
x <- time_seq(today(), length.out = 10, time_by = "week")
x <- x[-c(3, 7)]
# A new ID when more than a week has passed since the last time point
time_seq_id(x, time_by = "week")
# A new ID when >= 2 weeks has passed since the last time point
time_seq_id(x, time_by = "weeks", threshold = 2, switch_on_boundary = TRUE)
# A new ID when at least 4 cumulative weeks have passed
time_seq_id(x, time_by = "4 weeks",
            switch_on_boundary = TRUE, rolling = FALSE)
# A new ID when more than 4 cumulative weeks have passed
time_seq_id(x, time_by = "4 weeks",
            switch_on_boundary = FALSE, rolling = FALSE)
\dontshow{
data.table::setDTthreads(threads = .n_dt_threads)
collapse::set_collapse(nthreads = .n_collapse_threads)
}
}
