% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/add_sig.R
\name{add_sig}
\alias{add_sig}
\title{Add Significant Test Results}
\usage{
add_sig(object, ..., standardized = FALSE, na_str = "", use = "pvalue")
}
\arguments{
\item{object}{A \linkS4class{lavaan} object
or the output of
\code{\link[lavaan:parameterEstimates]{lavaan::parameterEstimates()}} or
\code{\link[lavaan:standardizedSolution]{lavaan::standardizedSolution()}}.
May also work on an \code{est_table}-class
object returned by functions
like \code{\link[=group_by_dvs]{group_by_dvs()}} but there
is no guarantee.}

\item{...}{Optional arguments to be
passed to
\code{\link[lavaan:parameterEstimates]{lavaan::parameterEstimates()}} or
\code{\link[lavaan:standardizedSolution]{lavaan::standardizedSolution()}}.}

\item{standardized}{Whether
standardized solution is needed. If
\code{TRUE},
\code{\link[lavaan:standardizedSolution]{lavaan::standardizedSolution()}} will
be called. If \code{FALSE}, the default,
\code{\link[lavaan:parameterEstimates]{lavaan::parameterEstimates()}} will
be called. Ignored if a table if
estimates is supplied.}

\item{na_str}{The string to be used
for parameters with no significant
tests. For example, fixed parameters.
Default is \code{""}.}

\item{use}{A character vector of one
or more strings. If \code{"pvalue"} is in
the vector, \emph{p}-values will be used.
If \code{"ci"} is in the vector,
confidence intervals appeared in
\code{ci.lower} and \code{ci.upper} will be
used. If \code{"boot.ci"} is in the vector
and the columns \code{boot.ci.lower} and
\code{boot.ci.upper} are available, these
columns will be used. Note that
\code{ci.lower} and \code{ci.upper} can also be
bootstrap confidence intervals in
some tables if \code{se = "boot"} is used.}
}
\value{
The output of
\code{\link[lavaan:parameterEstimates]{lavaan::parameterEstimates()}} or
\code{\link[lavaan:standardizedSolution]{lavaan::standardizedSolution()}},
with one or two columns inserted
after the parameter estimates to
denote the significant test results.
}
\description{
It inserts columns to
denote whether a parameter is
significant.
}
\details{
The function calls
\code{\link[lavaan:parameterEstimates]{lavaan::parameterEstimates()}} or
\code{\link[lavaan:standardizedSolution]{lavaan::standardizedSolution()}} and
checks the columns \code{pvalue},
\code{ci.lower} and \code{ci.upper}, and/or
\code{boot.ci.lower} and \code{boot.ci.upper}
and then inserts columns to denote
for each parameter estimate whether
it is significant based on the
requested criteria.
}
\examples{

library(lavaan)
set.seed(5478374)
n <- 50
x <- runif(n) - .5
m <- .40 * x + rnorm(n, 0, sqrt(1 - .40))
y <- .30 * m + rnorm(n, 0, sqrt(1 - .30))
dat <- data.frame(x = x, y = y, m = m)
model <-
'
m ~ a*x
y ~ b*m
ab := a*b
'
fit <- sem(model, data = dat, fixed.x = FALSE)

# Add "*" based on 'pvalue'
add_sig(fit)

# Add "*" for standardized solution
add_sig(fit, standardized = TRUE)

# Add "*" based on confidence interval
add_sig(fit, use = "ci")

# Add "*" for standardized solution based on confidence interval
add_sig(fit, standardized = TRUE, use = "ci")

# Add "*" for standardized solution based on confidence interval
# and 'pvalue'.
add_sig(fit, standardized = TRUE, use = c("ci", "pvalue"))

# Can also accept a parameter estimates table
est <- parameterEstimates(fit)
add_sig(est)

# So it can be used with some other functions in semhelpinghands
add_sig(filter_by(est, op = "~"))

# Piping can also be used
est |> filter_by(op = "~") |>
       add_sig()

}
\seealso{
\code{\link[lavaan:parameterEstimates]{lavaan::parameterEstimates()}} and
\code{\link[lavaan:standardizedSolution]{lavaan::standardizedSolution()}}
}
\author{
Shu Fai Cheung
\url{https://orcid.org/0000-0002-9871-9448}
}
