% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tt_dotabulation.R
\name{build_table}
\alias{build_table}
\title{Create a table from a layout and data}
\usage{
build_table(
  lyt,
  df,
  alt_counts_df = NULL,
  col_counts = NULL,
  col_total = if (is.null(alt_counts_df)) nrow(df) else nrow(alt_counts_df),
  topleft = NULL,
  hsep = default_hsep(),
  ...
)
}
\arguments{
\item{lyt}{layout object pre-data used for tabulation}

\item{df}{dataset (data.frame or tibble)}

\item{alt_counts_df}{dataset (data.frame or tibble). Alternative full data
the rtables framework will use (\emph{only}) when calculating column
counts.}

\item{col_counts}{numeric (or \code{NULL}). Deprecated. If non-null, column counts
which override those calculated automatically during tabulation. Must
specify "counts" for \emph{all} resulting columns if non-NULL. \code{NA}
elements will be replaced with the automatically calculated counts.}

\item{col_total}{integer(1). The total observations across all columns.
Defaults to \code{nrow(df)}.}

\item{topleft}{character. Override values for the "top left" material to be
displayed during printing.}

\item{hsep}{character(1). Set of character(s) to be repeated as the separator
between the header and body of the table when rendered as text. Defaults to
a connected horrizontal line (unicode 2014) in locals that use a UTF
charset, and to \code{-} elsewhere (with a once per session warning).}

\item{\dots}{currently ignored.}
}
\value{
A \code{TableTree} or \code{ElementaryTable} object representing the
table created by performing the tabulations declared in \code{lyt} to the
data \code{df}.
}
\description{
Layouts are used to describe a table pre-data. \code{build_table} is used to
create a table using a layout and a dataset.
}
\details{
When \code{alt_counts_df} is specified, column counts are calculated by
applying the exact column subsetting expressions determined when applying
column splitting to the main data (\code{df}) to \code{alt_counts_df} and
counting the observations in each resulting subset.

In particular, this means that in the case of splitting based on cuts of the
data, any dynamic cuts will have been calculated based on \code{df} and
simply re-used for the count calculation.
}
\note{
When overriding the column counts or totals care must be taken that,
e.g., \code{length()} or \code{nrow()} are not called within tabulation functions,
because those will NOT give the overridden counts. Writing/using tabulation
functions which accept \code{.N_col} and \code{.N_total} or do not rely on
column counts at all (even implicitly) is the only way to ensure overriden
counts are fully respected.
}
\examples{

l <- basic_table() \%>\%
  split_cols_by("Species") \%>\%
  analyze("Sepal.Length", afun = function(x) {
  list(
    "mean (sd)" = rcell(c(mean(x), sd(x)), format = "xx.xx (xx.xx)"),
    "range" = diff(range(x))
  )
})

l

build_table(l, iris)

# analyze multiple variables
l <- basic_table() \%>\%
  split_cols_by("Species") \%>\%
  analyze(c("Sepal.Length", "Petal.Width"), afun = function(x) {
  list(
    "mean (sd)" = rcell(c(mean(x), sd(x)), format = "xx.xx (xx.xx)"),
    "range" = diff(range(x))
  )
})

build_table(l, iris)

# an example more relevant for clinical trials
l <- basic_table() \%>\%
    split_cols_by("ARM") \%>\%
    analyze("AGE", afun = function(x) {
      setNames(as.list(fivenum(x)), c("minimum", "lower-hinge", "median",
                                      "upper-hinge", "maximum"))
    })

build_table(l, DM)

build_table(l, subset(DM, AGE > 40))

# with column counts
l2 <- l \%>\%
  add_colcounts()
build_table(l2, DM)


# with column counts calculated based on different data
miniDM <- DM[sample(1:NROW(DM), 100),]
build_table(l2, DM, alt_counts_df = miniDM)

build_table(l, DM, col_counts = 1:3)
}
\author{
Gabriel Becker
}
