% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tune.block.splsda.R
\name{tune.block.splsda}
\alias{tune.block.splsda}
\title{Tuning function for block.splsda method (N-integration with sparse
Discriminant Analysis)}
\usage{
tune.block.splsda(
  X,
  Y,
  indY,
  ncomp = 2,
  test.keepX,
  already.tested.X,
  validation = "Mfold",
  folds = 10,
  dist = "max.dist",
  measure = "BER",
  weighted = TRUE,
  progressBar = FALSE,
  tol = 1e-06,
  max.iter = 100,
  near.zero.var = FALSE,
  nrepeat = 1,
  design,
  scheme = "horst",
  scale = TRUE,
  init = "svd",
  light.output = TRUE,
  signif.threshold = 0.01,
  BPPARAM = SerialParam(),
  ...
)
}
\arguments{
\item{X}{A named list of data sets (called 'blocks') measured on the same 
samples. Data in the list should be arranged in matrices, samples x variables, 
with samples order matching in all data sets.}

\item{Y}{a factor or a class vector for the discrete outcome.}

\item{indY}{To supply if \code{Y} is missing, indicates the position of 
the matrix response in the list \code{X}.}

\item{ncomp}{the number of components to include in the model. Default to 2.
Applies to all blocks.}

\item{test.keepX}{A named list with the same length and names as X 
(without the outcome Y, if it is provided in X and designated using 
\code{indY}).  Each entry of this list is a numeric vector for the different 
keepX values to test for that specific block.}

\item{already.tested.X}{Optional, if \code{ncomp > 1} A named list of 
numeric vectors each of length \code{n_tested} indicating the number of 
variables to select from the \eqn{X} data set on the first \code{n_tested} 
components.}

\item{validation}{character.  What kind of (internal) validation to use,
matching one of \code{"Mfold"} or \code{"loo"} (see below). Default is
\code{"Mfold"}.}

\item{folds}{the folds in the Mfold cross-validation. See Details.}

\item{dist}{distance metric to estimate the
classification error rate, should be a subset of \code{"centroids.dist"},
\code{"mahalanobis.dist"} or \code{"max.dist"} (see Details).}

\item{measure}{The tuning measure used for different methods. See details.}

\item{weighted}{tune using either the performance of the Majority vote or
the Weighted vote.}

\item{progressBar}{by default set to \code{TRUE} to output the progress bar
of the computation.}

\item{tol}{Positive numeric used as convergence criteria/tolerance during the
iterative process. Default to \code{1e-06}.}

\item{max.iter}{Integer, the maximum number of iterations. Default to  100.}

\item{near.zero.var}{Logical, see the internal \code{\link{nearZeroVar}}
function (should be set to TRUE in particular for data with many zero
values). Setting this argument to FALSE (when appropriate) will speed up the
computations. Default value is FALSE.}

\item{nrepeat}{Number of times the Cross-Validation process is repeated.}

\item{design}{numeric matrix of size (number of blocks in X) x (number of
blocks in X) with values between 0 and 1. Each value indicates the strenght
of the relationship to be modelled between two blocks; a value of 0
indicates no relationship, 1 is the maximum value. Alternatively, one of
c('null', 'full') indicating a disconnected or fully connected design,
respecively, or a numeric between 0 and 1 which will designate all
off-diagonal elements of a fully connected design (see examples in
\code{block.splsda}). If \code{Y} is provided instead of \code{indY}, the
\code{design} matrix is changed to include relationships to \code{Y}.}

\item{scheme}{Either "horst", "factorial" or "centroid". Default =
\code{centroid}, see reference.}

\item{scale}{Logical. If scale = TRUE, each block is standardized to zero
means and unit variances (default: TRUE)}

\item{init}{Mode of initialization use in the algorithm, either by Singular
Value Decomposition of the product of each block of X with Y ('svd') or each
block independently ('svd.single'). Default = \code{svd.single}}

\item{light.output}{if set to FALSE, the prediction/classification of each
sample for each of \code{test.keepX} and each comp is returned.}

\item{signif.threshold}{numeric between 0 and 1 indicating the significance
threshold required for improvement in error rate of the components. Default
to 0.01.}

\item{BPPARAM}{A \linkS4class{BiocParallelParam} object indicating the type
of parallelisation. See examples.}

\item{...}{Optional arguments:
\itemize{
 \item \bold{seed} Integer. Seed number for reproducible parallel code.
 Default is \code{NULL}.
}
run in parallel when repeating the cross-validation, which is usually the
most computationally intensive process. If there is excess CPU, the
cross-vaidation is also parallelised on *nix-based OS which support
\code{mclapply}.}
}
\value{
A list that contains: \item{error.rate}{returns the prediction error
for each \code{test.keepX} on each component, averaged across all repeats
and subsampling folds. Standard deviation is also output. All error rates
are also available as a list.} \item{choice.keepX}{returns the number of
variables selected (optimal keepX) on each component, for each block.}
\item{choice.ncomp}{returns the optimal number of components for the model
fitted with \code{$choice.keepX}. } \item{error.rate.class}{returns the
error rate for each level of \code{Y} and for each component computed with
the optimal keepX}

\item{predict}{Prediction values for each sample, each \code{test.keepX},
each comp and each repeat. Only if light.output=FALSE}
\item{class}{Predicted class for each sample, each \code{test.keepX}, each
comp and each repeat. Only if light.output=FALSE}

\item{cor.value}{compute the correlation between latent variables for
two-factor sPLS-DA analysis.}
}
\description{
Computes M-fold or Leave-One-Out Cross-Validation scores based on a
user-input grid to determine the optimal parsity parameters values for
method \code{block.splsda}.
}
\details{
This tuning function should be used to tune the keepX parameters in the
\code{block.splsda} function (N-integration with sparse Discriminant
Analysis).

M-fold or LOO cross-validation is performed with stratified subsampling
where all classes are represented in each fold.

If \code{validation = "Mfold"}, M-fold cross-validation is performed. The
number of folds to generate is to be specified in the argument \code{folds}.

If \code{validation = "loo"}, leave-one-out cross-validation is performed.
By default \code{folds} is set to the number of unique individuals.

All combination of test.keepX values are tested. A message informs how many
will be fitted on each component for a given test.keepX.

More details about the prediction distances in \code{?predict} and the
supplemental material of the mixOmics article (Rohart et al. 2017). Details
about the PLS modes are in \code{?pls}.

BER is appropriate in case of an unbalanced number of samples per class as
it calculates the average proportion of wrongly classified samples in each
class, weighted by the number of samples in each class. BER is less biased
towards majority classes during the performance assessment.
}
\examples{
\dontrun{
data("breast.TCGA")
# this is the X data as a list of mRNA and miRNA; the Y data set is a single data set of proteins
data = list(mrna = breast.TCGA$data.train$mrna, mirna = breast.TCGA$data.train$mirna,
            protein = breast.TCGA$data.train$protein)
# set up a full design where every block is connected
# could also consider other weights, see our mixOmics manuscript
design = matrix(1, ncol = length(data), nrow = length(data),
                dimnames = list(names(data), names(data)))
diag(design) =  0
design
# set number of component per data set
ncomp = 3

# Tuning the first two components
# -------------
## Not run: 
# definition of the keepX value to be tested for each block mRNA miRNA and protein
# names of test.keepX must match the names of 'data'
test.keepX = list(mrna = c(10, 30), mirna = c(15, 25), protein = c(4, 8))

# the following may take some time to run, so we subset the data first.
# Note that for thorough tuning, nrepeat should be >= 3 so that significance of 
# the model improvement can be measured
## ---- subset by 3rd of samples
set.seed(100)
subset <- mixOmics:::stratified.subsampling(breast.TCGA$data.train$subtype, folds = 3)[[1]][[1]]
data <- lapply(data, function(omic) omic[subset,])
Y <- breast.TCGA$data.train$subtype[subset]
## ---- run
## setup cluster - use SnowParam() on Widnows
BPPARAM <- BiocParallel::MulticoreParam(workers = parallel::detectCores()-1)
tune <- tune.block.splsda(
    X = data,
    Y = Y,
    ncomp = ncomp,
    test.keepX = test.keepX,
    design = design,
    nrepeat = 2, 
    BPPARAM = BPPARAM
)

plot(tune)
tune$choice.ncomp
tune$choice.keepX

# Now tuning a new component given previous tuned keepX
already.tested.X = tune$choice.keepX
tune = tune.block.splsda(X = data, Y = Y,
                         ncomp = 4, test.keepX = test.keepX, design = design,
                         already.tested.X = already.tested.X,
                         BPPARAM = BPPARAM
                         )
tune$choice.keepX
}
}
\references{
Method:

Singh A., Gautier B., Shannon C., Vacher M., Rohart F., Tebbutt S. and Lê
Cao K.A. (2016). DIABLO: multi omics integration for biomarker discovery.

mixOmics article:

Rohart F, Gautier B, Singh A, Lê Cao K-A. mixOmics: an R package for 'omics
feature selection and multiple data integration. PLoS Comput Biol 13(11):
e1005752
}
\seealso{
\code{\link{block.splsda}} and http://www.mixOmics.org for more
details.
}
\author{
Florian Rohart, Amrit Singh, Kim-Anh Lê Cao, AL J Abadi
}
\keyword{multivariate}
\keyword{regression}
