% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mint.block.pls.R
\name{mint.block.pls}
\alias{mint.block.pls}
\title{NP-integration}
\usage{
mint.block.pls(
  X,
  Y,
  indY,
  study,
  ncomp = 2,
  design,
  scheme,
  mode,
  scale = TRUE,
  init,
  tol = 1e-06,
  max.iter = 100,
  near.zero.var = FALSE,
  all.outputs = TRUE
)
}
\arguments{
\item{X}{A named list of data sets (called 'blocks') measured on the same samples.
Data in the list should be arranged in samples x variables, with samples
order matching in all data sets.}

\item{Y}{Matrix or vector response for a multivariate regression framework.
Data should be continuous variables (see \code{?mint.block.splsda} for
supervised classification and factor response).}

\item{indY}{To be supplied if Y is missing, indicates the position of the
matrix / vector response in the list \code{X}}

\item{study}{Factor, indicating the membership of each sample to each of the
studies being combined}

\item{ncomp}{the number of components to include in the model. Default to 2.
Applies to all blocks.}

\item{design}{numeric matrix of size (number of blocks in X) x (number of
blocks in X) with values between 0 and 1. Each value indicates the strenght
of the relationship to be modelled between two blocks; a value of 0
indicates no relationship, 1 is the maximum value. Alternatively, one of
c('null', 'full') indicating a disconnected or fully connected design,
respecively, or a numeric between 0 and 1 which will designate all
off-diagonal elements of a fully connected design (see examples in
\code{block.splsda}). If \code{Y} is provided instead of \code{indY}, the
\code{design} matrix is changed to include relationships to \code{Y}.}

\item{scheme}{Character, one of 'horst', 'factorial' or 'centroid'. Default =
\code{'horst'}, see reference.}

\item{mode}{Character string indicating the type of PLS algorithm to use. One
of \code{"regression"}, \code{"canonical"}, \code{"invariant"} or \code{"classic"}. See Details.}

\item{scale}{Logical. If scale = TRUE, each block is standardized to zero
means and unit variances (default: TRUE)}

\item{init}{Mode of initialization use in the algorithm, either by Singular
Value Decomposition of the product of each block of X with Y ('svd') or each
block independently ('svd.single'). Default = \code{svd.single}}

\item{tol}{Positive numeric used as convergence criteria/tolerance during the
iterative process. Default to \code{1e-06}.}

\item{max.iter}{Integer, the maximum number of iterations. Default to  100.}

\item{near.zero.var}{Logical, see the internal \code{\link{nearZeroVar}}
function (should be set to TRUE in particular for data with many zero
values). Setting this argument to FALSE (when appropriate) will speed up the
computations. Default value is FALSE.}

\item{all.outputs}{Logical. Computation can be faster when some specific
(and non-essential) outputs are not calculated. Default = \code{TRUE}.}
}
\value{
\code{mint.block.pls} returns an object of class \code{"mint.pls",
"block.pls"}, a list that contains the following components:

\item{X}{the centered and standardized original predictor matrix.}
\item{Y}{the centered and standardized original response vector or matrix.}
\item{ncomp}{the number of components included in the model for each block.}
\item{mode}{the algorithm used to fit the model.} \item{mat.c}{matrix of
coefficients from the regression of X / residual matrices X on the
X-variates, to be used internally by \code{predict}.} \item{variates}{list
containing the \eqn{X} and \eqn{Y} variates.} \item{loadings}{list
containing the estimated loadings for the variates.} \item{names}{list
containing the names to be used for individuals and variables.}
\item{nzv}{list containing the zero- or near-zero predictors information.}
\item{tol}{the tolerance used in the iterative algorithm, used for
subsequent S3 methods} \item{max.iter}{the maximum number of iterations,
used for subsequent S3 methods} \item{iter}{Number of iterations of the
algorithm for each component}
}
\description{
Function to integrate data sets measured on the same samples (N-integration)
and to combine multiple independent studies measured on the same variables
or predictors (P-integration) using variants of multi-group and generalised
PLS (unsupervised analysis).
}
\details{
The function fits multi-group generalised PLS models with a specified number
of \code{ncomp} components. An outcome needs to be provided, either by
\code{Y} or by its position \code{indY} in the list of blocks \code{X}.

Multi (continuous)response are supported. \code{X} and \code{Y} can contain
missing values. Missing values are handled by being disregarded during the
cross product computations in the algorithm \code{block.pls} without having
to delete rows with missing data. Alternatively, missing data can be imputed
prior using the \code{nipals} function.

The type of algorithm to use is specified with the \code{mode} argument.
Four PLS algorithms are available: PLS regression \code{("regression")}, PLS
canonical analysis \code{("canonical")}, redundancy analysis
\code{("invariant")} and the classical PLS algorithm \code{("classic")} (see
References and more details in \code{?pls}).
}
\examples{

data(breast.TCGA)

# for the purpose of this example, we create data that fit in the context of
# this function.
# We consider the training set as study1 and the test set as another
# independent study2.

study = c(rep("study1",150), rep("study2",70))

# to put the data in the MINT format, we rbind the two studies
mrna = rbind(breast.TCGA$data.train$mrna, breast.TCGA$data.test$mrna)
mirna = rbind(breast.TCGA$data.train$mirna, breast.TCGA$data.test$mirna)

# For the purpose of this example, we create a continuous response by
# taking the first mrna variable, and removing it from the data
Y = mrna[,1]
mrna = mrna[,-1]

data = list(mrna = mrna, mirna = mirna)

# we can now apply the function
res = mint.block.plsda(data, Y, study=study, ncomp=2)

res
}
\references{
Rohart F, Eslami A, Matigian, N, Bougeard S, Lê Cao K-A (2017).
MINT: A multivariate integrative approach to identify a reproducible
biomarker signature across multiple experiments and platforms. BMC
Bioinformatics 18:128.

Eslami, A., Qannari, E. M., Kohler, A., and Bougeard, S. (2014). Algorithms
for multi-group PLS. J. Chemometrics, 28(3), 192-201.
}
\seealso{
\code{\link{spls}}, \code{\link{summary}}, \code{\link{plotIndiv}},
\code{\link{plotVar}}, \code{\link{predict}}, \code{\link{perf}},
\code{\link{mint.block.spls}}, \code{\link{mint.block.plsda}},
\code{\link{mint.block.splsda}} and http://www.mixOmics.org/mixMINT for more
details.
}
\author{
Florian Rohart, Benoit Gautier, Kim-Anh Lê Cao, Al J Abadi
}
\keyword{multivariate}
\keyword{regression}
